"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.isGeneratedWhenNeededMarker = exports.GeneratedWhenNeededMarker = exports.generatePhysicalName = void 0;
const crypto = require("crypto");
const stack_1 = require("../stack");
const token_1 = require("../token");
const token_map_1 = require("./token-map");
function generatePhysicalName(resource) {
    const stack = stack_1.Stack.of(resource);
    const stackPart = new PrefixNamePart(stack.stackName, 25);
    const idPart = new SuffixNamePart(resource.node.uniqueId, 24);
    const region = stack.region;
    if (token_1.Token.isUnresolved(region) || !region) {
        throw new Error(`Cannot generate a physical name for ${resource.node.path}, because the region is un-resolved or missing`);
    }
    const account = stack.account;
    if (token_1.Token.isUnresolved(account) || !account) {
        throw new Error(`Cannot generate a physical name for ${resource.node.path}, because the account is un-resolved or missing`);
    }
    const parts = [stackPart, idPart]
        .map(part => part.generate());
    const hashLength = 12;
    const sha256 = crypto.createHash('sha256')
        .update(stackPart.bareStr)
        .update(idPart.bareStr)
        .update(region)
        .update(account);
    const hash = sha256.digest('hex').slice(0, hashLength);
    const ret = [...parts, hash].join('');
    return ret.toLowerCase();
}
exports.generatePhysicalName = generatePhysicalName;
class NamePart {
    constructor(bareStr) {
        this.bareStr = bareStr;
    }
}
class PrefixNamePart extends NamePart {
    constructor(bareStr, prefixLength) {
        super(bareStr);
        this.prefixLength = prefixLength;
    }
    generate() {
        return this.bareStr.slice(0, this.prefixLength);
    }
}
class SuffixNamePart extends NamePart {
    constructor(str, suffixLength) {
        super(str);
        this.suffixLength = suffixLength;
    }
    generate() {
        const strLen = this.bareStr.length;
        const startIndex = Math.max(strLen - this.suffixLength, 0);
        return this.bareStr.slice(startIndex, strLen);
    }
}
const GENERATE_IF_NEEDED_SYMBOL = Symbol.for('@aws-cdk/core.<private>.GenerateIfNeeded');
/**
 * This marker token is used by PhysicalName.GENERATE_IF_NEEDED. When that token is passed to the
 * physicalName property of a Resource, it triggers different behavior in the Resource constructor
 * that will allow emission of a generated physical name (when the resource is used across
 * environments) or undefined (when the resource is not shared).
 *
 * This token throws an Error when it is resolved, as a way to prevent inadvertent mis-uses of it.
 */
class GeneratedWhenNeededMarker {
    constructor() {
        this.creationStack = [];
        Object.defineProperty(this, GENERATE_IF_NEEDED_SYMBOL, { value: true });
    }
    resolve(_ctx) {
        throw new Error('Invalid physical name passed to CloudFormation. Use "this.physicalName" instead');
    }
    toString() {
        return 'PhysicalName.GENERATE_IF_NEEDED';
    }
}
exports.GeneratedWhenNeededMarker = GeneratedWhenNeededMarker;
/**
 * Checks whether a stringified token resolves to a `GeneratedWhenNeededMarker`.
 */
function isGeneratedWhenNeededMarker(val) {
    const token = token_map_1.TokenMap.instance().lookupString(val);
    return !!token && GENERATE_IF_NEEDED_SYMBOL in token;
}
exports.isGeneratedWhenNeededMarker = isGeneratedWhenNeededMarker;
//# sourceMappingURL=data:application/json;base64,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