"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnReference = void 0;
const reference_1 = require("../reference");
const CFN_REFERENCE_SYMBOL = Symbol.for('@aws-cdk/core.CfnReference');
/**
 * A Token that represents a CloudFormation reference to another resource
 *
 * If these references are used in a different stack from where they are
 * defined, appropriate CloudFormation `Export`s and `Fn::ImportValue`s will be
 * synthesized automatically instead of the regular CloudFormation references.
 *
 * Additionally, the dependency between the stacks will be recorded, and the toolkit
 * will make sure to deploy producing stack before the consuming stack.
 *
 * This magic happens in the prepare() phase, where consuming stacks will call
 * `consumeFromStack` on these Tokens and if they happen to be exported by a different
 * Stack, we'll register the dependency.
 */
class CfnReference extends reference_1.Reference {
    constructor(value, displayName, target) {
        // prepend scope path to display name
        super(value, target, displayName);
        this.replacementTokens = new Map();
        this.targetStack = stack_1.Stack.of(target);
        Object.defineProperty(this, CFN_REFERENCE_SYMBOL, { value: true });
    }
    /**
     * Check whether this is actually a Reference
     */
    static isCfnReference(x) {
        return CFN_REFERENCE_SYMBOL in x;
    }
    /**
     * Return the CfnReference for the indicated target
     *
     * Will make sure that multiple invocations for the same target and intrinsic
     * return the same CfnReference. Because CfnReferences accumulate state in
     * the prepare() phase (for the purpose of cross-stack references), it's
     * important that the state isn't lost if it's lazily created, like so:
     *
     *     Lazy.stringValue({ produce: () => new CfnReference(...) })
     *
     * If fnSub is true, then this reference will resolve as ${logicalID}.
     * This allows cloudformation-include to correctly handle Fn::Sub.
     */
    static for(target, attribute, fnSub = false) {
        return CfnReference.singletonReference(target, attribute, fnSub, () => {
            const cfnIntrinsic = fnSub
                ? ('${' + target.logicalId + (attribute === 'Ref' ? '' : `.${attribute}`) + '}')
                : (attribute === 'Ref' ? { Ref: target.logicalId } : { 'Fn::GetAtt': [target.logicalId, attribute] });
            return new CfnReference(cfnIntrinsic, attribute, target);
        });
    }
    /**
     * Return a CfnReference that references a pseudo referencd
     */
    static forPseudo(pseudoName, scope) {
        return CfnReference.singletonReference(scope, `Pseudo:${pseudoName}`, false, () => {
            const cfnIntrinsic = { Ref: pseudoName };
            return new CfnReference(cfnIntrinsic, pseudoName, scope);
        });
    }
    /**
     * Get or create the table.
     * Passing fnSub = true allows cloudformation-include to correctly handle Fn::Sub.
     */
    static singletonReference(target, attribKey, fnSub, fresh) {
        let attribs = CfnReference.referenceTable.get(target);
        if (!attribs) {
            attribs = new Map();
            CfnReference.referenceTable.set(target, attribs);
        }
        const cacheKey = attribKey + (fnSub ? 'Fn::Sub' : '');
        let ref = attribs.get(cacheKey);
        if (!ref) {
            ref = fresh();
            attribs.set(cacheKey, ref);
        }
        return ref;
    }
    resolve(context) {
        // If we have a special token for this consuming stack, resolve that. Otherwise resolve as if
        // we are in the same stack.
        const consumingStack = stack_1.Stack.of(context.scope);
        const token = this.replacementTokens.get(consumingStack);
        // if (!token && this.isCrossStackReference(consumingStack) && !context.preparing) {
        // eslint-disable-next-line max-len
        //   throw new Error(`Cross-stack reference (${context.scope.node.path} -> ${this.target.node.path}) has not been assigned a value--call prepare() first`);
        // }
        if (token) {
            return token.resolve(context);
        }
        else {
            return super.resolve(context);
        }
    }
    hasValueForStack(stack) {
        if (stack === this.targetStack) {
            return true;
        }
        return this.replacementTokens.has(stack);
    }
    assignValueForStack(stack, value) {
        if (stack === this.targetStack) {
            throw new Error('cannot assign a value for the same stack');
        }
        if (this.hasValueForStack(stack)) {
            throw new Error('Cannot assign a reference value twice to the same stack. Use hasValueForStack to check first');
        }
        this.replacementTokens.set(stack, value);
    }
    /**
     * Implementation of toString() that will use the display name
     */
    toString() {
        return token_1.Token.asString(this, {
            displayHint: `${this.target.node.id}.${this.displayName}`,
        });
    }
}
exports.CfnReference = CfnReference;
/**
 * Static table where we keep singleton CfnReference instances
 */
CfnReference.referenceTable = new Map();
const stack_1 = require("../stack");
const token_1 = require("../token");
//# sourceMappingURL=data:application/json;base64,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