"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependableTrait = exports.ConcreteDependable = void 0;
/**
 * A set of constructs to be used as a dependable
 *
 * This class can be used when a set of constructs which are disjoint in the
 * construct tree needs to be combined to be used as a single dependable.
 *
 * @experimental
 */
class ConcreteDependable {
    constructor() {
        this._dependencyRoots = new Array();
        const self = this;
        DependableTrait.implement(this, {
            get dependencyRoots() { return self._dependencyRoots; },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(construct) {
        this._dependencyRoots.push(construct);
    }
}
exports.ConcreteDependable = ConcreteDependable;
const DEPENDABLE_SYMBOL = Symbol.for('@aws-cdk/core.DependableTrait');
/**
 * Trait for IDependable
 *
 * Traits are interfaces that are privately implemented by objects. Instead of
 * showing up in the public interface of a class, they need to be queried
 * explicitly. This is used to implement certain framework features that are
 * not intended to be used by Construct consumers, and so should be hidden
 * from accidental use.
 *
 * @example
 *
 * // Usage
 * const roots = DependableTrait.get(construct).dependencyRoots;
 *
 * // Definition
 * DependableTrait.implement(construct, {
 *   get dependencyRoots() { return []; }
 * });
 *
 * @experimental
 */
class DependableTrait {
    /**
     * Register `instance` to have the given DependableTrait
     *
     * Should be called in the class constructor.
     */
    static implement(instance, trait) {
        // I would also like to reference classes (to cut down on the list of objects
        // we need to manage), but we can't do that either since jsii doesn't have the
        // concept of a class reference.
        instance[DEPENDABLE_SYMBOL] = trait;
    }
    /**
     * Return the matching DependableTrait for the given class instance.
     */
    static get(instance) {
        const ret = instance[DEPENDABLE_SYMBOL];
        if (!ret) {
            throw new Error(`${instance} does not implement DependableTrait`);
        }
        return ret;
    }
}
exports.DependableTrait = DependableTrait;
//# sourceMappingURL=data:application/json;base64,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