"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResource = void 0;
const cfn_resource_1 = require("./cfn-resource");
const removal_policy_1 = require("./removal-policy");
const resource_1 = require("./resource");
const token_1 = require("./token");
/**
 * Custom resource that is implemented using a Lambda
 *
 * As a custom resource author, you should be publishing a subclass of this class
 * that hides the choice of provider, and accepts a strongly-typed properties
 * object with the properties your provider accepts.
 *
 * @resource AWS::CloudFormation::CustomResource
 */
class CustomResource extends resource_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const type = renderResourceType(props.resourceType);
        const pascalCaseProperties = (_a = props.pascalCaseProperties) !== null && _a !== void 0 ? _a : false;
        const properties = pascalCaseProperties ? uppercaseProperties(props.properties || {}) : (props.properties || {});
        this.resource = new cfn_resource_1.CfnResource(this, 'Default', {
            type,
            properties: {
                ServiceToken: props.serviceToken,
                ...properties,
            },
        });
        this.resource.applyRemovalPolicy(props.removalPolicy, {
            default: removal_policy_1.RemovalPolicy.DESTROY,
        });
    }
    /**
     * The physical name of this custom resource.
     */
    get ref() {
        return this.resource.ref;
    }
    /**
     * Returns the value of an attribute of the custom resource of an arbitrary
     * type. Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt`. Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getAttString` for string attributes.
     */
    getAtt(attributeName) {
        return this.resource.getAtt(attributeName);
    }
    /**
     * Returns the value of an attribute of the custom resource of type string.
     * Attributes are returned from the custom resource provider through the
     * `Data` map where the key is the attribute name.
     *
     * @param attributeName the name of the attribute
     * @returns a token for `Fn::GetAtt` encoded as a string.
     */
    getAttString(attributeName) {
        return token_1.Token.asString(this.getAtt(attributeName));
    }
}
exports.CustomResource = CustomResource;
/**
 * Uppercase the first letter of every property name
 *
 * It's customary for CloudFormation properties to start with capitals, and our
 * properties to start with lowercase, so this function translates from one
 * to the other
 */
function uppercaseProperties(props) {
    const ret = {};
    Object.keys(props).forEach(key => {
        const upper = key.substr(0, 1).toUpperCase() + key.substr(1);
        ret[upper] = props[key];
    });
    return ret;
}
function renderResourceType(resourceType) {
    if (!resourceType) {
        return 'AWS::CloudFormation::CustomResource';
    }
    if (!resourceType.startsWith('Custom::')) {
        throw new Error(`Custom resource type must begin with "Custom::" (${resourceType})`);
    }
    const typeName = resourceType.substr(resourceType.indexOf('::') + 2);
    if (typeName.length > 60) {
        throw new Error(`Custom resource type length > 60 (${resourceType})`);
    }
    if (!/^[a-z0-9_@-]+$/i.test(typeName)) {
        throw new Error(`Custom resource type name can only include alphanumeric characters and _@- (${typeName})`);
    }
    return resourceType;
}
//# sourceMappingURL=data:application/json;base64,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