"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContextProvider = void 0;
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const annotations_1 = require("./annotations");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * Base class for the model side of context providers
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 *
 * @experimental
 */
class ContextProvider {
    /**
     * @returns the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
     */
    static getKey(scope, options) {
        const stack = stack_1.Stack.of(scope);
        const props = {
            account: stack.account,
            region: stack.region,
            ...options.props || {},
        };
        if (Object.values(props).find(x => token_1.Token.isUnresolved(x))) {
            throw new Error(`Cannot determine scope for context provider ${options.provider}.\n` +
                'This usually happens when one or more of the provider props have unresolved tokens');
        }
        const propStrings = propsToArray(props);
        return {
            key: `${options.provider}:${propStrings.join(':')}`,
            props,
        };
    }
    static getValue(scope, options) {
        const stack = stack_1.Stack.of(scope);
        if (token_1.Token.isUnresolved(stack.account) || token_1.Token.isUnresolved(stack.region)) {
            throw new Error(`Cannot retrieve value from context provider ${options.provider} since account/region ` +
                'are not specified at the stack level. Either configure "env" with explicit account and region when ' +
                'you define your stack, or use the environment variables "CDK_DEFAULT_ACCOUNT" and "CDK_DEFAULT_REGION" ' +
                'to inherit environment information from the CLI (not recommended for production stacks)');
        }
        const { key, props } = this.getKey(scope, options);
        const value = scope.node.tryGetContext(key);
        const providerError = extractProviderError(value);
        // if context is missing or an error occurred during context retrieval,
        // report and return a dummy value.
        if (value === undefined || providerError !== undefined) {
            stack.reportMissingContext({
                key,
                provider: options.provider,
                props: props,
            });
            if (providerError !== undefined) {
                annotations_1.Annotations.of(scope).addError(providerError);
            }
            return { value: options.dummyValue };
        }
        return { value };
    }
    constructor() { }
}
exports.ContextProvider = ContextProvider;
/**
 * If the context value represents an error, return the error message
 */
function extractProviderError(value) {
    if (typeof value === 'object' && value !== null) {
        return value[cxapi.PROVIDER_ERROR_KEY];
    }
    return undefined;
}
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        // skip undefined values
        if (props[key] === undefined) {
            continue;
        }
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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