"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParser = exports.CfnParsingContext = exports.FromCloudFormation = void 0;
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_policy_1 = require("./cfn-resource-policy");
const lazy_1 = require("./lazy");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * This class contains static methods called when going from
 * translated values received from {@link CfnParser.parseValue}
 * to the actual L1 properties -
 * things like changing IResolvable to the appropriate type
 * (string, string array, or number), etc.
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code.
 *
 * @experimental
 */
class FromCloudFormation {
    // nothing to for any but return it
    static getAny(value) { return value; }
    // nothing to do - if 'value' is not a boolean or a Token,
    // a validator should report that at runtime
    static getBoolean(value) { return value; }
    static getDate(value) {
        // if the date is a deploy-time value, just return it
        if (token_1.isResolvableObject(value)) {
            return value;
        }
        // if the date has been given as a string, convert it
        if (typeof value === 'string') {
            return new Date(value);
        }
        // all other cases - just return the value,
        // if it's not a Date, a validator should catch it
        return value;
    }
    // won't always return a string; if the input can't be resolved to a string,
    // the input will be returned.
    static getString(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return value.toString();
        }
        // CloudFormation treats numbers and strings interchangeably;
        // so, if we get a number here, convert it to a string
        if (typeof value === 'number') {
            return value.toString();
        }
        // in all other cases, just return the input,
        // and let a validator handle it if it's not a string
        return value;
    }
    // won't always return a number; if the input can't be parsed to a number,
    // the input will be returned.
    static getNumber(value) {
        // if the string is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asNumber(value);
        }
        // return a number, if the input can be parsed as one
        let parsedValue;
        if (typeof value === 'string') {
            parsedValue = parseFloat(value);
            if (!isNaN(parsedValue)) {
                return parsedValue;
            }
        }
        // otherwise return the input,
        // and let a validator handle it if it's not a number
        return value;
    }
    static getStringArray(value) {
        // if the array is a deploy-time value, serialize it to a Token
        if (token_1.isResolvableObject(value)) {
            return token_1.Token.asList(value);
        }
        // in all other cases, delegate to the standard mapping logic
        return this.getArray(this.getString)(value);
    }
    static getArray(mapper) {
        return (value) => {
            if (!Array.isArray(value)) {
                // break the type system, and just return the given value,
                // which hopefully will be reported as invalid by the validator
                // of the property we're transforming
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            return value.map(mapper);
        };
    }
    static getMap(mapper) {
        return (value) => {
            if (typeof value !== 'object') {
                // if the input is not a map (= object in JS land),
                // just return it, and let the validator of this property handle it
                // (unless it's a deploy-time value,
                // which we can't map over at build time anyway)
                return value;
            }
            const ret = {};
            for (const [key, val] of Object.entries(value)) {
                ret[key] = mapper(val);
            }
            return ret;
        };
    }
    static getCfnTag(tag) {
        return tag == null
            ? {} // break the type system - this should be detected at runtime by a tag validator
            : {
                key: tag.Key,
                value: tag.Value,
            };
    }
    /**
     * Return a function that, when applied to a value, will return the first validly deserialized one
     */
    static getTypeUnion(validators, mappers) {
        return (value) => {
            for (let i = 0; i < validators.length; i++) {
                const candidate = mappers[i](value);
                if (validators[i](candidate).isSuccess) {
                    return candidate;
                }
            }
            // if nothing matches, just return the input unchanged, and let validators catch it
            return value;
        };
    }
}
exports.FromCloudFormation = FromCloudFormation;
/**
 * The context in which the parsing is taking place.
 *
 * Some fragments of CloudFormation templates behave differently than others
 * (for example, the 'Conditions' sections treats { "Condition": "NameOfCond" }
 * differently than the 'Resources' section).
 * This enum can be used to change the created {@link CfnParser} behavior,
 * based on the template context.
 */
var CfnParsingContext;
(function (CfnParsingContext) {
    /** We're currently parsing the 'Conditions' section. */
    CfnParsingContext[CfnParsingContext["CONDITIONS"] = 0] = "CONDITIONS";
    /** We're currently parsing the 'Rules' section. */
    CfnParsingContext[CfnParsingContext["RULES"] = 1] = "RULES";
})(CfnParsingContext = exports.CfnParsingContext || (exports.CfnParsingContext = {}));
/**
 * This class contains methods for translating from a pure CFN value
 * (like a JS object { "Ref": "Bucket" })
 * to a form CDK understands
 * (like Fn.ref('Bucket')).
 *
 * While this file not exported from the module
 * (to not make it part of the public API),
 * it is directly referenced in the generated L1 code,
 * so any renames of it need to be reflected in cfn2ts/codegen.ts as well.
 *
 * @experimental
 */
class CfnParser {
    constructor(options) {
        this.options = options;
    }
    handleAttributes(resource, resourceAttributes, logicalId) {
        var _a;
        const cfnOptions = resource.cfnOptions;
        cfnOptions.creationPolicy = this.parseCreationPolicy(resourceAttributes.CreationPolicy);
        cfnOptions.updatePolicy = this.parseUpdatePolicy(resourceAttributes.UpdatePolicy);
        cfnOptions.deletionPolicy = this.parseDeletionPolicy(resourceAttributes.DeletionPolicy);
        cfnOptions.updateReplacePolicy = this.parseDeletionPolicy(resourceAttributes.UpdateReplacePolicy);
        cfnOptions.metadata = this.parseValue(resourceAttributes.Metadata);
        // handle Condition
        if (resourceAttributes.Condition) {
            const condition = this.finder.findCondition(resourceAttributes.Condition);
            if (!condition) {
                throw new Error(`Resource '${logicalId}' uses Condition '${resourceAttributes.Condition}' that doesn't exist`);
            }
            cfnOptions.condition = condition;
        }
        // handle DependsOn
        resourceAttributes.DependsOn = (_a = resourceAttributes.DependsOn) !== null && _a !== void 0 ? _a : [];
        const dependencies = Array.isArray(resourceAttributes.DependsOn) ?
            resourceAttributes.DependsOn : [resourceAttributes.DependsOn];
        for (const dep of dependencies) {
            const depResource = this.finder.findResource(dep);
            if (!depResource) {
                throw new Error(`Resource '${logicalId}' depends on '${dep}' that doesn't exist`);
            }
            resource.node.addDependency(depResource);
        }
    }
    parseCreationPolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingCreationPolicy: parseAutoScalingCreationPolicy(policy.AutoScalingCreationPolicy),
            resourceSignal: parseResourceSignal(policy.ResourceSignal),
        });
        function parseAutoScalingCreationPolicy(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
            });
        }
        function parseResourceSignal(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                count: FromCloudFormation.getNumber(p.Count),
                timeout: FromCloudFormation.getString(p.Timeout),
            });
        }
    }
    parseUpdatePolicy(policy) {
        if (typeof policy !== 'object') {
            return undefined;
        }
        // change simple JS values to their CDK equivalents
        policy = this.parseValue(policy);
        return undefinedIfAllValuesAreEmpty({
            autoScalingReplacingUpdate: parseAutoScalingReplacingUpdate(policy.AutoScalingReplacingUpdate),
            autoScalingRollingUpdate: parseAutoScalingRollingUpdate(policy.AutoScalingRollingUpdate),
            autoScalingScheduledAction: parseAutoScalingScheduledAction(policy.AutoScalingScheduledAction),
            codeDeployLambdaAliasUpdate: parseCodeDeployLambdaAliasUpdate(policy.CodeDeployLambdaAliasUpdate),
            enableVersionUpgrade: policy.EnableVersionUpgrade,
            useOnlineResharding: policy.UseOnlineResharding,
        });
        function parseAutoScalingReplacingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                willReplace: p.WillReplace,
            });
        }
        function parseAutoScalingRollingUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                maxBatchSize: FromCloudFormation.getNumber(p.MaxBatchSize),
                minInstancesInService: FromCloudFormation.getNumber(p.MinInstancesInService),
                minSuccessfulInstancesPercent: FromCloudFormation.getNumber(p.MinSuccessfulInstancesPercent),
                pauseTime: FromCloudFormation.getString(p.PauseTime),
                suspendProcesses: FromCloudFormation.getStringArray(p.SuspendProcesses),
                waitOnResourceSignals: p.WaitOnResourceSignals,
            });
        }
        function parseCodeDeployLambdaAliasUpdate(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return {
                beforeAllowTrafficHook: FromCloudFormation.getString(p.BeforeAllowTrafficHook),
                afterAllowTrafficHook: FromCloudFormation.getString(p.AfterAllowTrafficHook),
                applicationName: FromCloudFormation.getString(p.ApplicationName),
                deploymentGroupName: FromCloudFormation.getString(p.DeploymentGroupName),
            };
        }
        function parseAutoScalingScheduledAction(p) {
            if (typeof p !== 'object') {
                return undefined;
            }
            return undefinedIfAllValuesAreEmpty({
                ignoreUnmodifiedGroupSizeProperties: p.IgnoreUnmodifiedGroupSizeProperties,
            });
        }
    }
    parseDeletionPolicy(policy) {
        switch (policy) {
            case null: return undefined;
            case undefined: return undefined;
            case 'Delete': return cfn_resource_policy_1.CfnDeletionPolicy.DELETE;
            case 'Retain': return cfn_resource_policy_1.CfnDeletionPolicy.RETAIN;
            case 'Snapshot': return cfn_resource_policy_1.CfnDeletionPolicy.SNAPSHOT;
            default: throw new Error(`Unrecognized DeletionPolicy '${policy}'`);
        }
    }
    parseValue(cfnValue) {
        // == null captures undefined as well
        if (cfnValue == null) {
            return undefined;
        }
        // if we have any late-bound values,
        // just return them
        if (token_1.isResolvableObject(cfnValue)) {
            return cfnValue;
        }
        if (Array.isArray(cfnValue)) {
            return cfnValue.map(el => this.parseValue(el));
        }
        if (typeof cfnValue === 'object') {
            // an object can be either a CFN intrinsic, or an actual object
            const cfnIntrinsic = this.parseIfCfnIntrinsic(cfnValue);
            if (cfnIntrinsic !== undefined) {
                return cfnIntrinsic;
            }
            const ret = {};
            for (const [key, val] of Object.entries(cfnValue)) {
                ret[key] = this.parseValue(val);
            }
            return ret;
        }
        // in all other cases, just return the input
        return cfnValue;
    }
    get finder() {
        return this.options.finder;
    }
    parseIfCfnIntrinsic(object) {
        const key = this.looksLikeCfnIntrinsic(object);
        switch (key) {
            case undefined:
                return undefined;
            case 'Ref': {
                const refTarget = object[key];
                const specialRef = this.specialCaseRefs(refTarget);
                if (specialRef !== undefined) {
                    return specialRef;
                }
                else {
                    const refElement = this.finder.findRefTarget(refTarget);
                    if (!refElement) {
                        throw new Error(`Element used in Ref expression with logical ID: '${refTarget}' not found`);
                    }
                    return cfn_reference_1.CfnReference.for(refElement, 'Ref');
                }
            }
            case 'Fn::GetAtt': {
                // Fn::GetAtt takes a 2-element list as its argument
                const value = object[key];
                const target = this.finder.findResource(value[0]);
                if (!target) {
                    throw new Error(`Resource used in GetAtt expression with logical ID: '${value[0]}' not found`);
                }
                return target.getAtt(value[1]);
            }
            case 'Fn::Join': {
                // Fn::Join takes a 2-element list as its argument,
                // where the first element is the delimiter,
                // and the second is the list of elements to join
                const value = this.parseValue(object[key]);
                // wrap the array as a Token,
                // as otherwise Fn.join() will try to concatenate
                // the non-token parts,
                // causing a diff with the original template
                return cfn_fn_1.Fn.join(value[0], lazy_1.Lazy.listValue({ produce: () => value[1] }));
            }
            case 'Fn::Cidr': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.cidr(value[0], value[1], value[2]);
            }
            case 'Fn::FindInMap': {
                const value = this.parseValue(object[key]);
                // the first argument to FindInMap is the mapping name
                const mapping = this.finder.findMapping(value[0]);
                if (!mapping) {
                    throw new Error(`Mapping used in FindInMap expression with name '${value[0]}' was not found in the template`);
                }
                return cfn_fn_1.Fn.findInMap(mapping.logicalId, value[1], value[2]);
            }
            case 'Fn::Select': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.select(value[0], value[1]);
            }
            case 'Fn::GetAZs': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.getAzs(value);
            }
            case 'Fn::ImportValue': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.importValue(value);
            }
            case 'Fn::Split': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.split(value[0], value[1]);
            }
            case 'Fn::Transform': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.transform(value.Name, value.Parameters);
            }
            case 'Fn::Base64': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.base64(value);
            }
            case 'Fn::If': {
                // Fn::If takes a 3-element list as its argument,
                // where the first element is the name of a Condition
                const value = this.parseValue(object[key]);
                const condition = this.finder.findCondition(value[0]);
                if (!condition) {
                    throw new Error(`Condition '${value[0]}' used in an Fn::If expression does not exist in the template`);
                }
                return cfn_fn_1.Fn.conditionIf(condition.logicalId, value[1], value[2]);
            }
            case 'Fn::Equals': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionEquals(value[0], value[1]);
            }
            case 'Fn::And': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionAnd(...value);
            }
            case 'Fn::Not': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionNot(value[0]);
            }
            case 'Fn::Or': {
                const value = this.parseValue(object[key]);
                return cfn_fn_1.Fn.conditionOr(...value);
            }
            case 'Fn::Sub': {
                const value = this.parseValue(object[key]);
                let fnSubString;
                let map;
                if (typeof value === 'string') {
                    fnSubString = value;
                    map = undefined;
                }
                else {
                    fnSubString = value[0];
                    map = value[1];
                }
                return cfn_fn_1.Fn.sub(this.parseFnSubString(fnSubString, map), map);
            }
            case 'Condition': {
                // a reference to a Condition from another Condition
                const condition = this.finder.findCondition(object[key]);
                if (!condition) {
                    throw new Error(`Referenced Condition with name '${object[key]}' was not found in the template`);
                }
                return { Condition: condition.logicalId };
            }
            default:
                if (this.options.context === CfnParsingContext.RULES) {
                    return this.handleRulesIntrinsic(key, object);
                }
                else {
                    throw new Error(`Unsupported CloudFormation function '${key}'`);
                }
        }
    }
    looksLikeCfnIntrinsic(object) {
        const objectKeys = Object.keys(object);
        // a CFN intrinsic is always an object with a single key
        if (objectKeys.length !== 1) {
            return undefined;
        }
        const key = objectKeys[0];
        return key === 'Ref' || key.startsWith('Fn::') ||
            // special intrinsic only available in the 'Conditions' section
            (this.options.context === CfnParsingContext.CONDITIONS && key === 'Condition')
            ? key
            : undefined;
    }
    parseFnSubString(value, map = {}) {
        const leftBrace = value.indexOf('${');
        const rightBrace = value.indexOf('}') + 1;
        // don't include left and right braces when searching for the target of the reference
        if (leftBrace === -1 || leftBrace >= rightBrace) {
            return value;
        }
        const leftHalf = value.substring(0, leftBrace);
        const rightHalf = value.substring(rightBrace);
        const refTarget = value.substring(leftBrace + 2, rightBrace - 1).trim();
        if (refTarget[0] === '!') {
            return value.substring(0, rightBrace) + this.parseFnSubString(rightHalf, map);
        }
        // lookup in map
        if (refTarget in map) {
            return leftHalf + '${' + refTarget + '}' + this.parseFnSubString(rightHalf, map);
        }
        // since it's not in the map, check if it's a pseudo parameter
        const specialRef = this.specialCaseSubRefs(refTarget);
        if (specialRef !== undefined) {
            return leftHalf + specialRef + this.parseFnSubString(rightHalf, map);
        }
        const dotIndex = refTarget.indexOf('.');
        const isRef = dotIndex === -1;
        if (isRef) {
            const refElement = this.finder.findRefTarget(refTarget);
            if (!refElement) {
                throw new Error(`Element referenced in Fn::Sub expression with logical ID: '${refTarget}' was not found in the template`);
            }
            return leftHalf + cfn_reference_1.CfnReference.for(refElement, 'Ref', true).toString() + this.parseFnSubString(rightHalf, map);
        }
        else {
            const targetId = refTarget.substring(0, dotIndex);
            const refResource = this.finder.findResource(targetId);
            if (!refResource) {
                throw new Error(`Resource referenced in Fn::Sub expression with logical ID: '${targetId}' was not found in the template`);
            }
            const attribute = refTarget.substring(dotIndex + 1);
            return leftHalf + cfn_reference_1.CfnReference.for(refResource, attribute, true).toString() + this.parseFnSubString(rightHalf, map);
        }
    }
    handleRulesIntrinsic(key, object) {
        // Rules have their own set of intrinsics:
        // https://docs.aws.amazon.com/servicecatalog/latest/adminguide/intrinsic-function-reference-rules.html
        switch (key) {
            case 'Fn::ValueOf': {
                // ValueOf is special,
                // as it takes the name of a Parameter as its first argument
                const value = this.parseValue(object[key]);
                const parameterName = value[0];
                if (parameterName in this.parameters) {
                    // since ValueOf returns the value of a specific attribute,
                    // fail here - this substitution is not allowed
                    throw new Error(`Cannot substitute parameter '${parameterName}' used in Fn::ValueOf expression with attribute '${value[1]}'`);
                }
                const param = this.finder.findRefTarget(parameterName);
                if (!param) {
                    throw new Error(`Rule references parameter '${parameterName}' which was not found in the template`);
                }
                // create an explicit IResolvable,
                // as Fn.valueOf() returns a string,
                // which is incorrect
                // (Fn::ValueOf can also return an array)
                return lazy_1.Lazy.anyValue({ produce: () => ({ 'Fn::ValueOf': [param.logicalId, value[1]] }) });
            }
            default:
                // I don't want to hard-code the list of supported Rules-specific intrinsics in this function;
                // so, just return undefined here,
                // and they will be treated as a regular JSON object
                return undefined;
        }
    }
    specialCaseRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        switch (value) {
            case 'AWS::AccountId': return cfn_pseudo_1.Aws.ACCOUNT_ID;
            case 'AWS::Region': return cfn_pseudo_1.Aws.REGION;
            case 'AWS::Partition': return cfn_pseudo_1.Aws.PARTITION;
            case 'AWS::URLSuffix': return cfn_pseudo_1.Aws.URL_SUFFIX;
            case 'AWS::NotificationARNs': return cfn_pseudo_1.Aws.NOTIFICATION_ARNS;
            case 'AWS::StackId': return cfn_pseudo_1.Aws.STACK_ID;
            case 'AWS::StackName': return cfn_pseudo_1.Aws.STACK_NAME;
            case 'AWS::NoValue': return cfn_pseudo_1.Aws.NO_VALUE;
            default: return undefined;
        }
    }
    specialCaseSubRefs(value) {
        if (value in this.parameters) {
            return this.parameters[value];
        }
        return value.indexOf('::') === -1 ? undefined : '${' + value + '}';
    }
    get parameters() {
        return this.options.parameters || {};
    }
}
exports.CfnParser = CfnParser;
function undefinedIfAllValuesAreEmpty(object) {
    return Object.values(object).some(v => v !== undefined) ? object : undefined;
}
//# sourceMappingURL=data:application/json;base64,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