"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnParameter = void 0;
const cfn_element_1 = require("./cfn-element");
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
/**
 * A CloudFormation parameter.
 *
 * Use the optional Parameters section to customize your templates.
 * Parameters enable you to input custom values to your template each time you create or
 * update a stack.
 */
class CfnParameter extends cfn_element_1.CfnElement {
    /**
     * Creates a parameter construct.
     * Note that the name (logical ID) of the parameter will derive from it's `coname` and location
     * within the stack. Therefore, it is recommended that parameters are defined at the stack level.
     *
     * @param scope The parent construct.
     * @param props The parameter properties.
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        this._type = props.type || 'String';
        this._default = props.default;
        this._allowedPattern = props.allowedPattern;
        this._allowedValues = props.allowedValues;
        this._constraintDescription = props.constraintDescription;
        this._description = props.description;
        this._maxLength = props.maxLength;
        this._maxValue = props.maxValue;
        this._minLength = props.minLength;
        this._minValue = props.minValue;
        this._noEcho = props.noEcho;
    }
    /**
     * The data type for the parameter (DataType).
     *
     * @default String
     */
    get type() {
        return this._type;
    }
    set type(type) {
        this._type = type;
    }
    /**
     * A value of the appropriate type for the template to use if no value is specified
     * when a stack is created. If you define constraints for the parameter, you must specify
     * a value that adheres to those constraints.
     *
     * @default - No default value for parameter.
     */
    get default() {
        return this._default;
    }
    set default(value) {
        this._default = value;
    }
    /**
     * A regular expression that represents the patterns to allow for String types.
     *
     * @default - No constraints on patterns allowed for parameter.
     */
    get allowedPattern() {
        return this._allowedPattern;
    }
    set allowedPattern(pattern) {
        this._allowedPattern = pattern;
    }
    /**
     * An array containing the list of values allowed for the parameter.
     *
     * @default - No constraints on values allowed for parameter.
     */
    get allowedValues() {
        return this._allowedValues;
    }
    set allowedValues(values) {
        this._allowedValues = values;
    }
    /**
     * A string that explains a constraint when the constraint is violated.
     * For example, without a constraint description, a parameter that has an allowed
     * pattern of [A-Za-z0-9]+ displays the following error message when the user specifies
     * an invalid value:
     *
     * @default - No description with customized error message when user specifies invalid values.
     */
    get constraintDescription() {
        return this._constraintDescription;
    }
    set constraintDescription(desc) {
        this._constraintDescription = desc;
    }
    /**
     * A string of up to 4000 characters that describes the parameter.
     *
     * @default - No description for the parameter.
     */
    get description() {
        return this._description;
    }
    set description(desc) {
        this._description = desc;
    }
    /**
     * An integer value that determines the largest number of characters you want to allow for String types.
     *
     * @default - None.
     */
    get maxLength() {
        return this._maxLength;
    }
    set maxLength(len) {
        this._maxLength = len;
    }
    /**
     * An integer value that determines the smallest number of characters you want to allow for String types.
     *
     * @default - None.
     */
    get minLength() {
        return this._minLength;
    }
    set minLength(len) {
        this._minLength = len;
    }
    /**
     * A numeric value that determines the largest numeric value you want to allow for Number types.
     *
     * @default - None.
     */
    get maxValue() {
        return this._maxValue;
    }
    set maxValue(len) {
        this._maxValue = len;
    }
    /**
     * A numeric value that determines the smallest numeric value you want to allow for Number types.
     *
     * @default - None.
     */
    get minValue() {
        return this._minValue;
    }
    set minValue(len) {
        this._minValue = len;
    }
    /**
     * Indicates if this parameter is configured with "NoEcho" enabled.
     */
    get noEcho() {
        return !!this._noEcho;
    }
    set noEcho(echo) {
        this._noEcho = echo;
    }
    /**
     * The parameter value as a Token
     */
    get value() {
        return cfn_reference_1.CfnReference.for(this, 'Ref');
    }
    /**
     * The parameter value, if it represents a string.
     */
    get valueAsString() {
        if (!isStringType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string type`);
        }
        return token_1.Token.asString(this.value);
    }
    /**
     * The parameter value, if it represents a string list.
     */
    get valueAsList() {
        if (!isListType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a string list type`);
        }
        return token_1.Token.asList(this.value);
    }
    /**
     * The parameter value, if it represents a number.
     */
    get valueAsNumber() {
        if (!isNumberType(this.type)) {
            throw new Error(`Parameter type (${this.type}) is not a number type`);
        }
        return token_1.Token.asNumber(this.value);
    }
    /**
     * @internal
     */
    _toCloudFormation() {
        return {
            Parameters: {
                [this.logicalId]: {
                    Type: this.type,
                    Default: this.default,
                    AllowedPattern: this.allowedPattern,
                    AllowedValues: this.allowedValues,
                    ConstraintDescription: this.constraintDescription,
                    Description: this.description,
                    MaxLength: this.maxLength,
                    MaxValue: this.maxValue,
                    MinLength: this.minLength,
                    MinValue: this.minValue,
                    NoEcho: this._noEcho,
                },
            },
        };
    }
    resolve(_context) {
        return this.value;
    }
}
exports.CfnParameter = CfnParameter;
/**
 * Whether the given parameter type looks like a list type
 */
function isListType(type) {
    return type.indexOf('List<') >= 0 || type.indexOf('CommaDelimitedList') >= 0;
}
/**
 * Whether the given parameter type looks like a number type
 */
function isNumberType(type) {
    return type === 'Number';
}
/**
 * Whether the given parameter type looks like a string type
 */
function isStringType(type) {
    return !isListType(type) && !isNumberType(type);
}
//# sourceMappingURL=data:application/json;base64,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