"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnRefElement = exports.CfnElement = void 0;
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const token_1 = require("./token");
const CFN_ELEMENT_SYMBOL = Symbol.for('@aws-cdk/core.CfnElement');
/**
 * An element of a CloudFormation stack.
 */
class CfnElement extends construct_compat_1.Construct {
    /**
     * Creates an entity and binds it to a tree.
     * Note that the root of the tree must be a Stack object (not just any Root).
     *
     * @param scope The parent construct
     * @param props Construct properties
     */
    constructor(scope, id) {
        super(scope, id);
        Object.defineProperty(this, CFN_ELEMENT_SYMBOL, { value: true });
        this.stack = stack_1.Stack.of(this);
        this.logicalId = lazy_1.Lazy.stringValue({ produce: () => this.synthesizeLogicalId() }, {
            displayHint: `${notTooLong(this.node.path)}.LogicalID`,
        });
        this.node.addMetadata(cxschema.ArtifactMetadataEntryType.LOGICAL_ID, this.logicalId, this.constructor);
    }
    /**
     * Returns `true` if a construct is a stack element (i.e. part of the
     * synthesized cloudformation template).
     *
     * Uses duck-typing instead of `instanceof` to allow stack elements from different
     * versions of this library to be included in the same stack.
     *
     * @returns The construct as a stack element or undefined if it is not a stack element.
     */
    static isCfnElement(x) {
        return CFN_ELEMENT_SYMBOL in x;
    }
    /**
     * Overrides the auto-generated logical ID with a specific ID.
     * @param newLogicalId The new logical ID to use for this stack element.
     */
    overrideLogicalId(newLogicalId) {
        this._logicalIdOverride = newLogicalId;
    }
    /**
     * @returns the stack trace of the point where this Resource was created from, sourced
     *      from the +metadata+ entry typed +aws:cdk:logicalId+, and with the bottom-most
     *      node +internal+ entries filtered.
     */
    get creationStack() {
        const trace = this.node.metadata.find(md => md.type === cxschema.ArtifactMetadataEntryType.LOGICAL_ID).trace;
        if (!trace) {
            return [];
        }
        return filterStackTrace(trace);
        function filterStackTrace(stack) {
            const result = Array.of(...stack);
            while (result.length > 0 && shouldFilter(result[result.length - 1])) {
                result.pop();
            }
            // It's weird if we filtered everything, so return the whole stack...
            return result.length === 0 ? stack : result;
        }
        function shouldFilter(str) {
            return str.match(/[^(]+\(internal\/.*/) !== null;
        }
    }
    /**
     * Called during synthesize to render the logical ID of this element. If
     * `overrideLogicalId` was it will be used, otherwise, we will allocate the
     * logical ID through the stack.
     */
    synthesizeLogicalId() {
        if (this._logicalIdOverride) {
            return this._logicalIdOverride;
        }
        else {
            return this.stack.getLogicalId(this);
        }
    }
}
exports.CfnElement = CfnElement;
/**
 * Base class for referenceable CloudFormation constructs which are not Resources
 *
 * These constructs are things like Conditions and Parameters, can be
 * referenced by taking the `.ref` attribute.
 *
 * Resource constructs do not inherit from CfnRefElement because they have their
 * own, more specific types returned from the .ref attribute. Also, some
 * resources aren't referenceable at all (such as BucketPolicies or GatewayAttachments).
 */
class CfnRefElement extends CfnElement {
    /**
     * Return a string that will be resolved to a CloudFormation `{ Ref }` for this element.
     *
     * If, by any chance, the intrinsic reference of a resource is not a string, you could
     * coerce it to an IResolvable through `Lazy.any({ produce: resource.ref })`.
     */
    get ref() {
        return token_1.Token.asString(cfn_reference_1.CfnReference.for(this, 'Ref'));
    }
}
exports.CfnRefElement = CfnRefElement;
function notTooLong(x) {
    if (x.length < 100) {
        return x;
    }
    return x.substr(0, 47) + '...' + x.substr(x.length - 47);
}
const cfn_reference_1 = require("./private/cfn-reference");
const stack_1 = require("./stack");
//# sourceMappingURL=data:application/json;base64,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