"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.readYamlSync = exports.readJsonSync = void 0;
const fs = require("fs");
const yaml = require("yaml");
function readJsonSync(filePath) {
    const fileContents = fs.readFileSync(filePath);
    return JSON.parse(fileContents.toString());
}
exports.readJsonSync = readJsonSync;
function readYamlSync(filePath) {
    const fileContents = fs.readFileSync(filePath);
    return parseYamlStrWithCfnTags(fileContents.toString());
}
exports.readYamlSync = readYamlSync;
function makeTagForCfnIntrinsic(intrinsicName, addFnPrefix = true, resolveFun) {
    return {
        identify(value) { return typeof value === 'string'; },
        tag: `!${intrinsicName}`,
        resolve: resolveFun || ((_doc, cstNode) => {
            const ret = {};
            ret[addFnPrefix ? `Fn::${intrinsicName}` : intrinsicName] =
                // the +1 is to account for the ! the short form begins with
                parseYamlStrWithCfnTags(cstNode.toString().substring(intrinsicName.length + 1));
            return ret;
        }),
    };
}
const shortForms = [
    'Base64', 'Cidr', 'FindInMap', 'GetAZs', 'ImportValue', 'Join', 'Sub',
    'Select', 'Split', 'Transform', 'And', 'Equals', 'If', 'Not', 'Or',
].map(name => makeTagForCfnIntrinsic(name)).concat(makeTagForCfnIntrinsic('Ref', false), makeTagForCfnIntrinsic('Condition', false), makeTagForCfnIntrinsic('GetAtt', true, (_doc, cstNode) => {
    const parsedArguments = parseYamlStrWithCfnTags(cstNode.toString().substring('!GetAtt'.length));
    let value;
    if (typeof parsedArguments === 'string') {
        // if the arguments to !GetAtt are a string,
        // the part before the first '.' is the logical ID,
        // and the rest is the attribute name
        // (which can contain '.')
        const firstDot = parsedArguments.indexOf('.');
        if (firstDot === -1) {
            throw new Error(`Short-form Fn::GetAtt must contain a '.' in its string argument, got: '${parsedArguments}'`);
        }
        value = [
            parsedArguments.substring(0, firstDot),
            parsedArguments.substring(firstDot + 1),
        ];
    }
    else {
        // this is the form where the arguments to Fn::GetAtt are already an array -
        // in this case, nothing more to do
        value = parsedArguments;
    }
    return { 'Fn::GetAtt': value };
}));
function parseYamlStrWithCfnTags(text) {
    return yaml.parse(text, {
        customTags: shortForms,
        schema: 'yaml-1.1',
    });
}
//# sourceMappingURL=data:application/json;base64,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