"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CfnInclude = void 0;
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_parse = require("../../core/lib/cfn-parse"); // Automatically re-written from '@aws-cdk/core/lib/cfn-parse'
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        this.parameters = {};
        this.mappings = {};
        this.rules = {};
        this.hooks = {};
        this.outputs = {};
        this.nestedStacks = {};
        this.parametersToReplace = props.parameters || {};
        // read the template into a JS object
        this.template = futils.readYamlSync(props.templateFile);
        this.preserveLogicalIds = (_a = props.preserveLogicalIds) !== null && _a !== void 0 ? _a : true;
        // check if all user specified parameter values exist in the template
        for (const logicalId of Object.keys(this.parametersToReplace)) {
            if (!(logicalId in (this.template.Parameters || {}))) {
                throw new Error(`Parameter with logical ID '${logicalId}' was not found in the template`);
            }
        }
        // instantiate the Mappings
        this.mappingsScope = new core.Construct(this, '$Mappings');
        for (const mappingName of Object.keys(this.template.Mappings || {})) {
            this.createMapping(mappingName);
        }
        // instantiate all parameters
        for (const logicalId of Object.keys(this.template.Parameters || {})) {
            this.createParameter(logicalId);
        }
        // instantiate the conditions
        this.conditionsScope = new core.Construct(this, '$Conditions');
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.getOrCreateCondition(conditionName);
        }
        // instantiate the rules
        this.rulesScope = new core.Construct(this, '$Rules');
        for (const ruleName of Object.keys(this.template.Rules || {})) {
            this.createRule(ruleName);
        }
        this.nestedStacksToInclude = props.nestedStacks || {};
        // instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
        // verify that all nestedStacks have been instantiated
        for (const nestedStackId of Object.keys(props.nestedStacks || {})) {
            if (!(nestedStackId in this.resources)) {
                throw new Error(`Nested Stack with logical ID '${nestedStackId}' was not found in the template`);
            }
        }
        // instantiate the Hooks
        this.hooksScope = new core.Construct(this, '$Hooks');
        for (const hookName of Object.keys(this.template.Hooks || {})) {
            this.createHook(hookName);
        }
        const outputScope = new core.Construct(this, '$Ouputs');
        for (const logicalId of Object.keys(this.template.Outputs || {})) {
            this.createOutput(logicalId, outputScope);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnParameter object from the 'Parameters'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Parameter with the given name is not present in the template,
     * throws an exception.
     *
     * @param parameterName the name of the parameter to retrieve
     */
    getParameter(parameterName) {
        const ret = this.parameters[parameterName];
        if (!ret) {
            throw new Error(`Parameter with name '${parameterName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnMapping object from the 'Mappings' section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Mapping with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param mappingName the name of the Mapping in the template to retrieve
     */
    getMapping(mappingName) {
        const ret = this.mappings[mappingName];
        if (!ret) {
            throw new Error(`Mapping with name '${mappingName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnOutput object from the 'Outputs'
     * section of the included template.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If an Output with the given name is not present in the template,
     * throws an exception.
     *
     * @param logicalId the name of the output to retrieve
     */
    getOutput(logicalId) {
        const ret = this.outputs[logicalId];
        if (!ret) {
            throw new Error(`Output with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnRule object from the 'Rules'
     * section of the CloudFormation template with the given name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Rule with the given name is not present in the template,
     * an exception will be thrown.
     *
     * @param ruleName the name of the Rule in the CloudFormation template
     */
    getRule(ruleName) {
        const ret = this.rules[ruleName];
        if (!ret) {
            throw new Error(`Rule with name '${ruleName}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnHook object from the 'Hooks'
     * section of the included CloudFormation template with the given logical ID.
     * Any modifications performed on the returned object will be reflected in the resulting CDK template.
     *
     * If a Hook with the given logical ID is not present in the template,
     * an exception will be thrown.
     *
     * @param hookLogicalId the logical ID of the Hook in the included CloudFormation template's 'Hooks' section
     */
    getHook(hookLogicalId) {
        const ret = this.hooks[hookLogicalId];
        if (!ret) {
            throw new Error(`Hook with logical ID '${hookLogicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the NestedStack with name logicalId.
     * For a nested stack to be returned by this method, it must be specified in the {@link CfnIncludeProps.nestedStacks}
     * property.
     *
     * @param logicalId the ID of the stack to retrieve, as it appears in the template
     */
    getNestedStack(logicalId) {
        if (!this.nestedStacks[logicalId]) {
            if (!this.template.Resources[logicalId]) {
                throw new Error(`Nested Stack with logical ID '${logicalId}' was not found in the template`);
            }
            else if (this.template.Resources[logicalId].Type !== 'AWS::CloudFormation::Stack') {
                throw new Error(`Resource with logical ID '${logicalId}' is not a CloudFormation Stack`);
            }
            else {
                throw new Error(`Nested Stack '${logicalId}' was not included in the nestedStacks property when including the parent template`);
            }
        }
        return this.nestedStacks[logicalId];
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            const self = this;
            const finder = {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            };
            const cfnParser = new cfn_parse.CfnParser({
                finder,
                parameters: this.parametersToReplace,
            });
            switch (section) {
                case 'Conditions':
                case 'Mappings':
                case 'Resources':
                case 'Parameters':
                case 'Rules':
                case 'Hooks':
                case 'Outputs':
                    // these are rendered as a side effect of instantiating the L1s
                    break;
                default:
                    ret[section] = cfnParser.parseValue(this.template[section]);
            }
        }
        return ret;
    }
    createMapping(mappingName) {
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findCondition() { throw new Error('Referring to Conditions in Mapping definitions is not allowed'); },
                findMapping() { throw new Error('Referring to other Mappings in Mapping definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Mapping definitions is not allowed'); },
                findResource() { throw new Error('Using GetAtt expressions in Mapping definitions is not allowed'); },
            },
            parameters: {},
        });
        const cfnMapping = new core.CfnMapping(this.mappingsScope, mappingName, {
            mapping: cfnParser.parseValue(this.template.Mappings[mappingName]),
        });
        this.mappings[mappingName] = cfnMapping;
        this.overrideLogicalIdIfNeeded(cfnMapping, mappingName);
    }
    createParameter(logicalId) {
        if (logicalId in this.parametersToReplace) {
            return;
        }
        const expression = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt expressions in Parameter definitions is not allowed'); },
                findRefTarget() { throw new Error('Using Ref expressions in Parameter definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Parameter definitions is not allowed'); },
                findMapping() { throw new Error('Referring to Mappings in Parameter definitions is not allowed'); },
            },
            parameters: {},
        }).parseValue(this.template.Parameters[logicalId]);
        const cfnParameter = new core.CfnParameter(this, logicalId, {
            type: expression.Type,
            default: expression.Default,
            allowedPattern: expression.AllowedPattern,
            allowedValues: expression.AllowedValues,
            constraintDescription: expression.ConstraintDescription,
            description: expression.Description,
            maxLength: expression.MaxLength,
            maxValue: expression.MaxValue,
            minLength: expression.MinLength,
            minValue: expression.MinValue,
            noEcho: expression.NoEcho,
        });
        this.overrideLogicalIdIfNeeded(cfnParameter, logicalId);
        this.parameters[logicalId] = cfnParameter;
    }
    createRule(ruleName) {
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findRefTarget(refTarget) {
                    // only parameters can be referenced in Rules
                    return self.parameters[refTarget];
                },
                findResource() { throw new Error('Using GetAtt expressions in Rule definitions is not allowed'); },
                findCondition() { throw new Error('Referring to Conditions in Rule definitions is not allowed'); },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
            context: cfn_parse.CfnParsingContext.RULES,
        });
        const ruleProperties = cfnParser.parseValue(this.template.Rules[ruleName]);
        const rule = new core.CfnRule(this.rulesScope, ruleName, {
            ruleCondition: ruleProperties.RuleCondition,
            assertions: ruleProperties.Assertions,
        });
        this.rules[ruleName] = rule;
        this.overrideLogicalIdIfNeeded(rule, ruleName);
    }
    createHook(hookName) {
        var _a;
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition(conditionName) {
                    return self.conditions[conditionName];
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        });
        const hookAttributes = this.template.Hooks[hookName];
        let hook;
        switch (hookAttributes.Type) {
            case 'AWS::CodeDeploy::BlueGreen':
                hook = core.CfnCodeDeployBlueGreenHook._fromCloudFormation(this.hooksScope, hookName, hookAttributes, {
                    parser: cfnParser,
                });
                break;
            default: {
                const hookProperties = (_a = cfnParser.parseValue(hookAttributes.Properties)) !== null && _a !== void 0 ? _a : {};
                hook = new core.CfnHook(this.hooksScope, hookName, {
                    type: hookAttributes.Type,
                    properties: hookProperties,
                });
            }
        }
        this.hooks[hookName] = hook;
        this.overrideLogicalIdIfNeeded(hook, hookName);
    }
    createOutput(logicalId, scope) {
        const self = this;
        const outputAttributes = new cfn_parse.CfnParser({
            finder: {
                findResource(lId) {
                    return self.resources[lId];
                },
                findRefTarget(elementName) {
                    var _a;
                    return (_a = self.resources[elementName]) !== null && _a !== void 0 ? _a : self.parameters[elementName];
                },
                findCondition() {
                    return undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            parameters: this.parametersToReplace,
        }).parseValue(this.template.Outputs[logicalId]);
        const cfnOutput = new core.CfnOutput(scope, logicalId, {
            value: outputAttributes.Value,
            description: outputAttributes.Description,
            exportName: outputAttributes.Export ? outputAttributes.Export.Name : undefined,
            condition: (() => {
                if (!outputAttributes.Condition) {
                    return undefined;
                }
                else if (this.conditions[outputAttributes.Condition]) {
                    return self.getCondition(outputAttributes.Condition);
                }
                throw new Error(`Output with name '${logicalId}' refers to a Condition with name ` +
                    `'${outputAttributes.Condition}' which was not found in this template`);
            })(),
        });
        this.overrideLogicalIdIfNeeded(cfnOutput, logicalId);
        this.outputs[logicalId] = cfnOutput;
    }
    getOrCreateCondition(conditionName) {
        if (conditionName in this.conditions) {
            return this.conditions[conditionName];
        }
        const self = this;
        const cfnParser = new cfn_parse.CfnParser({
            finder: {
                findResource() { throw new Error('Using GetAtt in Condition definitions is not allowed'); },
                findRefTarget(elementName) {
                    // only Parameters can be referenced in the 'Conditions' section
                    return self.parameters[elementName];
                },
                findCondition(cName) {
                    return cName in (self.template.Conditions || {})
                        ? self.getOrCreateCondition(cName)
                        : undefined;
                },
                findMapping(mappingName) {
                    return self.mappings[mappingName];
                },
            },
            context: cfn_parse.CfnParsingContext.CONDITIONS,
            parameters: this.parametersToReplace,
        });
        const cfnCondition = new core.CfnCondition(this.conditionsScope, conditionName, {
            expression: cfnParser.parseValue(this.template.Conditions[conditionName]),
        });
        this.overrideLogicalIdIfNeeded(cfnCondition, conditionName);
        this.conditions[conditionName] = cfnCondition;
        return cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        // fail early for resource attributes we don't support yet
        const knownAttributes = [
            'Type', 'Properties', 'Condition', 'DependsOn', 'Metadata',
            'CreationPolicy', 'UpdatePolicy', 'DeletionPolicy', 'UpdateReplacePolicy',
        ];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The ${attribute} resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findMapping(mappingName) {
                return self.mappings[mappingName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
            findRefTarget(elementName) {
                if (elementName in self.parameters) {
                    return self.parameters[elementName];
                }
                return this.findResource(elementName);
            },
        };
        const cfnParser = new cfn_parse.CfnParser({
            finder,
            parameters: this.parametersToReplace,
        });
        let l1Instance;
        if (this.nestedStacksToInclude[logicalId]) {
            l1Instance = this.createNestedStack(logicalId, cfnParser);
        }
        else {
            const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
            if (l1ClassFqn) {
                const options = {
                    parser: cfnParser,
                };
                const [moduleName, ...className] = l1ClassFqn.split('.');
                const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
                const jsClassFromModule = module[className.join('.')];
                l1Instance = jsClassFromModule._fromCloudFormation(this, logicalId, resourceAttributes, options);
            }
            else {
                l1Instance = new core.CfnResource(this, logicalId, {
                    type: resourceAttributes.Type,
                    properties: cfnParser.parseValue(resourceAttributes.Properties),
                });
                cfnParser.handleAttributes(l1Instance, resourceAttributes, logicalId);
            }
        }
        this.overrideLogicalIdIfNeeded(l1Instance, logicalId);
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
    createNestedStack(nestedStackId, cfnParser) {
        const templateResources = this.template.Resources || {};
        const nestedStackAttributes = templateResources[nestedStackId] || {};
        if (nestedStackAttributes.Type !== 'AWS::CloudFormation::Stack') {
            throw new Error(`Nested Stack with logical ID '${nestedStackId}' is not an AWS::CloudFormation::Stack resource`);
        }
        if (nestedStackAttributes.CreationPolicy) {
            throw new Error('CreationPolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        if (nestedStackAttributes.UpdatePolicy) {
            throw new Error('UpdatePolicy is not supported by the AWS::CloudFormation::Stack resource');
        }
        const nestedStackProps = cfnParser.parseValue(nestedStackAttributes.Properties);
        const nestedStack = new core.NestedStack(this, nestedStackId, {
            parameters: this.parametersForNestedStack(nestedStackProps.Parameters, nestedStackId),
            notificationArns: nestedStackProps.NotificationArns,
            timeout: nestedStackProps.Timeout,
        });
        const template = new CfnInclude(nestedStack, nestedStackId, this.nestedStacksToInclude[nestedStackId]);
        this.nestedStacks[nestedStackId] = { stack: nestedStack, includedTemplate: template };
        // we know this is never undefined for nested stacks
        const nestedStackResource = nestedStack.nestedStackResource;
        cfnParser.handleAttributes(nestedStackResource, nestedStackAttributes, nestedStackId);
        return nestedStackResource;
    }
    parametersForNestedStack(parameters, nestedStackId) {
        var _a;
        if (parameters == null) {
            return undefined;
        }
        const parametersToReplace = (_a = this.nestedStacksToInclude[nestedStackId].parameters) !== null && _a !== void 0 ? _a : {};
        const ret = {};
        for (const paramName of Object.keys(parameters)) {
            if (!(paramName in parametersToReplace)) {
                ret[paramName] = parameters[paramName];
            }
        }
        return ret;
    }
    overrideLogicalIdIfNeeded(element, id) {
        if (this.preserveLogicalIds) {
            element.overrideLogicalId(id);
        }
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2ZuLWluY2x1ZGUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJjZm4taW5jbHVkZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxtQ0FBbUMsQ0FBQyxnREFBZ0Q7QUFDcEYsc0RBQXNELENBQUMsOERBQThEO0FBQ3JILG1FQUFtRTtBQUNuRSx1Q0FBdUM7QUFtRXZDOzs7O0dBSUc7QUFDSCxNQUFhLFVBQVcsU0FBUSxJQUFJLENBQUMsVUFBVTtJQXFDM0MsWUFBWSxLQUFxQixFQUFFLEVBQVUsRUFBRSxLQUFzQjs7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQXJDSixlQUFVLEdBRXZCLEVBQUUsQ0FBQztRQUVVLGNBQVMsR0FFdEIsRUFBRSxDQUFDO1FBQ1UsZUFBVSxHQUV2QixFQUFFLENBQUM7UUFLVSxhQUFRLEdBRXJCLEVBQUUsQ0FBQztRQUNVLFVBQUssR0FFbEIsRUFBRSxDQUFDO1FBRVUsVUFBSyxHQUVsQixFQUFFLENBQUM7UUFFVSxZQUFPLEdBRXBCLEVBQUUsQ0FBQztRQUNVLGlCQUFZLEdBRXpCLEVBQUUsQ0FBQztRQVFILElBQUksQ0FBQyxtQkFBbUIsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FBQztRQUNsRCxxQ0FBcUM7UUFDckMsSUFBSSxDQUFDLFFBQVEsR0FBRyxNQUFNLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsa0JBQWtCLFNBQUcsS0FBSyxDQUFDLGtCQUFrQixtQ0FBSSxJQUFJLENBQUM7UUFDM0QscUVBQXFFO1FBQ3JFLEtBQUssTUFBTSxTQUFTLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsbUJBQW1CLENBQUMsRUFBRTtZQUMzRCxJQUFJLENBQUMsQ0FBQyxTQUFTLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUMsQ0FBQyxFQUFFO2dCQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLDhCQUE4QixTQUFTLGlDQUFpQyxDQUFDLENBQUM7YUFDN0Y7U0FDSjtRQUNELDJCQUEyQjtRQUMzQixJQUFJLENBQUMsYUFBYSxHQUFHLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsV0FBVyxDQUFDLENBQUM7UUFDM0QsS0FBSyxNQUFNLFdBQVcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsUUFBUSxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQ2pFLElBQUksQ0FBQyxhQUFhLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDbkM7UUFDRCw2QkFBNkI7UUFDN0IsS0FBSyxNQUFNLFNBQVMsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQ2pFLElBQUksQ0FBQyxlQUFlLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDbkM7UUFDRCw2QkFBNkI7UUFDN0IsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLGFBQWEsQ0FBQyxDQUFDO1FBQy9ELEtBQUssTUFBTSxhQUFhLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUMsRUFBRTtZQUNyRSxJQUFJLENBQUMsb0JBQW9CLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDNUM7UUFDRCx3QkFBd0I7UUFDeEIsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLFFBQVEsQ0FBQyxDQUFDO1FBQ3JELEtBQUssTUFBTSxRQUFRLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEtBQUssSUFBSSxFQUFFLENBQUMsRUFBRTtZQUMzRCxJQUFJLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1NBQzdCO1FBQ0QsSUFBSSxDQUFDLHFCQUFxQixHQUFHLEtBQUssQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDO1FBQ3RELDhDQUE4QztRQUM5QyxLQUFLLE1BQU0sU0FBUyxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLElBQUksRUFBRSxDQUFDLEVBQUU7WUFDaEUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLFNBQVMsQ0FBQyxDQUFDO1NBQ3ZDO1FBQ0Qsc0RBQXNEO1FBQ3RELEtBQUssTUFBTSxhQUFhLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQy9ELElBQUksQ0FBQyxDQUFDLGFBQWEsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLEVBQUU7Z0JBQ3BDLE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLGFBQWEsaUNBQWlDLENBQUMsQ0FBQzthQUNwRztTQUNKO1FBQ0Qsd0JBQXdCO1FBQ3hCLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztRQUNyRCxLQUFLLE1BQU0sUUFBUSxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxLQUFLLElBQUksRUFBRSxDQUFDLEVBQUU7WUFDM0QsSUFBSSxDQUFDLFVBQVUsQ0FBQyxRQUFRLENBQUMsQ0FBQztTQUM3QjtRQUNELE1BQU0sV0FBVyxHQUFHLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDeEQsS0FBSyxNQUFNLFNBQVMsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQzlELElBQUksQ0FBQyxZQUFZLENBQUMsU0FBUyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1NBQzdDO0lBQ0wsQ0FBQztJQUNEOzs7Ozs7Ozs7Ozs7Ozs7T0FlRztJQUNJLFdBQVcsQ0FBQyxTQUFpQjtRQUNoQyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3RDLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDTixNQUFNLElBQUksS0FBSyxDQUFDLDZCQUE2QixTQUFTLGlDQUFpQyxDQUFDLENBQUM7U0FDNUY7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7SUFDRDs7Ozs7Ozs7O09BU0c7SUFDSSxZQUFZLENBQUMsYUFBcUI7UUFDckMsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ04sTUFBTSxJQUFJLEtBQUssQ0FBQyx3QkFBd0IsYUFBYSxpQ0FBaUMsQ0FBQyxDQUFDO1NBQzNGO1FBQ0QsT0FBTyxHQUFHLENBQUM7SUFDZixDQUFDO0lBQ0Q7Ozs7Ozs7OztPQVNHO0lBQ0ksWUFBWSxDQUFDLGFBQXFCO1FBQ3JDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDM0MsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNOLE1BQU0sSUFBSSxLQUFLLENBQUMsd0JBQXdCLGFBQWEsaUNBQWlDLENBQUMsQ0FBQztTQUMzRjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2YsQ0FBQztJQUNEOzs7Ozs7OztPQVFHO0lBQ0ksVUFBVSxDQUFDLFdBQW1CO1FBQ2pDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNOLE1BQU0sSUFBSSxLQUFLLENBQUMsc0JBQXNCLFdBQVcsaUNBQWlDLENBQUMsQ0FBQztTQUN2RjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2YsQ0FBQztJQUNEOzs7Ozs7Ozs7T0FTRztJQUNJLFNBQVMsQ0FBQyxTQUFpQjtRQUM5QixNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3BDLElBQUksQ0FBQyxHQUFHLEVBQUU7WUFDTixNQUFNLElBQUksS0FBSyxDQUFDLDJCQUEyQixTQUFTLGlDQUFpQyxDQUFDLENBQUM7U0FDMUY7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7SUFDRDs7Ozs7Ozs7O09BU0c7SUFDSSxPQUFPLENBQUMsUUFBZ0I7UUFDM0IsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUNqQyxJQUFJLENBQUMsR0FBRyxFQUFFO1lBQ04sTUFBTSxJQUFJLEtBQUssQ0FBQyxtQkFBbUIsUUFBUSxpQ0FBaUMsQ0FBQyxDQUFDO1NBQ2pGO1FBQ0QsT0FBTyxHQUFHLENBQUM7SUFDZixDQUFDO0lBQ0Q7Ozs7Ozs7OztPQVNHO0lBQ0ksT0FBTyxDQUFDLGFBQXFCO1FBQ2hDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDdEMsSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUNOLE1BQU0sSUFBSSxLQUFLLENBQUMseUJBQXlCLGFBQWEsaUNBQWlDLENBQUMsQ0FBQztTQUM1RjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2YsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLGNBQWMsQ0FBQyxTQUFpQjtRQUNuQyxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUMvQixJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLEVBQUU7Z0JBQ3JDLE1BQU0sSUFBSSxLQUFLLENBQUMsaUNBQWlDLFNBQVMsaUNBQWlDLENBQUMsQ0FBQzthQUNoRztpQkFDSSxJQUFJLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDLElBQUksS0FBSyw0QkFBNEIsRUFBRTtnQkFDL0UsTUFBTSxJQUFJLEtBQUssQ0FBQyw2QkFBNkIsU0FBUyxpQ0FBaUMsQ0FBQyxDQUFDO2FBQzVGO2lCQUNJO2dCQUNELE1BQU0sSUFBSSxLQUFLLENBQUMsaUJBQWlCLFNBQVMsb0ZBQW9GLENBQUMsQ0FBQzthQUNuSTtTQUNKO1FBQ0QsT0FBTyxJQUFJLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ3hDLENBQUM7SUFDRCxnQkFBZ0I7SUFDVCxpQkFBaUI7UUFDcEIsTUFBTSxHQUFHLEdBRUwsRUFBRSxDQUFDO1FBQ1AsS0FBSyxNQUFNLE9BQU8sSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUM5QyxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7WUFDbEIsTUFBTSxNQUFNLEdBQXlCO2dCQUNqQyxZQUFZLENBQUMsR0FBRztvQkFDWixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQy9CLENBQUM7Z0JBQ0QsYUFBYSxDQUFDLFdBQW1COztvQkFDN0IsYUFBTyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxtQ0FBSSxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxDQUFDO2dCQUN2RSxDQUFDO2dCQUNELGFBQWEsQ0FBQyxhQUFxQjtvQkFDL0IsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO2dCQUMxQyxDQUFDO2dCQUNELFdBQVcsQ0FBQyxXQUFXO29CQUNuQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUM7Z0JBQ3RDLENBQUM7YUFDSixDQUFDO1lBQ0YsTUFBTSxTQUFTLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO2dCQUN0QyxNQUFNO2dCQUNOLFVBQVUsRUFBRSxJQUFJLENBQUMsbUJBQW1CO2FBQ3ZDLENBQUMsQ0FBQztZQUNILFFBQVEsT0FBTyxFQUFFO2dCQUNiLEtBQUssWUFBWSxDQUFDO2dCQUNsQixLQUFLLFVBQVUsQ0FBQztnQkFDaEIsS0FBSyxXQUFXLENBQUM7Z0JBQ2pCLEtBQUssWUFBWSxDQUFDO2dCQUNsQixLQUFLLE9BQU8sQ0FBQztnQkFDYixLQUFLLE9BQU8sQ0FBQztnQkFDYixLQUFLLFNBQVM7b0JBQ1YsK0RBQStEO29CQUMvRCxNQUFNO2dCQUNWO29CQUNJLEdBQUcsQ0FBQyxPQUFPLENBQUMsR0FBRyxTQUFTLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQzthQUNuRTtTQUNKO1FBQ0QsT0FBTyxHQUFHLENBQUM7SUFDZixDQUFDO0lBQ08sYUFBYSxDQUFDLFdBQW1CO1FBQ3JDLE1BQU0sU0FBUyxHQUFHLElBQUksU0FBUyxDQUFDLFNBQVMsQ0FBQztZQUN0QyxNQUFNLEVBQUU7Z0JBQ0osYUFBYSxLQUFLLE1BQU0sSUFBSSxLQUFLLENBQUMsK0RBQStELENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3JHLFdBQVcsS0FBSyxNQUFNLElBQUksS0FBSyxDQUFDLG1FQUFtRSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN2RyxhQUFhLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQyw2REFBNkQsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDbkcsWUFBWSxLQUFLLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLENBQUMsQ0FBQyxDQUFDLENBQUM7YUFDeEc7WUFDRCxVQUFVLEVBQUUsRUFBRTtTQUNqQixDQUFDLENBQUM7UUFDSCxNQUFNLFVBQVUsR0FBRyxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRSxXQUFXLEVBQUU7WUFDcEUsT0FBTyxFQUFFLFNBQVMsQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUM7U0FDckUsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsR0FBRyxVQUFVLENBQUM7UUFDeEMsSUFBSSxDQUFDLHlCQUF5QixDQUFDLFVBQVUsRUFBRSxXQUFXLENBQUMsQ0FBQztJQUM1RCxDQUFDO0lBQ08sZUFBZSxDQUFDLFNBQWlCO1FBQ3JDLElBQUksU0FBUyxJQUFJLElBQUksQ0FBQyxtQkFBbUIsRUFBRTtZQUN2QyxPQUFPO1NBQ1Y7UUFDRCxNQUFNLFVBQVUsR0FBRyxJQUFJLFNBQVMsQ0FBQyxTQUFTLENBQUM7WUFDdkMsTUFBTSxFQUFFO2dCQUNKLFlBQVksS0FBSyxNQUFNLElBQUksS0FBSyxDQUFDLGtFQUFrRSxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUN2RyxhQUFhLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQywrREFBK0QsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDckcsYUFBYSxLQUFLLE1BQU0sSUFBSSxLQUFLLENBQUMsaUVBQWlFLENBQUMsQ0FBQyxDQUFDLENBQUM7Z0JBQ3ZHLFdBQVcsS0FBSyxNQUFNLElBQUksS0FBSyxDQUFDLCtEQUErRCxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQ3RHO1lBQ0QsVUFBVSxFQUFFLEVBQUU7U0FDakIsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ25ELE1BQU0sWUFBWSxHQUFHLElBQUksSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQ3hELElBQUksRUFBRSxVQUFVLENBQUMsSUFBSTtZQUNyQixPQUFPLEVBQUUsVUFBVSxDQUFDLE9BQU87WUFDM0IsY0FBYyxFQUFFLFVBQVUsQ0FBQyxjQUFjO1lBQ3pDLGFBQWEsRUFBRSxVQUFVLENBQUMsYUFBYTtZQUN2QyxxQkFBcUIsRUFBRSxVQUFVLENBQUMscUJBQXFCO1lBQ3ZELFdBQVcsRUFBRSxVQUFVLENBQUMsV0FBVztZQUNuQyxTQUFTLEVBQUUsVUFBVSxDQUFDLFNBQVM7WUFDL0IsUUFBUSxFQUFFLFVBQVUsQ0FBQyxRQUFRO1lBQzdCLFNBQVMsRUFBRSxVQUFVLENBQUMsU0FBUztZQUMvQixRQUFRLEVBQUUsVUFBVSxDQUFDLFFBQVE7WUFDN0IsTUFBTSxFQUFFLFVBQVUsQ0FBQyxNQUFNO1NBQzVCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxZQUFZLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDeEQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsR0FBRyxZQUFZLENBQUM7SUFDOUMsQ0FBQztJQUNPLFVBQVUsQ0FBQyxRQUFnQjtRQUMvQixNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsTUFBTSxTQUFTLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO1lBQ3RDLE1BQU0sRUFBRTtnQkFDSixhQUFhLENBQUMsU0FBaUI7b0JBQzNCLDZDQUE2QztvQkFDN0MsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxDQUFDO2dCQUN0QyxDQUFDO2dCQUNELFlBQVksS0FBSyxNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxDQUFDLENBQUMsQ0FBQyxDQUFDO2dCQUNsRyxhQUFhLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQyw0REFBNEQsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDbEcsV0FBVyxDQUFDLFdBQW1CO29CQUMzQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUM7Z0JBQ3RDLENBQUM7YUFDSjtZQUNELFVBQVUsRUFBRSxJQUFJLENBQUMsbUJBQW1CO1lBQ3BDLE9BQU8sRUFBRSxTQUFTLENBQUMsaUJBQWlCLENBQUMsS0FBSztTQUM3QyxDQUFDLENBQUM7UUFDSCxNQUFNLGNBQWMsR0FBRyxTQUFTLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUM7UUFDM0UsTUFBTSxJQUFJLEdBQUcsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsUUFBUSxFQUFFO1lBQ3JELGFBQWEsRUFBRSxjQUFjLENBQUMsYUFBYTtZQUMzQyxVQUFVLEVBQUUsY0FBYyxDQUFDLFVBQVU7U0FDeEMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsR0FBRyxJQUFJLENBQUM7UUFDNUIsSUFBSSxDQUFDLHlCQUF5QixDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBQ08sVUFBVSxDQUFDLFFBQWdCOztRQUMvQixNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsTUFBTSxTQUFTLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO1lBQ3RDLE1BQU0sRUFBRTtnQkFDSixZQUFZLENBQUMsR0FBRztvQkFDWixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQy9CLENBQUM7Z0JBQ0QsYUFBYSxDQUFDLFdBQW1COztvQkFDN0IsYUFBTyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxtQ0FBSSxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxDQUFDO2dCQUN2RSxDQUFDO2dCQUNELGFBQWEsQ0FBQyxhQUFxQjtvQkFDL0IsT0FBTyxJQUFJLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO2dCQUMxQyxDQUFDO2dCQUNELFdBQVcsQ0FBQyxXQUFXO29CQUNuQixPQUFPLElBQUksQ0FBQyxRQUFRLENBQUMsV0FBVyxDQUFDLENBQUM7Z0JBQ3RDLENBQUM7YUFDSjtZQUNELFVBQVUsRUFBRSxJQUFJLENBQUMsbUJBQW1CO1NBQ3ZDLENBQUMsQ0FBQztRQUNILE1BQU0sY0FBYyxHQUFHLElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3JELElBQUksSUFBa0IsQ0FBQztRQUN2QixRQUFRLGNBQWMsQ0FBQyxJQUFJLEVBQUU7WUFDekIsS0FBSyw0QkFBNEI7Z0JBQzdCLElBQUksR0FBSSxJQUFJLENBQUMsMEJBQWtDLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxRQUFRLEVBQUUsY0FBYyxFQUFFO29CQUMzRyxNQUFNLEVBQUUsU0FBUztpQkFDcEIsQ0FBQyxDQUFDO2dCQUNILE1BQU07WUFDVixPQUFPLENBQUMsQ0FBQztnQkFDTCxNQUFNLGNBQWMsU0FBRyxTQUFTLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FBQyxVQUFVLENBQUMsbUNBQUksRUFBRSxDQUFDO2dCQUM3RSxJQUFJLEdBQUcsSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsUUFBUSxFQUFFO29CQUMvQyxJQUFJLEVBQUUsY0FBYyxDQUFDLElBQUk7b0JBQ3pCLFVBQVUsRUFBRSxjQUFjO2lCQUM3QixDQUFDLENBQUM7YUFDTjtTQUNKO1FBQ0QsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsR0FBRyxJQUFJLENBQUM7UUFDNUIsSUFBSSxDQUFDLHlCQUF5QixDQUFDLElBQUksRUFBRSxRQUFRLENBQUMsQ0FBQztJQUNuRCxDQUFDO0lBQ08sWUFBWSxDQUFDLFNBQWlCLEVBQUUsS0FBcUI7UUFDekQsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO1lBQzdDLE1BQU0sRUFBRTtnQkFDSixZQUFZLENBQUMsR0FBRztvQkFDWixPQUFPLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUM7Z0JBQy9CLENBQUM7Z0JBQ0QsYUFBYSxDQUFDLFdBQW1COztvQkFDN0IsYUFBTyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxtQ0FBSSxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsQ0FBQyxDQUFDO2dCQUN2RSxDQUFDO2dCQUNELGFBQWE7b0JBQ1QsT0FBTyxTQUFTLENBQUM7Z0JBQ3JCLENBQUM7Z0JBQ0QsV0FBVyxDQUFDLFdBQVc7b0JBQ25CLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDdEMsQ0FBQzthQUNKO1lBQ0QsVUFBVSxFQUFFLElBQUksQ0FBQyxtQkFBbUI7U0FDdkMsQ0FBQyxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1FBQ2hELE1BQU0sU0FBUyxHQUFHLElBQUksSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLEVBQUUsU0FBUyxFQUFFO1lBQ25ELEtBQUssRUFBRSxnQkFBZ0IsQ0FBQyxLQUFLO1lBQzdCLFdBQVcsRUFBRSxnQkFBZ0IsQ0FBQyxXQUFXO1lBQ3pDLFVBQVUsRUFBRSxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDOUUsU0FBUyxFQUFFLENBQUMsR0FBRyxFQUFFO2dCQUNiLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLEVBQUU7b0JBQzdCLE9BQU8sU0FBUyxDQUFDO2lCQUNwQjtxQkFDSSxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsZ0JBQWdCLENBQUMsU0FBUyxDQUFDLEVBQUU7b0JBQ2xELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsQ0FBQztpQkFDeEQ7Z0JBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxxQkFBcUIsU0FBUyxvQ0FBb0M7b0JBQzlFLElBQUksZ0JBQWdCLENBQUMsU0FBUyx3Q0FBd0MsQ0FBQyxDQUFDO1lBQ2hGLENBQUMsQ0FBQyxFQUFFO1NBQ1AsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLHlCQUF5QixDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNyRCxJQUFJLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsQ0FBQztJQUN4QyxDQUFDO0lBQ08sb0JBQW9CLENBQUMsYUFBcUI7UUFDOUMsSUFBSSxhQUFhLElBQUksSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNsQyxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDekM7UUFDRCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsTUFBTSxTQUFTLEdBQUcsSUFBSSxTQUFTLENBQUMsU0FBUyxDQUFDO1lBQ3RDLE1BQU0sRUFBRTtnQkFDSixZQUFZLEtBQUssTUFBTSxJQUFJLEtBQUssQ0FBQyxzREFBc0QsQ0FBQyxDQUFDLENBQUMsQ0FBQztnQkFDM0YsYUFBYSxDQUFDLFdBQW1CO29CQUM3QixnRUFBZ0U7b0JBQ2hFLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLENBQUMsQ0FBQztnQkFDeEMsQ0FBQztnQkFDRCxhQUFhLENBQUMsS0FBYTtvQkFDdkIsT0FBTyxLQUFLLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUM7d0JBQzVDLENBQUMsQ0FBQyxJQUFJLENBQUMsb0JBQW9CLENBQUMsS0FBSyxDQUFDO3dCQUNsQyxDQUFDLENBQUMsU0FBUyxDQUFDO2dCQUNwQixDQUFDO2dCQUNELFdBQVcsQ0FBQyxXQUFtQjtvQkFDM0IsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLFdBQVcsQ0FBQyxDQUFDO2dCQUN0QyxDQUFDO2FBQ0o7WUFDRCxPQUFPLEVBQUUsU0FBUyxDQUFDLGlCQUFpQixDQUFDLFVBQVU7WUFDL0MsVUFBVSxFQUFFLElBQUksQ0FBQyxtQkFBbUI7U0FDdkMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxZQUFZLEdBQUcsSUFBSSxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsYUFBYSxFQUFFO1lBQzVFLFVBQVUsRUFBRSxTQUFTLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxDQUFDO1NBQzVFLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxZQUFZLEVBQUUsYUFBYSxDQUFDLENBQUM7UUFDNUQsSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsR0FBRyxZQUFZLENBQUM7UUFDOUMsT0FBTyxZQUFZLENBQUM7SUFDeEIsQ0FBQztJQUNPLG1CQUFtQixDQUFDLFNBQWlCO1FBQ3pDLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdEMsSUFBSSxHQUFHLEVBQUU7WUFDTCxPQUFPLEdBQUcsQ0FBQztTQUNkO1FBQ0QsTUFBTSxrQkFBa0IsR0FBUSxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUNuRSwwREFBMEQ7UUFDMUQsTUFBTSxlQUFlLEdBQUc7WUFDcEIsTUFBTSxFQUFFLFlBQVksRUFBRSxXQUFXLEVBQUUsV0FBVyxFQUFFLFVBQVU7WUFDMUQsZ0JBQWdCLEVBQUUsY0FBYyxFQUFFLGdCQUFnQixFQUFFLHFCQUFxQjtTQUM1RSxDQUFDO1FBQ0YsS0FBSyxNQUFNLFNBQVMsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEVBQUU7WUFDckQsSUFBSSxDQUFDLGVBQWUsQ0FBQyxRQUFRLENBQUMsU0FBUyxDQUFDLEVBQUU7Z0JBQ3RDLE1BQU0sSUFBSSxLQUFLLENBQUMsT0FBTyxTQUFTLHNFQUFzRTtvQkFDbEcsZ0dBQWdHLENBQUMsQ0FBQzthQUN6RztTQUNKO1FBQ0QsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDO1FBQ2xCLE1BQU0sTUFBTSxHQUF5QjtZQUNqQyxhQUFhLENBQUMsYUFBcUI7Z0JBQy9CLE9BQU8sSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUMxQyxDQUFDO1lBQ0QsV0FBVyxDQUFDLFdBQVc7Z0JBQ25CLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUN0QyxDQUFDO1lBQ0QsWUFBWSxDQUFDLEdBQVc7Z0JBQ3BCLElBQUksQ0FBQyxDQUFDLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsU0FBUyxJQUFJLEVBQUUsQ0FBQyxDQUFDLEVBQUU7b0JBQzNDLE9BQU8sU0FBUyxDQUFDO2lCQUNwQjtnQkFDRCxPQUFPLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLENBQUMsQ0FBQztZQUN6QyxDQUFDO1lBQ0QsYUFBYSxDQUFDLFdBQW1CO2dCQUM3QixJQUFJLFdBQVcsSUFBSSxJQUFJLENBQUMsVUFBVSxFQUFFO29CQUNoQyxPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxDQUFDLENBQUM7aUJBQ3ZDO2dCQUNELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUMxQyxDQUFDO1NBQ0osQ0FBQztRQUNGLE1BQU0sU0FBUyxHQUFHLElBQUksU0FBUyxDQUFDLFNBQVMsQ0FBQztZQUN0QyxNQUFNO1lBQ04sVUFBVSxFQUFFLElBQUksQ0FBQyxtQkFBbUI7U0FDdkMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxVQUE0QixDQUFDO1FBQ2pDLElBQUksSUFBSSxDQUFDLHFCQUFxQixDQUFDLFNBQVMsQ0FBQyxFQUFFO1lBQ3ZDLFVBQVUsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1NBQzdEO2FBQ0k7WUFDRCxNQUFNLFVBQVUsR0FBRyxzQkFBc0IsQ0FBQyxNQUFNLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLENBQUM7WUFDMUUsSUFBSSxVQUFVLEVBQUU7Z0JBQ1osTUFBTSxPQUFPLEdBQXdDO29CQUNqRCxNQUFNLEVBQUUsU0FBUztpQkFDcEIsQ0FBQztnQkFDRixNQUFNLENBQUMsVUFBVSxFQUFFLEdBQUcsU0FBUyxDQUFDLEdBQUcsVUFBVSxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDekQsTUFBTSxNQUFNLEdBQUcsT0FBTyxDQUFDLFVBQVUsQ0FBQyxDQUFDLENBQUMsNERBQTREO2dCQUNoRyxNQUFNLGlCQUFpQixHQUFHLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUM7Z0JBQ3RELFVBQVUsR0FBRyxpQkFBaUIsQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxDQUFDO2FBQ3BHO2lCQUNJO2dCQUNELFVBQVUsR0FBRyxJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtvQkFDL0MsSUFBSSxFQUFFLGtCQUFrQixDQUFDLElBQUk7b0JBQzdCLFVBQVUsRUFBRSxTQUFTLENBQUMsVUFBVSxDQUFDLGtCQUFrQixDQUFDLFVBQVUsQ0FBQztpQkFDbEUsQ0FBQyxDQUFDO2dCQUNILFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxVQUFVLEVBQUUsa0JBQWtCLEVBQUUsU0FBUyxDQUFDLENBQUM7YUFDekU7U0FDSjtRQUNELElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxVQUFVLEVBQUUsU0FBUyxDQUFDLENBQUM7UUFDdEQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxTQUFTLENBQUMsR0FBRyxVQUFVLENBQUM7UUFDdkMsT0FBTyxVQUFVLENBQUM7SUFDdEIsQ0FBQztJQUNPLGlCQUFpQixDQUFDLGFBQXFCLEVBQUUsU0FBOEI7UUFDM0UsTUFBTSxpQkFBaUIsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLFNBQVMsSUFBSSxFQUFFLENBQUM7UUFDeEQsTUFBTSxxQkFBcUIsR0FBRyxpQkFBaUIsQ0FBQyxhQUFhLENBQUMsSUFBSSxFQUFFLENBQUM7UUFDckUsSUFBSSxxQkFBcUIsQ0FBQyxJQUFJLEtBQUssNEJBQTRCLEVBQUU7WUFDN0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQ0FBaUMsYUFBYSxpREFBaUQsQ0FBQyxDQUFDO1NBQ3BIO1FBQ0QsSUFBSSxxQkFBcUIsQ0FBQyxjQUFjLEVBQUU7WUFDdEMsTUFBTSxJQUFJLEtBQUssQ0FBQyw0RUFBNEUsQ0FBQyxDQUFDO1NBQ2pHO1FBQ0QsSUFBSSxxQkFBcUIsQ0FBQyxZQUFZLEVBQUU7WUFDcEMsTUFBTSxJQUFJLEtBQUssQ0FBQywwRUFBMEUsQ0FBQyxDQUFDO1NBQy9GO1FBQ0QsTUFBTSxnQkFBZ0IsR0FBRyxTQUFTLENBQUMsVUFBVSxDQUFDLHFCQUFxQixDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2hGLE1BQU0sV0FBVyxHQUFHLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsYUFBYSxFQUFFO1lBQzFELFVBQVUsRUFBRSxJQUFJLENBQUMsd0JBQXdCLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxFQUFFLGFBQWEsQ0FBQztZQUNyRixnQkFBZ0IsRUFBRSxnQkFBZ0IsQ0FBQyxnQkFBZ0I7WUFDbkQsT0FBTyxFQUFFLGdCQUFnQixDQUFDLE9BQU87U0FDcEMsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxRQUFRLEdBQUcsSUFBSSxVQUFVLENBQUMsV0FBVyxFQUFFLGFBQWEsRUFBRSxJQUFJLENBQUMscUJBQXFCLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQztRQUN2RyxJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUUsS0FBSyxFQUFFLFdBQVcsRUFBRSxnQkFBZ0IsRUFBRSxRQUFRLEVBQUUsQ0FBQztRQUN0RixvREFBb0Q7UUFDcEQsTUFBTSxtQkFBbUIsR0FBcUIsV0FBVyxDQUFDLG1CQUFvQixDQUFDO1FBQy9FLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxtQkFBbUIsRUFBRSxxQkFBcUIsRUFBRSxhQUFhLENBQUMsQ0FBQztRQUN0RixPQUFPLG1CQUFtQixDQUFDO0lBQy9CLENBQUM7SUFDTyx3QkFBd0IsQ0FBQyxVQUFlLEVBQUUsYUFBcUI7O1FBR25FLElBQUksVUFBVSxJQUFJLElBQUksRUFBRTtZQUNwQixPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE1BQU0sbUJBQW1CLFNBQUcsSUFBSSxDQUFDLHFCQUFxQixDQUFDLGFBQWEsQ0FBQyxDQUFDLFVBQVUsbUNBQUksRUFBRSxDQUFDO1FBQ3ZGLE1BQU0sR0FBRyxHQUVMLEVBQUUsQ0FBQztRQUNQLEtBQUssTUFBTSxTQUFTLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUM3QyxJQUFJLENBQUMsQ0FBQyxTQUFTLElBQUksbUJBQW1CLENBQUMsRUFBRTtnQkFDckMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxHQUFHLFVBQVUsQ0FBQyxTQUFTLENBQUMsQ0FBQzthQUMxQztTQUNKO1FBQ0QsT0FBTyxHQUFHLENBQUM7SUFDZixDQUFDO0lBQ08seUJBQXlCLENBQUMsT0FBd0IsRUFBRSxFQUFVO1FBQ2xFLElBQUksSUFBSSxDQUFDLGtCQUFrQixFQUFFO1lBQ3pCLE9BQU8sQ0FBQyxpQkFBaUIsQ0FBQyxFQUFFLENBQUMsQ0FBQztTQUNqQztJQUNMLENBQUM7Q0FDSjtBQXZqQkQsZ0NBdWpCQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNvcmUgZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgKiBhcyBjZm5fcGFyc2UgZnJvbSBcIi4uLy4uL2NvcmUvbGliL2Nmbi1wYXJzZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZS9saWIvY2ZuLXBhcnNlJ1xuaW1wb3J0ICogYXMgY2ZuX3R5cGVfdG9fbDFfbWFwcGluZyBmcm9tICcuL2Nmbi10eXBlLXRvLWwxLW1hcHBpbmcnO1xuaW1wb3J0ICogYXMgZnV0aWxzIGZyb20gJy4vZmlsZS11dGlscyc7XG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIG9mIHtAbGluayBDZm5JbmNsdWRlfS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDZm5JbmNsdWRlUHJvcHMge1xuICAgIC8qKlxuICAgICAqIFBhdGggdG8gdGhlIHRlbXBsYXRlIGZpbGUuXG4gICAgICpcbiAgICAgKiBCb3RoIEpTT04gYW5kIFlBTUwgdGVtcGxhdGUgZm9ybWF0cyBhcmUgc3VwcG9ydGVkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRlbXBsYXRlRmlsZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgdGhlIHJlc291cmNlcyBzaG91bGQgaGF2ZSB0aGUgc2FtZSBsb2dpY2FsIElEcyBpbiB0aGUgcmVzdWx0aW5nIENESyB0ZW1wbGF0ZVxuICAgICAqIGFzIHRoZXkgZGlkIGluIHRoZSBvcmlnaW5hbCBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZSBmaWxlLlxuICAgICAqIElmIHlvdSdyZSB2ZW5kaW5nIGEgQ29uc3RydWN0IHVzaW5nIGFuIGV4aXN0aW5nIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlLFxuICAgICAqIG1ha2Ugc3VyZSB0byBwYXNzIHRoaXMgYXMgYGZhbHNlYC5cbiAgICAgKlxuICAgICAqICoqTm90ZSoqOiByZWdhcmRsZXNzIG9mIHdoZXRoZXIgdGhpcyBvcHRpb24gaXMgdHJ1ZSBvciBmYWxzZSxcbiAgICAgKiB0aGUge0BsaW5rIENmbkluY2x1ZGUuZ2V0UmVzb3VyY2V9IGFuZCByZWxhdGVkIG1ldGhvZHMgYWx3YXlzIHVzZXMgdGhlIG9yaWdpbmFsIGxvZ2ljYWwgSUQgb2YgdGhlIHJlc291cmNlL2VsZW1lbnQsXG4gICAgICogYXMgc3BlY2lmaWVkIGluIHRoZSB0ZW1wbGF0ZSBmaWxlLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByZXNlcnZlTG9naWNhbElkcz86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogU3BlY2lmaWVzIHRoZSB0ZW1wbGF0ZSBmaWxlcyB0aGF0IGRlZmluZSBuZXN0ZWQgc3RhY2tzIHRoYXQgc2hvdWxkIGJlIGluY2x1ZGVkLlxuICAgICAqXG4gICAgICogSWYgeW91ciB0ZW1wbGF0ZSBzcGVjaWZpZXMgYSBzdGFjayB0aGF0IGlzbid0IGluY2x1ZGVkIGhlcmUsIGl0IHdvbid0IGJlIGNyZWF0ZWQgYXMgYSBOZXN0ZWRTdGFja1xuICAgICAqIHJlc291cmNlLCBhbmQgaXQgd29uJ3QgYmUgYWNjZXNzaWJsZSBmcm9tIHtAbGluayBDZm5JbmNsdWRlLmdldE5lc3RlZFN0YWNrfS5cbiAgICAgKlxuICAgICAqIElmIHlvdSBpbmNsdWRlIGEgc3RhY2sgaGVyZSB3aXRoIGFuIElEIHRoYXQgaXNuJ3QgaW4gdGhlIHRlbXBsYXRlLFxuICAgICAqIG9yIGlzIGluIHRoZSB0ZW1wbGF0ZSBidXQgaXMgbm90IGEgbmVzdGVkIHN0YWNrLFxuICAgICAqIHRlbXBsYXRlIGNyZWF0aW9uIHdpbGwgZmFpbCBhbmQgYW4gZXJyb3Igd2lsbCBiZSB0aHJvd24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIG5vIG5lc3RlZCBzdGFja3Mgd2lsbCBiZSBpbmNsdWRlZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IG5lc3RlZFN0YWNrcz86IHtcbiAgICAgICAgW3N0YWNrTmFtZTogc3RyaW5nXTogQ2ZuSW5jbHVkZVByb3BzO1xuICAgIH07XG4gICAgLyoqXG4gICAgICogU3BlY2lmaWVzIHBhcmFtZXRlcnMgdG8gYmUgcmVwbGFjZWQgYnkgdGhlIHZhbHVlcyBpbiB0aGlzIG1hcHBpbmcuXG4gICAgICogQW55IHBhcmFtZXRlcnMgaW4gdGhlIHRlbXBsYXRlIHRoYXQgYXJlbid0IHNwZWNpZmllZCBoZXJlIHdpbGwgYmUgbGVmdCB1bm1vZGlmaWVkLlxuICAgICAqIElmIHlvdSBpbmNsdWRlIGEgcGFyYW1ldGVyIGhlcmUgd2l0aCBhbiBJRCB0aGF0IGlzbid0IGluIHRoZSB0ZW1wbGF0ZSxcbiAgICAgKiB0ZW1wbGF0ZSBjcmVhdGlvbiB3aWxsIGZhaWwgYW5kIGFuIGVycm9yIHdpbGwgYmUgdGhyb3duLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBubyBwYXJhbWV0ZXJzIHdpbGwgYmUgcmVwbGFjZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBwYXJhbWV0ZXJzPzoge1xuICAgICAgICBbcGFyYW1ldGVyTmFtZTogc3RyaW5nXTogYW55O1xuICAgIH07XG59XG4vKipcbiAqIFRoZSB0eXBlIHJldHVybmVkIGZyb20ge0BsaW5rIENmbkluY2x1ZGUuZ2V0TmVzdGVkU3RhY2t9LlxuICogQ29udGFpbnMgYm90aCB0aGUgTmVzdGVkU3RhY2sgb2JqZWN0IGFuZFxuICogQ2ZuSW5jbHVkZSByZXByZXNlbnRhdGlvbnMgb2YgdGhlIGNoaWxkIHN0YWNrLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEluY2x1ZGVkTmVzdGVkU3RhY2sge1xuICAgIC8qKlxuICAgICAqIFRoZSBOZXN0ZWRTdGFjayBvYmplY3Qgd2hpY2ggcmVwcmVzZW50cyB0aGUgc2NvcGUgb2YgdGhlIHRlbXBsYXRlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHN0YWNrOiBjb3JlLk5lc3RlZFN0YWNrO1xuICAgIC8qKlxuICAgICAqIFRoZSBDZm5JbmNsdWRlIHRoYXQgcmVwcmVzZW50cyB0aGUgdGVtcGxhdGUsIHdoaWNoIGNhblxuICAgICAqIGJlIHVzZWQgdG8gYWNjZXNzIFJlc291cmNlcyBhbmQgb3RoZXIgdGVtcGxhdGUgZWxlbWVudHMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5jbHVkZWRUZW1wbGF0ZTogQ2ZuSW5jbHVkZTtcbn1cbi8qKlxuICogQ29uc3RydWN0IHRvIGltcG9ydCBhbiBleGlzdGluZyBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZSBmaWxlIGludG8gYSBDREsgYXBwbGljYXRpb24uXG4gKiBBbGwgcmVzb3VyY2VzIGRlZmluZWQgaW4gdGhlIHRlbXBsYXRlIGZpbGUgY2FuIGJlIHJldHJpZXZlZCBieSBjYWxsaW5nIHRoZSB7QGxpbmsgZ2V0UmVzb3VyY2V9IG1ldGhvZC5cbiAqIEFueSBtb2RpZmljYXRpb25zIG1hZGUgb24gdGhlIHJldHVybmVkIHJlc291cmNlIG9iamVjdHMgd2lsbCBiZSByZWZsZWN0ZWQgaW4gdGhlIHJlc3VsdGluZyBDREsgdGVtcGxhdGUuXG4gKi9cbmV4cG9ydCBjbGFzcyBDZm5JbmNsdWRlIGV4dGVuZHMgY29yZS5DZm5FbGVtZW50IHtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGNvbmRpdGlvbnM6IHtcbiAgICAgICAgW2NvbmRpdGlvbk5hbWU6IHN0cmluZ106IGNvcmUuQ2ZuQ29uZGl0aW9uO1xuICAgIH0gPSB7fTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGNvbmRpdGlvbnNTY29wZTogY29yZS5Db25zdHJ1Y3Q7XG4gICAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZXM6IHtcbiAgICAgICAgW2xvZ2ljYWxJZDogc3RyaW5nXTogY29yZS5DZm5SZXNvdXJjZTtcbiAgICB9ID0ge307XG4gICAgcHJpdmF0ZSByZWFkb25seSBwYXJhbWV0ZXJzOiB7XG4gICAgICAgIFtsb2dpY2FsSWQ6IHN0cmluZ106IGNvcmUuQ2ZuUGFyYW1ldGVyO1xuICAgIH0gPSB7fTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHBhcmFtZXRlcnNUb1JlcGxhY2U6IHtcbiAgICAgICAgW3BhcmFtZXRlck5hbWU6IHN0cmluZ106IGFueTtcbiAgICB9O1xuICAgIHByaXZhdGUgcmVhZG9ubHkgbWFwcGluZ3NTY29wZTogY29yZS5Db25zdHJ1Y3Q7XG4gICAgcHJpdmF0ZSByZWFkb25seSBtYXBwaW5nczoge1xuICAgICAgICBbbWFwcGluZ05hbWU6IHN0cmluZ106IGNvcmUuQ2ZuTWFwcGluZztcbiAgICB9ID0ge307XG4gICAgcHJpdmF0ZSByZWFkb25seSBydWxlczoge1xuICAgICAgICBbcnVsZU5hbWU6IHN0cmluZ106IGNvcmUuQ2ZuUnVsZTtcbiAgICB9ID0ge307XG4gICAgcHJpdmF0ZSByZWFkb25seSBydWxlc1Njb3BlOiBjb3JlLkNvbnN0cnVjdDtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGhvb2tzOiB7XG4gICAgICAgIFtob29rTmFtZTogc3RyaW5nXTogY29yZS5DZm5Ib29rO1xuICAgIH0gPSB7fTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGhvb2tzU2NvcGU6IGNvcmUuQ29uc3RydWN0O1xuICAgIHByaXZhdGUgcmVhZG9ubHkgb3V0cHV0czoge1xuICAgICAgICBbbG9naWNhbElkOiBzdHJpbmddOiBjb3JlLkNmbk91dHB1dDtcbiAgICB9ID0ge307XG4gICAgcHJpdmF0ZSByZWFkb25seSBuZXN0ZWRTdGFja3M6IHtcbiAgICAgICAgW2xvZ2ljYWxJZDogc3RyaW5nXTogSW5jbHVkZWROZXN0ZWRTdGFjaztcbiAgICB9ID0ge307XG4gICAgcHJpdmF0ZSByZWFkb25seSBuZXN0ZWRTdGFja3NUb0luY2x1ZGU6IHtcbiAgICAgICAgW25hbWU6IHN0cmluZ106IENmbkluY2x1ZGVQcm9wcztcbiAgICB9O1xuICAgIHByaXZhdGUgcmVhZG9ubHkgdGVtcGxhdGU6IGFueTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHByZXNlcnZlTG9naWNhbElkczogYm9vbGVhbjtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogY29yZS5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBDZm5JbmNsdWRlUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgdGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlID0gcHJvcHMucGFyYW1ldGVycyB8fCB7fTtcbiAgICAgICAgLy8gcmVhZCB0aGUgdGVtcGxhdGUgaW50byBhIEpTIG9iamVjdFxuICAgICAgICB0aGlzLnRlbXBsYXRlID0gZnV0aWxzLnJlYWRZYW1sU3luYyhwcm9wcy50ZW1wbGF0ZUZpbGUpO1xuICAgICAgICB0aGlzLnByZXNlcnZlTG9naWNhbElkcyA9IHByb3BzLnByZXNlcnZlTG9naWNhbElkcyA/PyB0cnVlO1xuICAgICAgICAvLyBjaGVjayBpZiBhbGwgdXNlciBzcGVjaWZpZWQgcGFyYW1ldGVyIHZhbHVlcyBleGlzdCBpbiB0aGUgdGVtcGxhdGVcbiAgICAgICAgZm9yIChjb25zdCBsb2dpY2FsSWQgb2YgT2JqZWN0LmtleXModGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlKSkge1xuICAgICAgICAgICAgaWYgKCEobG9naWNhbElkIGluICh0aGlzLnRlbXBsYXRlLlBhcmFtZXRlcnMgfHwge30pKSkge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgUGFyYW1ldGVyIHdpdGggbG9naWNhbCBJRCAnJHtsb2dpY2FsSWR9JyB3YXMgbm90IGZvdW5kIGluIHRoZSB0ZW1wbGF0ZWApO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIC8vIGluc3RhbnRpYXRlIHRoZSBNYXBwaW5nc1xuICAgICAgICB0aGlzLm1hcHBpbmdzU2NvcGUgPSBuZXcgY29yZS5Db25zdHJ1Y3QodGhpcywgJyRNYXBwaW5ncycpO1xuICAgICAgICBmb3IgKGNvbnN0IG1hcHBpbmdOYW1lIG9mIE9iamVjdC5rZXlzKHRoaXMudGVtcGxhdGUuTWFwcGluZ3MgfHwge30pKSB7XG4gICAgICAgICAgICB0aGlzLmNyZWF0ZU1hcHBpbmcobWFwcGluZ05hbWUpO1xuICAgICAgICB9XG4gICAgICAgIC8vIGluc3RhbnRpYXRlIGFsbCBwYXJhbWV0ZXJzXG4gICAgICAgIGZvciAoY29uc3QgbG9naWNhbElkIG9mIE9iamVjdC5rZXlzKHRoaXMudGVtcGxhdGUuUGFyYW1ldGVycyB8fCB7fSkpIHtcbiAgICAgICAgICAgIHRoaXMuY3JlYXRlUGFyYW1ldGVyKGxvZ2ljYWxJZCk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gaW5zdGFudGlhdGUgdGhlIGNvbmRpdGlvbnNcbiAgICAgICAgdGhpcy5jb25kaXRpb25zU2NvcGUgPSBuZXcgY29yZS5Db25zdHJ1Y3QodGhpcywgJyRDb25kaXRpb25zJyk7XG4gICAgICAgIGZvciAoY29uc3QgY29uZGl0aW9uTmFtZSBvZiBPYmplY3Qua2V5cyh0aGlzLnRlbXBsYXRlLkNvbmRpdGlvbnMgfHwge30pKSB7XG4gICAgICAgICAgICB0aGlzLmdldE9yQ3JlYXRlQ29uZGl0aW9uKGNvbmRpdGlvbk5hbWUpO1xuICAgICAgICB9XG4gICAgICAgIC8vIGluc3RhbnRpYXRlIHRoZSBydWxlc1xuICAgICAgICB0aGlzLnJ1bGVzU2NvcGUgPSBuZXcgY29yZS5Db25zdHJ1Y3QodGhpcywgJyRSdWxlcycpO1xuICAgICAgICBmb3IgKGNvbnN0IHJ1bGVOYW1lIG9mIE9iamVjdC5rZXlzKHRoaXMudGVtcGxhdGUuUnVsZXMgfHwge30pKSB7XG4gICAgICAgICAgICB0aGlzLmNyZWF0ZVJ1bGUocnVsZU5hbWUpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMubmVzdGVkU3RhY2tzVG9JbmNsdWRlID0gcHJvcHMubmVzdGVkU3RhY2tzIHx8IHt9O1xuICAgICAgICAvLyBpbnN0YW50aWF0ZSBhbGwgcmVzb3VyY2VzIGFzIENESyBMMSBvYmplY3RzXG4gICAgICAgIGZvciAoY29uc3QgbG9naWNhbElkIG9mIE9iamVjdC5rZXlzKHRoaXMudGVtcGxhdGUuUmVzb3VyY2VzIHx8IHt9KSkge1xuICAgICAgICAgICAgdGhpcy5nZXRPckNyZWF0ZVJlc291cmNlKGxvZ2ljYWxJZCk7XG4gICAgICAgIH1cbiAgICAgICAgLy8gdmVyaWZ5IHRoYXQgYWxsIG5lc3RlZFN0YWNrcyBoYXZlIGJlZW4gaW5zdGFudGlhdGVkXG4gICAgICAgIGZvciAoY29uc3QgbmVzdGVkU3RhY2tJZCBvZiBPYmplY3Qua2V5cyhwcm9wcy5uZXN0ZWRTdGFja3MgfHwge30pKSB7XG4gICAgICAgICAgICBpZiAoIShuZXN0ZWRTdGFja0lkIGluIHRoaXMucmVzb3VyY2VzKSkge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgTmVzdGVkIFN0YWNrIHdpdGggbG9naWNhbCBJRCAnJHtuZXN0ZWRTdGFja0lkfScgd2FzIG5vdCBmb3VuZCBpbiB0aGUgdGVtcGxhdGVgKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICAvLyBpbnN0YW50aWF0ZSB0aGUgSG9va3NcbiAgICAgICAgdGhpcy5ob29rc1Njb3BlID0gbmV3IGNvcmUuQ29uc3RydWN0KHRoaXMsICckSG9va3MnKTtcbiAgICAgICAgZm9yIChjb25zdCBob29rTmFtZSBvZiBPYmplY3Qua2V5cyh0aGlzLnRlbXBsYXRlLkhvb2tzIHx8IHt9KSkge1xuICAgICAgICAgICAgdGhpcy5jcmVhdGVIb29rKGhvb2tOYW1lKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBvdXRwdXRTY29wZSA9IG5ldyBjb3JlLkNvbnN0cnVjdCh0aGlzLCAnJE91cHV0cycpO1xuICAgICAgICBmb3IgKGNvbnN0IGxvZ2ljYWxJZCBvZiBPYmplY3Qua2V5cyh0aGlzLnRlbXBsYXRlLk91dHB1dHMgfHwge30pKSB7XG4gICAgICAgICAgICB0aGlzLmNyZWF0ZU91dHB1dChsb2dpY2FsSWQsIG91dHB1dFNjb3BlKTtcbiAgICAgICAgfVxuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm5zIHRoZSBsb3ctbGV2ZWwgQ2ZuUmVzb3VyY2UgZnJvbSB0aGUgdGVtcGxhdGUgd2l0aCB0aGUgZ2l2ZW4gbG9naWNhbCBJRC5cbiAgICAgKiBBbnkgbW9kaWZpY2F0aW9ucyBwZXJmb3JtZWQgb24gdGhhdCByZXNvdXJjZSB3aWxsIGJlIHJlZmxlY3RlZCBpbiB0aGUgcmVzdWx0aW5nIENESyB0ZW1wbGF0ZS5cbiAgICAgKlxuICAgICAqIFRoZSByZXR1cm5lZCBvYmplY3Qgd2lsbCBiZSBvZiB0aGUgcHJvcGVyIHVuZGVybHlpbmcgY2xhc3M7XG4gICAgICogeW91IGNhbiBhbHdheXMgY2FzdCBpdCB0byB0aGUgY29ycmVjdCB0eXBlIGluIHlvdXIgY29kZTpcbiAgICAgKlxuICAgICAqICAgICAvLyBhc3N1bWUgdGhlIHRlbXBsYXRlIGNvbnRhaW5zIGFuIEFXUzo6UzM6OkJ1Y2tldCB3aXRoIGxvZ2ljYWwgSUQgJ0J1Y2tldCdcbiAgICAgKiAgICAgY29uc3QgY2ZuQnVja2V0ID0gY2ZuVGVtcGxhdGUuZ2V0UmVzb3VyY2UoJ0J1Y2tldCcpIGFzIHMzLkNmbkJ1Y2tldDtcbiAgICAgKiAgICAgLy8gY2ZuQnVja2V0IGlzIG9mIHR5cGUgczMuQ2ZuQnVja2V0XG4gICAgICpcbiAgICAgKiBJZiB0aGUgdGVtcGxhdGUgZG9lcyBub3QgY29udGFpbiBhIHJlc291cmNlIHdpdGggdGhlIGdpdmVuIGxvZ2ljYWwgSUQsXG4gICAgICogYW4gZXhjZXB0aW9uIHdpbGwgYmUgdGhyb3duLlxuICAgICAqXG4gICAgICogQHBhcmFtIGxvZ2ljYWxJZCB0aGUgbG9naWNhbCBJRCBvZiB0aGUgcmVzb3VyY2UgaW4gdGhlIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlIGZpbGVcbiAgICAgKi9cbiAgICBwdWJsaWMgZ2V0UmVzb3VyY2UobG9naWNhbElkOiBzdHJpbmcpOiBjb3JlLkNmblJlc291cmNlIHtcbiAgICAgICAgY29uc3QgcmV0ID0gdGhpcy5yZXNvdXJjZXNbbG9naWNhbElkXTtcbiAgICAgICAgaWYgKCFyZXQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgUmVzb3VyY2Ugd2l0aCBsb2dpY2FsIElEICcke2xvZ2ljYWxJZH0nIHdhcyBub3QgZm91bmQgaW4gdGhlIHRlbXBsYXRlYCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJucyB0aGUgQ2ZuQ29uZGl0aW9uIG9iamVjdCBmcm9tIHRoZSAnQ29uZGl0aW9ucydcbiAgICAgKiBzZWN0aW9uIG9mIHRoZSBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZSB3aXRoIHRoZSBnaXZlbiBuYW1lLlxuICAgICAqIEFueSBtb2RpZmljYXRpb25zIHBlcmZvcm1lZCBvbiB0aGF0IG9iamVjdCB3aWxsIGJlIHJlZmxlY3RlZCBpbiB0aGUgcmVzdWx0aW5nIENESyB0ZW1wbGF0ZS5cbiAgICAgKlxuICAgICAqIElmIGEgQ29uZGl0aW9uIHdpdGggdGhlIGdpdmVuIG5hbWUgaXMgbm90IHByZXNlbnQgaW4gdGhlIHRlbXBsYXRlLFxuICAgICAqIHRocm93cyBhbiBleGNlcHRpb24uXG4gICAgICpcbiAgICAgKiBAcGFyYW0gY29uZGl0aW9uTmFtZSB0aGUgbmFtZSBvZiB0aGUgQ29uZGl0aW9uIGluIHRoZSBDbG91ZEZvcm1hdGlvbiB0ZW1wbGF0ZSBmaWxlXG4gICAgICovXG4gICAgcHVibGljIGdldENvbmRpdGlvbihjb25kaXRpb25OYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkNvbmRpdGlvbiB7XG4gICAgICAgIGNvbnN0IHJldCA9IHRoaXMuY29uZGl0aW9uc1tjb25kaXRpb25OYW1lXTtcbiAgICAgICAgaWYgKCFyZXQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgQ29uZGl0aW9uIHdpdGggbmFtZSAnJHtjb25kaXRpb25OYW1lfScgd2FzIG5vdCBmb3VuZCBpbiB0aGUgdGVtcGxhdGVgKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm5zIHRoZSBDZm5QYXJhbWV0ZXIgb2JqZWN0IGZyb20gdGhlICdQYXJhbWV0ZXJzJ1xuICAgICAqIHNlY3Rpb24gb2YgdGhlIGluY2x1ZGVkIHRlbXBsYXRlLlxuICAgICAqIEFueSBtb2RpZmljYXRpb25zIHBlcmZvcm1lZCBvbiB0aGF0IG9iamVjdCB3aWxsIGJlIHJlZmxlY3RlZCBpbiB0aGUgcmVzdWx0aW5nIENESyB0ZW1wbGF0ZS5cbiAgICAgKlxuICAgICAqIElmIGEgUGFyYW1ldGVyIHdpdGggdGhlIGdpdmVuIG5hbWUgaXMgbm90IHByZXNlbnQgaW4gdGhlIHRlbXBsYXRlLFxuICAgICAqIHRocm93cyBhbiBleGNlcHRpb24uXG4gICAgICpcbiAgICAgKiBAcGFyYW0gcGFyYW1ldGVyTmFtZSB0aGUgbmFtZSBvZiB0aGUgcGFyYW1ldGVyIHRvIHJldHJpZXZlXG4gICAgICovXG4gICAgcHVibGljIGdldFBhcmFtZXRlcihwYXJhbWV0ZXJOYW1lOiBzdHJpbmcpOiBjb3JlLkNmblBhcmFtZXRlciB7XG4gICAgICAgIGNvbnN0IHJldCA9IHRoaXMucGFyYW1ldGVyc1twYXJhbWV0ZXJOYW1lXTtcbiAgICAgICAgaWYgKCFyZXQpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgUGFyYW1ldGVyIHdpdGggbmFtZSAnJHtwYXJhbWV0ZXJOYW1lfScgd2FzIG5vdCBmb3VuZCBpbiB0aGUgdGVtcGxhdGVgKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm5zIHRoZSBDZm5NYXBwaW5nIG9iamVjdCBmcm9tIHRoZSAnTWFwcGluZ3MnIHNlY3Rpb24gb2YgdGhlIGluY2x1ZGVkIHRlbXBsYXRlLlxuICAgICAqIEFueSBtb2RpZmljYXRpb25zIHBlcmZvcm1lZCBvbiB0aGF0IG9iamVjdCB3aWxsIGJlIHJlZmxlY3RlZCBpbiB0aGUgcmVzdWx0aW5nIENESyB0ZW1wbGF0ZS5cbiAgICAgKlxuICAgICAqIElmIGEgTWFwcGluZyB3aXRoIHRoZSBnaXZlbiBuYW1lIGlzIG5vdCBwcmVzZW50IGluIHRoZSB0ZW1wbGF0ZSxcbiAgICAgKiBhbiBleGNlcHRpb24gd2lsbCBiZSB0aHJvd24uXG4gICAgICpcbiAgICAgKiBAcGFyYW0gbWFwcGluZ05hbWUgdGhlIG5hbWUgb2YgdGhlIE1hcHBpbmcgaW4gdGhlIHRlbXBsYXRlIHRvIHJldHJpZXZlXG4gICAgICovXG4gICAgcHVibGljIGdldE1hcHBpbmcobWFwcGluZ05hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuTWFwcGluZyB7XG4gICAgICAgIGNvbnN0IHJldCA9IHRoaXMubWFwcGluZ3NbbWFwcGluZ05hbWVdO1xuICAgICAgICBpZiAoIXJldCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBNYXBwaW5nIHdpdGggbmFtZSAnJHttYXBwaW5nTmFtZX0nIHdhcyBub3QgZm91bmQgaW4gdGhlIHRlbXBsYXRlYCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJucyB0aGUgQ2ZuT3V0cHV0IG9iamVjdCBmcm9tIHRoZSAnT3V0cHV0cydcbiAgICAgKiBzZWN0aW9uIG9mIHRoZSBpbmNsdWRlZCB0ZW1wbGF0ZS5cbiAgICAgKiBBbnkgbW9kaWZpY2F0aW9ucyBwZXJmb3JtZWQgb24gdGhhdCBvYmplY3Qgd2lsbCBiZSByZWZsZWN0ZWQgaW4gdGhlIHJlc3VsdGluZyBDREsgdGVtcGxhdGUuXG4gICAgICpcbiAgICAgKiBJZiBhbiBPdXRwdXQgd2l0aCB0aGUgZ2l2ZW4gbmFtZSBpcyBub3QgcHJlc2VudCBpbiB0aGUgdGVtcGxhdGUsXG4gICAgICogdGhyb3dzIGFuIGV4Y2VwdGlvbi5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBsb2dpY2FsSWQgdGhlIG5hbWUgb2YgdGhlIG91dHB1dCB0byByZXRyaWV2ZVxuICAgICAqL1xuICAgIHB1YmxpYyBnZXRPdXRwdXQobG9naWNhbElkOiBzdHJpbmcpOiBjb3JlLkNmbk91dHB1dCB7XG4gICAgICAgIGNvbnN0IHJldCA9IHRoaXMub3V0cHV0c1tsb2dpY2FsSWRdO1xuICAgICAgICBpZiAoIXJldCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBPdXRwdXQgd2l0aCBsb2dpY2FsIElEICcke2xvZ2ljYWxJZH0nIHdhcyBub3QgZm91bmQgaW4gdGhlIHRlbXBsYXRlYCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHJldDtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmV0dXJucyB0aGUgQ2ZuUnVsZSBvYmplY3QgZnJvbSB0aGUgJ1J1bGVzJ1xuICAgICAqIHNlY3Rpb24gb2YgdGhlIENsb3VkRm9ybWF0aW9uIHRlbXBsYXRlIHdpdGggdGhlIGdpdmVuIG5hbWUuXG4gICAgICogQW55IG1vZGlmaWNhdGlvbnMgcGVyZm9ybWVkIG9uIHRoYXQgb2JqZWN0IHdpbGwgYmUgcmVmbGVjdGVkIGluIHRoZSByZXN1bHRpbmcgQ0RLIHRlbXBsYXRlLlxuICAgICAqXG4gICAgICogSWYgYSBSdWxlIHdpdGggdGhlIGdpdmVuIG5hbWUgaXMgbm90IHByZXNlbnQgaW4gdGhlIHRlbXBsYXRlLFxuICAgICAqIGFuIGV4Y2VwdGlvbiB3aWxsIGJlIHRocm93bi5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBydWxlTmFtZSB0aGUgbmFtZSBvZiB0aGUgUnVsZSBpbiB0aGUgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGVcbiAgICAgKi9cbiAgICBwdWJsaWMgZ2V0UnVsZShydWxlTmFtZTogc3RyaW5nKTogY29yZS5DZm5SdWxlIHtcbiAgICAgICAgY29uc3QgcmV0ID0gdGhpcy5ydWxlc1tydWxlTmFtZV07XG4gICAgICAgIGlmICghcmV0KSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFJ1bGUgd2l0aCBuYW1lICcke3J1bGVOYW1lfScgd2FzIG5vdCBmb3VuZCBpbiB0aGUgdGVtcGxhdGVgKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm5zIHRoZSBDZm5Ib29rIG9iamVjdCBmcm9tIHRoZSAnSG9va3MnXG4gICAgICogc2VjdGlvbiBvZiB0aGUgaW5jbHVkZWQgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUgd2l0aCB0aGUgZ2l2ZW4gbG9naWNhbCBJRC5cbiAgICAgKiBBbnkgbW9kaWZpY2F0aW9ucyBwZXJmb3JtZWQgb24gdGhlIHJldHVybmVkIG9iamVjdCB3aWxsIGJlIHJlZmxlY3RlZCBpbiB0aGUgcmVzdWx0aW5nIENESyB0ZW1wbGF0ZS5cbiAgICAgKlxuICAgICAqIElmIGEgSG9vayB3aXRoIHRoZSBnaXZlbiBsb2dpY2FsIElEIGlzIG5vdCBwcmVzZW50IGluIHRoZSB0ZW1wbGF0ZSxcbiAgICAgKiBhbiBleGNlcHRpb24gd2lsbCBiZSB0aHJvd24uXG4gICAgICpcbiAgICAgKiBAcGFyYW0gaG9va0xvZ2ljYWxJZCB0aGUgbG9naWNhbCBJRCBvZiB0aGUgSG9vayBpbiB0aGUgaW5jbHVkZWQgQ2xvdWRGb3JtYXRpb24gdGVtcGxhdGUncyAnSG9va3MnIHNlY3Rpb25cbiAgICAgKi9cbiAgICBwdWJsaWMgZ2V0SG9vayhob29rTG9naWNhbElkOiBzdHJpbmcpOiBjb3JlLkNmbkhvb2sge1xuICAgICAgICBjb25zdCByZXQgPSB0aGlzLmhvb2tzW2hvb2tMb2dpY2FsSWRdO1xuICAgICAgICBpZiAoIXJldCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBIb29rIHdpdGggbG9naWNhbCBJRCAnJHtob29rTG9naWNhbElkfScgd2FzIG5vdCBmb3VuZCBpbiB0aGUgdGVtcGxhdGVgKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm5zIHRoZSBOZXN0ZWRTdGFjayB3aXRoIG5hbWUgbG9naWNhbElkLlxuICAgICAqIEZvciBhIG5lc3RlZCBzdGFjayB0byBiZSByZXR1cm5lZCBieSB0aGlzIG1ldGhvZCwgaXQgbXVzdCBiZSBzcGVjaWZpZWQgaW4gdGhlIHtAbGluayBDZm5JbmNsdWRlUHJvcHMubmVzdGVkU3RhY2tzfVxuICAgICAqIHByb3BlcnR5LlxuICAgICAqXG4gICAgICogQHBhcmFtIGxvZ2ljYWxJZCB0aGUgSUQgb2YgdGhlIHN0YWNrIHRvIHJldHJpZXZlLCBhcyBpdCBhcHBlYXJzIGluIHRoZSB0ZW1wbGF0ZVxuICAgICAqL1xuICAgIHB1YmxpYyBnZXROZXN0ZWRTdGFjayhsb2dpY2FsSWQ6IHN0cmluZyk6IEluY2x1ZGVkTmVzdGVkU3RhY2sge1xuICAgICAgICBpZiAoIXRoaXMubmVzdGVkU3RhY2tzW2xvZ2ljYWxJZF0pIHtcbiAgICAgICAgICAgIGlmICghdGhpcy50ZW1wbGF0ZS5SZXNvdXJjZXNbbG9naWNhbElkXSkge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgTmVzdGVkIFN0YWNrIHdpdGggbG9naWNhbCBJRCAnJHtsb2dpY2FsSWR9JyB3YXMgbm90IGZvdW5kIGluIHRoZSB0ZW1wbGF0ZWApO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgZWxzZSBpZiAodGhpcy50ZW1wbGF0ZS5SZXNvdXJjZXNbbG9naWNhbElkXS5UeXBlICE9PSAnQVdTOjpDbG91ZEZvcm1hdGlvbjo6U3RhY2snKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBSZXNvdXJjZSB3aXRoIGxvZ2ljYWwgSUQgJyR7bG9naWNhbElkfScgaXMgbm90IGEgQ2xvdWRGb3JtYXRpb24gU3RhY2tgKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgTmVzdGVkIFN0YWNrICcke2xvZ2ljYWxJZH0nIHdhcyBub3QgaW5jbHVkZWQgaW4gdGhlIG5lc3RlZFN0YWNrcyBwcm9wZXJ0eSB3aGVuIGluY2x1ZGluZyB0aGUgcGFyZW50IHRlbXBsYXRlYCk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHRoaXMubmVzdGVkU3RhY2tzW2xvZ2ljYWxJZF07XG4gICAgfVxuICAgIC8qKiBAaW50ZXJuYWwgKi9cbiAgICBwdWJsaWMgX3RvQ2xvdWRGb3JtYXRpb24oKTogb2JqZWN0IHtcbiAgICAgICAgY29uc3QgcmV0OiB7XG4gICAgICAgICAgICBbc2VjdGlvbjogc3RyaW5nXTogYW55O1xuICAgICAgICB9ID0ge307XG4gICAgICAgIGZvciAoY29uc3Qgc2VjdGlvbiBvZiBPYmplY3Qua2V5cyh0aGlzLnRlbXBsYXRlKSkge1xuICAgICAgICAgICAgY29uc3Qgc2VsZiA9IHRoaXM7XG4gICAgICAgICAgICBjb25zdCBmaW5kZXI6IGNmbl9wYXJzZS5JQ2ZuRmluZGVyID0ge1xuICAgICAgICAgICAgICAgIGZpbmRSZXNvdXJjZShsSWQpOiBjb3JlLkNmblJlc291cmNlIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYucmVzb3VyY2VzW2xJZF07XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBmaW5kUmVmVGFyZ2V0KGVsZW1lbnROYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkVsZW1lbnQgfCB1bmRlZmluZWQge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gc2VsZi5yZXNvdXJjZXNbZWxlbWVudE5hbWVdID8/IHNlbGYucGFyYW1ldGVyc1tlbGVtZW50TmFtZV07XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBmaW5kQ29uZGl0aW9uKGNvbmRpdGlvbk5hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuQ29uZGl0aW9uIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYuY29uZGl0aW9uc1tjb25kaXRpb25OYW1lXTtcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIGZpbmRNYXBwaW5nKG1hcHBpbmdOYW1lKTogY29yZS5DZm5NYXBwaW5nIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYubWFwcGluZ3NbbWFwcGluZ05hbWVdO1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9O1xuICAgICAgICAgICAgY29uc3QgY2ZuUGFyc2VyID0gbmV3IGNmbl9wYXJzZS5DZm5QYXJzZXIoe1xuICAgICAgICAgICAgICAgIGZpbmRlcixcbiAgICAgICAgICAgICAgICBwYXJhbWV0ZXJzOiB0aGlzLnBhcmFtZXRlcnNUb1JlcGxhY2UsXG4gICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIHN3aXRjaCAoc2VjdGlvbikge1xuICAgICAgICAgICAgICAgIGNhc2UgJ0NvbmRpdGlvbnMnOlxuICAgICAgICAgICAgICAgIGNhc2UgJ01hcHBpbmdzJzpcbiAgICAgICAgICAgICAgICBjYXNlICdSZXNvdXJjZXMnOlxuICAgICAgICAgICAgICAgIGNhc2UgJ1BhcmFtZXRlcnMnOlxuICAgICAgICAgICAgICAgIGNhc2UgJ1J1bGVzJzpcbiAgICAgICAgICAgICAgICBjYXNlICdIb29rcyc6XG4gICAgICAgICAgICAgICAgY2FzZSAnT3V0cHV0cyc6XG4gICAgICAgICAgICAgICAgICAgIC8vIHRoZXNlIGFyZSByZW5kZXJlZCBhcyBhIHNpZGUgZWZmZWN0IG9mIGluc3RhbnRpYXRpbmcgdGhlIEwxc1xuICAgICAgICAgICAgICAgICAgICBicmVhaztcbiAgICAgICAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgICAgICAgICByZXRbc2VjdGlvbl0gPSBjZm5QYXJzZXIucGFyc2VWYWx1ZSh0aGlzLnRlbXBsYXRlW3NlY3Rpb25dKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgICBwcml2YXRlIGNyZWF0ZU1hcHBpbmcobWFwcGluZ05hbWU6IHN0cmluZyk6IHZvaWQge1xuICAgICAgICBjb25zdCBjZm5QYXJzZXIgPSBuZXcgY2ZuX3BhcnNlLkNmblBhcnNlcih7XG4gICAgICAgICAgICBmaW5kZXI6IHtcbiAgICAgICAgICAgICAgICBmaW5kQ29uZGl0aW9uKCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1JlZmVycmluZyB0byBDb25kaXRpb25zIGluIE1hcHBpbmcgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgICAgICAgICBmaW5kTWFwcGluZygpIHsgdGhyb3cgbmV3IEVycm9yKCdSZWZlcnJpbmcgdG8gb3RoZXIgTWFwcGluZ3MgaW4gTWFwcGluZyBkZWZpbml0aW9ucyBpcyBub3QgYWxsb3dlZCcpOyB9LFxuICAgICAgICAgICAgICAgIGZpbmRSZWZUYXJnZXQoKSB7IHRocm93IG5ldyBFcnJvcignVXNpbmcgUmVmIGV4cHJlc3Npb25zIGluIE1hcHBpbmcgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgICAgICAgICBmaW5kUmVzb3VyY2UoKSB7IHRocm93IG5ldyBFcnJvcignVXNpbmcgR2V0QXR0IGV4cHJlc3Npb25zIGluIE1hcHBpbmcgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBwYXJhbWV0ZXJzOiB7fSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IGNmbk1hcHBpbmcgPSBuZXcgY29yZS5DZm5NYXBwaW5nKHRoaXMubWFwcGluZ3NTY29wZSwgbWFwcGluZ05hbWUsIHtcbiAgICAgICAgICAgIG1hcHBpbmc6IGNmblBhcnNlci5wYXJzZVZhbHVlKHRoaXMudGVtcGxhdGUuTWFwcGluZ3NbbWFwcGluZ05hbWVdKSxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMubWFwcGluZ3NbbWFwcGluZ05hbWVdID0gY2ZuTWFwcGluZztcbiAgICAgICAgdGhpcy5vdmVycmlkZUxvZ2ljYWxJZElmTmVlZGVkKGNmbk1hcHBpbmcsIG1hcHBpbmdOYW1lKTtcbiAgICB9XG4gICAgcHJpdmF0ZSBjcmVhdGVQYXJhbWV0ZXIobG9naWNhbElkOiBzdHJpbmcpOiB2b2lkIHtcbiAgICAgICAgaWYgKGxvZ2ljYWxJZCBpbiB0aGlzLnBhcmFtZXRlcnNUb1JlcGxhY2UpIHtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBleHByZXNzaW9uID0gbmV3IGNmbl9wYXJzZS5DZm5QYXJzZXIoe1xuICAgICAgICAgICAgZmluZGVyOiB7XG4gICAgICAgICAgICAgICAgZmluZFJlc291cmNlKCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1VzaW5nIEdldEF0dCBleHByZXNzaW9ucyBpbiBQYXJhbWV0ZXIgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgICAgICAgICBmaW5kUmVmVGFyZ2V0KCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1VzaW5nIFJlZiBleHByZXNzaW9ucyBpbiBQYXJhbWV0ZXIgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgICAgICAgICBmaW5kQ29uZGl0aW9uKCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1JlZmVycmluZyB0byBDb25kaXRpb25zIGluIFBhcmFtZXRlciBkZWZpbml0aW9ucyBpcyBub3QgYWxsb3dlZCcpOyB9LFxuICAgICAgICAgICAgICAgIGZpbmRNYXBwaW5nKCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1JlZmVycmluZyB0byBNYXBwaW5ncyBpbiBQYXJhbWV0ZXIgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBwYXJhbWV0ZXJzOiB7fSxcbiAgICAgICAgfSkucGFyc2VWYWx1ZSh0aGlzLnRlbXBsYXRlLlBhcmFtZXRlcnNbbG9naWNhbElkXSk7XG4gICAgICAgIGNvbnN0IGNmblBhcmFtZXRlciA9IG5ldyBjb3JlLkNmblBhcmFtZXRlcih0aGlzLCBsb2dpY2FsSWQsIHtcbiAgICAgICAgICAgIHR5cGU6IGV4cHJlc3Npb24uVHlwZSxcbiAgICAgICAgICAgIGRlZmF1bHQ6IGV4cHJlc3Npb24uRGVmYXVsdCxcbiAgICAgICAgICAgIGFsbG93ZWRQYXR0ZXJuOiBleHByZXNzaW9uLkFsbG93ZWRQYXR0ZXJuLFxuICAgICAgICAgICAgYWxsb3dlZFZhbHVlczogZXhwcmVzc2lvbi5BbGxvd2VkVmFsdWVzLFxuICAgICAgICAgICAgY29uc3RyYWludERlc2NyaXB0aW9uOiBleHByZXNzaW9uLkNvbnN0cmFpbnREZXNjcmlwdGlvbixcbiAgICAgICAgICAgIGRlc2NyaXB0aW9uOiBleHByZXNzaW9uLkRlc2NyaXB0aW9uLFxuICAgICAgICAgICAgbWF4TGVuZ3RoOiBleHByZXNzaW9uLk1heExlbmd0aCxcbiAgICAgICAgICAgIG1heFZhbHVlOiBleHByZXNzaW9uLk1heFZhbHVlLFxuICAgICAgICAgICAgbWluTGVuZ3RoOiBleHByZXNzaW9uLk1pbkxlbmd0aCxcbiAgICAgICAgICAgIG1pblZhbHVlOiBleHByZXNzaW9uLk1pblZhbHVlLFxuICAgICAgICAgICAgbm9FY2hvOiBleHByZXNzaW9uLk5vRWNobyxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMub3ZlcnJpZGVMb2dpY2FsSWRJZk5lZWRlZChjZm5QYXJhbWV0ZXIsIGxvZ2ljYWxJZCk7XG4gICAgICAgIHRoaXMucGFyYW1ldGVyc1tsb2dpY2FsSWRdID0gY2ZuUGFyYW1ldGVyO1xuICAgIH1cbiAgICBwcml2YXRlIGNyZWF0ZVJ1bGUocnVsZU5hbWU6IHN0cmluZyk6IHZvaWQge1xuICAgICAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICAgICAgY29uc3QgY2ZuUGFyc2VyID0gbmV3IGNmbl9wYXJzZS5DZm5QYXJzZXIoe1xuICAgICAgICAgICAgZmluZGVyOiB7XG4gICAgICAgICAgICAgICAgZmluZFJlZlRhcmdldChyZWZUYXJnZXQ6IHN0cmluZyk6IGNvcmUuQ2ZuRWxlbWVudCB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgICAgICAgICAgIC8vIG9ubHkgcGFyYW1ldGVycyBjYW4gYmUgcmVmZXJlbmNlZCBpbiBSdWxlc1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gc2VsZi5wYXJhbWV0ZXJzW3JlZlRhcmdldF07XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBmaW5kUmVzb3VyY2UoKSB7IHRocm93IG5ldyBFcnJvcignVXNpbmcgR2V0QXR0IGV4cHJlc3Npb25zIGluIFJ1bGUgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgICAgICAgICBmaW5kQ29uZGl0aW9uKCkgeyB0aHJvdyBuZXcgRXJyb3IoJ1JlZmVycmluZyB0byBDb25kaXRpb25zIGluIFJ1bGUgZGVmaW5pdGlvbnMgaXMgbm90IGFsbG93ZWQnKTsgfSxcbiAgICAgICAgICAgICAgICBmaW5kTWFwcGluZyhtYXBwaW5nTmFtZTogc3RyaW5nKTogY29yZS5DZm5NYXBwaW5nIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYubWFwcGluZ3NbbWFwcGluZ05hbWVdO1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgcGFyYW1ldGVyczogdGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlLFxuICAgICAgICAgICAgY29udGV4dDogY2ZuX3BhcnNlLkNmblBhcnNpbmdDb250ZXh0LlJVTEVTLFxuICAgICAgICB9KTtcbiAgICAgICAgY29uc3QgcnVsZVByb3BlcnRpZXMgPSBjZm5QYXJzZXIucGFyc2VWYWx1ZSh0aGlzLnRlbXBsYXRlLlJ1bGVzW3J1bGVOYW1lXSk7XG4gICAgICAgIGNvbnN0IHJ1bGUgPSBuZXcgY29yZS5DZm5SdWxlKHRoaXMucnVsZXNTY29wZSwgcnVsZU5hbWUsIHtcbiAgICAgICAgICAgIHJ1bGVDb25kaXRpb246IHJ1bGVQcm9wZXJ0aWVzLlJ1bGVDb25kaXRpb24sXG4gICAgICAgICAgICBhc3NlcnRpb25zOiBydWxlUHJvcGVydGllcy5Bc3NlcnRpb25zLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5ydWxlc1tydWxlTmFtZV0gPSBydWxlO1xuICAgICAgICB0aGlzLm92ZXJyaWRlTG9naWNhbElkSWZOZWVkZWQocnVsZSwgcnVsZU5hbWUpO1xuICAgIH1cbiAgICBwcml2YXRlIGNyZWF0ZUhvb2soaG9va05hbWU6IHN0cmluZyk6IHZvaWQge1xuICAgICAgICBjb25zdCBzZWxmID0gdGhpcztcbiAgICAgICAgY29uc3QgY2ZuUGFyc2VyID0gbmV3IGNmbl9wYXJzZS5DZm5QYXJzZXIoe1xuICAgICAgICAgICAgZmluZGVyOiB7XG4gICAgICAgICAgICAgICAgZmluZFJlc291cmNlKGxJZCk6IGNvcmUuQ2ZuUmVzb3VyY2UgfCB1bmRlZmluZWQge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gc2VsZi5yZXNvdXJjZXNbbElkXTtcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIGZpbmRSZWZUYXJnZXQoZWxlbWVudE5hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuRWxlbWVudCB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBzZWxmLnJlc291cmNlc1tlbGVtZW50TmFtZV0gPz8gc2VsZi5wYXJhbWV0ZXJzW2VsZW1lbnROYW1lXTtcbiAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIGZpbmRDb25kaXRpb24oY29uZGl0aW9uTmFtZTogc3RyaW5nKTogY29yZS5DZm5Db25kaXRpb24gfCB1bmRlZmluZWQge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gc2VsZi5jb25kaXRpb25zW2NvbmRpdGlvbk5hbWVdO1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgZmluZE1hcHBpbmcobWFwcGluZ05hbWUpOiBjb3JlLkNmbk1hcHBpbmcgfCB1bmRlZmluZWQge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gc2VsZi5tYXBwaW5nc1ttYXBwaW5nTmFtZV07XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBwYXJhbWV0ZXJzOiB0aGlzLnBhcmFtZXRlcnNUb1JlcGxhY2UsXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCBob29rQXR0cmlidXRlcyA9IHRoaXMudGVtcGxhdGUuSG9va3NbaG9va05hbWVdO1xuICAgICAgICBsZXQgaG9vazogY29yZS5DZm5Ib29rO1xuICAgICAgICBzd2l0Y2ggKGhvb2tBdHRyaWJ1dGVzLlR5cGUpIHtcbiAgICAgICAgICAgIGNhc2UgJ0FXUzo6Q29kZURlcGxveTo6Qmx1ZUdyZWVuJzpcbiAgICAgICAgICAgICAgICBob29rID0gKGNvcmUuQ2ZuQ29kZURlcGxveUJsdWVHcmVlbkhvb2sgYXMgYW55KS5fZnJvbUNsb3VkRm9ybWF0aW9uKHRoaXMuaG9va3NTY29wZSwgaG9va05hbWUsIGhvb2tBdHRyaWJ1dGVzLCB7XG4gICAgICAgICAgICAgICAgICAgIHBhcnNlcjogY2ZuUGFyc2VyLFxuICAgICAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgICAgIGJyZWFrO1xuICAgICAgICAgICAgZGVmYXVsdDoge1xuICAgICAgICAgICAgICAgIGNvbnN0IGhvb2tQcm9wZXJ0aWVzID0gY2ZuUGFyc2VyLnBhcnNlVmFsdWUoaG9va0F0dHJpYnV0ZXMuUHJvcGVydGllcykgPz8ge307XG4gICAgICAgICAgICAgICAgaG9vayA9IG5ldyBjb3JlLkNmbkhvb2sodGhpcy5ob29rc1Njb3BlLCBob29rTmFtZSwge1xuICAgICAgICAgICAgICAgICAgICB0eXBlOiBob29rQXR0cmlidXRlcy5UeXBlLFxuICAgICAgICAgICAgICAgICAgICBwcm9wZXJ0aWVzOiBob29rUHJvcGVydGllcyxcbiAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICB0aGlzLmhvb2tzW2hvb2tOYW1lXSA9IGhvb2s7XG4gICAgICAgIHRoaXMub3ZlcnJpZGVMb2dpY2FsSWRJZk5lZWRlZChob29rLCBob29rTmFtZSk7XG4gICAgfVxuICAgIHByaXZhdGUgY3JlYXRlT3V0cHV0KGxvZ2ljYWxJZDogc3RyaW5nLCBzY29wZTogY29yZS5Db25zdHJ1Y3QpOiB2b2lkIHtcbiAgICAgICAgY29uc3Qgc2VsZiA9IHRoaXM7XG4gICAgICAgIGNvbnN0IG91dHB1dEF0dHJpYnV0ZXMgPSBuZXcgY2ZuX3BhcnNlLkNmblBhcnNlcih7XG4gICAgICAgICAgICBmaW5kZXI6IHtcbiAgICAgICAgICAgICAgICBmaW5kUmVzb3VyY2UobElkKTogY29yZS5DZm5SZXNvdXJjZSB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBzZWxmLnJlc291cmNlc1tsSWRdO1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgZmluZFJlZlRhcmdldChlbGVtZW50TmFtZTogc3RyaW5nKTogY29yZS5DZm5FbGVtZW50IHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYucmVzb3VyY2VzW2VsZW1lbnROYW1lXSA/PyBzZWxmLnBhcmFtZXRlcnNbZWxlbWVudE5hbWVdO1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgZmluZENvbmRpdGlvbigpOiB1bmRlZmluZWQge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgZmluZE1hcHBpbmcobWFwcGluZ05hbWUpOiBjb3JlLkNmbk1hcHBpbmcgfCB1bmRlZmluZWQge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gc2VsZi5tYXBwaW5nc1ttYXBwaW5nTmFtZV07XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBwYXJhbWV0ZXJzOiB0aGlzLnBhcmFtZXRlcnNUb1JlcGxhY2UsXG4gICAgICAgIH0pLnBhcnNlVmFsdWUodGhpcy50ZW1wbGF0ZS5PdXRwdXRzW2xvZ2ljYWxJZF0pO1xuICAgICAgICBjb25zdCBjZm5PdXRwdXQgPSBuZXcgY29yZS5DZm5PdXRwdXQoc2NvcGUsIGxvZ2ljYWxJZCwge1xuICAgICAgICAgICAgdmFsdWU6IG91dHB1dEF0dHJpYnV0ZXMuVmFsdWUsXG4gICAgICAgICAgICBkZXNjcmlwdGlvbjogb3V0cHV0QXR0cmlidXRlcy5EZXNjcmlwdGlvbixcbiAgICAgICAgICAgIGV4cG9ydE5hbWU6IG91dHB1dEF0dHJpYnV0ZXMuRXhwb3J0ID8gb3V0cHV0QXR0cmlidXRlcy5FeHBvcnQuTmFtZSA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIGNvbmRpdGlvbjogKCgpID0+IHtcbiAgICAgICAgICAgICAgICBpZiAoIW91dHB1dEF0dHJpYnV0ZXMuQ29uZGl0aW9uKSB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIGVsc2UgaWYgKHRoaXMuY29uZGl0aW9uc1tvdXRwdXRBdHRyaWJ1dGVzLkNvbmRpdGlvbl0pIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYuZ2V0Q29uZGl0aW9uKG91dHB1dEF0dHJpYnV0ZXMuQ29uZGl0aW9uKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBPdXRwdXQgd2l0aCBuYW1lICcke2xvZ2ljYWxJZH0nIHJlZmVycyB0byBhIENvbmRpdGlvbiB3aXRoIG5hbWUgYCArXG4gICAgICAgICAgICAgICAgICAgIGAnJHtvdXRwdXRBdHRyaWJ1dGVzLkNvbmRpdGlvbn0nIHdoaWNoIHdhcyBub3QgZm91bmQgaW4gdGhpcyB0ZW1wbGF0ZWApO1xuICAgICAgICAgICAgfSkoKSxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMub3ZlcnJpZGVMb2dpY2FsSWRJZk5lZWRlZChjZm5PdXRwdXQsIGxvZ2ljYWxJZCk7XG4gICAgICAgIHRoaXMub3V0cHV0c1tsb2dpY2FsSWRdID0gY2ZuT3V0cHV0O1xuICAgIH1cbiAgICBwcml2YXRlIGdldE9yQ3JlYXRlQ29uZGl0aW9uKGNvbmRpdGlvbk5hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuQ29uZGl0aW9uIHtcbiAgICAgICAgaWYgKGNvbmRpdGlvbk5hbWUgaW4gdGhpcy5jb25kaXRpb25zKSB7XG4gICAgICAgICAgICByZXR1cm4gdGhpcy5jb25kaXRpb25zW2NvbmRpdGlvbk5hbWVdO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHNlbGYgPSB0aGlzO1xuICAgICAgICBjb25zdCBjZm5QYXJzZXIgPSBuZXcgY2ZuX3BhcnNlLkNmblBhcnNlcih7XG4gICAgICAgICAgICBmaW5kZXI6IHtcbiAgICAgICAgICAgICAgICBmaW5kUmVzb3VyY2UoKSB7IHRocm93IG5ldyBFcnJvcignVXNpbmcgR2V0QXR0IGluIENvbmRpdGlvbiBkZWZpbml0aW9ucyBpcyBub3QgYWxsb3dlZCcpOyB9LFxuICAgICAgICAgICAgICAgIGZpbmRSZWZUYXJnZXQoZWxlbWVudE5hbWU6IHN0cmluZyk6IGNvcmUuQ2ZuRWxlbWVudCB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgICAgICAgICAgIC8vIG9ubHkgUGFyYW1ldGVycyBjYW4gYmUgcmVmZXJlbmNlZCBpbiB0aGUgJ0NvbmRpdGlvbnMnIHNlY3Rpb25cbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYucGFyYW1ldGVyc1tlbGVtZW50TmFtZV07XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBmaW5kQ29uZGl0aW9uKGNOYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkNvbmRpdGlvbiB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBjTmFtZSBpbiAoc2VsZi50ZW1wbGF0ZS5Db25kaXRpb25zIHx8IHt9KVxuICAgICAgICAgICAgICAgICAgICAgICAgPyBzZWxmLmdldE9yQ3JlYXRlQ29uZGl0aW9uKGNOYW1lKVxuICAgICAgICAgICAgICAgICAgICAgICAgOiB1bmRlZmluZWQ7XG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBmaW5kTWFwcGluZyhtYXBwaW5nTmFtZTogc3RyaW5nKTogY29yZS5DZm5NYXBwaW5nIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYubWFwcGluZ3NbbWFwcGluZ05hbWVdO1xuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgY29udGV4dDogY2ZuX3BhcnNlLkNmblBhcnNpbmdDb250ZXh0LkNPTkRJVElPTlMsXG4gICAgICAgICAgICBwYXJhbWV0ZXJzOiB0aGlzLnBhcmFtZXRlcnNUb1JlcGxhY2UsXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCBjZm5Db25kaXRpb24gPSBuZXcgY29yZS5DZm5Db25kaXRpb24odGhpcy5jb25kaXRpb25zU2NvcGUsIGNvbmRpdGlvbk5hbWUsIHtcbiAgICAgICAgICAgIGV4cHJlc3Npb246IGNmblBhcnNlci5wYXJzZVZhbHVlKHRoaXMudGVtcGxhdGUuQ29uZGl0aW9uc1tjb25kaXRpb25OYW1lXSksXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLm92ZXJyaWRlTG9naWNhbElkSWZOZWVkZWQoY2ZuQ29uZGl0aW9uLCBjb25kaXRpb25OYW1lKTtcbiAgICAgICAgdGhpcy5jb25kaXRpb25zW2NvbmRpdGlvbk5hbWVdID0gY2ZuQ29uZGl0aW9uO1xuICAgICAgICByZXR1cm4gY2ZuQ29uZGl0aW9uO1xuICAgIH1cbiAgICBwcml2YXRlIGdldE9yQ3JlYXRlUmVzb3VyY2UobG9naWNhbElkOiBzdHJpbmcpOiBjb3JlLkNmblJlc291cmNlIHtcbiAgICAgICAgY29uc3QgcmV0ID0gdGhpcy5yZXNvdXJjZXNbbG9naWNhbElkXTtcbiAgICAgICAgaWYgKHJldCkge1xuICAgICAgICAgICAgcmV0dXJuIHJldDtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCByZXNvdXJjZUF0dHJpYnV0ZXM6IGFueSA9IHRoaXMudGVtcGxhdGUuUmVzb3VyY2VzW2xvZ2ljYWxJZF07XG4gICAgICAgIC8vIGZhaWwgZWFybHkgZm9yIHJlc291cmNlIGF0dHJpYnV0ZXMgd2UgZG9uJ3Qgc3VwcG9ydCB5ZXRcbiAgICAgICAgY29uc3Qga25vd25BdHRyaWJ1dGVzID0gW1xuICAgICAgICAgICAgJ1R5cGUnLCAnUHJvcGVydGllcycsICdDb25kaXRpb24nLCAnRGVwZW5kc09uJywgJ01ldGFkYXRhJyxcbiAgICAgICAgICAgICdDcmVhdGlvblBvbGljeScsICdVcGRhdGVQb2xpY3knLCAnRGVsZXRpb25Qb2xpY3knLCAnVXBkYXRlUmVwbGFjZVBvbGljeScsXG4gICAgICAgIF07XG4gICAgICAgIGZvciAoY29uc3QgYXR0cmlidXRlIG9mIE9iamVjdC5rZXlzKHJlc291cmNlQXR0cmlidXRlcykpIHtcbiAgICAgICAgICAgIGlmICgha25vd25BdHRyaWJ1dGVzLmluY2x1ZGVzKGF0dHJpYnV0ZSkpIHtcbiAgICAgICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSAke2F0dHJpYnV0ZX0gcmVzb3VyY2UgYXR0cmlidXRlIGlzIG5vdCBzdXBwb3J0ZWQgYnkgY2xvdWRmb3JtYXRpb24taW5jbHVkZSB5ZXQuIGAgK1xuICAgICAgICAgICAgICAgICAgICAnRWl0aGVyIHJlbW92ZSBpdCBmcm9tIHRoZSB0ZW1wbGF0ZSwgb3IgdXNlIHRoZSBDZGtJbmNsdWRlIGNsYXNzIGZyb20gdGhlIGNvcmUgcGFja2FnZSBpbnN0ZWFkLicpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHNlbGYgPSB0aGlzO1xuICAgICAgICBjb25zdCBmaW5kZXI6IGNmbl9wYXJzZS5JQ2ZuRmluZGVyID0ge1xuICAgICAgICAgICAgZmluZENvbmRpdGlvbihjb25kaXRpb25OYW1lOiBzdHJpbmcpOiBjb3JlLkNmbkNvbmRpdGlvbiB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYuY29uZGl0aW9uc1tjb25kaXRpb25OYW1lXTtcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBmaW5kTWFwcGluZyhtYXBwaW5nTmFtZSk6IGNvcmUuQ2ZuTWFwcGluZyB8IHVuZGVmaW5lZCB7XG4gICAgICAgICAgICAgICAgcmV0dXJuIHNlbGYubWFwcGluZ3NbbWFwcGluZ05hbWVdO1xuICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIGZpbmRSZXNvdXJjZShsSWQ6IHN0cmluZyk6IGNvcmUuQ2ZuUmVzb3VyY2UgfCB1bmRlZmluZWQge1xuICAgICAgICAgICAgICAgIGlmICghKGxJZCBpbiAoc2VsZi50ZW1wbGF0ZS5SZXNvdXJjZXMgfHwge30pKSkge1xuICAgICAgICAgICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICByZXR1cm4gc2VsZi5nZXRPckNyZWF0ZVJlc291cmNlKGxJZCk7XG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgZmluZFJlZlRhcmdldChlbGVtZW50TmFtZTogc3RyaW5nKTogY29yZS5DZm5FbGVtZW50IHwgdW5kZWZpbmVkIHtcbiAgICAgICAgICAgICAgICBpZiAoZWxlbWVudE5hbWUgaW4gc2VsZi5wYXJhbWV0ZXJzKSB7XG4gICAgICAgICAgICAgICAgICAgIHJldHVybiBzZWxmLnBhcmFtZXRlcnNbZWxlbWVudE5hbWVdO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICByZXR1cm4gdGhpcy5maW5kUmVzb3VyY2UoZWxlbWVudE5hbWUpO1xuICAgICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICAgICAgY29uc3QgY2ZuUGFyc2VyID0gbmV3IGNmbl9wYXJzZS5DZm5QYXJzZXIoe1xuICAgICAgICAgICAgZmluZGVyLFxuICAgICAgICAgICAgcGFyYW1ldGVyczogdGhpcy5wYXJhbWV0ZXJzVG9SZXBsYWNlLFxuICAgICAgICB9KTtcbiAgICAgICAgbGV0IGwxSW5zdGFuY2U6IGNvcmUuQ2ZuUmVzb3VyY2U7XG4gICAgICAgIGlmICh0aGlzLm5lc3RlZFN0YWNrc1RvSW5jbHVkZVtsb2dpY2FsSWRdKSB7XG4gICAgICAgICAgICBsMUluc3RhbmNlID0gdGhpcy5jcmVhdGVOZXN0ZWRTdGFjayhsb2dpY2FsSWQsIGNmblBhcnNlcik7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSB7XG4gICAgICAgICAgICBjb25zdCBsMUNsYXNzRnFuID0gY2ZuX3R5cGVfdG9fbDFfbWFwcGluZy5sb29rdXAocmVzb3VyY2VBdHRyaWJ1dGVzLlR5cGUpO1xuICAgICAgICAgICAgaWYgKGwxQ2xhc3NGcW4pIHtcbiAgICAgICAgICAgICAgICBjb25zdCBvcHRpb25zOiBjZm5fcGFyc2UuRnJvbUNsb3VkRm9ybWF0aW9uT3B0aW9ucyA9IHtcbiAgICAgICAgICAgICAgICAgICAgcGFyc2VyOiBjZm5QYXJzZXIsXG4gICAgICAgICAgICAgICAgfTtcbiAgICAgICAgICAgICAgICBjb25zdCBbbW9kdWxlTmFtZSwgLi4uY2xhc3NOYW1lXSA9IGwxQ2xhc3NGcW4uc3BsaXQoJy4nKTtcbiAgICAgICAgICAgICAgICBjb25zdCBtb2R1bGUgPSByZXF1aXJlKG1vZHVsZU5hbWUpOyAvLyBlc2xpbnQtZGlzYWJsZS1saW5lIEB0eXBlc2NyaXB0LWVzbGludC9uby1yZXF1aXJlLWltcG9ydHNcbiAgICAgICAgICAgICAgICBjb25zdCBqc0NsYXNzRnJvbU1vZHVsZSA9IG1vZHVsZVtjbGFzc05hbWUuam9pbignLicpXTtcbiAgICAgICAgICAgICAgICBsMUluc3RhbmNlID0ganNDbGFzc0Zyb21Nb2R1bGUuX2Zyb21DbG91ZEZvcm1hdGlvbih0aGlzLCBsb2dpY2FsSWQsIHJlc291cmNlQXR0cmlidXRlcywgb3B0aW9ucyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICBsMUluc3RhbmNlID0gbmV3IGNvcmUuQ2ZuUmVzb3VyY2UodGhpcywgbG9naWNhbElkLCB7XG4gICAgICAgICAgICAgICAgICAgIHR5cGU6IHJlc291cmNlQXR0cmlidXRlcy5UeXBlLFxuICAgICAgICAgICAgICAgICAgICBwcm9wZXJ0aWVzOiBjZm5QYXJzZXIucGFyc2VWYWx1ZShyZXNvdXJjZUF0dHJpYnV0ZXMuUHJvcGVydGllcyksXG4gICAgICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAgICAgY2ZuUGFyc2VyLmhhbmRsZUF0dHJpYnV0ZXMobDFJbnN0YW5jZSwgcmVzb3VyY2VBdHRyaWJ1dGVzLCBsb2dpY2FsSWQpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHRoaXMub3ZlcnJpZGVMb2dpY2FsSWRJZk5lZWRlZChsMUluc3RhbmNlLCBsb2dpY2FsSWQpO1xuICAgICAgICB0aGlzLnJlc291cmNlc1tsb2dpY2FsSWRdID0gbDFJbnN0YW5jZTtcbiAgICAgICAgcmV0dXJuIGwxSW5zdGFuY2U7XG4gICAgfVxuICAgIHByaXZhdGUgY3JlYXRlTmVzdGVkU3RhY2sobmVzdGVkU3RhY2tJZDogc3RyaW5nLCBjZm5QYXJzZXI6IGNmbl9wYXJzZS5DZm5QYXJzZXIpOiBjb3JlLkNmblJlc291cmNlIHtcbiAgICAgICAgY29uc3QgdGVtcGxhdGVSZXNvdXJjZXMgPSB0aGlzLnRlbXBsYXRlLlJlc291cmNlcyB8fCB7fTtcbiAgICAgICAgY29uc3QgbmVzdGVkU3RhY2tBdHRyaWJ1dGVzID0gdGVtcGxhdGVSZXNvdXJjZXNbbmVzdGVkU3RhY2tJZF0gfHwge307XG4gICAgICAgIGlmIChuZXN0ZWRTdGFja0F0dHJpYnV0ZXMuVHlwZSAhPT0gJ0FXUzo6Q2xvdWRGb3JtYXRpb246OlN0YWNrJykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBOZXN0ZWQgU3RhY2sgd2l0aCBsb2dpY2FsIElEICcke25lc3RlZFN0YWNrSWR9JyBpcyBub3QgYW4gQVdTOjpDbG91ZEZvcm1hdGlvbjo6U3RhY2sgcmVzb3VyY2VgKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAobmVzdGVkU3RhY2tBdHRyaWJ1dGVzLkNyZWF0aW9uUG9saWN5KSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0NyZWF0aW9uUG9saWN5IGlzIG5vdCBzdXBwb3J0ZWQgYnkgdGhlIEFXUzo6Q2xvdWRGb3JtYXRpb246OlN0YWNrIHJlc291cmNlJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKG5lc3RlZFN0YWNrQXR0cmlidXRlcy5VcGRhdGVQb2xpY3kpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignVXBkYXRlUG9saWN5IGlzIG5vdCBzdXBwb3J0ZWQgYnkgdGhlIEFXUzo6Q2xvdWRGb3JtYXRpb246OlN0YWNrIHJlc291cmNlJyk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgbmVzdGVkU3RhY2tQcm9wcyA9IGNmblBhcnNlci5wYXJzZVZhbHVlKG5lc3RlZFN0YWNrQXR0cmlidXRlcy5Qcm9wZXJ0aWVzKTtcbiAgICAgICAgY29uc3QgbmVzdGVkU3RhY2sgPSBuZXcgY29yZS5OZXN0ZWRTdGFjayh0aGlzLCBuZXN0ZWRTdGFja0lkLCB7XG4gICAgICAgICAgICBwYXJhbWV0ZXJzOiB0aGlzLnBhcmFtZXRlcnNGb3JOZXN0ZWRTdGFjayhuZXN0ZWRTdGFja1Byb3BzLlBhcmFtZXRlcnMsIG5lc3RlZFN0YWNrSWQpLFxuICAgICAgICAgICAgbm90aWZpY2F0aW9uQXJuczogbmVzdGVkU3RhY2tQcm9wcy5Ob3RpZmljYXRpb25Bcm5zLFxuICAgICAgICAgICAgdGltZW91dDogbmVzdGVkU3RhY2tQcm9wcy5UaW1lb3V0LFxuICAgICAgICB9KTtcbiAgICAgICAgY29uc3QgdGVtcGxhdGUgPSBuZXcgQ2ZuSW5jbHVkZShuZXN0ZWRTdGFjaywgbmVzdGVkU3RhY2tJZCwgdGhpcy5uZXN0ZWRTdGFja3NUb0luY2x1ZGVbbmVzdGVkU3RhY2tJZF0pO1xuICAgICAgICB0aGlzLm5lc3RlZFN0YWNrc1tuZXN0ZWRTdGFja0lkXSA9IHsgc3RhY2s6IG5lc3RlZFN0YWNrLCBpbmNsdWRlZFRlbXBsYXRlOiB0ZW1wbGF0ZSB9O1xuICAgICAgICAvLyB3ZSBrbm93IHRoaXMgaXMgbmV2ZXIgdW5kZWZpbmVkIGZvciBuZXN0ZWQgc3RhY2tzXG4gICAgICAgIGNvbnN0IG5lc3RlZFN0YWNrUmVzb3VyY2U6IGNvcmUuQ2ZuUmVzb3VyY2UgPSBuZXN0ZWRTdGFjay5uZXN0ZWRTdGFja1Jlc291cmNlITtcbiAgICAgICAgY2ZuUGFyc2VyLmhhbmRsZUF0dHJpYnV0ZXMobmVzdGVkU3RhY2tSZXNvdXJjZSwgbmVzdGVkU3RhY2tBdHRyaWJ1dGVzLCBuZXN0ZWRTdGFja0lkKTtcbiAgICAgICAgcmV0dXJuIG5lc3RlZFN0YWNrUmVzb3VyY2U7XG4gICAgfVxuICAgIHByaXZhdGUgcGFyYW1ldGVyc0Zvck5lc3RlZFN0YWNrKHBhcmFtZXRlcnM6IGFueSwgbmVzdGVkU3RhY2tJZDogc3RyaW5nKToge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBhbnk7XG4gICAgfSB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGlmIChwYXJhbWV0ZXJzID09IG51bGwpIHtcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgcGFyYW1ldGVyc1RvUmVwbGFjZSA9IHRoaXMubmVzdGVkU3RhY2tzVG9JbmNsdWRlW25lc3RlZFN0YWNrSWRdLnBhcmFtZXRlcnMgPz8ge307XG4gICAgICAgIGNvbnN0IHJldDoge1xuICAgICAgICAgICAgW2tleTogc3RyaW5nXTogc3RyaW5nO1xuICAgICAgICB9ID0ge307XG4gICAgICAgIGZvciAoY29uc3QgcGFyYW1OYW1lIG9mIE9iamVjdC5rZXlzKHBhcmFtZXRlcnMpKSB7XG4gICAgICAgICAgICBpZiAoIShwYXJhbU5hbWUgaW4gcGFyYW1ldGVyc1RvUmVwbGFjZSkpIHtcbiAgICAgICAgICAgICAgICByZXRbcGFyYW1OYW1lXSA9IHBhcmFtZXRlcnNbcGFyYW1OYW1lXTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gcmV0O1xuICAgIH1cbiAgICBwcml2YXRlIG92ZXJyaWRlTG9naWNhbElkSWZOZWVkZWQoZWxlbWVudDogY29yZS5DZm5FbGVtZW50LCBpZDogc3RyaW5nKTogdm9pZCB7XG4gICAgICAgIGlmICh0aGlzLnByZXNlcnZlTG9naWNhbElkcykge1xuICAgICAgICAgICAgZWxlbWVudC5vdmVycmlkZUxvZ2ljYWxJZChpZCk7XG4gICAgICAgIH1cbiAgICB9XG59XG4iXX0=