"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StateMachine = exports.LogLevel = exports.StateMachineType = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const state_graph_1 = require("./state-graph");
const stepfunctions_generated_1 = require("./stepfunctions.generated");
/**
 * Two types of state machines are available in AWS Step Functions: EXPRESS AND STANDARD.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-standard-vs-express.html
 *
 * @default STANDARD
 */
var StateMachineType;
(function (StateMachineType) {
    /**
     * Express Workflows are ideal for high-volume, event processing workloads.
     */
    StateMachineType["EXPRESS"] = "EXPRESS";
    /**
     * Standard Workflows are ideal for long-running, durable, and auditable workflows.
     */
    StateMachineType["STANDARD"] = "STANDARD";
})(StateMachineType = exports.StateMachineType || (exports.StateMachineType = {}));
/**
 * Defines which category of execution history events are logged.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/cloudwatch-log-level.html
 *
 * @default ERROR
 */
var LogLevel;
(function (LogLevel) {
    /**
     * No Logging
     */
    LogLevel["OFF"] = "OFF";
    /**
     * Log everything
     */
    LogLevel["ALL"] = "ALL";
    /**
     * Log all errors
     */
    LogLevel["ERROR"] = "ERROR";
    /**
     * Log fatal errors
     */
    LogLevel["FATAL"] = "FATAL";
})(LogLevel = exports.LogLevel || (exports.LogLevel = {}));
/**
 * A new or imported state machine.
 */
class StateMachineBase extends core_1.Resource {
    /**
     * Import a state machine
     */
    static fromStateMachineArn(scope, id, stateMachineArn) {
        class Import extends StateMachineBase {
            constructor() {
                super(...arguments);
                this.stateMachineArn = stateMachineArn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Grant the given identity permissions to start an execution of this state
     * machine.
     */
    grantStartExecution(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: ['states:StartExecution'],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions to read results from state
     * machine.
     */
    grantRead(identity) {
        iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:ListExecutions',
                'states:ListStateMachines',
            ],
            resourceArns: [this.stateMachineArn],
        });
        iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:DescribeExecution',
                'states:DescribeStateMachineForExecution',
                'states:GetExecutionHistory',
            ],
            resourceArns: [`${this.executionArn()}:*`],
        });
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:ListActivities',
                'states:DescribeStateMachine',
                'states:DescribeActivity',
            ],
            resourceArns: ['*'],
        });
    }
    /**
     * Grant the given identity task response permissions on a state machine
     */
    grantTaskResponse(identity) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions: [
                'states:SendTaskSuccess',
                'states:SendTaskFailure',
                'states:SendTaskHeartbeat',
            ],
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Grant the given identity permissions on all executions of the state machine
     */
    grantExecution(identity, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions,
            resourceArns: [`${this.executionArn()}:*`],
        });
    }
    /**
     * Grant the given identity custom permissions
     */
    grant(identity, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee: identity,
            actions,
            resourceArns: [this.stateMachineArn],
        });
    }
    /**
     * Returns the pattern for the execution ARN's of the state machine
     */
    executionArn() {
        return core_1.Stack.of(this).formatArn({
            resource: 'execution',
            service: 'states',
            resourceName: core_1.Arn.parse(this.stateMachineArn, ':').resourceName,
            sep: ':',
        });
    }
}
/**
 * Define a StepFunctions State Machine
 */
class StateMachine extends StateMachineBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.stateMachineName,
        });
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('states.amazonaws.com'),
        });
        const graph = new state_graph_1.StateGraph(props.definition.startState, `State Machine ${id} definition`);
        graph.timeout = props.timeout;
        this.stateMachineType = props.stateMachineType ? props.stateMachineType : StateMachineType.STANDARD;
        let loggingConfiguration;
        if (props.logs) {
            const conf = props.logs;
            loggingConfiguration = {
                destinations: [{ cloudWatchLogsLogGroup: { logGroupArn: conf.destination.logGroupArn } }],
                includeExecutionData: conf.includeExecutionData,
                level: conf.level || 'ERROR',
            };
            // https://docs.aws.amazon.com/step-functions/latest/dg/cw-logs.html#cloudwatch-iam-policy
            this.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: [
                    'logs:CreateLogDelivery',
                    'logs:GetLogDelivery',
                    'logs:UpdateLogDelivery',
                    'logs:DeleteLogDelivery',
                    'logs:ListLogDeliveries',
                    'logs:PutResourcePolicy',
                    'logs:DescribeResourcePolicies',
                    'logs:DescribeLogGroups',
                ],
                resources: ['*'],
            }));
        }
        const resource = new stepfunctions_generated_1.CfnStateMachine(this, 'Resource', {
            stateMachineName: this.physicalName,
            stateMachineType: props.stateMachineType ? props.stateMachineType : undefined,
            roleArn: this.role.roleArn,
            definitionString: core_1.Stack.of(this).toJsonString(graph.toGraphJson()),
            loggingConfiguration,
        });
        resource.node.addDependency(this.role);
        for (const statement of graph.policyStatements) {
            this.addToRolePolicy(statement);
        }
        this.stateMachineName = this.getResourceNameAttribute(resource.attrName);
        this.stateMachineArn = this.getResourceArnAttribute(resource.ref, {
            service: 'states',
            resource: 'stateMachine',
            resourceName: this.physicalName,
            sep: ':',
        });
    }
    /**
     * Add the given statement to the role's policy
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Return the given named metric for this State Machine's executions
     *
     * @default sum over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/States',
            metricName,
            dimensions: { StateMachineArn: this.stateMachineArn },
            statistic: 'sum',
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the number of executions that failed
     *
     * @default sum over 5 minutes
     */
    metricFailed(props) {
        return this.metric('ExecutionsFailed', props);
    }
    /**
     * Metric for the number of executions that were throttled
     *
     * @default sum over 5 minutes
     */
    metricThrottled(props) {
        return this.metric('ExecutionThrottled', props);
    }
    /**
     * Metric for the number of executions that were aborted
     *
     * @default sum over 5 minutes
     */
    metricAborted(props) {
        return this.metric('ExecutionsAborted', props);
    }
    /**
     * Metric for the number of executions that succeeded
     *
     * @default sum over 5 minutes
     */
    metricSucceeded(props) {
        return this.metric('ExecutionsSucceeded', props);
    }
    /**
     * Metric for the number of executions that succeeded
     *
     * @default sum over 5 minutes
     */
    metricTimedOut(props) {
        return this.metric('ExecutionsTimedOut', props);
    }
    /**
     * Metric for the number of executions that were started
     *
     * @default sum over 5 minutes
     */
    metricStarted(props) {
        return this.metric('ExecutionsStarted', props);
    }
    /**
     * Metric for the interval, in milliseconds, between the time the execution starts and the time it closes
     *
     * @default sum over 5 minutes
     */
    metricTime(props) {
        return this.metric('ExecutionTime', props);
    }
}
exports.StateMachine = StateMachine;
//# sourceMappingURL=data:application/json;base64,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