"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubscriptionProtocol = exports.Subscription = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const sns_generated_1 = require("./sns.generated");
/**
 * A new subscription.
 *
 * Prefer to use the `ITopic.addSubscription()` methods to create instances of
 * this class.
 */
class Subscription extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.rawMessageDelivery && ['http', 'https', 'sqs'].indexOf(props.protocol) < 0) {
            throw new Error('Raw message delivery can only be enabled for HTTP/S and SQS subscriptions.');
        }
        if (props.filterPolicy) {
            if (Object.keys(props.filterPolicy).length > 5) {
                throw new Error('A filter policy can have a maximum of 5 attribute names.');
            }
            this.filterPolicy = Object.entries(props.filterPolicy)
                .reduce((acc, [k, v]) => ({ ...acc, [k]: v.conditions }), {});
            let total = 1;
            Object.values(this.filterPolicy).forEach(filter => { total *= filter.length; });
            if (total > 100) {
                throw new Error(`The total combination of values (${total}) must not exceed 100.`);
            }
        }
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        new sns_generated_1.CfnSubscription(this, 'Resource', {
            endpoint: props.endpoint,
            protocol: props.protocol,
            topicArn: props.topic.topicArn,
            rawMessageDelivery: props.rawMessageDelivery,
            filterPolicy: this.filterPolicy,
            region: props.region,
            redrivePolicy: this.buildDeadLetterConfig(this.deadLetterQueue),
        });
    }
    buildDeadLetterQueue(props) {
        if (!props.deadLetterQueue) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue;
        deadLetterQueue.addToResourcePolicy(new aws_iam_1.PolicyStatement({
            resources: [deadLetterQueue.queueArn],
            actions: ['sqs:SendMessage'],
            principals: [new aws_iam_1.ServicePrincipal('sns.amazonaws.com')],
            conditions: {
                ArnEquals: { 'aws:SourceArn': props.topic.topicArn },
            },
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                deadLetterTargetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
}
exports.Subscription = Subscription;
/**
 * The type of subscription, controlling the type of the endpoint parameter.
 */
var SubscriptionProtocol;
(function (SubscriptionProtocol) {
    /**
     * JSON-encoded message is POSTED to an HTTP url.
     */
    SubscriptionProtocol["HTTP"] = "http";
    /**
     * JSON-encoded message is POSTed to an HTTPS url.
     */
    SubscriptionProtocol["HTTPS"] = "https";
    /**
     * Notifications are sent via email.
     */
    SubscriptionProtocol["EMAIL"] = "email";
    /**
     * Notifications are JSON-encoded and sent via mail.
     */
    SubscriptionProtocol["EMAIL_JSON"] = "email-json";
    /**
     * Notification is delivered by SMS
     */
    SubscriptionProtocol["SMS"] = "sms";
    /**
     * Notifications are enqueued into an SQS queue.
     */
    SubscriptionProtocol["SQS"] = "sqs";
    /**
     * JSON-encoded notifications are sent to a mobile app endpoint.
     */
    SubscriptionProtocol["APPLICATION"] = "application";
    /**
     * Notifications trigger a Lambda function.
     */
    SubscriptionProtocol["LAMBDA"] = "lambda";
})(SubscriptionProtocol = exports.SubscriptionProtocol || (exports.SubscriptionProtocol = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic3Vic2NyaXB0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsic3Vic2NyaXB0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDJDQUFrRSxDQUFDLG1EQUFtRDtBQUV0SCxxQ0FBaUQsQ0FBQyxnREFBZ0Q7QUFDbEcsbURBQWtEO0FBd0RsRDs7Ozs7R0FLRztBQUNILE1BQWEsWUFBYSxTQUFRLGVBQVE7SUFRdEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF3QjtRQUM5RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBQ2pCLElBQUksS0FBSyxDQUFDLGtCQUFrQixJQUFJLENBQUMsTUFBTSxFQUFFLE9BQU8sRUFBRSxLQUFLLENBQUMsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUNsRixNQUFNLElBQUksS0FBSyxDQUFDLDRFQUE0RSxDQUFDLENBQUM7U0FDakc7UUFDRCxJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDcEIsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO2dCQUM1QyxNQUFNLElBQUksS0FBSyxDQUFDLDBEQUEwRCxDQUFDLENBQUM7YUFDL0U7WUFDRCxJQUFJLENBQUMsWUFBWSxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQztpQkFDakQsTUFBTSxDQUFDLENBQUMsR0FBRyxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLEVBQUUsR0FBRyxHQUFHLEVBQUUsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUMsVUFBVSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztZQUNsRSxJQUFJLEtBQUssR0FBRyxDQUFDLENBQUM7WUFDZCxNQUFNLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUUsR0FBRyxLQUFLLElBQUksTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1lBQ2hGLElBQUksS0FBSyxHQUFHLEdBQUcsRUFBRTtnQkFDYixNQUFNLElBQUksS0FBSyxDQUFDLG9DQUFvQyxLQUFLLHdCQUF3QixDQUFDLENBQUM7YUFDdEY7U0FDSjtRQUNELElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDLG9CQUFvQixDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3hELElBQUksK0JBQWUsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2xDLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUTtZQUN4QixRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsUUFBUSxFQUFFLEtBQUssQ0FBQyxLQUFLLENBQUMsUUFBUTtZQUM5QixrQkFBa0IsRUFBRSxLQUFLLENBQUMsa0JBQWtCO1lBQzVDLFlBQVksRUFBRSxJQUFJLENBQUMsWUFBWTtZQUMvQixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU07WUFDcEIsYUFBYSxFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDO1NBQ2xFLENBQUMsQ0FBQztJQUNQLENBQUM7SUFDTyxvQkFBb0IsQ0FBQyxLQUF3QjtRQUNqRCxJQUFJLENBQUMsS0FBSyxDQUFDLGVBQWUsRUFBRTtZQUN4QixPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE1BQU0sZUFBZSxHQUFHLEtBQUssQ0FBQyxlQUFlLENBQUM7UUFDOUMsZUFBZSxDQUFDLG1CQUFtQixDQUFDLElBQUkseUJBQWUsQ0FBQztZQUNwRCxTQUFTLEVBQUUsQ0FBQyxlQUFlLENBQUMsUUFBUSxDQUFDO1lBQ3JDLE9BQU8sRUFBRSxDQUFDLGlCQUFpQixDQUFDO1lBQzVCLFVBQVUsRUFBRSxDQUFDLElBQUksMEJBQWdCLENBQUMsbUJBQW1CLENBQUMsQ0FBQztZQUN2RCxVQUFVLEVBQUU7Z0JBQ1IsU0FBUyxFQUFFLEVBQUUsZUFBZSxFQUFFLEtBQUssQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFO2FBQ3ZEO1NBQ0osQ0FBQyxDQUFDLENBQUM7UUFDSixPQUFPLGVBQWUsQ0FBQztJQUMzQixDQUFDO0lBQ08scUJBQXFCLENBQUMsZUFBd0I7UUFDbEQsSUFBSSxlQUFlLEVBQUU7WUFDakIsT0FBTztnQkFDSCxtQkFBbUIsRUFBRSxlQUFlLENBQUMsUUFBUTthQUNoRCxDQUFDO1NBQ0w7YUFDSTtZQUNELE9BQU8sU0FBUyxDQUFDO1NBQ3BCO0lBQ0wsQ0FBQztDQUNKO0FBN0RELG9DQTZEQztBQUNEOztHQUVHO0FBQ0gsSUFBWSxvQkFpQ1g7QUFqQ0QsV0FBWSxvQkFBb0I7SUFDNUI7O09BRUc7SUFDSCxxQ0FBYSxDQUFBO0lBQ2I7O09BRUc7SUFDSCx1Q0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCx1Q0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCxpREFBeUIsQ0FBQTtJQUN6Qjs7T0FFRztJQUNILG1DQUFXLENBQUE7SUFDWDs7T0FFRztJQUNILG1DQUFXLENBQUE7SUFDWDs7T0FFRztJQUNILG1EQUEyQixDQUFBO0lBQzNCOztPQUVHO0lBQ0gseUNBQWlCLENBQUE7QUFDckIsQ0FBQyxFQWpDVyxvQkFBb0IsR0FBcEIsNEJBQW9CLEtBQXBCLDRCQUFvQixRQWlDL0IiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgeyBQb2xpY3lTdGF0ZW1lbnQsIFNlcnZpY2VQcmluY2lwYWwgfSBmcm9tIFwiLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCB7IElRdWV1ZSB9IGZyb20gXCIuLi8uLi9hd3Mtc3FzXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3Mtc3FzJ1xuaW1wb3J0IHsgQ29uc3RydWN0LCBSZXNvdXJjZSB9IGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQ2ZuU3Vic2NyaXB0aW9uIH0gZnJvbSAnLi9zbnMuZ2VuZXJhdGVkJztcbmltcG9ydCB7IFN1YnNjcmlwdGlvbkZpbHRlciB9IGZyb20gJy4vc3Vic2NyaXB0aW9uLWZpbHRlcic7XG5pbXBvcnQgeyBJVG9waWMgfSBmcm9tICcuL3RvcGljLWJhc2UnO1xuLyoqXG4gKiBPcHRpb25zIGZvciBjcmVhdGluZyBhIG5ldyBzdWJzY3JpcHRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTdWJzY3JpcHRpb25PcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBXaGF0IHR5cGUgb2Ygc3Vic2NyaXB0aW9uIHRvIGFkZC5cbiAgICAgKi9cbiAgICByZWFkb25seSBwcm90b2NvbDogU3Vic2NyaXB0aW9uUHJvdG9jb2w7XG4gICAgLyoqXG4gICAgICogVGhlIHN1YnNjcmlwdGlvbiBlbmRwb2ludC5cbiAgICAgKlxuICAgICAqIFRoZSBtZWFuaW5nIG9mIHRoaXMgdmFsdWUgZGVwZW5kcyBvbiB0aGUgdmFsdWUgZm9yICdwcm90b2NvbCcuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW5kcG9pbnQ6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiB0cnVlIGlmIHJhdyBtZXNzYWdlIGRlbGl2ZXJ5IGlzIGVuYWJsZWQgZm9yIHRoZSBzdWJzY3JpcHRpb24uIFJhdyBtZXNzYWdlcyBhcmUgZnJlZSBvZiBKU09OIGZvcm1hdHRpbmcgYW5kIGNhbiBiZVxuICAgICAqIHNlbnQgdG8gSFRUUC9TIGFuZCBBbWF6b24gU1FTIGVuZHBvaW50cy4gRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBHZXRTdWJzY3JpcHRpb25BdHRyaWJ1dGVzIGluIHRoZSBBbWF6b24gU2ltcGxlXG4gICAgICogTm90aWZpY2F0aW9uIFNlcnZpY2UgQVBJIFJlZmVyZW5jZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmF3TWVzc2FnZURlbGl2ZXJ5PzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBUaGUgZmlsdGVyIHBvbGljeS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gYWxsIG1lc3NhZ2VzIGFyZSBkZWxpdmVyZWRcbiAgICAgKi9cbiAgICByZWFkb25seSBmaWx0ZXJQb2xpY3k/OiB7XG4gICAgICAgIFthdHRyaWJ1dGU6IHN0cmluZ106IFN1YnNjcmlwdGlvbkZpbHRlcjtcbiAgICB9O1xuICAgIC8qKlxuICAgICAqIFRoZSByZWdpb24gd2hlcmUgdGhlIHRvcGljIHJlc2lkZXMsIGluIHRoZSBjYXNlIG9mIGNyb3NzLXJlZ2lvbiBzdWJzY3JpcHRpb25zXG4gICAgICogQGxpbmsgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXJlc291cmNlLXNucy1zdWJzY3JpcHRpb24uaHRtbCNjZm4tc25zLXN1YnNjcmlwdGlvbi1yZWdpb25cbiAgICAgKiBAZGVmYXVsdCAtIHRoZSByZWdpb24gd2hlcmUgdGhlIENsb3VkRm9ybWF0aW9uIHN0YWNrIGlzIGJlaW5nIGRlcGxveWVkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlZ2lvbj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBRdWV1ZSB0byBiZSB1c2VkIGFzIGRlYWQgbGV0dGVyIHF1ZXVlLlxuICAgICAqIElmIG5vdCBwYXNzZWQgbm8gZGVhZCBsZXR0ZXIgcXVldWUgaXMgZW5hYmxlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gZGVhZCBsZXR0ZXIgcXVldWUgZW5hYmxlZC5cbiAgICAgKi9cbiAgICByZWFkb25seSBkZWFkTGV0dGVyUXVldWU/OiBJUXVldWU7XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGNyZWF0aW5nIGEgbmV3IHN1YnNjcmlwdGlvblxuICovXG5leHBvcnQgaW50ZXJmYWNlIFN1YnNjcmlwdGlvblByb3BzIGV4dGVuZHMgU3Vic2NyaXB0aW9uT3B0aW9ucyB7XG4gICAgLyoqXG4gICAgICogVGhlIHRvcGljIHRvIHN1YnNjcmliZSB0by5cbiAgICAgKi9cbiAgICByZWFkb25seSB0b3BpYzogSVRvcGljO1xufVxuLyoqXG4gKiBBIG5ldyBzdWJzY3JpcHRpb24uXG4gKlxuICogUHJlZmVyIHRvIHVzZSB0aGUgYElUb3BpYy5hZGRTdWJzY3JpcHRpb24oKWAgbWV0aG9kcyB0byBjcmVhdGUgaW5zdGFuY2VzIG9mXG4gKiB0aGlzIGNsYXNzLlxuICovXG5leHBvcnQgY2xhc3MgU3Vic2NyaXB0aW9uIGV4dGVuZHMgUmVzb3VyY2Uge1xuICAgIC8qKlxuICAgICAqIFRoZSBETFEgYXNzb2NpYXRlZCB3aXRoIHRoaXMgc3Vic2NyaXB0aW9uIGlmIHByZXNlbnQuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGRlYWRMZXR0ZXJRdWV1ZT86IElRdWV1ZTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGZpbHRlclBvbGljeT86IHtcbiAgICAgICAgW2F0dHJpYnV0ZTogc3RyaW5nXTogYW55W107XG4gICAgfTtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogU3Vic2NyaXB0aW9uUHJvcHMpIHtcbiAgICAgICAgc3VwZXIoc2NvcGUsIGlkKTtcbiAgICAgICAgaWYgKHByb3BzLnJhd01lc3NhZ2VEZWxpdmVyeSAmJiBbJ2h0dHAnLCAnaHR0cHMnLCAnc3FzJ10uaW5kZXhPZihwcm9wcy5wcm90b2NvbCkgPCAwKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1JhdyBtZXNzYWdlIGRlbGl2ZXJ5IGNhbiBvbmx5IGJlIGVuYWJsZWQgZm9yIEhUVFAvUyBhbmQgU1FTIHN1YnNjcmlwdGlvbnMuJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmZpbHRlclBvbGljeSkge1xuICAgICAgICAgICAgaWYgKE9iamVjdC5rZXlzKHByb3BzLmZpbHRlclBvbGljeSkubGVuZ3RoID4gNSkge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQSBmaWx0ZXIgcG9saWN5IGNhbiBoYXZlIGEgbWF4aW11bSBvZiA1IGF0dHJpYnV0ZSBuYW1lcy4nKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIHRoaXMuZmlsdGVyUG9saWN5ID0gT2JqZWN0LmVudHJpZXMocHJvcHMuZmlsdGVyUG9saWN5KVxuICAgICAgICAgICAgICAgIC5yZWR1Y2UoKGFjYywgW2ssIHZdKSA9PiAoeyAuLi5hY2MsIFtrXTogdi5jb25kaXRpb25zIH0pLCB7fSk7XG4gICAgICAgICAgICBsZXQgdG90YWwgPSAxO1xuICAgICAgICAgICAgT2JqZWN0LnZhbHVlcyh0aGlzLmZpbHRlclBvbGljeSkuZm9yRWFjaChmaWx0ZXIgPT4geyB0b3RhbCAqPSBmaWx0ZXIubGVuZ3RoOyB9KTtcbiAgICAgICAgICAgIGlmICh0b3RhbCA+IDEwMCkge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgVGhlIHRvdGFsIGNvbWJpbmF0aW9uIG9mIHZhbHVlcyAoJHt0b3RhbH0pIG11c3Qgbm90IGV4Y2VlZCAxMDAuYCk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5kZWFkTGV0dGVyUXVldWUgPSB0aGlzLmJ1aWxkRGVhZExldHRlclF1ZXVlKHByb3BzKTtcbiAgICAgICAgbmV3IENmblN1YnNjcmlwdGlvbih0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBlbmRwb2ludDogcHJvcHMuZW5kcG9pbnQsXG4gICAgICAgICAgICBwcm90b2NvbDogcHJvcHMucHJvdG9jb2wsXG4gICAgICAgICAgICB0b3BpY0FybjogcHJvcHMudG9waWMudG9waWNBcm4sXG4gICAgICAgICAgICByYXdNZXNzYWdlRGVsaXZlcnk6IHByb3BzLnJhd01lc3NhZ2VEZWxpdmVyeSxcbiAgICAgICAgICAgIGZpbHRlclBvbGljeTogdGhpcy5maWx0ZXJQb2xpY3ksXG4gICAgICAgICAgICByZWdpb246IHByb3BzLnJlZ2lvbixcbiAgICAgICAgICAgIHJlZHJpdmVQb2xpY3k6IHRoaXMuYnVpbGREZWFkTGV0dGVyQ29uZmlnKHRoaXMuZGVhZExldHRlclF1ZXVlKSxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIHByaXZhdGUgYnVpbGREZWFkTGV0dGVyUXVldWUocHJvcHM6IFN1YnNjcmlwdGlvblByb3BzKSB7XG4gICAgICAgIGlmICghcHJvcHMuZGVhZExldHRlclF1ZXVlKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGRlYWRMZXR0ZXJRdWV1ZSA9IHByb3BzLmRlYWRMZXR0ZXJRdWV1ZTtcbiAgICAgICAgZGVhZExldHRlclF1ZXVlLmFkZFRvUmVzb3VyY2VQb2xpY3kobmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICByZXNvdXJjZXM6IFtkZWFkTGV0dGVyUXVldWUucXVldWVBcm5dLFxuICAgICAgICAgICAgYWN0aW9uczogWydzcXM6U2VuZE1lc3NhZ2UnXSxcbiAgICAgICAgICAgIHByaW5jaXBhbHM6IFtuZXcgU2VydmljZVByaW5jaXBhbCgnc25zLmFtYXpvbmF3cy5jb20nKV0sXG4gICAgICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgICAgICAgQXJuRXF1YWxzOiB7ICdhd3M6U291cmNlQXJuJzogcHJvcHMudG9waWMudG9waWNBcm4gfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pKTtcbiAgICAgICAgcmV0dXJuIGRlYWRMZXR0ZXJRdWV1ZTtcbiAgICB9XG4gICAgcHJpdmF0ZSBidWlsZERlYWRMZXR0ZXJDb25maWcoZGVhZExldHRlclF1ZXVlPzogSVF1ZXVlKSB7XG4gICAgICAgIGlmIChkZWFkTGV0dGVyUXVldWUpIHtcbiAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgZGVhZExldHRlclRhcmdldEFybjogZGVhZExldHRlclF1ZXVlLnF1ZXVlQXJuLFxuICAgICAgICAgICAgfTtcbiAgICAgICAgfVxuICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIH1cbiAgICB9XG59XG4vKipcbiAqIFRoZSB0eXBlIG9mIHN1YnNjcmlwdGlvbiwgY29udHJvbGxpbmcgdGhlIHR5cGUgb2YgdGhlIGVuZHBvaW50IHBhcmFtZXRlci5cbiAqL1xuZXhwb3J0IGVudW0gU3Vic2NyaXB0aW9uUHJvdG9jb2wge1xuICAgIC8qKlxuICAgICAqIEpTT04tZW5jb2RlZCBtZXNzYWdlIGlzIFBPU1RFRCB0byBhbiBIVFRQIHVybC5cbiAgICAgKi9cbiAgICBIVFRQID0gJ2h0dHAnLFxuICAgIC8qKlxuICAgICAqIEpTT04tZW5jb2RlZCBtZXNzYWdlIGlzIFBPU1RlZCB0byBhbiBIVFRQUyB1cmwuXG4gICAgICovXG4gICAgSFRUUFMgPSAnaHR0cHMnLFxuICAgIC8qKlxuICAgICAqIE5vdGlmaWNhdGlvbnMgYXJlIHNlbnQgdmlhIGVtYWlsLlxuICAgICAqL1xuICAgIEVNQUlMID0gJ2VtYWlsJyxcbiAgICAvKipcbiAgICAgKiBOb3RpZmljYXRpb25zIGFyZSBKU09OLWVuY29kZWQgYW5kIHNlbnQgdmlhIG1haWwuXG4gICAgICovXG4gICAgRU1BSUxfSlNPTiA9ICdlbWFpbC1qc29uJyxcbiAgICAvKipcbiAgICAgKiBOb3RpZmljYXRpb24gaXMgZGVsaXZlcmVkIGJ5IFNNU1xuICAgICAqL1xuICAgIFNNUyA9ICdzbXMnLFxuICAgIC8qKlxuICAgICAqIE5vdGlmaWNhdGlvbnMgYXJlIGVucXVldWVkIGludG8gYW4gU1FTIHF1ZXVlLlxuICAgICAqL1xuICAgIFNRUyA9ICdzcXMnLFxuICAgIC8qKlxuICAgICAqIEpTT04tZW5jb2RlZCBub3RpZmljYXRpb25zIGFyZSBzZW50IHRvIGEgbW9iaWxlIGFwcCBlbmRwb2ludC5cbiAgICAgKi9cbiAgICBBUFBMSUNBVElPTiA9ICdhcHBsaWNhdGlvbicsXG4gICAgLyoqXG4gICAgICogTm90aWZpY2F0aW9ucyB0cmlnZ2VyIGEgTGFtYmRhIGZ1bmN0aW9uLlxuICAgICAqL1xuICAgIExBTUJEQSA9ICdsYW1iZGEnXG59XG4iXX0=