"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DropSpamReceiptRule = exports.ReceiptRule = exports.TlsPolicy = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const ses_generated_1 = require("./ses.generated");
/**
 * The type of TLS policy for a receipt rule.
 */
var TlsPolicy;
(function (TlsPolicy) {
    /**
     * Do not check for TLS.
     */
    TlsPolicy["OPTIONAL"] = "Optional";
    /**
     * Bounce emails that are not received over TLS.
     */
    TlsPolicy["REQUIRE"] = "Require";
})(TlsPolicy = exports.TlsPolicy || (exports.TlsPolicy = {}));
/**
 * A new receipt rule.
 */
class ReceiptRule extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.receiptRuleName,
        });
        this.actions = new Array();
        const resource = new ses_generated_1.CfnReceiptRule(this, 'Resource', {
            after: props.after ? props.after.receiptRuleName : undefined,
            rule: {
                actions: core_1.Lazy.anyValue({ produce: () => this.renderActions() }),
                enabled: props.enabled === undefined ? true : props.enabled,
                name: this.physicalName,
                recipients: props.recipients,
                scanEnabled: props.scanEnabled,
                tlsPolicy: props.tlsPolicy,
            },
            ruleSetName: props.ruleSet.receiptRuleSetName,
        });
        this.receiptRuleName = resource.ref;
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    static fromReceiptRuleName(scope, id, receiptRuleName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.receiptRuleName = receiptRuleName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an action to this receipt rule.
     */
    addAction(action) {
        this.actions.push(action.bind(this));
    }
    renderActions() {
        if (this.actions.length === 0) {
            return undefined;
        }
        return this.actions;
    }
}
exports.ReceiptRule = ReceiptRule;
/**
 * A rule added at the top of the rule set to drop spam/virus.
 *
 * @see https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
 */
class DropSpamReceiptRule extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fn = new lambda.SingletonFunction(this, 'Function', {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromInline(`exports.handler = ${dropSpamCode}`),
            uuid: '224e77f9-a32e-4b4d-ac32-983477abba16',
        });
        fn.addPermission('AllowSes', {
            action: 'lambda:InvokeFunction',
            principal: new iam.ServicePrincipal('ses.amazonaws.com'),
            sourceAccount: core_1.Aws.ACCOUNT_ID,
        });
        this.rule = new ReceiptRule(this, 'Rule', {
            actions: [
                {
                    bind: () => ({
                        lambdaAction: {
                            functionArn: fn.functionArn,
                            invocationType: 'RequestResponse',
                        },
                    }),
                },
            ],
            scanEnabled: true,
            ruleSet: props.ruleSet,
        });
    }
}
exports.DropSpamReceiptRule = DropSpamReceiptRule;
// Adapted from https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html
/* eslint-disable no-console */
function dropSpamCode(event, _, callback) {
    console.log('Spam filter');
    const sesNotification = event.Records[0].ses;
    console.log('SES Notification:\n', JSON.stringify(sesNotification, null, 2));
    // Check if any spam check failed
    if (sesNotification.receipt.spfVerdict.status === 'FAIL'
        || sesNotification.receipt.dkimVerdict.status === 'FAIL'
        || sesNotification.receipt.spamVerdict.status === 'FAIL'
        || sesNotification.receipt.virusVerdict.status === 'FAIL') {
        console.log('Dropping spam');
        // Stop processing rule set, dropping message
        callback(null, { disposition: 'STOP_RULE_SET' });
    }
    else {
        callback(null, null);
    }
}
//# sourceMappingURL=data:application/json;base64,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