"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheckType = exports.RoutingPolicy = exports.DnsRecordType = exports.Service = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const alias_target_instance_1 = require("./alias-target-instance");
const cname_instance_1 = require("./cname-instance");
const ip_instance_1 = require("./ip-instance");
const namespace_1 = require("./namespace");
const non_ip_instance_1 = require("./non-ip-instance");
const servicediscovery_generated_1 = require("./servicediscovery.generated");
class ServiceBase extends core_1.Resource {
}
/**
 * Define a CloudMap Service
 */
class Service extends ServiceBase {
    constructor(scope, id, props) {
        super(scope, id);
        const namespaceType = props.namespace.type;
        // Validations
        if (namespaceType === namespace_1.NamespaceType.HTTP && (props.routingPolicy || props.dnsRecordType)) {
            throw new Error('Cannot specify `routingPolicy` or `dnsRecord` when using an HTTP namespace.');
        }
        if (props.healthCheck && props.customHealthCheck) {
            throw new Error('Cannot specify both `healthCheckConfig` and `healthCheckCustomConfig`.');
        }
        if (namespaceType === namespace_1.NamespaceType.DNS_PRIVATE && props.healthCheck) {
            throw new Error('Cannot specify `healthCheckConfig` for a Private DNS namespace.');
        }
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.dnsRecordType === DnsRecordType.CNAME) {
            throw new Error('Cannot use `CNAME` record when routing policy is `Multivalue`.');
        }
        // Additional validation for eventual attachment of LBs
        // The same validation happens later on during the actual attachment
        // of LBs, but we need the property for the correct configuration of
        // routingPolicy anyway, so might as well do the validation as well.
        if (props.routingPolicy === RoutingPolicy.MULTIVALUE
            && props.loadBalancer) {
            throw new Error('Cannot register loadbalancers when routing policy is `Multivalue`.');
        }
        if (props.healthCheck
            && props.healthCheck.type === HealthCheckType.TCP
            && props.healthCheck.resourcePath) {
            throw new Error('Cannot specify `resourcePath` when using a `TCP` health check.');
        }
        // Set defaults where necessary
        const routingPolicy = (props.dnsRecordType === DnsRecordType.CNAME) || props.loadBalancer
            ? RoutingPolicy.WEIGHTED
            : RoutingPolicy.MULTIVALUE;
        const dnsRecordType = props.dnsRecordType || DnsRecordType.A;
        if (props.loadBalancer
            && (!(dnsRecordType === DnsRecordType.A
                || dnsRecordType === DnsRecordType.AAAA
                || dnsRecordType === DnsRecordType.A_AAAA))) {
            throw new Error('Must support `A` or `AAAA` records to register loadbalancers.');
        }
        const dnsConfig = props.namespace.type === namespace_1.NamespaceType.HTTP
            ? undefined
            : {
                dnsRecords: renderDnsRecords(dnsRecordType, props.dnsTtl),
                namespaceId: props.namespace.namespaceId,
                routingPolicy,
            };
        const healthCheckConfigDefaults = {
            type: HealthCheckType.HTTP,
            failureThreshold: 1,
            resourcePath: props.healthCheck && props.healthCheck.type !== HealthCheckType.TCP
                ? '/'
                : undefined,
        };
        const healthCheckConfig = props.healthCheck && { ...healthCheckConfigDefaults, ...props.healthCheck };
        const healthCheckCustomConfig = props.customHealthCheck;
        // Create service
        const service = new servicediscovery_generated_1.CfnService(this, 'Resource', {
            name: props.name,
            description: props.description,
            dnsConfig,
            healthCheckConfig,
            healthCheckCustomConfig,
            namespaceId: props.namespace.namespaceId,
        });
        this.serviceName = service.attrName;
        this.serviceArn = service.attrArn;
        this.serviceId = service.attrId;
        this.namespace = props.namespace;
        this.dnsRecordType = dnsRecordType;
        this.routingPolicy = routingPolicy;
    }
    static fromServiceAttributes(scope, id, attrs) {
        class Import extends ServiceBase {
            constructor() {
                super(...arguments);
                this.namespace = attrs.namespace;
                this.serviceId = attrs.serviceId;
                this.serviceArn = attrs.serviceArn;
                this.dnsRecordType = attrs.dnsRecordType;
                this.routingPolicy = attrs.routingPolicy;
                this.serviceName = attrs.serviceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Registers an ELB as a new instance with unique name instanceId in this service.
     */
    registerLoadBalancer(id, loadBalancer, customAttributes) {
        return new alias_target_instance_1.AliasTargetInstance(this, id, {
            service: this,
            dnsName: loadBalancer.loadBalancerDnsName,
            customAttributes,
        });
    }
    /**
     * Registers a resource that is accessible using values other than an IP address or a domain name (CNAME).
     */
    registerNonIpInstance(id, props) {
        return new non_ip_instance_1.NonIpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * Registers a resource that is accessible using an IP address.
     */
    registerIpInstance(id, props) {
        return new ip_instance_1.IpInstance(this, id, {
            service: this,
            ...props,
        });
    }
    /**
     * Registers a resource that is accessible using a CNAME.
     */
    registerCnameInstance(id, props) {
        return new cname_instance_1.CnameInstance(this, id, {
            service: this,
            ...props,
        });
    }
}
exports.Service = Service;
function renderDnsRecords(dnsRecordType, dnsTtl = core_1.Duration.minutes(1)) {
    const ttl = dnsTtl.toSeconds();
    if (dnsRecordType === DnsRecordType.A_AAAA) {
        return [{
                type: DnsRecordType.A,
                ttl,
            }, {
                type: DnsRecordType.AAAA,
                ttl,
            }];
    }
    else {
        return [{ type: dnsRecordType, ttl }];
    }
}
var DnsRecordType;
(function (DnsRecordType) {
    /**
     * An A record
     */
    DnsRecordType["A"] = "A";
    /**
     * An AAAA record
     */
    DnsRecordType["AAAA"] = "AAAA";
    /**
     * Both an A and AAAA record
     */
    DnsRecordType["A_AAAA"] = "A, AAAA";
    /**
     * A Srv record
     */
    DnsRecordType["SRV"] = "SRV";
    /**
     * A CNAME record
     */
    DnsRecordType["CNAME"] = "CNAME";
})(DnsRecordType = exports.DnsRecordType || (exports.DnsRecordType = {}));
var RoutingPolicy;
(function (RoutingPolicy) {
    /**
     * Route 53 returns the applicable value from one randomly selected instance from among the instances that you
     * registered using the same service.
     */
    RoutingPolicy["WEIGHTED"] = "WEIGHTED";
    /**
     * If you define a health check for the service and the health check is healthy, Route 53 returns the applicable value
     * for up to eight instances.
     */
    RoutingPolicy["MULTIVALUE"] = "MULTIVALUE";
})(RoutingPolicy = exports.RoutingPolicy || (exports.RoutingPolicy = {}));
var HealthCheckType;
(function (HealthCheckType) {
    /**
     * Route 53 tries to establish a TCP connection. If successful, Route 53 submits an HTTP request and waits for an HTTP
     * status code of 200 or greater and less than 400.
     */
    HealthCheckType["HTTP"] = "HTTP";
    /**
     * Route 53 tries to establish a TCP connection. If successful, Route 53 submits an HTTPS request and waits for an
     * HTTP status code of 200 or greater and less than 400.  If you specify HTTPS for the value of Type, the endpoint
     * must support TLS v1.0 or later.
     */
    HealthCheckType["HTTPS"] = "HTTPS";
    /**
     * Route 53 tries to establish a TCP connection.
     * If you specify TCP for Type, don't specify a value for ResourcePath.
     */
    HealthCheckType["TCP"] = "TCP";
})(HealthCheckType = exports.HealthCheckType || (exports.HealthCheckType = {}));
//# sourceMappingURL=data:application/json;base64,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