"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecretRotation = exports.SecretRotationApplication = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const serverless = require("../../aws-sam"); // Automatically re-written from '@aws-cdk/aws-sam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A secret rotation serverless application.
 */
class SecretRotationApplication {
    constructor(applicationId, semanticVersion, options) {
        this.applicationId = `arn:aws:serverlessrepo:us-east-1:297356227824:applications/${applicationId}`;
        this.semanticVersion = semanticVersion;
        this.isMultiUser = options && options.isMultiUser;
    }
}
exports.SecretRotationApplication = SecretRotationApplication;
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the single user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the multi user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the single user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the multi user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the single user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the multi user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the single user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the multi user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the single user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the multi user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the single user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the multi user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the single user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationSingleUser', '1.1.60');
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the multi user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationMultiUser', '1.1.60', {
    isMultiUser: true,
});
/**
 * Secret rotation for a service or database
 */
class SecretRotation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.target.connections.defaultPort) {
            throw new Error('The `target` connections must have a default port range.');
        }
        if (props.application.isMultiUser && !props.masterSecret) {
            throw new Error('The `masterSecret` must be specified for application using the multi user scheme.');
        }
        // Max length of 64 chars, get the last 64 chars
        const uniqueId = this.node.uniqueId;
        const rotationFunctionName = uniqueId.substring(Math.max(uniqueId.length - 64, 0), uniqueId.length);
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
        });
        props.target.connections.allowDefaultPortFrom(securityGroup);
        const parameters = {
            endpoint: `https://secretsmanager.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}`,
            functionName: rotationFunctionName,
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds.join(','),
            vpcSecurityGroupIds: securityGroup.securityGroupId,
        };
        if (props.excludeCharacters) {
            parameters.excludeCharacters = props.excludeCharacters;
        }
        if (props.secret.encryptionKey) {
            parameters.kmsKeyArn = props.secret.encryptionKey.keyArn;
        }
        if (props.masterSecret) {
            parameters.masterSecretArn = props.masterSecret.secretArn;
            if (props.masterSecret.encryptionKey) {
                parameters.masterSecretKmsKeyArn = props.masterSecret.encryptionKey.keyArn;
            }
        }
        const application = new serverless.CfnApplication(this, 'Resource', {
            location: props.application,
            parameters,
        });
        // This creates a CF a dependency between the rotation schedule and the
        // serverless application. This is needed because it's the application
        // that creates the Lambda permission to invoke the function.
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html
        const rotationLambda = lambda.Function.fromFunctionArn(this, 'RotationLambda', core_1.Token.asString(application.getAtt('Outputs.RotationLambdaARN')));
        props.secret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
            automaticallyAfter: props.automaticallyAfter,
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
        if (props.masterSecret) {
            props.masterSecret.denyAccountRootDelete();
        }
    }
}
exports.SecretRotation = SecretRotation;
//# sourceMappingURL=data:application/json;base64,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