"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ReplaceKey = exports.BucketAccessControl = exports.EventType = exports.BucketEncryption = exports.Bucket = exports.InventoryObjectVersion = exports.InventoryFrequency = exports.InventoryFormat = exports.RedirectProtocol = exports.HttpMethods = exports.BlockPublicAccess = void 0;
const os_1 = require("os");
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const bucket_policy_1 = require("./bucket-policy");
const notifications_resource_1 = require("./notifications-resource");
const perms = require("./perms");
const s3_generated_1 = require("./s3.generated");
const util_1 = require("./util");
/**
 * Represents an S3 Bucket.
 *
 * Buckets can be either defined within this stack:
 *
 *   new Bucket(this, 'MyBucket', { props });
 *
 * Or imported from an existing bucket:
 *
 *   Bucket.import(this, 'MyImportedBucket', { bucketArn: ... });
 *
 * You can also export a bucket and import it into another stack:
 *
 *   const ref = myBucket.export();
 *   Bucket.import(this, 'MyImportedBucket', ref);
 *
 */
class BucketBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Indicates if a bucket resource policy should automatically created upon
         * the first call to `addToResourcePolicy`.
         */
        this.autoCreatePolicy = false;
    }
    /**
     * Define a CloudWatch event that triggers when something happens to this repository
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            source: ['aws.s3'],
            detailType: ['AWS API Call via CloudTrail'],
            detail: {
                resources: {
                    ARN: options.paths ? options.paths.map(p => this.arnForObjects(p)) : [this.bucketArn],
                },
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object is uploaded
     * to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailPutObject(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: ['PutObject'],
            },
        });
        return rule;
    }
    /**
     * Defines an AWS CloudWatch event that triggers when an object at the
     * specified paths (keys) in this bucket are written to.  This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule
     * @param options Options for adding the rule
     */
    onCloudTrailWriteObject(id, options = {}) {
        const rule = this.onCloudTrailEvent(id, options);
        rule.addEventPattern({
            detail: {
                eventName: [
                    'CompleteMultipartUpload',
                    'CopyObject',
                    'PutObject',
                ],
                requestParameters: {
                    bucketName: [this.bucketName],
                    key: options.paths,
                },
            },
        });
        return rule;
    }
    /**
     * Adds a statement to the resource policy for a principal (i.e.
     * account/role/service) to perform actions on this bucket and/or it's
     * contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for
     * this bucket or objects.
     */
    addToResourcePolicy(permission) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new bucket_policy_1.BucketPolicy(this, 'Policy', { bucket: this });
        }
        if (this.policy) {
            this.policy.document.addStatements(permission);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.policy) === null || _a === void 0 ? void 0 : _a.document.validateForResourcePolicy()) || []);
        return errors;
    }
    /**
     * The https URL of an S3 object. For example:
     * @example https://s3.us-west-1.amazonaws.com/onlybucket
     * @example https://s3.us-west-1.amazonaws.com/bucket/key
     * @example https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     * @param key The S3 key of the object. If not specified, the URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    urlForObject(key) {
        const stack = core_1.Stack.of(this);
        const prefix = `https://s3.${stack.region}.${stack.urlSuffix}/`;
        return this.buildUrl(prefix, key);
    }
    /**
     * The S3 URL of an S3 object. For example:
     * @example s3://onlybucket
     * @example s3://bucket/key
     * @param key The S3 key of the object. If not specified, the S3 URL of the
     *      bucket is returned.
     * @returns an ObjectS3Url token
     */
    s3UrlForObject(key) {
        return this.buildUrl('s3://', key);
    }
    /**
     * Returns an ARN that represents all objects within the bucket that match
     * the key pattern specified. To represent all keys, specify ``"*"``.
     *
     * If you specify multiple components for keyPattern, they will be concatenated::
     *
     *   arnForObjects('home/', team, '/', user, '/*')
     *
     */
    arnForObjects(keyPattern) {
        return `${this.bucketArn}/${keyPattern}`;
    }
    /**
     * Grant read permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantRead(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_READ_ACTIONS, perms.KEY_READ_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantWrite(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_WRITE_ACTIONS, perms.KEY_WRITE_ACTIONS, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantPut(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_PUT_ACTIONS, perms.KEY_WRITE_ACTIONS, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants s3:DeleteObject* permission to an IAM pricipal for objects
     * in this bucket.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantDelete(identity, objectsKeyPattern = '*') {
        return this.grant(identity, perms.BUCKET_DELETE_ACTIONS, [], this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Grants read/write permissions for this bucket and it's contents to an IAM
     * principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * @param identity The principal
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*')
     */
    grantReadWrite(identity, objectsKeyPattern = '*') {
        const bucketActions = perms.BUCKET_READ_ACTIONS.concat(perms.BUCKET_WRITE_ACTIONS);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.grant(identity, bucketActions, keyActions, this.bucketArn, this.arnForObjects(objectsKeyPattern));
    }
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *     const grant = bucket.grantPublicAccess();
     *     grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow. Default is "s3:GetObject".
     */
    grantPublicAccess(keyPrefix = '*', ...allowedActions) {
        if (this.disallowPublicAccess) {
            throw new Error("Cannot grant public access when 'blockPublicPolicy' is enabled");
        }
        allowedActions = allowedActions.length > 0 ? allowedActions : ['s3:GetObject'];
        return iam.Grant.addToPrincipalOrResource({
            actions: allowedActions,
            resourceArns: [this.arnForObjects(keyPrefix)],
            grantee: new iam.Anyone(),
            resource: this,
        });
    }
    buildUrl(prefix, key) {
        const components = [
            prefix,
            this.bucketName,
        ];
        if (key) {
            // trim prepending '/'
            if (typeof key === 'string' && key.startsWith('/')) {
                key = key.substr(1);
            }
            components.push('/');
            components.push(key);
        }
        return components.join('');
    }
    grant(grantee, bucketActions, keyActions, resourceArn, ...otherResourceArns) {
        const resources = [resourceArn, ...otherResourceArns];
        const ret = iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: bucketActions,
            resourceArns: resources,
            resource: this,
        });
        if (this.encryptionKey && keyActions && keyActions.length !== 0) {
            this.encryptionKey.grant(grantee, ...keyActions);
        }
        return ret;
    }
}
class BlockPublicAccess {
    constructor(options) {
        this.blockPublicAcls = options.blockPublicAcls;
        this.blockPublicPolicy = options.blockPublicPolicy;
        this.ignorePublicAcls = options.ignorePublicAcls;
        this.restrictPublicBuckets = options.restrictPublicBuckets;
    }
}
exports.BlockPublicAccess = BlockPublicAccess;
BlockPublicAccess.BLOCK_ALL = new BlockPublicAccess({
    blockPublicAcls: true,
    blockPublicPolicy: true,
    ignorePublicAcls: true,
    restrictPublicBuckets: true,
});
BlockPublicAccess.BLOCK_ACLS = new BlockPublicAccess({
    blockPublicAcls: true,
    ignorePublicAcls: true,
});
/**
 * All http request methods
 */
var HttpMethods;
(function (HttpMethods) {
    /**
     * The GET method requests a representation of the specified resource.
     */
    HttpMethods["GET"] = "GET";
    /**
     * The PUT method replaces all current representations of the target resource with the request payload.
     */
    HttpMethods["PUT"] = "PUT";
    /**
     * The HEAD method asks for a response identical to that of a GET request, but without the response body.
     */
    HttpMethods["HEAD"] = "HEAD";
    /**
     * The POST method is used to submit an entity to the specified resource, often causing a change in state or side effects on the server.
     */
    HttpMethods["POST"] = "POST";
    /**
     * The DELETE method deletes the specified resource.
     */
    HttpMethods["DELETE"] = "DELETE";
})(HttpMethods = exports.HttpMethods || (exports.HttpMethods = {}));
/**
 * All http request methods
 */
var RedirectProtocol;
(function (RedirectProtocol) {
    RedirectProtocol["HTTP"] = "http";
    RedirectProtocol["HTTPS"] = "https";
})(RedirectProtocol = exports.RedirectProtocol || (exports.RedirectProtocol = {}));
/**
 * All supported inventory list formats.
 */
var InventoryFormat;
(function (InventoryFormat) {
    /**
     * Generate the inventory list as CSV.
     */
    InventoryFormat["CSV"] = "CSV";
    /**
     * Generate the inventory list as Parquet.
     */
    InventoryFormat["PARQUET"] = "Parquet";
    /**
     * Generate the inventory list as Parquet.
     */
    InventoryFormat["ORC"] = "ORC";
})(InventoryFormat = exports.InventoryFormat || (exports.InventoryFormat = {}));
/**
 * All supported inventory frequencies.
 */
var InventoryFrequency;
(function (InventoryFrequency) {
    /**
     * A report is generated every day.
     */
    InventoryFrequency["DAILY"] = "Daily";
    /**
     * A report is generated every Sunday (UTC timezone) after the initial report.
     */
    InventoryFrequency["WEEKLY"] = "Weekly";
})(InventoryFrequency = exports.InventoryFrequency || (exports.InventoryFrequency = {}));
/**
 * Inventory version support.
 */
var InventoryObjectVersion;
(function (InventoryObjectVersion) {
    /**
     * Includes all versions of each object in the report.
     */
    InventoryObjectVersion["ALL"] = "All";
    /**
     * Includes only the current version of each object in the report.
     */
    InventoryObjectVersion["CURRENT"] = "Current";
})(InventoryObjectVersion = exports.InventoryObjectVersion || (exports.InventoryObjectVersion = {}));
/**
 * An S3 bucket with associated policy objects
 *
 * This bucket does not yet have all features that exposed by the underlying
 * BucketResource.
 */
class Bucket extends BucketBase {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: props.bucketName,
        });
        this.autoCreatePolicy = true;
        this.lifecycleRules = [];
        this.metrics = [];
        this.cors = [];
        this.inventories = [];
        const { bucketEncryption, encryptionKey } = this.parseEncryption(props);
        this.validateBucketName(this.physicalName);
        const websiteConfiguration = this.renderWebsiteConfiguration(props);
        this.isWebsite = (websiteConfiguration !== undefined);
        const resource = new s3_generated_1.CfnBucket(this, 'Resource', {
            bucketName: this.physicalName,
            bucketEncryption,
            versioningConfiguration: props.versioned ? { status: 'Enabled' } : undefined,
            lifecycleConfiguration: core_1.Lazy.anyValue({ produce: () => this.parseLifecycleConfiguration() }),
            websiteConfiguration,
            publicAccessBlockConfiguration: props.blockPublicAccess,
            metricsConfigurations: core_1.Lazy.anyValue({ produce: () => this.parseMetricConfiguration() }),
            corsConfiguration: core_1.Lazy.anyValue({ produce: () => this.parseCorsConfiguration() }),
            accessControl: core_1.Lazy.stringValue({ produce: () => this.accessControl }),
            loggingConfiguration: this.parseServerAccessLogs(props),
            inventoryConfigurations: core_1.Lazy.anyValue({ produce: () => this.parseInventoryConfiguration() }),
        });
        resource.applyRemovalPolicy(props.removalPolicy);
        this.versioned = props.versioned;
        this.encryptionKey = encryptionKey;
        this.bucketName = this.getResourceNameAttribute(resource.ref);
        this.bucketArn = this.getResourceArnAttribute(resource.attrArn, {
            region: '',
            account: '',
            service: 's3',
            resource: this.physicalName,
        });
        this.bucketDomainName = resource.attrDomainName;
        this.bucketWebsiteUrl = resource.attrWebsiteUrl;
        this.bucketWebsiteDomainName = core_1.Fn.select(2, core_1.Fn.split('/', this.bucketWebsiteUrl));
        this.bucketDualStackDomainName = resource.attrDualStackDomainName;
        this.bucketRegionalDomainName = resource.attrRegionalDomainName;
        this.disallowPublicAccess = props.blockPublicAccess && props.blockPublicAccess.blockPublicPolicy;
        this.accessControl = props.accessControl;
        if (props.serverAccessLogsBucket instanceof Bucket) {
            props.serverAccessLogsBucket.allowLogDelivery();
        }
        for (const inventory of (_a = props.inventories) !== null && _a !== void 0 ? _a : []) {
            this.addInventory(inventory);
        }
        // Add all bucket metric configurations rules
        (props.metrics || []).forEach(this.addMetric.bind(this));
        // Add all cors configuration rules
        (props.cors || []).forEach(this.addCorsRule.bind(this));
        // Add all lifecycle rules
        (props.lifecycleRules || []).forEach(this.addLifecycleRule.bind(this));
        // defines a BucketNotifications construct. Notice that an actual resource will only
        // be added if there are notifications added, so we don't need to condition this.
        this.notifications = new notifications_resource_1.BucketNotifications(this, 'Notifications', { bucket: this });
        if (props.publicReadAccess) {
            this.grantPublicAccess();
        }
    }
    static fromBucketArn(scope, id, bucketArn) {
        return Bucket.fromBucketAttributes(scope, id, { bucketArn });
    }
    static fromBucketName(scope, id, bucketName) {
        return Bucket.fromBucketAttributes(scope, id, { bucketName });
    }
    /**
     * Creates a Bucket construct that represents an external bucket.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `BucketAttributes` object. Can be obtained from a call to
     * `bucket.export()` or manually created.
     */
    static fromBucketAttributes(scope, id, attrs) {
        var _a;
        const stack = core_1.Stack.of(scope);
        const region = (_a = attrs.region) !== null && _a !== void 0 ? _a : stack.region;
        const urlSuffix = stack.urlSuffix;
        const bucketName = util_1.parseBucketName(scope, attrs);
        if (!bucketName) {
            throw new Error('Bucket name is required');
        }
        const newUrlFormat = attrs.bucketWebsiteNewUrlFormat === undefined
            ? false
            : attrs.bucketWebsiteNewUrlFormat;
        const websiteDomain = newUrlFormat
            ? `${bucketName}.s3-website.${region}.${urlSuffix}`
            : `${bucketName}.s3-website-${region}.${urlSuffix}`;
        class Import extends BucketBase {
            constructor() {
                var _a;
                super(...arguments);
                this.bucketName = bucketName;
                this.bucketArn = util_1.parseBucketArn(scope, attrs);
                this.bucketDomainName = attrs.bucketDomainName || `${bucketName}.s3.${urlSuffix}`;
                this.bucketWebsiteUrl = attrs.bucketWebsiteUrl || `http://${websiteDomain}`;
                this.bucketWebsiteDomainName = attrs.bucketWebsiteUrl ? core_1.Fn.select(2, core_1.Fn.split('/', attrs.bucketWebsiteUrl)) : websiteDomain;
                this.bucketRegionalDomainName = attrs.bucketRegionalDomainName || `${bucketName}.s3.${region}.${urlSuffix}`;
                this.bucketDualStackDomainName = attrs.bucketDualStackDomainName || `${bucketName}.s3.dualstack.${region}.${urlSuffix}`;
                this.bucketWebsiteNewUrlFormat = newUrlFormat;
                this.encryptionKey = attrs.encryptionKey;
                this.isWebsite = (_a = attrs.isWebsite) !== null && _a !== void 0 ? _a : false;
                this.policy = undefined;
                this.autoCreatePolicy = false;
                this.disallowPublicAccess = false;
            }
            /**
             * Exports this bucket from the stack.
             */
            export() {
                return attrs;
            }
        }
        return new Import(scope, id, {
            account: attrs.account,
            region: attrs.region,
        });
    }
    /**
     * Add a lifecycle rule to the bucket
     *
     * @param rule The rule to add
     */
    addLifecycleRule(rule) {
        if ((rule.noncurrentVersionExpiration !== undefined
            || (rule.noncurrentVersionTransitions && rule.noncurrentVersionTransitions.length > 0))
            && !this.versioned) {
            throw new Error("Cannot use 'noncurrent' rules on a nonversioned bucket");
        }
        this.lifecycleRules.push(rule);
    }
    /**
     * Adds a metrics configuration for the CloudWatch request metrics from the bucket.
     *
     * @param metric The metric configuration to add
     */
    addMetric(metric) {
        this.metrics.push(metric);
    }
    /**
     * Adds a cross-origin access configuration for objects in an Amazon S3 bucket
     *
     * @param rule The CORS configuration rule to add
     */
    addCorsRule(rule) {
        this.cors.push(rule);
    }
    /**
     * Adds a bucket notification event destination.
     * @param event The event to trigger the notification
     * @param dest The notification destination (Lambda, SNS Topic or SQS Queue)
     *
     * @param filters S3 object key filter rules to determine which objects
     * trigger this event. Each filter must include a `prefix` and/or `suffix`
     * that will be matched against the s3 object key. Refer to the S3 Developer Guide
     * for details about allowed filter rules.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html#notification-how-to-filtering
     *
     * @example
     *
     *    bucket.addEventNotification(EventType.OnObjectCreated, myLambda, 'home/myusername/*')
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html
     */
    addEventNotification(event, dest, ...filters) {
        this.notifications.addNotification(event, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notificatins when an object is
     * created in the bucket. This is identical to calling
     * `onEvent(EventType.ObjectCreated)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectCreatedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.OBJECT_CREATED, dest, ...filters);
    }
    /**
     * Subscribes a destination to receive notificatins when an object is
     * removed from the bucket. This is identical to calling
     * `onEvent(EventType.ObjectRemoved)`.
     *
     * @param dest The notification destination (see onEvent)
     * @param filters Filters (see onEvent)
     */
    addObjectRemovedNotification(dest, ...filters) {
        return this.addEventNotification(EventType.OBJECT_REMOVED, dest, ...filters);
    }
    /**
     * Add an inventory configuration.
     *
     * @param inventory configuration to add
     */
    addInventory(inventory) {
        this.inventories.push(inventory);
    }
    validateBucketName(physicalName) {
        const bucketName = physicalName;
        if (!bucketName || core_1.Token.isUnresolved(bucketName)) {
            // the name is a late-bound value, not a defined string,
            // so skip validation
            return;
        }
        const errors = [];
        // Rules codified from https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html
        if (bucketName.length < 3 || bucketName.length > 63) {
            errors.push('Bucket name must be at least 3 and no more than 63 characters');
        }
        const charsetMatch = bucketName.match(/[^a-z0-9.-]/);
        if (charsetMatch) {
            errors.push('Bucket name must only contain lowercase characters and the symbols, period (.) and dash (-) '
                + `(offset: ${charsetMatch.index})`);
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(0))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + '(offset: 0)');
        }
        if (!/[a-z0-9]/.test(bucketName.charAt(bucketName.length - 1))) {
            errors.push('Bucket name must start and end with a lowercase character or number '
                + `(offset: ${bucketName.length - 1})`);
        }
        const consecSymbolMatch = bucketName.match(/\.-|-\.|\.\./);
        if (consecSymbolMatch) {
            errors.push('Bucket name must not have dash next to period, or period next to dash, or consecutive periods '
                + `(offset: ${consecSymbolMatch.index})`);
        }
        if (/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/.test(bucketName)) {
            errors.push('Bucket name must not resemble an IP address');
        }
        if (errors.length > 0) {
            throw new Error(`Invalid S3 bucket name (value: ${bucketName})${os_1.EOL}${errors.join(os_1.EOL)}`);
        }
    }
    /**
     * Set up key properties and return the Bucket encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        // default based on whether encryptionKey is specified
        let encryptionType = props.encryption;
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey ? BucketEncryption.KMS : BucketEncryption.UNENCRYPTED;
        }
        // if encryption key is set, encryption must be set to KMS.
        if (encryptionType !== BucketEncryption.KMS && props.encryptionKey) {
            throw new Error(`encryptionKey is specified, so 'encryption' must be set to KMS (value: ${encryptionType})`);
        }
        if (encryptionType === BucketEncryption.UNENCRYPTED) {
            return { bucketEncryption: undefined, encryptionKey: undefined };
        }
        if (encryptionType === BucketEncryption.KMS) {
            const encryptionKey = props.encryptionKey || new kms.Key(this, 'Key', {
                description: `Created by ${this.node.path}`,
            });
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    {
                        serverSideEncryptionByDefault: {
                            sseAlgorithm: 'aws:kms',
                            kmsMasterKeyId: encryptionKey.keyArn,
                        },
                    },
                ],
            };
            return { encryptionKey, bucketEncryption };
        }
        if (encryptionType === BucketEncryption.S3_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'AES256' } },
                ],
            };
            return { bucketEncryption };
        }
        if (encryptionType === BucketEncryption.KMS_MANAGED) {
            const bucketEncryption = {
                serverSideEncryptionConfiguration: [
                    { serverSideEncryptionByDefault: { sseAlgorithm: 'aws:kms' } },
                ],
            };
            return { bucketEncryption };
        }
        throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
    }
    /**
     * Parse the lifecycle configuration out of the uucket props
     * @param props Par
     */
    parseLifecycleConfiguration() {
        if (!this.lifecycleRules || this.lifecycleRules.length === 0) {
            return undefined;
        }
        const self = this;
        return { rules: this.lifecycleRules.map(parseLifecycleRule) };
        function parseLifecycleRule(rule) {
            const enabled = rule.enabled !== undefined ? rule.enabled : true;
            const x = {
                // eslint-disable-next-line max-len
                abortIncompleteMultipartUpload: rule.abortIncompleteMultipartUploadAfter !== undefined ? { daysAfterInitiation: rule.abortIncompleteMultipartUploadAfter.toDays() } : undefined,
                expirationDate: rule.expirationDate,
                expirationInDays: rule.expiration && rule.expiration.toDays(),
                id: rule.id,
                noncurrentVersionExpirationInDays: rule.noncurrentVersionExpiration && rule.noncurrentVersionExpiration.toDays(),
                noncurrentVersionTransitions: mapOrUndefined(rule.noncurrentVersionTransitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionInDays: t.transitionAfter.toDays(),
                })),
                prefix: rule.prefix,
                status: enabled ? 'Enabled' : 'Disabled',
                transitions: mapOrUndefined(rule.transitions, t => ({
                    storageClass: t.storageClass.value,
                    transitionDate: t.transitionDate,
                    transitionInDays: t.transitionAfter && t.transitionAfter.toDays(),
                })),
                tagFilters: self.parseTagFilters(rule.tagFilters),
            };
            return x;
        }
    }
    parseServerAccessLogs(props) {
        var _a;
        if (!props.serverAccessLogsBucket && !props.serverAccessLogsPrefix) {
            return undefined;
        }
        return {
            destinationBucketName: (_a = props.serverAccessLogsBucket) === null || _a === void 0 ? void 0 : _a.bucketName,
            logFilePrefix: props.serverAccessLogsPrefix,
        };
    }
    parseMetricConfiguration() {
        if (!this.metrics || this.metrics.length === 0) {
            return undefined;
        }
        const self = this;
        return this.metrics.map(parseMetric);
        function parseMetric(metric) {
            return {
                id: metric.id,
                prefix: metric.prefix,
                tagFilters: self.parseTagFilters(metric.tagFilters),
            };
        }
    }
    parseCorsConfiguration() {
        if (!this.cors || this.cors.length === 0) {
            return undefined;
        }
        return { corsRules: this.cors.map(parseCors) };
        function parseCors(rule) {
            return {
                id: rule.id,
                maxAge: rule.maxAge,
                allowedHeaders: rule.allowedHeaders,
                allowedMethods: rule.allowedMethods,
                allowedOrigins: rule.allowedOrigins,
                exposedHeaders: rule.exposedHeaders,
            };
        }
    }
    parseTagFilters(tagFilters) {
        if (!tagFilters || tagFilters.length === 0) {
            return undefined;
        }
        return Object.keys(tagFilters).map(tag => ({
            key: tag,
            value: tagFilters[tag],
        }));
    }
    renderWebsiteConfiguration(props) {
        if (!props.websiteErrorDocument && !props.websiteIndexDocument && !props.websiteRedirect && !props.websiteRoutingRules) {
            return undefined;
        }
        if (props.websiteErrorDocument && !props.websiteIndexDocument) {
            throw new Error('"websiteIndexDocument" is required if "websiteErrorDocument" is set');
        }
        if (props.websiteRedirect && (props.websiteErrorDocument || props.websiteIndexDocument || props.websiteRoutingRules)) {
            throw new Error('"websiteIndexDocument", "websiteErrorDocument" and, "websiteRoutingRules" cannot be set if "websiteRedirect" is used');
        }
        const routingRules = props.websiteRoutingRules ? props.websiteRoutingRules.map((rule) => {
            if (rule.condition && !rule.condition.httpErrorCodeReturnedEquals && !rule.condition.keyPrefixEquals) {
                throw new Error('The condition property cannot be an empty object');
            }
            return {
                redirectRule: {
                    hostName: rule.hostName,
                    httpRedirectCode: rule.httpRedirectCode,
                    protocol: rule.protocol,
                    replaceKeyWith: rule.replaceKey && rule.replaceKey.withKey,
                    replaceKeyPrefixWith: rule.replaceKey && rule.replaceKey.prefixWithKey,
                },
                routingRuleCondition: rule.condition,
            };
        }) : undefined;
        return {
            indexDocument: props.websiteIndexDocument,
            errorDocument: props.websiteErrorDocument,
            redirectAllRequestsTo: props.websiteRedirect,
            routingRules,
        };
    }
    /**
     * Allows the LogDelivery group to write, fails if ACL was set differently.
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl
     */
    allowLogDelivery() {
        if (this.accessControl && this.accessControl !== BucketAccessControl.LOG_DELIVERY_WRITE) {
            throw new Error("Cannot enable log delivery to this bucket because the bucket's ACL has been set and can't be changed");
        }
        this.accessControl = BucketAccessControl.LOG_DELIVERY_WRITE;
    }
    parseInventoryConfiguration() {
        if (!this.inventories || this.inventories.length === 0) {
            return undefined;
        }
        return this.inventories.map((inventory, index) => {
            var _a, _b, _c, _d, _e, _f;
            const format = (_a = inventory.format) !== null && _a !== void 0 ? _a : InventoryFormat.CSV;
            const frequency = (_b = inventory.frequency) !== null && _b !== void 0 ? _b : InventoryFrequency.WEEKLY;
            const id = (_c = inventory.inventoryId) !== null && _c !== void 0 ? _c : `${this.node.id}Inventory${index}`;
            if (inventory.destination.bucket instanceof Bucket) {
                inventory.destination.bucket.addToResourcePolicy(new iam.PolicyStatement({
                    effect: iam.Effect.ALLOW,
                    actions: ['s3:PutObject'],
                    resources: [
                        inventory.destination.bucket.bucketArn,
                        inventory.destination.bucket.arnForObjects(`${(_d = inventory.destination.prefix) !== null && _d !== void 0 ? _d : ''}*`),
                    ],
                    principals: [new iam.ServicePrincipal('s3.amazonaws.com')],
                    conditions: {
                        ArnLike: {
                            'aws:SourceArn': this.bucketArn,
                        },
                    },
                }));
            }
            return {
                id,
                destination: {
                    bucketArn: inventory.destination.bucket.bucketArn,
                    bucketAccountId: inventory.destination.bucketOwner,
                    prefix: inventory.destination.prefix,
                    format,
                },
                enabled: (_e = inventory.enabled) !== null && _e !== void 0 ? _e : true,
                includedObjectVersions: (_f = inventory.includeObjectVersions) !== null && _f !== void 0 ? _f : InventoryObjectVersion.ALL,
                scheduleFrequency: frequency,
                optionalFields: inventory.optionalFields,
                prefix: inventory.objectsPrefix,
            };
        });
    }
}
exports.Bucket = Bucket;
/**
 * What kind of server-side encryption to apply to this bucket
 */
var BucketEncryption;
(function (BucketEncryption) {
    /**
     * Objects in the bucket are not encrypted.
     */
    BucketEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by KMS.
     */
    BucketEncryption["KMS_MANAGED"] = "MANAGED";
    /**
     * Server-side encryption with a master key managed by S3.
     */
    BucketEncryption["S3_MANAGED"] = "S3MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    BucketEncryption["KMS"] = "KMS";
})(BucketEncryption = exports.BucketEncryption || (exports.BucketEncryption = {}));
/**
 * Notification event types.
 */
var EventType;
(function (EventType) {
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED"] = "s3:ObjectCreated:*";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_PUT"] = "s3:ObjectCreated:Put";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_POST"] = "s3:ObjectCreated:Post";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_COPY"] = "s3:ObjectCreated:Copy";
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object. Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     */
    EventType["OBJECT_CREATED_COMPLETE_MULTIPART_UPLOAD"] = "s3:ObjectCreated:CompleteMultipartUpload";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED"] = "s3:ObjectRemoved:*";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED_DELETE"] = "s3:ObjectRemoved:Delete";
    /**
     * By using the ObjectRemoved event types, you can enable notification when
     * an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     */
    EventType["OBJECT_REMOVED_DELETE_MARKER_CREATED"] = "s3:ObjectRemoved:DeleteMarkerCreated";
    /**
     * You can use this event type to request Amazon S3 to send a notification
     * message when Amazon S3 detects that an object of the RRS storage class is
     * lost.
     */
    EventType["REDUCED_REDUNDANCY_LOST_OBJECT"] = "s3:ReducedRedundancyLostObject";
})(EventType = exports.EventType || (exports.EventType = {}));
/**
 * Default bucket access control types.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html
 */
var BucketAccessControl;
(function (BucketAccessControl) {
    /**
     * Owner gets FULL_CONTROL. No one else has access rights.
     */
    BucketAccessControl["PRIVATE"] = "Private";
    /**
     * Owner gets FULL_CONTROL. The AllUsers group gets READ access.
     */
    BucketAccessControl["PUBLIC_READ"] = "PublicRead";
    /**
     * Owner gets FULL_CONTROL. The AllUsers group gets READ and WRITE access.
     * Granting this on a bucket is generally not recommended.
     */
    BucketAccessControl["PUBLIC_READ_WRITE"] = "PublicReadWrite";
    /**
     * Owner gets FULL_CONTROL. The AuthenticatedUsers group gets READ access.
     */
    BucketAccessControl["AUTHENTICATED_READ"] = "AuthenticatedRead";
    /**
     * The LogDelivery group gets WRITE and READ_ACP permissions on the bucket.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerLogs.html
     */
    BucketAccessControl["LOG_DELIVERY_WRITE"] = "LogDeliveryWrite";
    /**
     * Object owner gets FULL_CONTROL. Bucket owner gets READ access.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     */
    BucketAccessControl["BUCKET_OWNER_READ"] = "BucketOwnerRead";
    /**
     * Both the object owner and the bucket owner get FULL_CONTROL over the object.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     */
    BucketAccessControl["BUCKET_OWNER_FULL_CONTROL"] = "BucketOwnerFullControl";
    /**
     * Owner gets FULL_CONTROL. Amazon EC2 gets READ access to GET an Amazon Machine Image (AMI) bundle from Amazon S3.
     */
    BucketAccessControl["AWS_EXEC_READ"] = "AwsExecRead";
})(BucketAccessControl = exports.BucketAccessControl || (exports.BucketAccessControl = {}));
class ReplaceKey {
    constructor(withKey, prefixWithKey) {
        this.withKey = withKey;
        this.prefixWithKey = prefixWithKey;
    }
    /**
     * The specific object key to use in the redirect request
     */
    static with(keyReplacement) {
        return new this(keyReplacement);
    }
    /**
     * The object key prefix to use in the redirect request
     */
    static prefixWith(keyReplacement) {
        return new this(undefined, keyReplacement);
    }
}
exports.ReplaceKey = ReplaceKey;
function mapOrUndefined(list, callback) {
    if (!list || list.length === 0) {
        return undefined;
    }
    return list.map(callback);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVja2V0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYnVja2V0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDJCQUF5QjtBQUN6QiwyQ0FBMkMsQ0FBQyxzREFBc0Q7QUFDbEcscUNBQXFDLENBQUMsbURBQW1EO0FBQ3pGLHFDQUFxQyxDQUFDLG1EQUFtRDtBQUN6RixxQ0FBbUcsQ0FBQyxnREFBZ0Q7QUFDcEosbURBQStDO0FBRS9DLHFFQUErRDtBQUMvRCxpQ0FBaUM7QUFFakMsaURBQTJDO0FBQzNDLGlDQUF5RDtBQXVRekQ7Ozs7Ozs7Ozs7Ozs7Ozs7R0FnQkc7QUFDSCxNQUFlLFVBQVcsU0FBUSxlQUFRO0lBQTFDOztRQXVCSTs7O1dBR0c7UUFDZ0IscUJBQWdCLEdBQUcsS0FBSyxDQUFDO0lBd1FoRCxDQUFDO0lBblFHOzs7Ozs7OztPQVFHO0lBQ0ksaUJBQWlCLENBQUMsRUFBVSxFQUFFLFVBQTBDLEVBQUU7UUFDN0UsTUFBTSxJQUFJLEdBQUcsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxFQUFFLEVBQUUsT0FBTyxDQUFDLENBQUM7UUFDaEQsSUFBSSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLENBQUM7UUFDL0IsSUFBSSxDQUFDLGVBQWUsQ0FBQztZQUNqQixNQUFNLEVBQUUsQ0FBQyxRQUFRLENBQUM7WUFDbEIsVUFBVSxFQUFFLENBQUMsNkJBQTZCLENBQUM7WUFDM0MsTUFBTSxFQUFFO2dCQUNKLFNBQVMsRUFBRTtvQkFDUCxHQUFHLEVBQUUsT0FBTyxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQztpQkFDeEY7YUFDSjtTQUNKLENBQUMsQ0FBQztRQUNILE9BQU8sSUFBSSxDQUFDO0lBQ2hCLENBQUM7SUFDRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ0kscUJBQXFCLENBQUMsRUFBVSxFQUFFLFVBQTBDLEVBQUU7UUFDakYsTUFBTSxJQUFJLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsRUFBRSxPQUFPLENBQUMsQ0FBQztRQUNqRCxJQUFJLENBQUMsZUFBZSxDQUFDO1lBQ2pCLE1BQU0sRUFBRTtnQkFDSixTQUFTLEVBQUUsQ0FBQyxXQUFXLENBQUM7YUFDM0I7U0FDSixDQUFDLENBQUM7UUFDSCxPQUFPLElBQUksQ0FBQztJQUNoQixDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSSx1QkFBdUIsQ0FBQyxFQUFVLEVBQUUsVUFBMEMsRUFBRTtRQUNuRixNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsRUFBRSxFQUFFLE9BQU8sQ0FBQyxDQUFDO1FBQ2pELElBQUksQ0FBQyxlQUFlLENBQUM7WUFDakIsTUFBTSxFQUFFO2dCQUNKLFNBQVMsRUFBRTtvQkFDUCx5QkFBeUI7b0JBQ3pCLFlBQVk7b0JBQ1osV0FBVztpQkFDZDtnQkFDRCxpQkFBaUIsRUFBRTtvQkFDZixVQUFVLEVBQUUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDO29CQUM3QixHQUFHLEVBQUUsT0FBTyxDQUFDLEtBQUs7aUJBQ3JCO2FBQ0o7U0FDSixDQUFDLENBQUM7UUFDSCxPQUFPLElBQUksQ0FBQztJQUNoQixDQUFDO0lBQ0Q7Ozs7O09BS0c7SUFDSSxtQkFBbUIsQ0FBQyxVQUErQjtRQUN0RCxJQUFJLENBQUMsSUFBSSxDQUFDLE1BQU0sSUFBSSxJQUFJLENBQUMsZ0JBQWdCLEVBQUU7WUFDdkMsSUFBSSxDQUFDLE1BQU0sR0FBRyxJQUFJLDRCQUFZLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRSxFQUFFLE1BQU0sRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1NBQ3BFO1FBQ0QsSUFBSSxJQUFJLENBQUMsTUFBTSxFQUFFO1lBQ2IsSUFBSSxDQUFDLE1BQU0sQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1lBQy9DLE9BQU8sRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLElBQUksQ0FBQyxNQUFNLEVBQUUsQ0FBQztTQUNsRTtRQUNELE9BQU8sRUFBRSxjQUFjLEVBQUUsS0FBSyxFQUFFLENBQUM7SUFDckMsQ0FBQztJQUNTLFFBQVE7O1FBQ2QsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLFFBQVEsRUFBRSxDQUFDO1FBQ2hDLE1BQU0sQ0FBQyxJQUFJLENBQUMsR0FBRyxPQUFBLElBQUksQ0FBQyxNQUFNLDBDQUFFLFFBQVEsQ0FBQyx5QkFBeUIsT0FBTSxFQUFFLENBQUMsQ0FBQztRQUN4RSxPQUFPLE1BQU0sQ0FBQztJQUNsQixDQUFDO0lBQ0Q7Ozs7Ozs7O09BUUc7SUFDSSxZQUFZLENBQUMsR0FBWTtRQUM1QixNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDO1FBQzdCLE1BQU0sTUFBTSxHQUFHLGNBQWMsS0FBSyxDQUFDLE1BQU0sSUFBSSxLQUFLLENBQUMsU0FBUyxHQUFHLENBQUM7UUFDaEUsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sRUFBRSxHQUFHLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBQ0Q7Ozs7Ozs7T0FPRztJQUNJLGNBQWMsQ0FBQyxHQUFZO1FBQzlCLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUUsR0FBRyxDQUFDLENBQUM7SUFDdkMsQ0FBQztJQUNEOzs7Ozs7OztPQVFHO0lBQ0ksYUFBYSxDQUFDLFVBQWtCO1FBQ25DLE9BQU8sR0FBRyxJQUFJLENBQUMsU0FBUyxJQUFJLFVBQVUsRUFBRSxDQUFDO0lBQzdDLENBQUM7SUFDRDs7Ozs7Ozs7O09BU0c7SUFDSSxTQUFTLENBQUMsUUFBd0IsRUFBRSxvQkFBeUIsR0FBRztRQUNuRSxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxtQkFBbUIsRUFBRSxLQUFLLENBQUMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQztJQUMxSSxDQUFDO0lBQ0Q7Ozs7Ozs7O09BUUc7SUFDSSxVQUFVLENBQUMsUUFBd0IsRUFBRSxvQkFBeUIsR0FBRztRQUNwRSxPQUFPLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxvQkFBb0IsRUFBRSxLQUFLLENBQUMsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQztJQUM1SSxDQUFDO0lBQ0Q7Ozs7Ozs7T0FPRztJQUNJLFFBQVEsQ0FBQyxRQUF3QixFQUFFLG9CQUF5QixHQUFHO1FBQ2xFLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixFQUFFLEtBQUssQ0FBQyxpQkFBaUIsRUFBRSxJQUFJLENBQUMsYUFBYSxDQUFDLGlCQUFpQixDQUFDLENBQUMsQ0FBQztJQUMxSCxDQUFDO0lBQ0Q7Ozs7OztPQU1HO0lBQ0ksV0FBVyxDQUFDLFFBQXdCLEVBQUUsb0JBQXlCLEdBQUc7UUFDckUsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMscUJBQXFCLEVBQUUsRUFBRSxFQUFFLElBQUksQ0FBQyxhQUFhLENBQUMsaUJBQWlCLENBQUMsQ0FBQyxDQUFDO0lBQ3hHLENBQUM7SUFDRDs7Ozs7Ozs7O09BU0c7SUFDSSxjQUFjLENBQUMsUUFBd0IsRUFBRSxvQkFBeUIsR0FBRztRQUN4RSxNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsbUJBQW1CLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDO1FBQ25GLE1BQU0sVUFBVSxHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLENBQUM7UUFDMUUsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxhQUFhLEVBQUUsVUFBVSxFQUFFLElBQUksQ0FBQyxTQUFTLEVBQUUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUM7SUFDbEgsQ0FBQztJQUNEOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7T0FxQkc7SUFDSSxpQkFBaUIsQ0FBQyxTQUFTLEdBQUcsR0FBRyxFQUFFLEdBQUcsY0FBd0I7UUFDakUsSUFBSSxJQUFJLENBQUMsb0JBQW9CLEVBQUU7WUFDM0IsTUFBTSxJQUFJLEtBQUssQ0FBQyxnRUFBZ0UsQ0FBQyxDQUFDO1NBQ3JGO1FBQ0QsY0FBYyxHQUFHLGNBQWMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLENBQUM7UUFDL0UsT0FBTyxHQUFHLENBQUMsS0FBSyxDQUFDLHdCQUF3QixDQUFDO1lBQ3RDLE9BQU8sRUFBRSxjQUFjO1lBQ3ZCLFlBQVksRUFBRSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDN0MsT0FBTyxFQUFFLElBQUksR0FBRyxDQUFDLE1BQU0sRUFBRTtZQUN6QixRQUFRLEVBQUUsSUFBSTtTQUNqQixDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ08sUUFBUSxDQUFDLE1BQWMsRUFBRSxHQUFZO1FBQ3pDLE1BQU0sVUFBVSxHQUFHO1lBQ2YsTUFBTTtZQUNOLElBQUksQ0FBQyxVQUFVO1NBQ2xCLENBQUM7UUFDRixJQUFJLEdBQUcsRUFBRTtZQUNMLHNCQUFzQjtZQUN0QixJQUFJLE9BQU8sR0FBRyxLQUFLLFFBQVEsSUFBSSxHQUFHLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxFQUFFO2dCQUNoRCxHQUFHLEdBQUcsR0FBRyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQzthQUN2QjtZQUNELFVBQVUsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7WUFDckIsVUFBVSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUN4QjtRQUNELE9BQU8sVUFBVSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUMvQixDQUFDO0lBQ08sS0FBSyxDQUFDLE9BQXVCLEVBQUUsYUFBdUIsRUFBRSxVQUFvQixFQUFFLFdBQW1CLEVBQUUsR0FBRyxpQkFBMkI7UUFDckksTUFBTSxTQUFTLEdBQUcsQ0FBQyxXQUFXLEVBQUUsR0FBRyxpQkFBaUIsQ0FBQyxDQUFDO1FBQ3RELE1BQU0sR0FBRyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsd0JBQXdCLENBQUM7WUFDM0MsT0FBTztZQUNQLE9BQU8sRUFBRSxhQUFhO1lBQ3RCLFlBQVksRUFBRSxTQUFTO1lBQ3ZCLFFBQVEsRUFBRSxJQUFJO1NBQ2pCLENBQUMsQ0FBQztRQUNILElBQUksSUFBSSxDQUFDLGFBQWEsSUFBSSxVQUFVLElBQUksVUFBVSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDN0QsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLEdBQUcsVUFBVSxDQUFDLENBQUM7U0FDcEQ7UUFDRCxPQUFPLEdBQUcsQ0FBQztJQUNmLENBQUM7Q0FDSjtBQTJCRCxNQUFhLGlCQUFpQjtJQWUxQixZQUFZLE9BQWlDO1FBQ3pDLElBQUksQ0FBQyxlQUFlLEdBQUcsT0FBTyxDQUFDLGVBQWUsQ0FBQztRQUMvQyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsT0FBTyxDQUFDLGlCQUFpQixDQUFDO1FBQ25ELElBQUksQ0FBQyxnQkFBZ0IsR0FBRyxPQUFPLENBQUMsZ0JBQWdCLENBQUM7UUFDakQsSUFBSSxDQUFDLHFCQUFxQixHQUFHLE9BQU8sQ0FBQyxxQkFBcUIsQ0FBQztJQUMvRCxDQUFDOztBQXBCTCw4Q0FxQkM7QUFwQjBCLDJCQUFTLEdBQUcsSUFBSSxpQkFBaUIsQ0FBQztJQUNyRCxlQUFlLEVBQUUsSUFBSTtJQUNyQixpQkFBaUIsRUFBRSxJQUFJO0lBQ3ZCLGdCQUFnQixFQUFFLElBQUk7SUFDdEIscUJBQXFCLEVBQUUsSUFBSTtDQUM5QixDQUFDLENBQUM7QUFDb0IsNEJBQVUsR0FBRyxJQUFJLGlCQUFpQixDQUFDO0lBQ3RELGVBQWUsRUFBRSxJQUFJO0lBQ3JCLGdCQUFnQixFQUFFLElBQUk7Q0FDekIsQ0FBQyxDQUFDO0FBZ0NQOztHQUVHO0FBQ0gsSUFBWSxXQXFCWDtBQXJCRCxXQUFZLFdBQVc7SUFDbkI7O09BRUc7SUFDSCwwQkFBVyxDQUFBO0lBQ1g7O09BRUc7SUFDSCwwQkFBVyxDQUFBO0lBQ1g7O09BRUc7SUFDSCw0QkFBYSxDQUFBO0lBQ2I7O09BRUc7SUFDSCw0QkFBYSxDQUFBO0lBQ2I7O09BRUc7SUFDSCxnQ0FBaUIsQ0FBQTtBQUNyQixDQUFDLEVBckJXLFdBQVcsR0FBWCxtQkFBVyxLQUFYLG1CQUFXLFFBcUJ0QjtBQXNDRDs7R0FFRztBQUNILElBQVksZ0JBR1g7QUFIRCxXQUFZLGdCQUFnQjtJQUN4QixpQ0FBYSxDQUFBO0lBQ2IsbUNBQWUsQ0FBQTtBQUNuQixDQUFDLEVBSFcsZ0JBQWdCLEdBQWhCLHdCQUFnQixLQUFoQix3QkFBZ0IsUUFHM0I7QUFnQkQ7O0dBRUc7QUFDSCxJQUFZLGVBYVg7QUFiRCxXQUFZLGVBQWU7SUFDdkI7O09BRUc7SUFDSCw4QkFBVyxDQUFBO0lBQ1g7O09BRUc7SUFDSCxzQ0FBbUIsQ0FBQTtJQUNuQjs7T0FFRztJQUNILDhCQUFXLENBQUE7QUFDZixDQUFDLEVBYlcsZUFBZSxHQUFmLHVCQUFlLEtBQWYsdUJBQWUsUUFhMUI7QUFDRDs7R0FFRztBQUNILElBQVksa0JBU1g7QUFURCxXQUFZLGtCQUFrQjtJQUMxQjs7T0FFRztJQUNILHFDQUFlLENBQUE7SUFDZjs7T0FFRztJQUNILHVDQUFpQixDQUFBO0FBQ3JCLENBQUMsRUFUVyxrQkFBa0IsR0FBbEIsMEJBQWtCLEtBQWxCLDBCQUFrQixRQVM3QjtBQUNEOztHQUVHO0FBQ0gsSUFBWSxzQkFTWDtBQVRELFdBQVksc0JBQXNCO0lBQzlCOztPQUVHO0lBQ0gscUNBQVcsQ0FBQTtJQUNYOztPQUVHO0lBQ0gsNkNBQW1CLENBQUE7QUFDdkIsQ0FBQyxFQVRXLHNCQUFzQixHQUF0Qiw4QkFBc0IsS0FBdEIsOEJBQXNCLFFBU2pDO0FBZ05EOzs7OztHQUtHO0FBQ0gsTUFBYSxNQUFPLFNBQVEsVUFBVTtJQTBFbEMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxRQUFxQixFQUFFOztRQUM3RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNiLFlBQVksRUFBRSxLQUFLLENBQUMsVUFBVTtTQUNqQyxDQUFDLENBQUM7UUFaRyxxQkFBZ0IsR0FBRyxJQUFJLENBQUM7UUFHakIsbUJBQWMsR0FBb0IsRUFBRSxDQUFDO1FBR3JDLFlBQU8sR0FBb0IsRUFBRSxDQUFDO1FBQzlCLFNBQUksR0FBZSxFQUFFLENBQUM7UUFDdEIsZ0JBQVcsR0FBZ0IsRUFBRSxDQUFDO1FBSzNDLE1BQU0sRUFBRSxnQkFBZ0IsRUFBRSxhQUFhLEVBQUUsR0FBRyxJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3hFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUM7UUFDM0MsTUFBTSxvQkFBb0IsR0FBRyxJQUFJLENBQUMsMEJBQTBCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDcEUsSUFBSSxDQUFDLFNBQVMsR0FBRyxDQUFDLG9CQUFvQixLQUFLLFNBQVMsQ0FBQyxDQUFDO1FBQ3RELE1BQU0sUUFBUSxHQUFHLElBQUksd0JBQVMsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzdDLFVBQVUsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUM3QixnQkFBZ0I7WUFDaEIsdUJBQXVCLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRSxNQUFNLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDNUUsc0JBQXNCLEVBQUUsV0FBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsMkJBQTJCLEVBQUUsRUFBRSxDQUFDO1lBQzVGLG9CQUFvQjtZQUNwQiw4QkFBOEIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO1lBQ3ZELHFCQUFxQixFQUFFLFdBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLHdCQUF3QixFQUFFLEVBQUUsQ0FBQztZQUN4RixpQkFBaUIsRUFBRSxXQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxzQkFBc0IsRUFBRSxFQUFFLENBQUM7WUFDbEYsYUFBYSxFQUFFLFdBQUksQ0FBQyxXQUFXLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRSxDQUFDO1lBQ3RFLG9CQUFvQixFQUFFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxLQUFLLENBQUM7WUFDdkQsdUJBQXVCLEVBQUUsV0FBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsMkJBQTJCLEVBQUUsRUFBRSxDQUFDO1NBQ2hHLENBQUMsQ0FBQztRQUNILFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLFNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO1FBQ2pDLElBQUksQ0FBQyxhQUFhLEdBQUcsYUFBYSxDQUFDO1FBQ25DLElBQUksQ0FBQyxVQUFVLEdBQUcsSUFBSSxDQUFDLHdCQUF3QixDQUFDLFFBQVEsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUM5RCxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxRQUFRLENBQUMsT0FBTyxFQUFFO1lBQzVELE1BQU0sRUFBRSxFQUFFO1lBQ1YsT0FBTyxFQUFFLEVBQUU7WUFDWCxPQUFPLEVBQUUsSUFBSTtZQUNiLFFBQVEsRUFBRSxJQUFJLENBQUMsWUFBWTtTQUM5QixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsUUFBUSxDQUFDLGNBQWMsQ0FBQztRQUNoRCxJQUFJLENBQUMsZ0JBQWdCLEdBQUcsUUFBUSxDQUFDLGNBQWMsQ0FBQztRQUNoRCxJQUFJLENBQUMsdUJBQXVCLEdBQUcsU0FBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsU0FBRSxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQztRQUNsRixJQUFJLENBQUMseUJBQXlCLEdBQUcsUUFBUSxDQUFDLHVCQUF1QixDQUFDO1FBQ2xFLElBQUksQ0FBQyx3QkFBd0IsR0FBRyxRQUFRLENBQUMsc0JBQXNCLENBQUM7UUFDaEUsSUFBSSxDQUFDLG9CQUFvQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxLQUFLLENBQUMsaUJBQWlCLENBQUMsaUJBQWlCLENBQUM7UUFDakcsSUFBSSxDQUFDLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO1FBQ3pDLElBQUksS0FBSyxDQUFDLHNCQUFzQixZQUFZLE1BQU0sRUFBRTtZQUNoRCxLQUFLLENBQUMsc0JBQXNCLENBQUMsZ0JBQWdCLEVBQUUsQ0FBQztTQUNuRDtRQUNELEtBQUssTUFBTSxTQUFTLFVBQUksS0FBSyxDQUFDLFdBQVcsbUNBQUksRUFBRSxFQUFFO1lBQzdDLElBQUksQ0FBQyxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUM7U0FDaEM7UUFDRCw2Q0FBNkM7UUFDN0MsQ0FBQyxLQUFLLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQ3pELG1DQUFtQztRQUNuQyxDQUFDLEtBQUssQ0FBQyxJQUFJLElBQUksRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDeEQsMEJBQTBCO1FBQzFCLENBQUMsS0FBSyxDQUFDLGNBQWMsSUFBSSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQ3ZFLG9GQUFvRjtRQUNwRixpRkFBaUY7UUFDakYsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLDRDQUFtQixDQUFDLElBQUksRUFBRSxlQUFlLEVBQUUsRUFBRSxNQUFNLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQztRQUN0RixJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUN4QixJQUFJLENBQUMsaUJBQWlCLEVBQUUsQ0FBQztTQUM1QjtJQUNMLENBQUM7SUFqSU0sTUFBTSxDQUFDLGFBQWEsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxTQUFpQjtRQUN2RSxPQUFPLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQztJQUNqRSxDQUFDO0lBQ00sTUFBTSxDQUFDLGNBQWMsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxVQUFrQjtRQUN6RSxPQUFPLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQztJQUNsRSxDQUFDO0lBQ0Q7Ozs7Ozs7T0FPRztJQUNJLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUF1Qjs7UUFDcEYsTUFBTSxLQUFLLEdBQUcsWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM5QixNQUFNLE1BQU0sU0FBRyxLQUFLLENBQUMsTUFBTSxtQ0FBSSxLQUFLLENBQUMsTUFBTSxDQUFDO1FBQzVDLE1BQU0sU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7UUFDbEMsTUFBTSxVQUFVLEdBQUcsc0JBQWUsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDakQsSUFBSSxDQUFDLFVBQVUsRUFBRTtZQUNiLE1BQU0sSUFBSSxLQUFLLENBQUMseUJBQXlCLENBQUMsQ0FBQztTQUM5QztRQUNELE1BQU0sWUFBWSxHQUFHLEtBQUssQ0FBQyx5QkFBeUIsS0FBSyxTQUFTO1lBQzlELENBQUMsQ0FBQyxLQUFLO1lBQ1AsQ0FBQyxDQUFDLEtBQUssQ0FBQyx5QkFBeUIsQ0FBQztRQUN0QyxNQUFNLGFBQWEsR0FBRyxZQUFZO1lBQzlCLENBQUMsQ0FBQyxHQUFHLFVBQVUsZUFBZSxNQUFNLElBQUksU0FBUyxFQUFFO1lBQ25ELENBQUMsQ0FBQyxHQUFHLFVBQVUsZUFBZSxNQUFNLElBQUksU0FBUyxFQUFFLENBQUM7UUFDeEQsTUFBTSxNQUFPLFNBQVEsVUFBVTtZQUEvQjs7O2dCQUNvQixlQUFVLEdBQUcsVUFBVyxDQUFDO2dCQUN6QixjQUFTLEdBQUcscUJBQWMsQ0FBQyxLQUFLLEVBQUUsS0FBSyxDQUFDLENBQUM7Z0JBQ3pDLHFCQUFnQixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsSUFBSSxHQUFHLFVBQVUsT0FBTyxTQUFTLEVBQUUsQ0FBQztnQkFDN0UscUJBQWdCLEdBQUcsS0FBSyxDQUFDLGdCQUFnQixJQUFJLFVBQVUsYUFBYSxFQUFFLENBQUM7Z0JBQ3ZFLDRCQUF1QixHQUFHLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsU0FBRSxDQUFDLE1BQU0sQ0FBQyxDQUFDLEVBQUUsU0FBRSxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsYUFBYSxDQUFDO2dCQUN2SCw2QkFBd0IsR0FBRyxLQUFLLENBQUMsd0JBQXdCLElBQUksR0FBRyxVQUFVLE9BQU8sTUFBTSxJQUFJLFNBQVMsRUFBRSxDQUFDO2dCQUN2Ryw4QkFBeUIsR0FBRyxLQUFLLENBQUMseUJBQXlCLElBQUksR0FBRyxVQUFVLGlCQUFpQixNQUFNLElBQUksU0FBUyxFQUFFLENBQUM7Z0JBQ25ILDhCQUF5QixHQUFHLFlBQVksQ0FBQztnQkFDekMsa0JBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxDQUFDO2dCQUNwQyxjQUFTLFNBQUcsS0FBSyxDQUFDLFNBQVMsbUNBQUksS0FBSyxDQUFDO2dCQUM5QyxXQUFNLEdBQWtCLFNBQVMsQ0FBQztnQkFDL0IscUJBQWdCLEdBQUcsS0FBSyxDQUFDO2dCQUN6Qix5QkFBb0IsR0FBRyxLQUFLLENBQUM7WUFPM0MsQ0FBQztZQU5HOztlQUVHO1lBQ0ksTUFBTTtnQkFDVCxPQUFPLEtBQUssQ0FBQztZQUNqQixDQUFDO1NBQ0o7UUFDRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDekIsT0FBTyxFQUFFLEtBQUssQ0FBQyxPQUFPO1lBQ3RCLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTtTQUN2QixDQUFDLENBQUM7SUFDUCxDQUFDO0lBNkVEOzs7O09BSUc7SUFDSSxnQkFBZ0IsQ0FBQyxJQUFtQjtRQUN2QyxJQUFJLENBQUMsSUFBSSxDQUFDLDJCQUEyQixLQUFLLFNBQVM7ZUFDNUMsQ0FBQyxJQUFJLENBQUMsNEJBQTRCLElBQUksSUFBSSxDQUFDLDRCQUE0QixDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQztlQUNwRixDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUU7WUFDcEIsTUFBTSxJQUFJLEtBQUssQ0FBQyx3REFBd0QsQ0FBQyxDQUFDO1NBQzdFO1FBQ0QsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxTQUFTLENBQUMsTUFBcUI7UUFDbEMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsSUFBYztRQUM3QixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztJQUN6QixDQUFDO0lBQ0Q7Ozs7Ozs7Ozs7Ozs7Ozs7OztPQWtCRztJQUNJLG9CQUFvQixDQUFDLEtBQWdCLEVBQUUsSUFBb0MsRUFBRSxHQUFHLE9BQWdDO1FBQ25ILElBQUksQ0FBQyxhQUFhLENBQUMsZUFBZSxDQUFDLEtBQUssRUFBRSxJQUFJLEVBQUUsR0FBRyxPQUFPLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBQ0Q7Ozs7Ozs7T0FPRztJQUNJLDRCQUE0QixDQUFDLElBQW9DLEVBQUUsR0FBRyxPQUFnQztRQUN6RyxPQUFPLElBQUksQ0FBQyxvQkFBb0IsQ0FBQyxTQUFTLENBQUMsY0FBYyxFQUFFLElBQUksRUFBRSxHQUFHLE9BQU8sQ0FBQyxDQUFDO0lBQ2pGLENBQUM7SUFDRDs7Ozs7OztPQU9HO0lBQ0ksNEJBQTRCLENBQUMsSUFBb0MsRUFBRSxHQUFHLE9BQWdDO1FBQ3pHLE9BQU8sSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxjQUFjLEVBQUUsSUFBSSxFQUFFLEdBQUcsT0FBTyxDQUFDLENBQUM7SUFDakYsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxZQUFZLENBQUMsU0FBb0I7UUFDcEMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDckMsQ0FBQztJQUNPLGtCQUFrQixDQUFDLFlBQW9CO1FBQzNDLE1BQU0sVUFBVSxHQUFHLFlBQVksQ0FBQztRQUNoQyxJQUFJLENBQUMsVUFBVSxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDL0Msd0RBQXdEO1lBQ3hELHFCQUFxQjtZQUNyQixPQUFPO1NBQ1Y7UUFDRCxNQUFNLE1BQU0sR0FBYSxFQUFFLENBQUM7UUFDNUIsOEZBQThGO1FBQzlGLElBQUksVUFBVSxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksVUFBVSxDQUFDLE1BQU0sR0FBRyxFQUFFLEVBQUU7WUFDakQsTUFBTSxDQUFDLElBQUksQ0FBQywrREFBK0QsQ0FBQyxDQUFDO1NBQ2hGO1FBQ0QsTUFBTSxZQUFZLEdBQUcsVUFBVSxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztRQUNyRCxJQUFJLFlBQVksRUFBRTtZQUNkLE1BQU0sQ0FBQyxJQUFJLENBQUMsOEZBQThGO2tCQUNwRyxZQUFZLFlBQVksQ0FBQyxLQUFLLEdBQUcsQ0FBQyxDQUFDO1NBQzVDO1FBQ0QsSUFBSSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQ3hDLE1BQU0sQ0FBQyxJQUFJLENBQUMsc0VBQXNFO2tCQUM1RSxhQUFhLENBQUMsQ0FBQztTQUN4QjtRQUNELElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzVELE1BQU0sQ0FBQyxJQUFJLENBQUMsc0VBQXNFO2tCQUM1RSxZQUFZLFVBQVUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztTQUMvQztRQUNELE1BQU0saUJBQWlCLEdBQUcsVUFBVSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsQ0FBQztRQUMzRCxJQUFJLGlCQUFpQixFQUFFO1lBQ25CLE1BQU0sQ0FBQyxJQUFJLENBQUMsZ0dBQWdHO2tCQUN0RyxZQUFZLGlCQUFpQixDQUFDLEtBQUssR0FBRyxDQUFDLENBQUM7U0FDakQ7UUFDRCxJQUFJLHNDQUFzQyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsRUFBRTtZQUN6RCxNQUFNLENBQUMsSUFBSSxDQUFDLDZDQUE2QyxDQUFDLENBQUM7U0FDOUQ7UUFDRCxJQUFJLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ25CLE1BQU0sSUFBSSxLQUFLLENBQUMsa0NBQWtDLFVBQVUsSUFBSSxRQUFHLEdBQUcsTUFBTSxDQUFDLElBQUksQ0FBQyxRQUFHLENBQUMsRUFBRSxDQUFDLENBQUM7U0FDN0Y7SUFDTCxDQUFDO0lBQ0Q7OztPQUdHO0lBQ0ssZUFBZSxDQUFDLEtBQWtCO1FBSXRDLHNEQUFzRDtRQUN0RCxJQUFJLGNBQWMsR0FBRyxLQUFLLENBQUMsVUFBVSxDQUFDO1FBQ3RDLElBQUksY0FBYyxLQUFLLFNBQVMsRUFBRTtZQUM5QixjQUFjLEdBQUcsS0FBSyxDQUFDLGFBQWEsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxnQkFBZ0IsQ0FBQyxXQUFXLENBQUM7U0FDOUY7UUFDRCwyREFBMkQ7UUFDM0QsSUFBSSxjQUFjLEtBQUssZ0JBQWdCLENBQUMsR0FBRyxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDaEUsTUFBTSxJQUFJLEtBQUssQ0FBQywwRUFBMEUsY0FBYyxHQUFHLENBQUMsQ0FBQztTQUNoSDtRQUNELElBQUksY0FBYyxLQUFLLGdCQUFnQixDQUFDLFdBQVcsRUFBRTtZQUNqRCxPQUFPLEVBQUUsZ0JBQWdCLEVBQUUsU0FBUyxFQUFFLGFBQWEsRUFBRSxTQUFTLEVBQUUsQ0FBQztTQUNwRTtRQUNELElBQUksY0FBYyxLQUFLLGdCQUFnQixDQUFDLEdBQUcsRUFBRTtZQUN6QyxNQUFNLGFBQWEsR0FBRyxLQUFLLENBQUMsYUFBYSxJQUFJLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO2dCQUNsRSxXQUFXLEVBQUUsY0FBYyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRTthQUM5QyxDQUFDLENBQUM7WUFDSCxNQUFNLGdCQUFnQixHQUFHO2dCQUNyQixpQ0FBaUMsRUFBRTtvQkFDL0I7d0JBQ0ksNkJBQTZCLEVBQUU7NEJBQzNCLFlBQVksRUFBRSxTQUFTOzRCQUN2QixjQUFjLEVBQUUsYUFBYSxDQUFDLE1BQU07eUJBQ3ZDO3FCQUNKO2lCQUNKO2FBQ0osQ0FBQztZQUNGLE9BQU8sRUFBRSxhQUFhLEVBQUUsZ0JBQWdCLEVBQUUsQ0FBQztTQUM5QztRQUNELElBQUksY0FBYyxLQUFLLGdCQUFnQixDQUFDLFVBQVUsRUFBRTtZQUNoRCxNQUFNLGdCQUFnQixHQUFHO2dCQUNyQixpQ0FBaUMsRUFBRTtvQkFDL0IsRUFBRSw2QkFBNkIsRUFBRSxFQUFFLFlBQVksRUFBRSxRQUFRLEVBQUUsRUFBRTtpQkFDaEU7YUFDSixDQUFDO1lBQ0YsT0FBTyxFQUFFLGdCQUFnQixFQUFFLENBQUM7U0FDL0I7UUFDRCxJQUFJLGNBQWMsS0FBSyxnQkFBZ0IsQ0FBQyxXQUFXLEVBQUU7WUFDakQsTUFBTSxnQkFBZ0IsR0FBRztnQkFDckIsaUNBQWlDLEVBQUU7b0JBQy9CLEVBQUUsNkJBQTZCLEVBQUUsRUFBRSxZQUFZLEVBQUUsU0FBUyxFQUFFLEVBQUU7aUJBQ2pFO2FBQ0osQ0FBQztZQUNGLE9BQU8sRUFBRSxnQkFBZ0IsRUFBRSxDQUFDO1NBQy9CO1FBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxnQ0FBZ0MsY0FBYyxFQUFFLENBQUMsQ0FBQztJQUN0RSxDQUFDO0lBQ0Q7OztPQUdHO0lBQ0ssMkJBQTJCO1FBQy9CLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxJQUFJLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUMxRCxPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE1BQU0sSUFBSSxHQUFHLElBQUksQ0FBQztRQUNsQixPQUFPLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLEVBQUUsQ0FBQztRQUM5RCxTQUFTLGtCQUFrQixDQUFDLElBQW1CO1lBQzNDLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxPQUFPLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUM7WUFDakUsTUFBTSxDQUFDLEdBQTJCO2dCQUM5QixtQ0FBbUM7Z0JBQ25DLDhCQUE4QixFQUFFLElBQUksQ0FBQyxtQ0FBbUMsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLEVBQUUsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLG1DQUFtQyxDQUFDLE1BQU0sRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLFNBQVM7Z0JBQy9LLGNBQWMsRUFBRSxJQUFJLENBQUMsY0FBYztnQkFDbkMsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLFVBQVUsSUFBSSxJQUFJLENBQUMsVUFBVSxDQUFDLE1BQU0sRUFBRTtnQkFDN0QsRUFBRSxFQUFFLElBQUksQ0FBQyxFQUFFO2dCQUNYLGlDQUFpQyxFQUFFLElBQUksQ0FBQywyQkFBMkIsSUFBSSxJQUFJLENBQUMsMkJBQTJCLENBQUMsTUFBTSxFQUFFO2dCQUNoSCw0QkFBNEIsRUFBRSxjQUFjLENBQUMsSUFBSSxDQUFDLDRCQUE0QixFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztvQkFDbEYsWUFBWSxFQUFFLENBQUMsQ0FBQyxZQUFZLENBQUMsS0FBSztvQkFDbEMsZ0JBQWdCLEVBQUUsQ0FBQyxDQUFDLGVBQWUsQ0FBQyxNQUFNLEVBQUU7aUJBQy9DLENBQUMsQ0FBQztnQkFDSCxNQUFNLEVBQUUsSUFBSSxDQUFDLE1BQU07Z0JBQ25CLE1BQU0sRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsVUFBVTtnQkFDeEMsV0FBVyxFQUFFLGNBQWMsQ0FBQyxJQUFJLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztvQkFDaEQsWUFBWSxFQUFFLENBQUMsQ0FBQyxZQUFZLENBQUMsS0FBSztvQkFDbEMsY0FBYyxFQUFFLENBQUMsQ0FBQyxjQUFjO29CQUNoQyxnQkFBZ0IsRUFBRSxDQUFDLENBQUMsZUFBZSxJQUFJLENBQUMsQ0FBQyxlQUFlLENBQUMsTUFBTSxFQUFFO2lCQUNwRSxDQUFDLENBQUM7Z0JBQ0gsVUFBVSxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQzthQUNwRCxDQUFDO1lBQ0YsT0FBTyxDQUFDLENBQUM7UUFDYixDQUFDO0lBQ0wsQ0FBQztJQUNPLHFCQUFxQixDQUFDLEtBQWtCOztRQUM1QyxJQUFJLENBQUMsS0FBSyxDQUFDLHNCQUFzQixJQUFJLENBQUMsS0FBSyxDQUFDLHNCQUFzQixFQUFFO1lBQ2hFLE9BQU8sU0FBUyxDQUFDO1NBQ3BCO1FBQ0QsT0FBTztZQUNILHFCQUFxQixRQUFFLEtBQUssQ0FBQyxzQkFBc0IsMENBQUUsVUFBVTtZQUMvRCxhQUFhLEVBQUUsS0FBSyxDQUFDLHNCQUFzQjtTQUM5QyxDQUFDO0lBQ04sQ0FBQztJQUNPLHdCQUF3QjtRQUM1QixJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUMsT0FBTyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDNUMsT0FBTyxTQUFTLENBQUM7U0FDcEI7UUFDRCxNQUFNLElBQUksR0FBRyxJQUFJLENBQUM7UUFDbEIsT0FBTyxJQUFJLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxXQUFXLENBQUMsQ0FBQztRQUNyQyxTQUFTLFdBQVcsQ0FBQyxNQUFxQjtZQUN0QyxPQUFPO2dCQUNILEVBQUUsRUFBRSxNQUFNLENBQUMsRUFBRTtnQkFDYixNQUFNLEVBQUUsTUFBTSxDQUFDLE1BQU07Z0JBQ3JCLFVBQVUsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUM7YUFDdEQsQ0FBQztRQUNOLENBQUM7SUFDTCxDQUFDO0lBQ08sc0JBQXNCO1FBQzFCLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUN0QyxPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE9BQU8sRUFBRSxTQUFTLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsU0FBUyxDQUFDLEVBQUUsQ0FBQztRQUMvQyxTQUFTLFNBQVMsQ0FBQyxJQUFjO1lBQzdCLE9BQU87Z0JBQ0gsRUFBRSxFQUFFLElBQUksQ0FBQyxFQUFFO2dCQUNYLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTtnQkFDbkIsY0FBYyxFQUFFLElBQUksQ0FBQyxjQUFjO2dCQUNuQyxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWM7Z0JBQ25DLGNBQWMsRUFBRSxJQUFJLENBQUMsY0FBYztnQkFDbkMsY0FBYyxFQUFFLElBQUksQ0FBQyxjQUFjO2FBQ3RDLENBQUM7UUFDTixDQUFDO0lBQ0wsQ0FBQztJQUNPLGVBQWUsQ0FBQyxVQUV2QjtRQUNHLElBQUksQ0FBQyxVQUFVLElBQUksVUFBVSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDeEMsT0FBTyxTQUFTLENBQUM7U0FDcEI7UUFDRCxPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLENBQUMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQztZQUN2QyxHQUFHLEVBQUUsR0FBRztZQUNSLEtBQUssRUFBRSxVQUFVLENBQUMsR0FBRyxDQUFDO1NBQ3pCLENBQUMsQ0FBQyxDQUFDO0lBQ1IsQ0FBQztJQUNPLDBCQUEwQixDQUFDLEtBQWtCO1FBQ2pELElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLElBQUksQ0FBQyxLQUFLLENBQUMsZUFBZSxJQUFJLENBQUMsS0FBSyxDQUFDLG1CQUFtQixFQUFFO1lBQ3BILE9BQU8sU0FBUyxDQUFDO1NBQ3BCO1FBQ0QsSUFBSSxLQUFLLENBQUMsb0JBQW9CLElBQUksQ0FBQyxLQUFLLENBQUMsb0JBQW9CLEVBQUU7WUFDM0QsTUFBTSxJQUFJLEtBQUssQ0FBQyxxRUFBcUUsQ0FBQyxDQUFDO1NBQzFGO1FBQ0QsSUFBSSxLQUFLLENBQUMsZUFBZSxJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQixJQUFJLEtBQUssQ0FBQyxvQkFBb0IsSUFBSSxLQUFLLENBQUMsbUJBQW1CLENBQUMsRUFBRTtZQUNsSCxNQUFNLElBQUksS0FBSyxDQUFDLHNIQUFzSCxDQUFDLENBQUM7U0FDM0k7UUFDRCxNQUFNLFlBQVksR0FBRyxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsQ0FBQyxHQUFHLENBQWdDLENBQUMsSUFBSSxFQUFFLEVBQUU7WUFDbkgsSUFBSSxJQUFJLENBQUMsU0FBUyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQywyQkFBMkIsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsZUFBZSxFQUFFO2dCQUNsRyxNQUFNLElBQUksS0FBSyxDQUFDLGtEQUFrRCxDQUFDLENBQUM7YUFDdkU7WUFDRCxPQUFPO2dCQUNILFlBQVksRUFBRTtvQkFDVixRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVE7b0JBQ3ZCLGdCQUFnQixFQUFFLElBQUksQ0FBQyxnQkFBZ0I7b0JBQ3ZDLFFBQVEsRUFBRSxJQUFJLENBQUMsUUFBUTtvQkFDdkIsY0FBYyxFQUFFLElBQUksQ0FBQyxVQUFVLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxPQUFPO29CQUMxRCxvQkFBb0IsRUFBRSxJQUFJLENBQUMsVUFBVSxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsYUFBYTtpQkFDekU7Z0JBQ0Qsb0JBQW9CLEVBQUUsSUFBSSxDQUFDLFNBQVM7YUFDdkMsQ0FBQztRQUNOLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDZixPQUFPO1lBQ0gsYUFBYSxFQUFFLEtBQUssQ0FBQyxvQkFBb0I7WUFDekMsYUFBYSxFQUFFLEtBQUssQ0FBQyxvQkFBb0I7WUFDekMscUJBQXFCLEVBQUUsS0FBSyxDQUFDLGVBQWU7WUFDNUMsWUFBWTtTQUNmLENBQUM7SUFDTixDQUFDO0lBQ0Q7Ozs7O09BS0c7SUFDSyxnQkFBZ0I7UUFDcEIsSUFBSSxJQUFJLENBQUMsYUFBYSxJQUFJLElBQUksQ0FBQyxhQUFhLEtBQUssbUJBQW1CLENBQUMsa0JBQWtCLEVBQUU7WUFDckYsTUFBTSxJQUFJLEtBQUssQ0FBQyxzR0FBc0csQ0FBQyxDQUFDO1NBQzNIO1FBQ0QsSUFBSSxDQUFDLGFBQWEsR0FBRyxtQkFBbUIsQ0FBQyxrQkFBa0IsQ0FBQztJQUNoRSxDQUFDO0lBQ08sMkJBQTJCO1FBQy9CLElBQUksQ0FBQyxJQUFJLENBQUMsV0FBVyxJQUFJLElBQUksQ0FBQyxXQUFXLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNwRCxPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE9BQU8sSUFBSSxDQUFDLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxTQUFTLEVBQUUsS0FBSyxFQUFFLEVBQUU7O1lBQzdDLE1BQU0sTUFBTSxTQUFHLFNBQVMsQ0FBQyxNQUFNLG1DQUFJLGVBQWUsQ0FBQyxHQUFHLENBQUM7WUFDdkQsTUFBTSxTQUFTLFNBQUcsU0FBUyxDQUFDLFNBQVMsbUNBQUksa0JBQWtCLENBQUMsTUFBTSxDQUFDO1lBQ25FLE1BQU0sRUFBRSxTQUFHLFNBQVMsQ0FBQyxXQUFXLG1DQUFJLEdBQUcsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLFlBQVksS0FBSyxFQUFFLENBQUM7WUFDdkUsSUFBSSxTQUFTLENBQUMsV0FBVyxDQUFDLE1BQU0sWUFBWSxNQUFNLEVBQUU7Z0JBQ2hELFNBQVMsQ0FBQyxXQUFXLENBQUMsTUFBTSxDQUFDLG1CQUFtQixDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztvQkFDckUsTUFBTSxFQUFFLEdBQUcsQ0FBQyxNQUFNLENBQUMsS0FBSztvQkFDeEIsT0FBTyxFQUFFLENBQUMsY0FBYyxDQUFDO29CQUN6QixTQUFTLEVBQUU7d0JBQ1AsU0FBUyxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsU0FBUzt3QkFDdEMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsYUFBYSxDQUFDLEdBQUcsTUFBQSxTQUFTLENBQUMsV0FBVyxDQUFDLE1BQU0sbUNBQUksRUFBRSxHQUFHLENBQUM7cUJBQ3ZGO29CQUNELFVBQVUsRUFBRSxDQUFDLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLGtCQUFrQixDQUFDLENBQUM7b0JBQzFELFVBQVUsRUFBRTt3QkFDUixPQUFPLEVBQUU7NEJBQ0wsZUFBZSxFQUFFLElBQUksQ0FBQyxTQUFTO3lCQUNsQztxQkFDSjtpQkFDSixDQUFDLENBQUMsQ0FBQzthQUNQO1lBQ0QsT0FBTztnQkFDSCxFQUFFO2dCQUNGLFdBQVcsRUFBRTtvQkFDVCxTQUFTLEVBQUUsU0FBUyxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsU0FBUztvQkFDakQsZUFBZSxFQUFFLFNBQVMsQ0FBQyxXQUFXLENBQUMsV0FBVztvQkFDbEQsTUFBTSxFQUFFLFNBQVMsQ0FBQyxXQUFXLENBQUMsTUFBTTtvQkFDcEMsTUFBTTtpQkFDVDtnQkFDRCxPQUFPLFFBQUUsU0FBUyxDQUFDLE9BQU8sbUNBQUksSUFBSTtnQkFDbEMsc0JBQXNCLFFBQUUsU0FBUyxDQUFDLHFCQUFxQixtQ0FBSSxzQkFBc0IsQ0FBQyxHQUFHO2dCQUNyRixpQkFBaUIsRUFBRSxTQUFTO2dCQUM1QixjQUFjLEVBQUUsU0FBUyxDQUFDLGNBQWM7Z0JBQ3hDLE1BQU0sRUFBRSxTQUFTLENBQUMsYUFBYTthQUNsQyxDQUFDO1FBQ04sQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0NBQ0o7QUF4ZEQsd0JBd2RDO0FBQ0Q7O0dBRUc7QUFDSCxJQUFZLGdCQWtCWDtBQWxCRCxXQUFZLGdCQUFnQjtJQUN4Qjs7T0FFRztJQUNILHdDQUFvQixDQUFBO0lBQ3BCOztPQUVHO0lBQ0gsMkNBQXVCLENBQUE7SUFDdkI7O09BRUc7SUFDSCw0Q0FBd0IsQ0FBQTtJQUN4Qjs7O09BR0c7SUFDSCwrQkFBVyxDQUFBO0FBQ2YsQ0FBQyxFQWxCVyxnQkFBZ0IsR0FBaEIsd0JBQWdCLEtBQWhCLHdCQUFnQixRQWtCM0I7QUFDRDs7R0FFRztBQUNILElBQVksU0ErRlg7QUEvRkQsV0FBWSxTQUFTO0lBQ2pCOzs7Ozs7T0FNRztJQUNILGtEQUFxQyxDQUFBO0lBQ3JDOzs7Ozs7T0FNRztJQUNILHdEQUEyQyxDQUFBO0lBQzNDOzs7Ozs7T0FNRztJQUNILDBEQUE2QyxDQUFBO0lBQzdDOzs7Ozs7T0FNRztJQUNILDBEQUE2QyxDQUFBO0lBQzdDOzs7Ozs7T0FNRztJQUNILGtHQUFxRixDQUFBO0lBQ3JGOzs7Ozs7Ozs7Ozs7OztPQWNHO0lBQ0gsa0RBQXFDLENBQUE7SUFDckM7Ozs7Ozs7Ozs7Ozs7O09BY0c7SUFDSCw4REFBaUQsQ0FBQTtJQUNqRDs7Ozs7Ozs7Ozs7Ozs7T0FjRztJQUNILDBGQUE2RSxDQUFBO0lBQzdFOzs7O09BSUc7SUFDSCw4RUFBaUUsQ0FBQTtBQUNyRSxDQUFDLEVBL0ZXLFNBQVMsR0FBVCxpQkFBUyxLQUFULGlCQUFTLFFBK0ZwQjtBQXNCRDs7OztHQUlHO0FBQ0gsSUFBWSxtQkFxQ1g7QUFyQ0QsV0FBWSxtQkFBbUI7SUFDM0I7O09BRUc7SUFDSCwwQ0FBbUIsQ0FBQTtJQUNuQjs7T0FFRztJQUNILGlEQUEwQixDQUFBO0lBQzFCOzs7T0FHRztJQUNILDREQUFxQyxDQUFBO0lBQ3JDOztPQUVHO0lBQ0gsK0RBQXdDLENBQUE7SUFDeEM7OztPQUdHO0lBQ0gsOERBQXVDLENBQUE7SUFDdkM7OztPQUdHO0lBQ0gsNERBQXFDLENBQUE7SUFDckM7OztPQUdHO0lBQ0gsMkVBQW9ELENBQUE7SUFDcEQ7O09BRUc7SUFDSCxvREFBNkIsQ0FBQTtBQUNqQyxDQUFDLEVBckNXLG1CQUFtQixHQUFuQiwyQkFBbUIsS0FBbkIsMkJBQW1CLFFBcUM5QjtBQXFCRCxNQUFhLFVBQVU7SUFhbkIsWUFBb0MsT0FBZ0IsRUFBa0IsYUFBc0I7UUFBeEQsWUFBTyxHQUFQLE9BQU8sQ0FBUztRQUFrQixrQkFBYSxHQUFiLGFBQWEsQ0FBUztJQUM1RixDQUFDO0lBYkQ7O09BRUc7SUFDSSxNQUFNLENBQUMsSUFBSSxDQUFDLGNBQXNCO1FBQ3JDLE9BQU8sSUFBSSxJQUFJLENBQUMsY0FBYyxDQUFDLENBQUM7SUFDcEMsQ0FBQztJQUNEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFVBQVUsQ0FBQyxjQUFzQjtRQUMzQyxPQUFPLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRSxjQUFjLENBQUMsQ0FBQztJQUMvQyxDQUFDO0NBR0o7QUFmRCxnQ0FlQztBQXNDRCxTQUFTLGNBQWMsQ0FBTyxJQUFxQixFQUFFLFFBQTJCO0lBQzVFLElBQUksQ0FBQyxJQUFJLElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7UUFDNUIsT0FBTyxTQUFTLENBQUM7S0FDcEI7SUFDRCxPQUFPLElBQUksQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLENBQUM7QUFDOUIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IEVPTCB9IGZyb20gJ29zJztcbmltcG9ydCAqIGFzIGV2ZW50cyBmcm9tIFwiLi4vLi4vYXdzLWV2ZW50c1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWV2ZW50cydcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIGttcyBmcm9tIFwiLi4vLi4vYXdzLWttc1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWttcydcbmltcG9ydCB7IENvbnN0cnVjdCwgRm4sIElSZXNvdXJjZSwgTGF6eSwgUmVtb3ZhbFBvbGljeSwgUmVzb3VyY2UsIFN0YWNrLCBUb2tlbiB9IGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQnVja2V0UG9saWN5IH0gZnJvbSAnLi9idWNrZXQtcG9saWN5JztcbmltcG9ydCB7IElCdWNrZXROb3RpZmljYXRpb25EZXN0aW5hdGlvbiB9IGZyb20gJy4vZGVzdGluYXRpb24nO1xuaW1wb3J0IHsgQnVja2V0Tm90aWZpY2F0aW9ucyB9IGZyb20gJy4vbm90aWZpY2F0aW9ucy1yZXNvdXJjZSc7XG5pbXBvcnQgKiBhcyBwZXJtcyBmcm9tICcuL3Blcm1zJztcbmltcG9ydCB7IExpZmVjeWNsZVJ1bGUgfSBmcm9tICcuL3J1bGUnO1xuaW1wb3J0IHsgQ2ZuQnVja2V0IH0gZnJvbSAnLi9zMy5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgcGFyc2VCdWNrZXRBcm4sIHBhcnNlQnVja2V0TmFtZSB9IGZyb20gJy4vdXRpbCc7XG5leHBvcnQgaW50ZXJmYWNlIElCdWNrZXQgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuICAgIC8qKlxuICAgICAqIFRoZSBBUk4gb2YgdGhlIGJ1Y2tldC5cbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgYnVja2V0QXJuOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIGJ1Y2tldC5cbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgYnVja2V0TmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBVUkwgb2YgdGhlIHN0YXRpYyB3ZWJzaXRlLlxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICByZWFkb25seSBidWNrZXRXZWJzaXRlVXJsOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIERvbWFpbiBuYW1lIG9mIHRoZSBzdGF0aWMgd2Vic2l0ZS5cbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgYnVja2V0V2Vic2l0ZURvbWFpbk5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgSVB2NCBETlMgbmFtZSBvZiB0aGUgc3BlY2lmaWVkIGJ1Y2tldC5cbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgYnVja2V0RG9tYWluTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBJUHY2IEROUyBuYW1lIG9mIHRoZSBzcGVjaWZpZWQgYnVja2V0LlxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICByZWFkb25seSBidWNrZXREdWFsU3RhY2tEb21haW5OYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHJlZ2lvbmFsIGRvbWFpbiBuYW1lIG9mIHRoZSBzcGVjaWZpZWQgYnVja2V0LlxuICAgICAqIEBhdHRyaWJ1dGVcbiAgICAgKi9cbiAgICByZWFkb25seSBidWNrZXRSZWdpb25hbERvbWFpbk5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBJZiB0aGlzIGJ1Y2tldCBoYXMgYmVlbiBjb25maWd1cmVkIGZvciBzdGF0aWMgd2Vic2l0ZSBob3N0aW5nLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGlzV2Vic2l0ZT86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogT3B0aW9uYWwgS01TIGVuY3J5cHRpb24ga2V5IGFzc29jaWF0ZWQgd2l0aCB0aGlzIGJ1Y2tldC5cbiAgICAgKi9cbiAgICByZWFkb25seSBlbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG4gICAgLyoqXG4gICAgICogVGhlIHJlc291cmNlIHBvbGljeSBhc3NvY2lhdGVkIHdpdGggdGhpcyBidWNrZXQuXG4gICAgICpcbiAgICAgKiBJZiBgYXV0b0NyZWF0ZVBvbGljeWAgaXMgdHJ1ZSwgYSBgQnVja2V0UG9saWN5YCB3aWxsIGJlIGNyZWF0ZWQgdXBvbiB0aGVcbiAgICAgKiBmaXJzdCBjYWxsIHRvIGFkZFRvUmVzb3VyY2VQb2xpY3kocykuXG4gICAgICovXG4gICAgcG9saWN5PzogQnVja2V0UG9saWN5O1xuICAgIC8qKlxuICAgICAqIEFkZHMgYSBzdGF0ZW1lbnQgdG8gdGhlIHJlc291cmNlIHBvbGljeSBmb3IgYSBwcmluY2lwYWwgKGkuZS5cbiAgICAgKiBhY2NvdW50L3JvbGUvc2VydmljZSkgdG8gcGVyZm9ybSBhY3Rpb25zIG9uIHRoaXMgYnVja2V0IGFuZC9vciBpdCdzXG4gICAgICogY29udGVudHMuIFVzZSBgYnVja2V0QXJuYCBhbmQgYGFybkZvck9iamVjdHMoa2V5cylgIHRvIG9idGFpbiBBUk5zIGZvclxuICAgICAqIHRoaXMgYnVja2V0IG9yIG9iamVjdHMuXG4gICAgICovXG4gICAgYWRkVG9SZXNvdXJjZVBvbGljeShwZXJtaXNzaW9uOiBpYW0uUG9saWN5U3RhdGVtZW50KTogaWFtLkFkZFRvUmVzb3VyY2VQb2xpY3lSZXN1bHQ7XG4gICAgLyoqXG4gICAgICogVGhlIGh0dHBzIFVSTCBvZiBhbiBTMyBvYmplY3QuIEZvciBleGFtcGxlOlxuICAgICAqIEBleGFtcGxlIGh0dHBzOi8vczMudXMtd2VzdC0xLmFtYXpvbmF3cy5jb20vb25seWJ1Y2tldFxuICAgICAqIEBleGFtcGxlIGh0dHBzOi8vczMudXMtd2VzdC0xLmFtYXpvbmF3cy5jb20vYnVja2V0L2tleVxuICAgICAqIEBleGFtcGxlIGh0dHBzOi8vczMuY24tbm9ydGgtMS5hbWF6b25hd3MuY29tLmNuL2NoaW5hLWJ1Y2tldC9teWtleVxuICAgICAqIEBwYXJhbSBrZXkgVGhlIFMzIGtleSBvZiB0aGUgb2JqZWN0LiBJZiBub3Qgc3BlY2lmaWVkLCB0aGUgVVJMIG9mIHRoZVxuICAgICAqICAgICAgYnVja2V0IGlzIHJldHVybmVkLlxuICAgICAqIEByZXR1cm5zIGFuIE9iamVjdFMzVXJsIHRva2VuXG4gICAgICovXG4gICAgdXJsRm9yT2JqZWN0KGtleT86IHN0cmluZyk6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgUzMgVVJMIG9mIGFuIFMzIG9iamVjdC4gRm9yIGV4YW1wbGU6XG4gICAgICogQGV4YW1wbGUgczM6Ly9vbmx5YnVja2V0XG4gICAgICogQGV4YW1wbGUgczM6Ly9idWNrZXQva2V5XG4gICAgICogQHBhcmFtIGtleSBUaGUgUzMga2V5IG9mIHRoZSBvYmplY3QuIElmIG5vdCBzcGVjaWZpZWQsIHRoZSBTMyBVUkwgb2YgdGhlXG4gICAgICogICAgICBidWNrZXQgaXMgcmV0dXJuZWQuXG4gICAgICogQHJldHVybnMgYW4gT2JqZWN0UzNVcmwgdG9rZW5cbiAgICAgKi9cbiAgICBzM1VybEZvck9iamVjdChrZXk/OiBzdHJpbmcpOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogUmV0dXJucyBhbiBBUk4gdGhhdCByZXByZXNlbnRzIGFsbCBvYmplY3RzIHdpdGhpbiB0aGUgYnVja2V0IHRoYXQgbWF0Y2hcbiAgICAgKiB0aGUga2V5IHBhdHRlcm4gc3BlY2lmaWVkLiBUbyByZXByZXNlbnQgYWxsIGtleXMsIHNwZWNpZnkgYGBcIipcImBgLlxuICAgICAqL1xuICAgIGFybkZvck9iamVjdHMoa2V5UGF0dGVybjogc3RyaW5nKTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEdyYW50IHJlYWQgcGVybWlzc2lvbnMgZm9yIHRoaXMgYnVja2V0IGFuZCBpdCdzIGNvbnRlbnRzIHRvIGFuIElBTVxuICAgICAqIHByaW5jaXBhbCAoUm9sZS9Hcm91cC9Vc2VyKS5cbiAgICAgKlxuICAgICAqIElmIGVuY3J5cHRpb24gaXMgdXNlZCwgcGVybWlzc2lvbiB0byB1c2UgdGhlIGtleSB0byBkZWNyeXB0IHRoZSBjb250ZW50c1xuICAgICAqIG9mIHRoZSBidWNrZXQgd2lsbCBhbHNvIGJlIGdyYW50ZWQgdG8gdGhlIHNhbWUgcHJpbmNpcGFsLlxuICAgICAqXG4gICAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICAgKiBAcGFyYW0gb2JqZWN0c0tleVBhdHRlcm4gUmVzdHJpY3QgdGhlIHBlcm1pc3Npb24gdG8gYSBjZXJ0YWluIGtleSBwYXR0ZXJuIChkZWZhdWx0ICcqJylcbiAgICAgKi9cbiAgICBncmFudFJlYWQoaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlLCBvYmplY3RzS2V5UGF0dGVybj86IGFueSk6IGlhbS5HcmFudDtcbiAgICAvKipcbiAgICAgKiBHcmFudCB3cml0ZSBwZXJtaXNzaW9ucyB0byB0aGlzIGJ1Y2tldCB0byBhbiBJQU0gcHJpbmNpcGFsLlxuICAgICAqXG4gICAgICogSWYgZW5jcnlwdGlvbiBpcyB1c2VkLCBwZXJtaXNzaW9uIHRvIHVzZSB0aGUga2V5IHRvIGVuY3J5cHQgdGhlIGNvbnRlbnRzXG4gICAgICogb2Ygd3JpdHRlbiBmaWxlcyB3aWxsIGFsc28gYmUgZ3JhbnRlZCB0byB0aGUgc2FtZSBwcmluY2lwYWwuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgICAqIEBwYXJhbSBvYmplY3RzS2V5UGF0dGVybiBSZXN0cmljdCB0aGUgcGVybWlzc2lvbiB0byBhIGNlcnRhaW4ga2V5IHBhdHRlcm4gKGRlZmF1bHQgJyonKVxuICAgICAqL1xuICAgIGdyYW50V3JpdGUoaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlLCBvYmplY3RzS2V5UGF0dGVybj86IGFueSk6IGlhbS5HcmFudDtcbiAgICAvKipcbiAgICAgKiBHcmFudHMgczM6UHV0T2JqZWN0KiBhbmQgczM6QWJvcnQqIHBlcm1pc3Npb25zIGZvciB0aGlzIGJ1Y2tldCB0byBhbiBJQU0gcHJpbmNpcGFsLlxuICAgICAqXG4gICAgICogSWYgZW5jcnlwdGlvbiBpcyB1c2VkLCBwZXJtaXNzaW9uIHRvIHVzZSB0aGUga2V5IHRvIGVuY3J5cHQgdGhlIGNvbnRlbnRzXG4gICAgICogb2Ygd3JpdHRlbiBmaWxlcyB3aWxsIGFsc28gYmUgZ3JhbnRlZCB0byB0aGUgc2FtZSBwcmluY2lwYWwuXG4gICAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICAgKiBAcGFyYW0gb2JqZWN0c0tleVBhdHRlcm4gUmVzdHJpY3QgdGhlIHBlcm1pc3Npb24gdG8gYSBjZXJ0YWluIGtleSBwYXR0ZXJuIChkZWZhdWx0ICcqJylcbiAgICAgKi9cbiAgICBncmFudFB1dChpZGVudGl0eTogaWFtLklHcmFudGFibGUsIG9iamVjdHNLZXlQYXR0ZXJuPzogYW55KTogaWFtLkdyYW50O1xuICAgIC8qKlxuICAgICAqIEdyYW50cyBzMzpEZWxldGVPYmplY3QqIHBlcm1pc3Npb24gdG8gYW4gSUFNIHByaWNpcGFsIGZvciBvYmplY3RzXG4gICAgICogaW4gdGhpcyBidWNrZXQuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gaWRlbnRpdHkgVGhlIHByaW5jaXBhbFxuICAgICAqIEBwYXJhbSBvYmplY3RzS2V5UGF0dGVybiBSZXN0cmljdCB0aGUgcGVybWlzc2lvbiB0byBhIGNlcnRhaW4ga2V5IHBhdHRlcm4gKGRlZmF1bHQgJyonKVxuICAgICAqL1xuICAgIGdyYW50RGVsZXRlKGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSwgb2JqZWN0c0tleVBhdHRlcm4/OiBhbnkpOiBpYW0uR3JhbnQ7XG4gICAgLyoqXG4gICAgICogR3JhbnRzIHJlYWQvd3JpdGUgcGVybWlzc2lvbnMgZm9yIHRoaXMgYnVja2V0IGFuZCBpdCdzIGNvbnRlbnRzIHRvIGFuIElBTVxuICAgICAqIHByaW5jaXBhbCAoUm9sZS9Hcm91cC9Vc2VyKS5cbiAgICAgKlxuICAgICAqIElmIGFuIGVuY3J5cHRpb24ga2V5IGlzIHVzZWQsIHBlcm1pc3Npb24gdG8gdXNlIHRoZSBrZXkgZm9yXG4gICAgICogZW5jcnlwdC9kZWNyeXB0IHdpbGwgYWxzbyBiZSBncmFudGVkLlxuICAgICAqXG4gICAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICAgKiBAcGFyYW0gb2JqZWN0c0tleVBhdHRlcm4gUmVzdHJpY3QgdGhlIHBlcm1pc3Npb24gdG8gYSBjZXJ0YWluIGtleSBwYXR0ZXJuIChkZWZhdWx0ICcqJylcbiAgICAgKi9cbiAgICBncmFudFJlYWRXcml0ZShpZGVudGl0eTogaWFtLklHcmFudGFibGUsIG9iamVjdHNLZXlQYXR0ZXJuPzogYW55KTogaWFtLkdyYW50O1xuICAgIC8qKlxuICAgICAqIEFsbG93cyB1bnJlc3RyaWN0ZWQgYWNjZXNzIHRvIG9iamVjdHMgZnJvbSB0aGlzIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIElNUE9SVEFOVDogVGhpcyBwZXJtaXNzaW9uIGFsbG93cyBhbnlvbmUgdG8gcGVyZm9ybSBhY3Rpb25zIG9uIFMzIG9iamVjdHNcbiAgICAgKiBpbiB0aGlzIGJ1Y2tldCwgd2hpY2ggaXMgdXNlZnVsIGZvciB3aGVuIHlvdSBjb25maWd1cmUgeW91ciBidWNrZXQgYXMgYVxuICAgICAqIHdlYnNpdGUgYW5kIHdhbnQgZXZlcnlvbmUgdG8gYmUgYWJsZSB0byByZWFkIG9iamVjdHMgaW4gdGhlIGJ1Y2tldCB3aXRob3V0XG4gICAgICogbmVlZGluZyB0byBhdXRoZW50aWNhdGUuXG4gICAgICpcbiAgICAgKiBXaXRob3V0IGFyZ3VtZW50cywgdGhpcyBtZXRob2Qgd2lsbCBncmFudCByZWFkIChcInMzOkdldE9iamVjdFwiKSBhY2Nlc3MgdG9cbiAgICAgKiBhbGwgb2JqZWN0cyAoXCIqXCIpIGluIHRoZSBidWNrZXQuXG4gICAgICpcbiAgICAgKiBUaGUgbWV0aG9kIHJldHVybnMgdGhlIGBpYW0uR3JhbnRgIG9iamVjdCwgd2hpY2ggY2FuIHRoZW4gYmUgbW9kaWZpZWRcbiAgICAgKiBhcyBuZWVkZWQuIEZvciBleGFtcGxlLCB5b3UgY2FuIGFkZCBhIGNvbmRpdGlvbiB0aGF0IHdpbGwgcmVzdHJpY3QgYWNjZXNzIG9ubHlcbiAgICAgKiB0byBhbiBJUHY0IHJhbmdlIGxpa2UgdGhpczpcbiAgICAgKlxuICAgICAqICAgICBjb25zdCBncmFudCA9IGJ1Y2tldC5ncmFudFB1YmxpY0FjY2VzcygpO1xuICAgICAqICAgICBncmFudC5yZXNvdXJjZVN0YXRlbWVudCEuYWRkQ29uZGl0aW9uKOKAmElwQWRkcmVzc+KAmSwgeyDigJxhd3M6U291cmNlSXDigJ06IOKAnDU0LjI0MC4xNDMuMC8yNOKAnSB9KTtcbiAgICAgKlxuICAgICAqXG4gICAgICogQHBhcmFtIGtleVByZWZpeCB0aGUgcHJlZml4IG9mIFMzIG9iamVjdCBrZXlzIChlLmcuIGBob21lLypgKS4gRGVmYXVsdCBpcyBcIipcIi5cbiAgICAgKiBAcGFyYW0gYWxsb3dlZEFjdGlvbnMgdGhlIHNldCBvZiBTMyBhY3Rpb25zIHRvIGFsbG93LiBEZWZhdWx0IGlzIFwiczM6R2V0T2JqZWN0XCIuXG4gICAgICogQHJldHVybnMgVGhlIGBpYW0uUG9saWN5U3RhdGVtZW50YCBvYmplY3QsIHdoaWNoIGNhbiBiZSB1c2VkIHRvIGFwcGx5IGUuZy4gY29uZGl0aW9ucy5cbiAgICAgKi9cbiAgICBncmFudFB1YmxpY0FjY2VzcyhrZXlQcmVmaXg/OiBzdHJpbmcsIC4uLmFsbG93ZWRBY3Rpb25zOiBzdHJpbmdbXSk6IGlhbS5HcmFudDtcbiAgICAvKipcbiAgICAgKiBEZWZpbmVzIGEgQ2xvdWRXYXRjaCBldmVudCB0aGF0IHRyaWdnZXJzIHdoZW4gc29tZXRoaW5nIGhhcHBlbnMgdG8gdGhpcyBidWNrZXRcbiAgICAgKlxuICAgICAqIFJlcXVpcmVzIHRoYXQgdGhlcmUgZXhpc3RzIGF0IGxlYXN0IG9uZSBDbG91ZFRyYWlsIFRyYWlsIGluIHlvdXIgYWNjb3VudFxuICAgICAqIHRoYXQgY2FwdHVyZXMgdGhlIGV2ZW50LiBUaGlzIG1ldGhvZCB3aWxsIG5vdCBjcmVhdGUgdGhlIFRyYWlsLlxuICAgICAqXG4gICAgICogQHBhcmFtIGlkIFRoZSBpZCBvZiB0aGUgcnVsZVxuICAgICAqIEBwYXJhbSBvcHRpb25zIE9wdGlvbnMgZm9yIGFkZGluZyB0aGUgcnVsZVxuICAgICAqL1xuICAgIG9uQ2xvdWRUcmFpbEV2ZW50KGlkOiBzdHJpbmcsIG9wdGlvbnM/OiBPbkNsb3VkVHJhaWxCdWNrZXRFdmVudE9wdGlvbnMpOiBldmVudHMuUnVsZTtcbiAgICAvKipcbiAgICAgKiBEZWZpbmVzIGFuIEFXUyBDbG91ZFdhdGNoIGV2ZW50IHRoYXQgdHJpZ2dlcnMgd2hlbiBhbiBvYmplY3QgaXMgdXBsb2FkZWRcbiAgICAgKiB0byB0aGUgc3BlY2lmaWVkIHBhdGhzIChrZXlzKSBpbiB0aGlzIGJ1Y2tldCB1c2luZyB0aGUgUHV0T2JqZWN0IEFQSSBjYWxsLlxuICAgICAqXG4gICAgICogTm90ZSB0aGF0IHNvbWUgdG9vbHMgbGlrZSBgYXdzIHMzIGNwYCB3aWxsIGF1dG9tYXRpY2FsbHkgdXNlIGVpdGhlclxuICAgICAqIFB1dE9iamVjdCBvciB0aGUgbXVsdGlwYXJ0IHVwbG9hZCBBUEkgZGVwZW5kaW5nIG9uIHRoZSBmaWxlIHNpemUsXG4gICAgICogc28gdXNpbmcgYG9uQ2xvdWRUcmFpbFdyaXRlT2JqZWN0YCBtYXkgYmUgcHJlZmVyYWJsZS5cbiAgICAgKlxuICAgICAqIFJlcXVpcmVzIHRoYXQgdGhlcmUgZXhpc3RzIGF0IGxlYXN0IG9uZSBDbG91ZFRyYWlsIFRyYWlsIGluIHlvdXIgYWNjb3VudFxuICAgICAqIHRoYXQgY2FwdHVyZXMgdGhlIGV2ZW50LiBUaGlzIG1ldGhvZCB3aWxsIG5vdCBjcmVhdGUgdGhlIFRyYWlsLlxuICAgICAqXG4gICAgICogQHBhcmFtIGlkIFRoZSBpZCBvZiB0aGUgcnVsZVxuICAgICAqIEBwYXJhbSBvcHRpb25zIE9wdGlvbnMgZm9yIGFkZGluZyB0aGUgcnVsZVxuICAgICAqL1xuICAgIG9uQ2xvdWRUcmFpbFB1dE9iamVjdChpZDogc3RyaW5nLCBvcHRpb25zPzogT25DbG91ZFRyYWlsQnVja2V0RXZlbnRPcHRpb25zKTogZXZlbnRzLlJ1bGU7XG4gICAgLyoqXG4gICAgICogRGVmaW5lcyBhbiBBV1MgQ2xvdWRXYXRjaCBldmVudCB0aGF0IHRyaWdnZXJzIHdoZW4gYW4gb2JqZWN0IGF0IHRoZVxuICAgICAqIHNwZWNpZmllZCBwYXRocyAoa2V5cykgaW4gdGhpcyBidWNrZXQgYXJlIHdyaXR0ZW4gdG8uICBUaGlzIGluY2x1ZGVzXG4gICAgICogdGhlIGV2ZW50cyBQdXRPYmplY3QsIENvcHlPYmplY3QsIGFuZCBDb21wbGV0ZU11bHRpcGFydFVwbG9hZC5cbiAgICAgKlxuICAgICAqIE5vdGUgdGhhdCBzb21lIHRvb2xzIGxpa2UgYGF3cyBzMyBjcGAgd2lsbCBhdXRvbWF0aWNhbGx5IHVzZSBlaXRoZXJcbiAgICAgKiBQdXRPYmplY3Qgb3IgdGhlIG11bHRpcGFydCB1cGxvYWQgQVBJIGRlcGVuZGluZyBvbiB0aGUgZmlsZSBzaXplLFxuICAgICAqIHNvIHVzaW5nIHRoaXMgbWV0aG9kIG1heSBiZSBwcmVmZXJhYmxlIHRvIGBvbkNsb3VkVHJhaWxQdXRPYmplY3RgLlxuICAgICAqXG4gICAgICogUmVxdWlyZXMgdGhhdCB0aGVyZSBleGlzdHMgYXQgbGVhc3Qgb25lIENsb3VkVHJhaWwgVHJhaWwgaW4geW91ciBhY2NvdW50XG4gICAgICogdGhhdCBjYXB0dXJlcyB0aGUgZXZlbnQuIFRoaXMgbWV0aG9kIHdpbGwgbm90IGNyZWF0ZSB0aGUgVHJhaWwuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gaWQgVGhlIGlkIG9mIHRoZSBydWxlXG4gICAgICogQHBhcmFtIG9wdGlvbnMgT3B0aW9ucyBmb3IgYWRkaW5nIHRoZSBydWxlXG4gICAgICovXG4gICAgb25DbG91ZFRyYWlsV3JpdGVPYmplY3QoaWQ6IHN0cmluZywgb3B0aW9ucz86IE9uQ2xvdWRUcmFpbEJ1Y2tldEV2ZW50T3B0aW9ucyk6IGV2ZW50cy5SdWxlO1xufVxuLyoqXG4gKiBBIHJlZmVyZW5jZSB0byBhIGJ1Y2tldC4gVGhlIGVhc2llc3Qgd2F5IHRvIGluc3RhbnRpYXRlIGlzIHRvIGNhbGxcbiAqIGBidWNrZXQuZXhwb3J0KClgLiBUaGVuLCB0aGUgY29uc3VtZXIgY2FuIHVzZSBgQnVja2V0LmltcG9ydCh0aGlzLCByZWYpYCBhbmRcbiAqIGdldCBhIGBCdWNrZXRgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEJ1Y2tldEF0dHJpYnV0ZXMge1xuICAgIC8qKlxuICAgICAqIFRoZSBBUk4gb2YgdGhlIGJ1Y2tldC4gQXQgbGVhc3Qgb25lIG9mIGJ1Y2tldEFybiBvciBidWNrZXROYW1lIG11c3QgYmVcbiAgICAgKiBkZWZpbmVkIGluIG9yZGVyIHRvIGluaXRpYWxpemUgYSBidWNrZXQgcmVmLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJ1Y2tldEFybj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgYnVja2V0LiBJZiB0aGUgdW5kZXJseWluZyB2YWx1ZSBvZiBBUk4gaXMgYSBzdHJpbmcsIHRoZVxuICAgICAqIG5hbWUgd2lsbCBiZSBwYXJzZWQgZnJvbSB0aGUgQVJOLiBPdGhlcndpc2UsIHRoZSBuYW1lIGlzIG9wdGlvbmFsLCBidXRcbiAgICAgKiBzb21lIGZlYXR1cmVzIHRoYXQgcmVxdWlyZSB0aGUgYnVja2V0IG5hbWUgc3VjaCBhcyBhdXRvLWNyZWF0aW5nIGEgYnVja2V0XG4gICAgICogcG9saWN5LCB3b24ndCB3b3JrLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJ1Y2tldE5hbWU/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIGRvbWFpbiBuYW1lIG9mIHRoZSBidWNrZXQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBJbmZlcnJlZCBmcm9tIGJ1Y2tldCBuYW1lXG4gICAgICovXG4gICAgcmVhZG9ubHkgYnVja2V0RG9tYWluTmFtZT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgd2Vic2l0ZSBVUkwgb2YgdGhlIGJ1Y2tldCAoaWYgc3RhdGljIHdlYiBob3N0aW5nIGlzIGVuYWJsZWQpLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgSW5mZXJyZWQgZnJvbSBidWNrZXQgbmFtZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJ1Y2tldFdlYnNpdGVVcmw/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHJlZ2lvbmFsIGRvbWFpbiBuYW1lIG9mIHRoZSBzcGVjaWZpZWQgYnVja2V0LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJ1Y2tldFJlZ2lvbmFsRG9tYWluTmFtZT86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgSVB2NiBETlMgbmFtZSBvZiB0aGUgc3BlY2lmaWVkIGJ1Y2tldC5cbiAgICAgKi9cbiAgICByZWFkb25seSBidWNrZXREdWFsU3RhY2tEb21haW5OYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBmb3JtYXQgb2YgdGhlIHdlYnNpdGUgVVJMIG9mIHRoZSBidWNrZXQuIFRoaXMgc2hvdWxkIGJlIHRydWUgZm9yXG4gICAgICogcmVnaW9ucyBsYXVuY2hlZCBzaW5jZSAyMDE0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgZmFsc2VcbiAgICAgKi9cbiAgICByZWFkb25seSBidWNrZXRXZWJzaXRlTmV3VXJsRm9ybWF0PzogYm9vbGVhbjtcbiAgICByZWFkb25seSBlbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG4gICAgLyoqXG4gICAgICogSWYgdGhpcyBidWNrZXQgaGFzIGJlZW4gY29uZmlndXJlZCBmb3Igc3RhdGljIHdlYnNpdGUgaG9zdGluZy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgaXNXZWJzaXRlPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBUaGUgYWNjb3VudCB0aGlzIGV4aXN0aW5nIGJ1Y2tldCBiZWxvbmdzIHRvLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBpdCdzIGFzc3VtZWQgdGhlIGJ1Y2tldCBiZWxvbmdzIHRvIHRoZSBzYW1lIGFjY291bnQgYXMgdGhlIHNjb3BlIGl0J3MgYmVpbmcgaW1wb3J0ZWQgaW50b1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGFjY291bnQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHJlZ2lvbiB0aGlzIGV4aXN0aW5nIGJ1Y2tldCBpcyBpbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gaXQncyBhc3N1bWVkIHRoZSBidWNrZXQgaXMgaW4gdGhlIHNhbWUgcmVnaW9uIGFzIHRoZSBzY29wZSBpdCdzIGJlaW5nIGltcG9ydGVkIGludG9cbiAgICAgKi9cbiAgICByZWFkb25seSByZWdpb24/OiBzdHJpbmc7XG59XG4vKipcbiAqIFJlcHJlc2VudHMgYW4gUzMgQnVja2V0LlxuICpcbiAqIEJ1Y2tldHMgY2FuIGJlIGVpdGhlciBkZWZpbmVkIHdpdGhpbiB0aGlzIHN0YWNrOlxuICpcbiAqICAgbmV3IEJ1Y2tldCh0aGlzLCAnTXlCdWNrZXQnLCB7IHByb3BzIH0pO1xuICpcbiAqIE9yIGltcG9ydGVkIGZyb20gYW4gZXhpc3RpbmcgYnVja2V0OlxuICpcbiAqICAgQnVja2V0LmltcG9ydCh0aGlzLCAnTXlJbXBvcnRlZEJ1Y2tldCcsIHsgYnVja2V0QXJuOiAuLi4gfSk7XG4gKlxuICogWW91IGNhbiBhbHNvIGV4cG9ydCBhIGJ1Y2tldCBhbmQgaW1wb3J0IGl0IGludG8gYW5vdGhlciBzdGFjazpcbiAqXG4gKiAgIGNvbnN0IHJlZiA9IG15QnVja2V0LmV4cG9ydCgpO1xuICogICBCdWNrZXQuaW1wb3J0KHRoaXMsICdNeUltcG9ydGVkQnVja2V0JywgcmVmKTtcbiAqXG4gKi9cbmFic3RyYWN0IGNsYXNzIEJ1Y2tldEJhc2UgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElCdWNrZXQge1xuICAgIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBidWNrZXRBcm46IHN0cmluZztcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgYnVja2V0TmFtZTogc3RyaW5nO1xuICAgIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBidWNrZXREb21haW5OYW1lOiBzdHJpbmc7XG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGJ1Y2tldFdlYnNpdGVVcmw6IHN0cmluZztcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgYnVja2V0V2Vic2l0ZURvbWFpbk5hbWU6IHN0cmluZztcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgYnVja2V0UmVnaW9uYWxEb21haW5OYW1lOiBzdHJpbmc7XG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGJ1Y2tldER1YWxTdGFja0RvbWFpbk5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBPcHRpb25hbCBLTVMgZW5jcnlwdGlvbiBrZXkgYXNzb2NpYXRlZCB3aXRoIHRoaXMgYnVja2V0LlxuICAgICAqL1xuICAgIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBlbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG4gICAgLyoqXG4gICAgICogSWYgdGhpcyBidWNrZXQgaGFzIGJlZW4gY29uZmlndXJlZCBmb3Igc3RhdGljIHdlYnNpdGUgaG9zdGluZy5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgaXNXZWJzaXRlPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBUaGUgcmVzb3VyY2UgcG9saWN5IGFzc29jaWF0ZWQgd2l0aCB0aGlzIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIElmIGBhdXRvQ3JlYXRlUG9saWN5YCBpcyB0cnVlLCBhIGBCdWNrZXRQb2xpY3lgIHdpbGwgYmUgY3JlYXRlZCB1cG9uIHRoZVxuICAgICAqIGZpcnN0IGNhbGwgdG8gYWRkVG9SZXNvdXJjZVBvbGljeShzKS5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWJzdHJhY3QgcG9saWN5PzogQnVja2V0UG9saWN5O1xuICAgIC8qKlxuICAgICAqIEluZGljYXRlcyBpZiBhIGJ1Y2tldCByZXNvdXJjZSBwb2xpY3kgc2hvdWxkIGF1dG9tYXRpY2FsbHkgY3JlYXRlZCB1cG9uXG4gICAgICogdGhlIGZpcnN0IGNhbGwgdG8gYGFkZFRvUmVzb3VyY2VQb2xpY3lgLlxuICAgICAqL1xuICAgIHByb3RlY3RlZCBhYnN0cmFjdCBhdXRvQ3JlYXRlUG9saWN5ID0gZmFsc2U7XG4gICAgLyoqXG4gICAgICogV2hldGhlciB0byBkaXNhbGxvdyBwdWJsaWMgYWNjZXNzXG4gICAgICovXG4gICAgcHJvdGVjdGVkIGFic3RyYWN0IGRpc2FsbG93UHVibGljQWNjZXNzPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBEZWZpbmUgYSBDbG91ZFdhdGNoIGV2ZW50IHRoYXQgdHJpZ2dlcnMgd2hlbiBzb21ldGhpbmcgaGFwcGVucyB0byB0aGlzIHJlcG9zaXRvcnlcbiAgICAgKlxuICAgICAqIFJlcXVpcmVzIHRoYXQgdGhlcmUgZXhpc3RzIGF0IGxlYXN0IG9uZSBDbG91ZFRyYWlsIFRyYWlsIGluIHlvdXIgYWNjb3VudFxuICAgICAqIHRoYXQgY2FwdHVyZXMgdGhlIGV2ZW50LiBUaGlzIG1ldGhvZCB3aWxsIG5vdCBjcmVhdGUgdGhlIFRyYWlsLlxuICAgICAqXG4gICAgICogQHBhcmFtIGlkIFRoZSBpZCBvZiB0aGUgcnVsZVxuICAgICAqIEBwYXJhbSBvcHRpb25zIE9wdGlvbnMgZm9yIGFkZGluZyB0aGUgcnVsZVxuICAgICAqL1xuICAgIHB1YmxpYyBvbkNsb3VkVHJhaWxFdmVudChpZDogc3RyaW5nLCBvcHRpb25zOiBPbkNsb3VkVHJhaWxCdWNrZXRFdmVudE9wdGlvbnMgPSB7fSk6IGV2ZW50cy5SdWxlIHtcbiAgICAgICAgY29uc3QgcnVsZSA9IG5ldyBldmVudHMuUnVsZSh0aGlzLCBpZCwgb3B0aW9ucyk7XG4gICAgICAgIHJ1bGUuYWRkVGFyZ2V0KG9wdGlvbnMudGFyZ2V0KTtcbiAgICAgICAgcnVsZS5hZGRFdmVudFBhdHRlcm4oe1xuICAgICAgICAgICAgc291cmNlOiBbJ2F3cy5zMyddLFxuICAgICAgICAgICAgZGV0YWlsVHlwZTogWydBV1MgQVBJIENhbGwgdmlhIENsb3VkVHJhaWwnXSxcbiAgICAgICAgICAgIGRldGFpbDoge1xuICAgICAgICAgICAgICAgIHJlc291cmNlczoge1xuICAgICAgICAgICAgICAgICAgICBBUk46IG9wdGlvbnMucGF0aHMgPyBvcHRpb25zLnBhdGhzLm1hcChwID0+IHRoaXMuYXJuRm9yT2JqZWN0cyhwKSkgOiBbdGhpcy5idWNrZXRBcm5dLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgICAgcmV0dXJuIHJ1bGU7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIERlZmluZXMgYW4gQVdTIENsb3VkV2F0Y2ggZXZlbnQgdGhhdCB0cmlnZ2VycyB3aGVuIGFuIG9iamVjdCBpcyB1cGxvYWRlZFxuICAgICAqIHRvIHRoZSBzcGVjaWZpZWQgcGF0aHMgKGtleXMpIGluIHRoaXMgYnVja2V0IHVzaW5nIHRoZSBQdXRPYmplY3QgQVBJIGNhbGwuXG4gICAgICpcbiAgICAgKiBOb3RlIHRoYXQgc29tZSB0b29scyBsaWtlIGBhd3MgczMgY3BgIHdpbGwgYXV0b21hdGljYWxseSB1c2UgZWl0aGVyXG4gICAgICogUHV0T2JqZWN0IG9yIHRoZSBtdWx0aXBhcnQgdXBsb2FkIEFQSSBkZXBlbmRpbmcgb24gdGhlIGZpbGUgc2l6ZSxcbiAgICAgKiBzbyB1c2luZyBgb25DbG91ZFRyYWlsV3JpdGVPYmplY3RgIG1heSBiZSBwcmVmZXJhYmxlLlxuICAgICAqXG4gICAgICogUmVxdWlyZXMgdGhhdCB0aGVyZSBleGlzdHMgYXQgbGVhc3Qgb25lIENsb3VkVHJhaWwgVHJhaWwgaW4geW91ciBhY2NvdW50XG4gICAgICogdGhhdCBjYXB0dXJlcyB0aGUgZXZlbnQuIFRoaXMgbWV0aG9kIHdpbGwgbm90IGNyZWF0ZSB0aGUgVHJhaWwuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gaWQgVGhlIGlkIG9mIHRoZSBydWxlXG4gICAgICogQHBhcmFtIG9wdGlvbnMgT3B0aW9ucyBmb3IgYWRkaW5nIHRoZSBydWxlXG4gICAgICovXG4gICAgcHVibGljIG9uQ2xvdWRUcmFpbFB1dE9iamVjdChpZDogc3RyaW5nLCBvcHRpb25zOiBPbkNsb3VkVHJhaWxCdWNrZXRFdmVudE9wdGlvbnMgPSB7fSk6IGV2ZW50cy5SdWxlIHtcbiAgICAgICAgY29uc3QgcnVsZSA9IHRoaXMub25DbG91ZFRyYWlsRXZlbnQoaWQsIG9wdGlvbnMpO1xuICAgICAgICBydWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAgICAgICAgICBkZXRhaWw6IHtcbiAgICAgICAgICAgICAgICBldmVudE5hbWU6IFsnUHV0T2JqZWN0J10sXG4gICAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICAgICAgcmV0dXJuIHJ1bGU7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIERlZmluZXMgYW4gQVdTIENsb3VkV2F0Y2ggZXZlbnQgdGhhdCB0cmlnZ2VycyB3aGVuIGFuIG9iamVjdCBhdCB0aGVcbiAgICAgKiBzcGVjaWZpZWQgcGF0aHMgKGtleXMpIGluIHRoaXMgYnVja2V0IGFyZSB3cml0dGVuIHRvLiAgVGhpcyBpbmNsdWRlc1xuICAgICAqIHRoZSBldmVudHMgUHV0T2JqZWN0LCBDb3B5T2JqZWN0LCBhbmQgQ29tcGxldGVNdWx0aXBhcnRVcGxvYWQuXG4gICAgICpcbiAgICAgKiBOb3RlIHRoYXQgc29tZSB0b29scyBsaWtlIGBhd3MgczMgY3BgIHdpbGwgYXV0b21hdGljYWxseSB1c2UgZWl0aGVyXG4gICAgICogUHV0T2JqZWN0IG9yIHRoZSBtdWx0aXBhcnQgdXBsb2FkIEFQSSBkZXBlbmRpbmcgb24gdGhlIGZpbGUgc2l6ZSxcbiAgICAgKiBzbyB1c2luZyB0aGlzIG1ldGhvZCBtYXkgYmUgcHJlZmVyYWJsZSB0byBgb25DbG91ZFRyYWlsUHV0T2JqZWN0YC5cbiAgICAgKlxuICAgICAqIFJlcXVpcmVzIHRoYXQgdGhlcmUgZXhpc3RzIGF0IGxlYXN0IG9uZSBDbG91ZFRyYWlsIFRyYWlsIGluIHlvdXIgYWNjb3VudFxuICAgICAqIHRoYXQgY2FwdHVyZXMgdGhlIGV2ZW50LiBUaGlzIG1ldGhvZCB3aWxsIG5vdCBjcmVhdGUgdGhlIFRyYWlsLlxuICAgICAqXG4gICAgICogQHBhcmFtIGlkIFRoZSBpZCBvZiB0aGUgcnVsZVxuICAgICAqIEBwYXJhbSBvcHRpb25zIE9wdGlvbnMgZm9yIGFkZGluZyB0aGUgcnVsZVxuICAgICAqL1xuICAgIHB1YmxpYyBvbkNsb3VkVHJhaWxXcml0ZU9iamVjdChpZDogc3RyaW5nLCBvcHRpb25zOiBPbkNsb3VkVHJhaWxCdWNrZXRFdmVudE9wdGlvbnMgPSB7fSk6IGV2ZW50cy5SdWxlIHtcbiAgICAgICAgY29uc3QgcnVsZSA9IHRoaXMub25DbG91ZFRyYWlsRXZlbnQoaWQsIG9wdGlvbnMpO1xuICAgICAgICBydWxlLmFkZEV2ZW50UGF0dGVybih7XG4gICAgICAgICAgICBkZXRhaWw6IHtcbiAgICAgICAgICAgICAgICBldmVudE5hbWU6IFtcbiAgICAgICAgICAgICAgICAgICAgJ0NvbXBsZXRlTXVsdGlwYXJ0VXBsb2FkJyxcbiAgICAgICAgICAgICAgICAgICAgJ0NvcHlPYmplY3QnLFxuICAgICAgICAgICAgICAgICAgICAnUHV0T2JqZWN0JyxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgIHJlcXVlc3RQYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICAgICAgICAgIGJ1Y2tldE5hbWU6IFt0aGlzLmJ1Y2tldE5hbWVdLFxuICAgICAgICAgICAgICAgICAgICBrZXk6IG9wdGlvbnMucGF0aHMsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgICByZXR1cm4gcnVsZTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkcyBhIHN0YXRlbWVudCB0byB0aGUgcmVzb3VyY2UgcG9saWN5IGZvciBhIHByaW5jaXBhbCAoaS5lLlxuICAgICAqIGFjY291bnQvcm9sZS9zZXJ2aWNlKSB0byBwZXJmb3JtIGFjdGlvbnMgb24gdGhpcyBidWNrZXQgYW5kL29yIGl0J3NcbiAgICAgKiBjb250ZW50cy4gVXNlIGBidWNrZXRBcm5gIGFuZCBgYXJuRm9yT2JqZWN0cyhrZXlzKWAgdG8gb2J0YWluIEFSTnMgZm9yXG4gICAgICogdGhpcyBidWNrZXQgb3Igb2JqZWN0cy5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkVG9SZXNvdXJjZVBvbGljeShwZXJtaXNzaW9uOiBpYW0uUG9saWN5U3RhdGVtZW50KTogaWFtLkFkZFRvUmVzb3VyY2VQb2xpY3lSZXN1bHQge1xuICAgICAgICBpZiAoIXRoaXMucG9saWN5ICYmIHRoaXMuYXV0b0NyZWF0ZVBvbGljeSkge1xuICAgICAgICAgICAgdGhpcy5wb2xpY3kgPSBuZXcgQnVja2V0UG9saWN5KHRoaXMsICdQb2xpY3knLCB7IGJ1Y2tldDogdGhpcyB9KTtcbiAgICAgICAgfVxuICAgICAgICBpZiAodGhpcy5wb2xpY3kpIHtcbiAgICAgICAgICAgIHRoaXMucG9saWN5LmRvY3VtZW50LmFkZFN0YXRlbWVudHMocGVybWlzc2lvbik7XG4gICAgICAgICAgICByZXR1cm4geyBzdGF0ZW1lbnRBZGRlZDogdHJ1ZSwgcG9saWN5RGVwZW5kYWJsZTogdGhpcy5wb2xpY3kgfTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4geyBzdGF0ZW1lbnRBZGRlZDogZmFsc2UgfTtcbiAgICB9XG4gICAgcHJvdGVjdGVkIHZhbGlkYXRlKCk6IHN0cmluZ1tdIHtcbiAgICAgICAgY29uc3QgZXJyb3JzID0gc3VwZXIudmFsaWRhdGUoKTtcbiAgICAgICAgZXJyb3JzLnB1c2goLi4udGhpcy5wb2xpY3k/LmRvY3VtZW50LnZhbGlkYXRlRm9yUmVzb3VyY2VQb2xpY3koKSB8fCBbXSk7XG4gICAgICAgIHJldHVybiBlcnJvcnM7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFRoZSBodHRwcyBVUkwgb2YgYW4gUzMgb2JqZWN0LiBGb3IgZXhhbXBsZTpcbiAgICAgKiBAZXhhbXBsZSBodHRwczovL3MzLnVzLXdlc3QtMS5hbWF6b25hd3MuY29tL29ubHlidWNrZXRcbiAgICAgKiBAZXhhbXBsZSBodHRwczovL3MzLnVzLXdlc3QtMS5hbWF6b25hd3MuY29tL2J1Y2tldC9rZXlcbiAgICAgKiBAZXhhbXBsZSBodHRwczovL3MzLmNuLW5vcnRoLTEuYW1hem9uYXdzLmNvbS5jbi9jaGluYS1idWNrZXQvbXlrZXlcbiAgICAgKiBAcGFyYW0ga2V5IFRoZSBTMyBrZXkgb2YgdGhlIG9iamVjdC4gSWYgbm90IHNwZWNpZmllZCwgdGhlIFVSTCBvZiB0aGVcbiAgICAgKiAgICAgIGJ1Y2tldCBpcyByZXR1cm5lZC5cbiAgICAgKiBAcmV0dXJucyBhbiBPYmplY3RTM1VybCB0b2tlblxuICAgICAqL1xuICAgIHB1YmxpYyB1cmxGb3JPYmplY3Qoa2V5Pzogc3RyaW5nKTogc3RyaW5nIHtcbiAgICAgICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcbiAgICAgICAgY29uc3QgcHJlZml4ID0gYGh0dHBzOi8vczMuJHtzdGFjay5yZWdpb259LiR7c3RhY2sudXJsU3VmZml4fS9gO1xuICAgICAgICByZXR1cm4gdGhpcy5idWlsZFVybChwcmVmaXgsIGtleSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFRoZSBTMyBVUkwgb2YgYW4gUzMgb2JqZWN0LiBGb3IgZXhhbXBsZTpcbiAgICAgKiBAZXhhbXBsZSBzMzovL29ubHlidWNrZXRcbiAgICAgKiBAZXhhbXBsZSBzMzovL2J1Y2tldC9rZXlcbiAgICAgKiBAcGFyYW0ga2V5IFRoZSBTMyBrZXkgb2YgdGhlIG9iamVjdC4gSWYgbm90IHNwZWNpZmllZCwgdGhlIFMzIFVSTCBvZiB0aGVcbiAgICAgKiAgICAgIGJ1Y2tldCBpcyByZXR1cm5lZC5cbiAgICAgKiBAcmV0dXJucyBhbiBPYmplY3RTM1VybCB0b2tlblxuICAgICAqL1xuICAgIHB1YmxpYyBzM1VybEZvck9iamVjdChrZXk/OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gdGhpcy5idWlsZFVybCgnczM6Ly8nLCBrZXkpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm5zIGFuIEFSTiB0aGF0IHJlcHJlc2VudHMgYWxsIG9iamVjdHMgd2l0aGluIHRoZSBidWNrZXQgdGhhdCBtYXRjaFxuICAgICAqIHRoZSBrZXkgcGF0dGVybiBzcGVjaWZpZWQuIFRvIHJlcHJlc2VudCBhbGwga2V5cywgc3BlY2lmeSBgYFwiKlwiYGAuXG4gICAgICpcbiAgICAgKiBJZiB5b3Ugc3BlY2lmeSBtdWx0aXBsZSBjb21wb25lbnRzIGZvciBrZXlQYXR0ZXJuLCB0aGV5IHdpbGwgYmUgY29uY2F0ZW5hdGVkOjpcbiAgICAgKlxuICAgICAqICAgYXJuRm9yT2JqZWN0cygnaG9tZS8nLCB0ZWFtLCAnLycsIHVzZXIsICcvKicpXG4gICAgICpcbiAgICAgKi9cbiAgICBwdWJsaWMgYXJuRm9yT2JqZWN0cyhrZXlQYXR0ZXJuOiBzdHJpbmcpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gYCR7dGhpcy5idWNrZXRBcm59LyR7a2V5UGF0dGVybn1gO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBHcmFudCByZWFkIHBlcm1pc3Npb25zIGZvciB0aGlzIGJ1Y2tldCBhbmQgaXQncyBjb250ZW50cyB0byBhbiBJQU1cbiAgICAgKiBwcmluY2lwYWwgKFJvbGUvR3JvdXAvVXNlcikuXG4gICAgICpcbiAgICAgKiBJZiBlbmNyeXB0aW9uIGlzIHVzZWQsIHBlcm1pc3Npb24gdG8gdXNlIHRoZSBrZXkgdG8gZGVjcnlwdCB0aGUgY29udGVudHNcbiAgICAgKiBvZiB0aGUgYnVja2V0IHdpbGwgYWxzbyBiZSBncmFudGVkIHRvIHRoZSBzYW1lIHByaW5jaXBhbC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBpZGVudGl0eSBUaGUgcHJpbmNpcGFsXG4gICAgICogQHBhcmFtIG9iamVjdHNLZXlQYXR0ZXJuIFJlc3RyaWN0IHRoZSBwZXJtaXNzaW9uIHRvIGEgY2VydGFpbiBrZXkgcGF0dGVybiAoZGVmYXVsdCAnKicpXG4gICAgICovXG4gICAgcHVibGljIGdyYW50UmVhZChpZGVudGl0eTogaWFtLklHcmFudGFibGUsIG9iamVjdHNLZXlQYXR0ZXJuOiBhbnkgPSAnKicpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuZ3JhbnQoaWRlbnRpdHksIHBlcm1zLkJVQ0tFVF9SRUFEX0FDVElPTlMsIHBlcm1zLktFWV9SRUFEX0FDVElPTlMsIHRoaXMuYnVja2V0QXJuLCB0aGlzLmFybkZvck9iamVjdHMob2JqZWN0c0tleVBhdHRlcm4pKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogR3JhbnQgd3JpdGUgcGVybWlzc2lvbnMgdG8gdGhpcyBidWNrZXQgdG8gYW4gSUFNIHByaW5jaXBhbC5cbiAgICAgKlxuICAgICAqIElmIGVuY3J5cHRpb24gaXMgdXNlZCwgcGVybWlzc2lvbiB0byB1c2UgdGhlIGtleSB0byBlbmNyeXB0IHRoZSBjb250ZW50c1xuICAgICAqIG9mIHdyaXR0ZW4gZmlsZXMgd2lsbCBhbHNvIGJlIGdyYW50ZWQgdG8gdGhlIHNhbWUgcHJpbmNpcGFsLlxuICAgICAqXG4gICAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICAgKiBAcGFyYW0gb2JqZWN0c0tleVBhdHRlcm4gUmVzdHJpY3QgdGhlIHBlcm1pc3Npb24gdG8gYSBjZXJ0YWluIGtleSBwYXR0ZXJuIChkZWZhdWx0ICcqJylcbiAgICAgKi9cbiAgICBwdWJsaWMgZ3JhbnRXcml0ZShpZGVudGl0eTogaWFtLklHcmFudGFibGUsIG9iamVjdHNLZXlQYXR0ZXJuOiBhbnkgPSAnKicpIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuZ3JhbnQoaWRlbnRpdHksIHBlcm1zLkJVQ0tFVF9XUklURV9BQ1RJT05TLCBwZXJtcy5LRVlfV1JJVEVfQUNUSU9OUywgdGhpcy5idWNrZXRBcm4sIHRoaXMuYXJuRm9yT2JqZWN0cyhvYmplY3RzS2V5UGF0dGVybikpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBHcmFudHMgczM6UHV0T2JqZWN0KiBhbmQgczM6QWJvcnQqIHBlcm1pc3Npb25zIGZvciB0aGlzIGJ1Y2tldCB0byBhbiBJQU0gcHJpbmNpcGFsLlxuICAgICAqXG4gICAgICogSWYgZW5jcnlwdGlvbiBpcyB1c2VkLCBwZXJtaXNzaW9uIHRvIHVzZSB0aGUga2V5IHRvIGVuY3J5cHQgdGhlIGNvbnRlbnRzXG4gICAgICogb2Ygd3JpdHRlbiBmaWxlcyB3aWxsIGFsc28gYmUgZ3JhbnRlZCB0byB0aGUgc2FtZSBwcmluY2lwYWwuXG4gICAgICogQHBhcmFtIGlkZW50aXR5IFRoZSBwcmluY2lwYWxcbiAgICAgKiBAcGFyYW0gb2JqZWN0c0tleVBhdHRlcm4gUmVzdHJpY3QgdGhlIHBlcm1pc3Npb24gdG8gYSBjZXJ0YWluIGtleSBwYXR0ZXJuIChkZWZhdWx0ICcqJylcbiAgICAgKi9cbiAgICBwdWJsaWMgZ3JhbnRQdXQoaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlLCBvYmplY3RzS2V5UGF0dGVybjogYW55ID0gJyonKSB7XG4gICAgICAgIHJldHVybiB0aGlzLmdyYW50KGlkZW50aXR5LCBwZXJtcy5CVUNLRVRfUFVUX0FDVElPTlMsIHBlcm1zLktFWV9XUklURV9BQ1RJT05TLCB0aGlzLmFybkZvck9iamVjdHMob2JqZWN0c0tleVBhdHRlcm4pKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogR3JhbnRzIHMzOkRlbGV0ZU9iamVjdCogcGVybWlzc2lvbiB0byBhbiBJQU0gcHJpY2lwYWwgZm9yIG9iamVjdHNcbiAgICAgKiBpbiB0aGlzIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBpZGVudGl0eSBUaGUgcHJpbmNpcGFsXG4gICAgICogQHBhcmFtIG9iamVjdHNLZXlQYXR0ZXJuIFJlc3RyaWN0IHRoZSBwZXJtaXNzaW9uIHRvIGEgY2VydGFpbiBrZXkgcGF0dGVybiAoZGVmYXVsdCAnKicpXG4gICAgICovXG4gICAgcHVibGljIGdyYW50RGVsZXRlKGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSwgb2JqZWN0c0tleVBhdHRlcm46IGFueSA9ICcqJykge1xuICAgICAgICByZXR1cm4gdGhpcy5ncmFudChpZGVudGl0eSwgcGVybXMuQlVDS0VUX0RFTEVURV9BQ1RJT05TLCBbXSwgdGhpcy5hcm5Gb3JPYmplY3RzKG9iamVjdHNLZXlQYXR0ZXJuKSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEdyYW50cyByZWFkL3dyaXRlIHBlcm1pc3Npb25zIGZvciB0aGlzIGJ1Y2tldCBhbmQgaXQncyBjb250ZW50cyB0byBhbiBJQU1cbiAgICAgKiBwcmluY2lwYWwgKFJvbGUvR3JvdXAvVXNlcikuXG4gICAgICpcbiAgICAgKiBJZiBhbiBlbmNyeXB0aW9uIGtleSBpcyB1c2VkLCBwZXJtaXNzaW9uIHRvIHVzZSB0aGUga2V5IGZvclxuICAgICAqIGVuY3J5cHQvZGVjcnlwdCB3aWxsIGFsc28gYmUgZ3JhbnRlZC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBpZGVudGl0eSBUaGUgcHJpbmNpcGFsXG4gICAgICogQHBhcmFtIG9iamVjdHNLZXlQYXR0ZXJuIFJlc3RyaWN0IHRoZSBwZXJtaXNzaW9uIHRvIGEgY2VydGFpbiBrZXkgcGF0dGVybiAoZGVmYXVsdCAnKicpXG4gICAgICovXG4gICAgcHVibGljIGdyYW50UmVhZFdyaXRlKGlkZW50aXR5OiBpYW0uSUdyYW50YWJsZSwgb2JqZWN0c0tleVBhdHRlcm46IGFueSA9ICcqJykge1xuICAgICAgICBjb25zdCBidWNrZXRBY3Rpb25zID0gcGVybXMuQlVDS0VUX1JFQURfQUNUSU9OUy5jb25jYXQocGVybXMuQlVDS0VUX1dSSVRFX0FDVElPTlMpO1xuICAgICAgICBjb25zdCBrZXlBY3Rpb25zID0gcGVybXMuS0VZX1JFQURfQUNUSU9OUy5jb25jYXQocGVybXMuS0VZX1dSSVRFX0FDVElPTlMpO1xuICAgICAgICByZXR1cm4gdGhpcy5ncmFudChpZGVudGl0eSwgYnVja2V0QWN0aW9ucywga2V5QWN0aW9ucywgdGhpcy5idWNrZXRBcm4sIHRoaXMuYXJuRm9yT2JqZWN0cyhvYmplY3RzS2V5UGF0dGVybikpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBbGxvd3MgdW5yZXN0cmljdGVkIGFjY2VzcyB0byBvYmplY3RzIGZyb20gdGhpcyBidWNrZXQuXG4gICAgICpcbiAgICAgKiBJTVBPUlRBTlQ6IFRoaXMgcGVybWlzc2lvbiBhbGxvd3MgYW55b25lIHRvIHBlcmZvcm0gYWN0aW9ucyBvbiBTMyBvYmplY3RzXG4gICAgICogaW4gdGhpcyBidWNrZXQsIHdoaWNoIGlzIHVzZWZ1bCBmb3Igd2hlbiB5b3UgY29uZmlndXJlIHlvdXIgYnVja2V0IGFzIGFcbiAgICAgKiB3ZWJzaXRlIGFuZCB3YW50IGV2ZXJ5b25lIHRvIGJlIGFibGUgdG8gcmVhZCBvYmplY3RzIGluIHRoZSBidWNrZXQgd2l0aG91dFxuICAgICAqIG5lZWRpbmcgdG8gYXV0aGVudGljYXRlLlxuICAgICAqXG4gICAgICogV2l0aG91dCBhcmd1bWVudHMsIHRoaXMgbWV0aG9kIHdpbGwgZ3JhbnQgcmVhZCAoXCJzMzpHZXRPYmplY3RcIikgYWNjZXNzIHRvXG4gICAgICogYWxsIG9iamVjdHMgKFwiKlwiKSBpbiB0aGUgYnVja2V0LlxuICAgICAqXG4gICAgICogVGhlIG1ldGhvZCByZXR1cm5zIHRoZSBgaWFtLkdyYW50YCBvYmplY3QsIHdoaWNoIGNhbiB0aGVuIGJlIG1vZGlmaWVkXG4gICAgICogYXMgbmVlZGVkLiBGb3IgZXhhbXBsZSwgeW91IGNhbiBhZGQgYSBjb25kaXRpb24gdGhhdCB3aWxsIHJlc3RyaWN0IGFjY2VzcyBvbmx5XG4gICAgICogdG8gYW4gSVB2NCByYW5nZSBsaWtlIHRoaXM6XG4gICAgICpcbiAgICAgKiAgICAgY29uc3QgZ3JhbnQgPSBidWNrZXQuZ3JhbnRQdWJsaWNBY2Nlc3MoKTtcbiAgICAgKiAgICAgZ3JhbnQucmVzb3VyY2VTdGF0ZW1lbnQhLmFkZENvbmRpdGlvbijigJhJcEFkZHJlc3PigJksIHsg4oCcYXdzOlNvdXJjZUlw4oCdOiDigJw1NC4yNDAuMTQzLjAvMjTigJ0gfSk7XG4gICAgICpcbiAgICAgKlxuICAgICAqIEBwYXJhbSBrZXlQcmVmaXggdGhlIHByZWZpeCBvZiBTMyBvYmplY3Qga2V5cyAoZS5nLiBgaG9tZS8qYCkuIERlZmF1bHQgaXMgXCIqXCIuXG4gICAgICogQHBhcmFtIGFsbG93ZWRBY3Rpb25zIHRoZSBzZXQgb2YgUzMgYWN0aW9ucyB0byBhbGxvdy4gRGVmYXVsdCBpcyBcInMzOkdldE9iamVjdFwiLlxuICAgICAqL1xuICAgIHB1YmxpYyBncmFudFB1YmxpY0FjY2VzcyhrZXlQcmVmaXggPSAnKicsIC4uLmFsbG93ZWRBY3Rpb25zOiBzdHJpbmdbXSkge1xuICAgICAgICBpZiAodGhpcy5kaXNhbGxvd1B1YmxpY0FjY2Vzcykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiQ2Fubm90IGdyYW50IHB1YmxpYyBhY2Nlc3Mgd2hlbiAnYmxvY2tQdWJsaWNQb2xpY3knIGlzIGVuYWJsZWRcIik7XG4gICAgICAgIH1cbiAgICAgICAgYWxsb3dlZEFjdGlvbnMgPSBhbGxvd2VkQWN0aW9ucy5sZW5ndGggPiAwID8gYWxsb3dlZEFjdGlvbnMgOiBbJ3MzOkdldE9iamVjdCddO1xuICAgICAgICByZXR1cm4gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsT3JSZXNvdXJjZSh7XG4gICAgICAgICAgICBhY3Rpb25zOiBhbGxvd2VkQWN0aW9ucyxcbiAgICAgICAgICAgIHJlc291cmNlQXJuczogW3RoaXMuYXJuRm9yT2JqZWN0cyhrZXlQcmVmaXgpXSxcbiAgICAgICAgICAgIGdyYW50ZWU6IG5ldyBpYW0uQW55b25lKCksXG4gICAgICAgICAgICByZXNvdXJjZTogdGhpcyxcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIHByaXZhdGUgYnVpbGRVcmwocHJlZml4OiBzdHJpbmcsIGtleT86IHN0cmluZyk6IHN0cmluZyB7XG4gICAgICAgIGNvbnN0IGNvbXBvbmVudHMgPSBbXG4gICAgICAgICAgICBwcmVmaXgsXG4gICAgICAgICAgICB0aGlzLmJ1Y2tldE5hbWUsXG4gICAgICAgIF07XG4gICAgICAgIGlmIChrZXkpIHtcbiAgICAgICAgICAgIC8vIHRyaW0gcHJlcGVuZGluZyAnLydcbiAgICAgICAgICAgIGlmICh0eXBlb2Yga2V5ID09PSAnc3RyaW5nJyAmJiBrZXkuc3RhcnRzV2l0aCgnLycpKSB7XG4gICAgICAgICAgICAgICAga2V5ID0ga2V5LnN1YnN0cigxKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGNvbXBvbmVudHMucHVzaCgnLycpO1xuICAgICAgICAgICAgY29tcG9uZW50cy5wdXNoKGtleSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIGNvbXBvbmVudHMuam9pbignJyk7XG4gICAgfVxuICAgIHByaXZhdGUgZ3JhbnQoZ3JhbnRlZTogaWFtLklHcmFudGFibGUsIGJ1Y2tldEFjdGlvbnM6IHN0cmluZ1tdLCBrZXlBY3Rpb25zOiBzdHJpbmdbXSwgcmVzb3VyY2VBcm46IHN0cmluZywgLi4ub3RoZXJSZXNvdXJjZUFybnM6IHN0cmluZ1tdKSB7XG4gICAgICAgIGNvbnN0IHJlc291cmNlcyA9IFtyZXNvdXJjZUFybiwgLi4ub3RoZXJSZXNvdXJjZUFybnNdO1xuICAgICAgICBjb25zdCByZXQgPSBpYW0uR3JhbnQuYWRkVG9QcmluY2lwYWxPclJlc291cmNlKHtcbiAgICAgICAgICAgIGdyYW50ZWUsXG4gICAgICAgICAgICBhY3Rpb25zOiBidWNrZXRBY3Rpb25zLFxuICAgICAgICAgICAgcmVzb3VyY2VBcm5zOiByZXNvdXJjZXMsXG4gICAgICAgICAgICByZXNvdXJjZTogdGhpcyxcbiAgICAgICAgfSk7XG4gICAgICAgIGlmICh0aGlzLmVuY3J5cHRpb25LZXkgJiYga2V5QWN0aW9ucyAmJiBrZXlBY3Rpb25zLmxlbmd0aCAhPT0gMCkge1xuICAgICAgICAgICAgdGhpcy5lbmNyeXB0aW9uS2V5LmdyYW50KGdyYW50ZWUsIC4uLmtleUFjdGlvbnMpO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiByZXQ7XG4gICAgfVxufVxuZXhwb3J0IGludGVyZmFjZSBCbG9ja1B1YmxpY0FjY2Vzc09wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgdG8gYmxvY2sgcHVibGljIEFDTHNcbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvYWNjZXNzLWNvbnRyb2wtYmxvY2stcHVibGljLWFjY2Vzcy5odG1sI2FjY2Vzcy1jb250cm9sLWJsb2NrLXB1YmxpYy1hY2Nlc3Mtb3B0aW9uc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGJsb2NrUHVibGljQWNscz86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogV2hldGhlciB0byBibG9jayBwdWJsaWMgcG9saWN5XG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2FjY2Vzcy1jb250cm9sLWJsb2NrLXB1YmxpYy1hY2Nlc3MuaHRtbCNhY2Nlc3MtY29udHJvbC1ibG9jay1wdWJsaWMtYWNjZXNzLW9wdGlvbnNcbiAgICAgKi9cbiAgICByZWFkb25seSBibG9ja1B1YmxpY1BvbGljeT86IGJvb2xlYW47XG4gICAgLyoqXG4gICAgICogV2hldGhlciB0byBpZ25vcmUgcHVibGljIEFDTHNcbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvYWNjZXNzLWNvbnRyb2wtYmxvY2stcHVibGljLWFjY2Vzcy5odG1sI2FjY2Vzcy1jb250cm9sLWJsb2NrLXB1YmxpYy1hY2Nlc3Mtb3B0aW9uc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGlnbm9yZVB1YmxpY0FjbHM/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgdG8gcmVzdHJpY3QgcHVibGljIGFjY2Vzc1xuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9hY2Nlc3MtY29udHJvbC1ibG9jay1wdWJsaWMtYWNjZXNzLmh0bWwjYWNjZXNzLWNvbnRyb2wtYmxvY2stcHVibGljLWFjY2Vzcy1vcHRpb25zXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVzdHJpY3RQdWJsaWNCdWNrZXRzPzogYm9vbGVhbjtcbn1cbmV4cG9ydCBjbGFzcyBCbG9ja1B1YmxpY0FjY2VzcyB7XG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBCTE9DS19BTEwgPSBuZXcgQmxvY2tQdWJsaWNBY2Nlc3Moe1xuICAgICAgICBibG9ja1B1YmxpY0FjbHM6IHRydWUsXG4gICAgICAgIGJsb2NrUHVibGljUG9saWN5OiB0cnVlLFxuICAgICAgICBpZ25vcmVQdWJsaWNBY2xzOiB0cnVlLFxuICAgICAgICByZXN0cmljdFB1YmxpY0J1Y2tldHM6IHRydWUsXG4gICAgfSk7XG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBCTE9DS19BQ0xTID0gbmV3IEJsb2NrUHVibGljQWNjZXNzKHtcbiAgICAgICAgYmxvY2tQdWJsaWNBY2xzOiB0cnVlLFxuICAgICAgICBpZ25vcmVQdWJsaWNBY2xzOiB0cnVlLFxuICAgIH0pO1xuICAgIHB1YmxpYyBibG9ja1B1YmxpY0FjbHM6IGJvb2xlYW4gfCB1bmRlZmluZWQ7XG4gICAgcHVibGljIGJsb2NrUHVibGljUG9saWN5OiBib29sZWFuIHwgdW5kZWZpbmVkO1xuICAgIHB1YmxpYyBpZ25vcmVQdWJsaWNBY2xzOiBib29sZWFuIHwgdW5kZWZpbmVkO1xuICAgIHB1YmxpYyByZXN0cmljdFB1YmxpY0J1Y2tldHM6IGJvb2xlYW4gfCB1bmRlZmluZWQ7XG4gICAgY29uc3RydWN0b3Iob3B0aW9uczogQmxvY2tQdWJsaWNBY2Nlc3NPcHRpb25zKSB7XG4gICAgICAgIHRoaXMuYmxvY2tQdWJsaWNBY2xzID0gb3B0aW9ucy5ibG9ja1B1YmxpY0FjbHM7XG4gICAgICAgIHRoaXMuYmxvY2tQdWJsaWNQb2xpY3kgPSBvcHRpb25zLmJsb2NrUHVibGljUG9saWN5O1xuICAgICAgICB0aGlzLmlnbm9yZVB1YmxpY0FjbHMgPSBvcHRpb25zLmlnbm9yZVB1YmxpY0FjbHM7XG4gICAgICAgIHRoaXMucmVzdHJpY3RQdWJsaWNCdWNrZXRzID0gb3B0aW9ucy5yZXN0cmljdFB1YmxpY0J1Y2tldHM7XG4gICAgfVxufVxuLyoqXG4gKiBTcGVjaWZpZXMgYSBtZXRyaWNzIGNvbmZpZ3VyYXRpb24gZm9yIHRoZSBDbG91ZFdhdGNoIHJlcXVlc3QgbWV0cmljcyBmcm9tIGFuIEFtYXpvbiBTMyBidWNrZXQuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQnVja2V0TWV0cmljcyB7XG4gICAgLyoqXG4gICAgICogVGhlIElEIHVzZWQgdG8gaWRlbnRpZnkgdGhlIG1ldHJpY3MgY29uZmlndXJhdGlvbi5cbiAgICAgKi9cbiAgICByZWFkb25seSBpZDogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBwcmVmaXggdGhhdCBhbiBvYmplY3QgbXVzdCBoYXZlIHRvIGJlIGluY2x1ZGVkIGluIHRoZSBtZXRyaWNzIHJlc3VsdHMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcHJlZml4Pzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFNwZWNpZmllcyBhIGxpc3Qgb2YgdGFnIGZpbHRlcnMgdG8gdXNlIGFzIGEgbWV0cmljcyBjb25maWd1cmF0aW9uIGZpbHRlci5cbiAgICAgKiBUaGUgbWV0cmljcyBjb25maWd1cmF0aW9uIGluY2x1ZGVzIG9ubHkgb2JqZWN0cyB0aGF0IG1lZXQgdGhlIGZpbHRlcidzIGNyaXRlcmlhLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHRhZ0ZpbHRlcnM/OiB7XG4gICAgICAgIFt0YWc6IHN0cmluZ106IGFueTtcbiAgICB9O1xufVxuLyoqXG4gKiBBbGwgaHR0cCByZXF1ZXN0IG1ldGhvZHNcbiAqL1xuZXhwb3J0IGVudW0gSHR0cE1ldGhvZHMge1xuICAgIC8qKlxuICAgICAqIFRoZSBHRVQgbWV0aG9kIHJlcXVlc3RzIGEgcmVwcmVzZW50YXRpb24gb2YgdGhlIHNwZWNpZmllZCByZXNvdXJjZS5cbiAgICAgKi9cbiAgICBHRVQgPSAnR0VUJyxcbiAgICAvKipcbiAgICAgKiBUaGUgUFVUIG1ldGhvZCByZXBsYWNlcyBhbGwgY3VycmVudCByZXByZXNlbnRhdGlvbnMgb2YgdGhlIHRhcmdldCByZXNvdXJjZSB3aXRoIHRoZSByZXF1ZXN0IHBheWxvYWQuXG4gICAgICovXG4gICAgUFVUID0gJ1BVVCcsXG4gICAgLyoqXG4gICAgICogVGhlIEhFQUQgbWV0aG9kIGFza3MgZm9yIGEgcmVzcG9uc2UgaWRlbnRpY2FsIHRvIHRoYXQgb2YgYSBHRVQgcmVxdWVzdCwgYnV0IHdpdGhvdXQgdGhlIHJlc3BvbnNlIGJvZHkuXG4gICAgICovXG4gICAgSEVBRCA9ICdIRUFEJyxcbiAgICAvKipcbiAgICAgKiBUaGUgUE9TVCBtZXRob2QgaXMgdXNlZCB0byBzdWJtaXQgYW4gZW50aXR5IHRvIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2UsIG9mdGVuIGNhdXNpbmcgYSBjaGFuZ2UgaW4gc3RhdGUgb3Igc2lkZSBlZmZlY3RzIG9uIHRoZSBzZXJ2ZXIuXG4gICAgICovXG4gICAgUE9TVCA9ICdQT1NUJyxcbiAgICAvKipcbiAgICAgKiBUaGUgREVMRVRFIG1ldGhvZCBkZWxldGVzIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2UuXG4gICAgICovXG4gICAgREVMRVRFID0gJ0RFTEVURSdcbn1cbi8qKlxuICogU3BlY2lmaWVzIGEgY3Jvc3Mtb3JpZ2luIGFjY2VzcyBydWxlIGZvciBhbiBBbWF6b24gUzMgYnVja2V0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIENvcnNSdWxlIHtcbiAgICAvKipcbiAgICAgKiBBIHVuaXF1ZSBpZGVudGlmaWVyIGZvciB0aGlzIHJ1bGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGlkIHNwZWNpZmllZC5cbiAgICAgKi9cbiAgICByZWFkb25seSBpZD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgdGltZSBpbiBzZWNvbmRzIHRoYXQgeW91ciBicm93c2VyIGlzIHRvIGNhY2hlIHRoZSBwcmVmbGlnaHQgcmVzcG9uc2UgZm9yIHRoZSBzcGVjaWZpZWQgcmVzb3VyY2UuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGNhY2hpbmcuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbWF4QWdlPzogbnVtYmVyO1xuICAgIC8qKlxuICAgICAqIEhlYWRlcnMgdGhhdCBhcmUgc3BlY2lmaWVkIGluIHRoZSBBY2Nlc3MtQ29udHJvbC1SZXF1ZXN0LUhlYWRlcnMgaGVhZGVyLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBoZWFkZXJzIGFsbG93ZWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWxsb3dlZEhlYWRlcnM/OiBzdHJpbmdbXTtcbiAgICAvKipcbiAgICAgKiBBbiBIVFRQIG1ldGhvZCB0aGF0IHlvdSBhbGxvdyB0aGUgb3JpZ2luIHRvIGV4ZWN1dGUuXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWxsb3dlZE1ldGhvZHM6IEh0dHBNZXRob2RzW107XG4gICAgLyoqXG4gICAgICogT25lIG9yIG1vcmUgb3JpZ2lucyB5b3Ugd2FudCBjdXN0b21lcnMgdG8gYmUgYWJsZSB0byBhY2Nlc3MgdGhlIGJ1Y2tldCBmcm9tLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFsbG93ZWRPcmlnaW5zOiBzdHJpbmdbXTtcbiAgICAvKipcbiAgICAgKiBPbmUgb3IgbW9yZSBoZWFkZXJzIGluIHRoZSByZXNwb25zZSB0aGF0IHlvdSB3YW50IGN1c3RvbWVycyB0byBiZSBhYmxlIHRvIGFjY2VzcyBmcm9tIHRoZWlyIGFwcGxpY2F0aW9ucy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gaGVhZGVycyBleHBvc2VkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGV4cG9zZWRIZWFkZXJzPzogc3RyaW5nW107XG59XG4vKipcbiAqIEFsbCBodHRwIHJlcXVlc3QgbWV0aG9kc1xuICovXG5leHBvcnQgZW51bSBSZWRpcmVjdFByb3RvY29sIHtcbiAgICBIVFRQID0gJ2h0dHAnLFxuICAgIEhUVFBTID0gJ2h0dHBzJ1xufVxuLyoqXG4gKiBTcGVjaWZpZXMgYSByZWRpcmVjdCBiZWhhdmlvciBvZiBhbGwgcmVxdWVzdHMgdG8gYSB3ZWJzaXRlIGVuZHBvaW50IG9mIGEgYnVja2V0LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlZGlyZWN0VGFyZ2V0IHtcbiAgICAvKipcbiAgICAgKiBOYW1lIG9mIHRoZSBob3N0IHdoZXJlIHJlcXVlc3RzIGFyZSByZWRpcmVjdGVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgaG9zdE5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBQcm90b2NvbCB0byB1c2Ugd2hlbiByZWRpcmVjdGluZyByZXF1ZXN0c1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBUaGUgcHJvdG9jb2wgdXNlZCBpbiB0aGUgb3JpZ2luYWwgcmVxdWVzdC5cbiAgICAgKi9cbiAgICByZWFkb25seSBwcm90b2NvbD86IFJlZGlyZWN0UHJvdG9jb2w7XG59XG4vKipcbiAqIEFsbCBzdXBwb3J0ZWQgaW52ZW50b3J5IGxpc3QgZm9ybWF0cy5cbiAqL1xuZXhwb3J0IGVudW0gSW52ZW50b3J5Rm9ybWF0IHtcbiAgICAvKipcbiAgICAgKiBHZW5lcmF0ZSB0aGUgaW52ZW50b3J5IGxpc3QgYXMgQ1NWLlxuICAgICAqL1xuICAgIENTViA9ICdDU1YnLFxuICAgIC8qKlxuICAgICAqIEdlbmVyYXRlIHRoZSBpbnZlbnRvcnkgbGlzdCBhcyBQYXJxdWV0LlxuICAgICAqL1xuICAgIFBBUlFVRVQgPSAnUGFycXVldCcsXG4gICAgLyoqXG4gICAgICogR2VuZXJhdGUgdGhlIGludmVudG9yeSBsaXN0IGFzIFBhcnF1ZXQuXG4gICAgICovXG4gICAgT1JDID0gJ09SQydcbn1cbi8qKlxuICogQWxsIHN1cHBvcnRlZCBpbnZlbnRvcnkgZnJlcXVlbmNpZXMuXG4gKi9cbmV4cG9ydCBlbnVtIEludmVudG9yeUZyZXF1ZW5jeSB7XG4gICAgLyoqXG4gICAgICogQSByZXBvcnQgaXMgZ2VuZXJhdGVkIGV2ZXJ5IGRheS5cbiAgICAgKi9cbiAgICBEQUlMWSA9ICdEYWlseScsXG4gICAgLyoqXG4gICAgICogQSByZXBvcnQgaXMgZ2VuZXJhdGVkIGV2ZXJ5IFN1bmRheSAoVVRDIHRpbWV6b25lKSBhZnRlciB0aGUgaW5pdGlhbCByZXBvcnQuXG4gICAgICovXG4gICAgV0VFS0xZID0gJ1dlZWtseSdcbn1cbi8qKlxuICogSW52ZW50b3J5IHZlcnNpb24gc3VwcG9ydC5cbiAqL1xuZXhwb3J0IGVudW0gSW52ZW50b3J5T2JqZWN0VmVyc2lvbiB7XG4gICAgLyoqXG4gICAgICogSW5jbHVkZXMgYWxsIHZlcnNpb25zIG9mIGVhY2ggb2JqZWN0IGluIHRoZSByZXBvcnQuXG4gICAgICovXG4gICAgQUxMID0gJ0FsbCcsXG4gICAgLyoqXG4gICAgICogSW5jbHVkZXMgb25seSB0aGUgY3VycmVudCB2ZXJzaW9uIG9mIGVhY2ggb2JqZWN0IGluIHRoZSByZXBvcnQuXG4gICAgICovXG4gICAgQ1VSUkVOVCA9ICdDdXJyZW50J1xufVxuLyoqXG4gKiBUaGUgZGVzdGluYXRpb24gb2YgdGhlIGludmVudG9yeS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJbnZlbnRvcnlEZXN0aW5hdGlvbiB7XG4gICAgLyoqXG4gICAgICogQnVja2V0IHdoZXJlIGFsbCBpbnZlbnRvcmllcyB3aWxsIGJlIHNhdmVkIGluLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJ1Y2tldDogSUJ1Y2tldDtcbiAgICAvKipcbiAgICAgKiBUaGUgcHJlZml4IHRvIGJlIHVzZWQgd2hlbiBzYXZpbmcgdGhlIGludmVudG9yeS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gcHJlZml4LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByZWZpeD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgYWNjb3VudCBJRCB0aGF0IG93bnMgdGhlIGRlc3RpbmF0aW9uIFMzIGJ1Y2tldC5cbiAgICAgKiBJZiBubyBhY2NvdW50IElEIGlzIHByb3ZpZGVkLCB0aGUgb3duZXIgaXMgbm90IHZhbGlkYXRlZCBiZWZvcmUgZXhwb3J0aW5nIGRhdGEuXG4gICAgICogSXQncyByZWNvbW1lbmRlZCB0byBzZXQgYW4gYWNjb3VudCBJRCB0byBwcmV2ZW50IHByb2JsZW1zIGlmIHRoZSBkZXN0aW5hdGlvbiBidWNrZXQgb3duZXJzaGlwIGNoYW5nZXMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGFjY291bnQgSUQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgYnVja2V0T3duZXI/OiBzdHJpbmc7XG59XG4vKipcbiAqIFNwZWNpZmllcyB0aGUgaW52ZW50b3J5IGNvbmZpZ3VyYXRpb24gb2YgYW4gUzMgQnVja2V0LlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvc3RvcmFnZS1pbnZlbnRvcnkuaHRtbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEludmVudG9yeSB7XG4gICAgLyoqXG4gICAgICogVGhlIGRlc3RpbmF0aW9uIG9mIHRoZSBpbnZlbnRvcnkuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGVzdGluYXRpb246IEludmVudG9yeURlc3RpbmF0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBpbnZlbnRvcnkgd2lsbCBvbmx5IGluY2x1ZGUgb2JqZWN0cyB0aGF0IG1lZXQgdGhlIHByZWZpeCBmaWx0ZXIgY3JpdGVyaWEuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIG9iamVjdHMgcHJlZml4XG4gICAgICovXG4gICAgcmVhZG9ubHkgb2JqZWN0c1ByZWZpeD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgZm9ybWF0IG9mIHRoZSBpbnZlbnRvcnkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBJbnZlbnRvcnlGb3JtYXQuQ1NWXG4gICAgICovXG4gICAgcmVhZG9ubHkgZm9ybWF0PzogSW52ZW50b3J5Rm9ybWF0O1xuICAgIC8qKlxuICAgICAqIFdoZXRoZXIgdGhlIGludmVudG9yeSBpcyBlbmFibGVkIG9yIGRpc2FibGVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGVuYWJsZWQ/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFRoZSBpbnZlbnRvcnkgY29uZmlndXJhdGlvbiBJRC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gZ2VuZXJhdGVkIElELlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGludmVudG9yeUlkPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEZyZXF1ZW5jeSBhdCB3aGljaCB0aGUgaW52ZW50b3J5IHNob3VsZCBiZSBnZW5lcmF0ZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBJbnZlbnRvcnlGcmVxdWVuY3kuV0VFS0xZXG4gICAgICovXG4gICAgcmVhZG9ubHkgZnJlcXVlbmN5PzogSW52ZW50b3J5RnJlcXVlbmN5O1xuICAgIC8qKlxuICAgICAqIElmIHRoZSBpbnZlbnRvcnkgc2hvdWxkIGNvbnRhaW4gYWxsIHRoZSBvYmplY3QgdmVyc2lvbnMgb3Igb25seSB0aGUgY3VycmVudCBvbmUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBJbnZlbnRvcnlPYmplY3RWZXJzaW9uLkFMTFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGluY2x1ZGVPYmplY3RWZXJzaW9ucz86IEludmVudG9yeU9iamVjdFZlcnNpb247XG4gICAgLyoqXG4gICAgICogQSBsaXN0IG9mIG9wdGlvbmFsIGZpZWxkcyB0byBiZSBpbmNsdWRlZCBpbiB0aGUgaW52ZW50b3J5IHJlc3VsdC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gb3B0aW9uYWwgZmllbGRzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG9wdGlvbmFsRmllbGRzPzogc3RyaW5nW107XG59XG5leHBvcnQgaW50ZXJmYWNlIEJ1Y2tldFByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUga2luZCBvZiBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIHRvIGFwcGx5IHRvIHRoaXMgYnVja2V0LlxuICAgICAqXG4gICAgICogSWYgeW91IGNob29zZSBLTVMsIHlvdSBjYW4gc3BlY2lmeSBhIEtNUyBrZXkgdmlhIGBlbmNyeXB0aW9uS2V5YC4gSWZcbiAgICAgKiBlbmNyeXB0aW9uIGtleSBpcyBub3Qgc3BlY2lmaWVkLCBhIGtleSB3aWxsIGF1dG9tYXRpY2FsbHkgYmUgY3JlYXRlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gYEttc2AgaWYgYGVuY3J5cHRpb25LZXlgIGlzIHNwZWNpZmllZCwgb3IgYFVuZW5jcnlwdGVkYCBvdGhlcndpc2UuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW5jcnlwdGlvbj86IEJ1Y2tldEVuY3J5cHRpb247XG4gICAgLyoqXG4gICAgICogRXh0ZXJuYWwgS01TIGtleSB0byB1c2UgZm9yIGJ1Y2tldCBlbmNyeXB0aW9uLlxuICAgICAqXG4gICAgICogVGhlICdlbmNyeXB0aW9uJyBwcm9wZXJ0eSBtdXN0IGJlIGVpdGhlciBub3Qgc3BlY2lmaWVkIG9yIHNldCB0byBcIkttc1wiLlxuICAgICAqIEFuIGVycm9yIHdpbGwgYmUgZW1pdHRlZCBpZiBlbmNyeXB0aW9uIGlzIHNldCB0byBcIlVuZW5jcnlwdGVkXCIgb3JcbiAgICAgKiBcIk1hbmFnZWRcIi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gSWYgZW5jcnlwdGlvbiBpcyBzZXQgdG8gXCJLbXNcIiBhbmQgdGhpcyBwcm9wZXJ0eSBpcyB1bmRlZmluZWQsXG4gICAgICogYSBuZXcgS01TIGtleSB3aWxsIGJlIGNyZWF0ZWQgYW5kIGFzc29jaWF0ZWQgd2l0aCB0aGlzIGJ1Y2tldC5cbiAgICAgKi9cbiAgICByZWFkb25seSBlbmNyeXB0aW9uS2V5Pzoga21zLklLZXk7XG4gICAgLyoqXG4gICAgICogUGh5c2ljYWwgbmFtZSBvZiB0aGlzIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gQXNzaWduZWQgYnkgQ2xvdWRGb3JtYXRpb24gKHJlY29tbWVuZGVkKS5cbiAgICAgKi9cbiAgICByZWFkb25seSBidWNrZXROYW1lPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFBvbGljeSB0byBhcHBseSB3aGVuIHRoZSBidWNrZXQgaXMgcmVtb3ZlZCBmcm9tIHRoaXMgc3RhY2suXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFRoZSBidWNrZXQgd2lsbCBiZSBvcnBoYW5lZC5cbiAgICAgKi9cbiAgICByZWFkb25seSByZW1vdmFsUG9saWN5PzogUmVtb3ZhbFBvbGljeTtcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIHRoaXMgYnVja2V0IHNob3VsZCBoYXZlIHZlcnNpb25pbmcgdHVybmVkIG9uIG9yIG5vdC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgdmVyc2lvbmVkPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBSdWxlcyB0aGF0IGRlZmluZSBob3cgQW1hem9uIFMzIG1hbmFnZXMgb2JqZWN0cyBkdXJpbmcgdGhlaXIgbGlmZXRpbWUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGxpZmVjeWNsZSBydWxlcy5cbiAgICAgKi9cbiAgICByZWFkb25seSBsaWZlY3ljbGVSdWxlcz86IExpZmVjeWNsZVJ1bGVbXTtcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgaW5kZXggZG9jdW1lbnQgKGUuZy4gXCJpbmRleC5odG1sXCIpIGZvciB0aGUgd2Vic2l0ZS4gRW5hYmxlcyBzdGF0aWMgd2Vic2l0ZVxuICAgICAqIGhvc3RpbmcgZm9yIHRoaXMgYnVja2V0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBpbmRleCBkb2N1bWVudC5cbiAgICAgKi9cbiAgICByZWFkb25seSB3ZWJzaXRlSW5kZXhEb2N1bWVudD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgbmFtZSBvZiB0aGUgZXJyb3IgZG9jdW1lbnQgKGUuZy4gXCI0MDQuaHRtbFwiKSBmb3IgdGhlIHdlYnNpdGUuXG4gICAgICogYHdlYnNpdGVJbmRleERvY3VtZW50YCBtdXN0IGFsc28gYmUgc2V0IGlmIHRoaXMgaXMgc2V0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBlcnJvciBkb2N1bWVudC5cbiAgICAgKi9cbiAgICByZWFkb25seSB3ZWJzaXRlRXJyb3JEb2N1bWVudD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBTcGVjaWZpZXMgdGhlIHJlZGlyZWN0IGJlaGF2aW9yIG9mIGFsbCByZXF1ZXN0cyB0byBhIHdlYnNpdGUgZW5kcG9pbnQgb2YgYSBidWNrZXQuXG4gICAgICpcbiAgICAgKiBJZiB5b3Ugc3BlY2lmeSB0aGlzIHByb3BlcnR5LCB5b3UgY2FuJ3Qgc3BlY2lmeSBcIndlYnNpdGVJbmRleERvY3VtZW50XCIsIFwid2Vic2l0ZUVycm9yRG9jdW1lbnRcIiBub3IgLCBcIndlYnNpdGVSb3V0aW5nUnVsZXNcIi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gcmVkaXJlY3Rpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgd2Vic2l0ZVJlZGlyZWN0PzogUmVkaXJlY3RUYXJnZXQ7XG4gICAgLyoqXG4gICAgICogUnVsZXMgdGhhdCBkZWZpbmUgd2hlbiBhIHJlZGlyZWN0IGlzIGFwcGxpZWQgYW5kIHRoZSByZWRpcmVjdCBiZWhhdmlvclxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyByZWRpcmVjdGlvbiBydWxlcy5cbiAgICAgKi9cbiAgICByZWFkb25seSB3ZWJzaXRlUm91dGluZ1J1bGVzPzogUm91dGluZ1J1bGVbXTtcbiAgICAvKipcbiAgICAgKiBTcGVjaWZpZXMgYSBjYW5uZWQgQUNMIHRoYXQgZ3JhbnRzIHByZWRlZmluZWQgcGVybWlzc2lvbnMgdG8gdGhlIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IEJ1Y2tldEFjY2Vzc0NvbnRyb2wuUFJJVkFURVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFjY2Vzc0NvbnRyb2w/OiBCdWNrZXRBY2Nlc3NDb250cm9sO1xuICAgIC8qKlxuICAgICAqIEdyYW50cyBwdWJsaWMgcmVhZCBhY2Nlc3MgdG8gYWxsIG9iamVjdHMgaW4gdGhlIGJ1Y2tldC5cbiAgICAgKiBTaW1pbGFyIHRvIGNhbGxpbmcgYGJ1Y2tldC5ncmFudFB1YmxpY0FjY2VzcygpYFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgZmFsc2VcbiAgICAgKi9cbiAgICByZWFkb25seSBwdWJsaWNSZWFkQWNjZXNzPzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBUaGUgYmxvY2sgcHVibGljIGFjY2VzcyBjb25maWd1cmF0aW9uIG9mIHRoaXMgYnVja2V0LlxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9hY2Nlc3MtY29udHJvbC1ibG9jay1wdWJsaWMtYWNjZXNzLmh0bWxcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlIE5ldyBidWNrZXRzIGFuZCBvYmplY3RzIGRvbid0IGFsbG93IHB1YmxpYyBhY2Nlc3MsIGJ1dCB1c2VycyBjYW4gbW9kaWZ5IGJ1Y2tldFxuICAgICAqIHBvbGljaWVzIG9yIG9iamVjdCBwZXJtaXNzaW9ucyB0byBhbGxvdyBwdWJsaWMgYWNjZXNzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGJsb2NrUHVibGljQWNjZXNzPzogQmxvY2tQdWJsaWNBY2Nlc3M7XG4gICAgLyoqXG4gICAgICogVGhlIG1ldHJpY3MgY29uZmlndXJhdGlvbiBvZiB0aGlzIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXByb3BlcnRpZXMtczMtYnVja2V0LW1ldHJpY3Njb25maWd1cmF0aW9uLmh0bWxcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gbWV0cmljcyBjb25maWd1cmF0aW9uLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1ldHJpY3M/OiBCdWNrZXRNZXRyaWNzW107XG4gICAgLyoqXG4gICAgICogVGhlIENPUlMgY29uZmlndXJhdGlvbiBvZiB0aGlzIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FXU0Nsb3VkRm9ybWF0aW9uL2xhdGVzdC9Vc2VyR3VpZGUvYXdzLXByb3BlcnRpZXMtczMtYnVja2V0LWNvcnMuaHRtbFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBDT1JTIGNvbmZpZ3VyYXRpb24uXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29ycz86IENvcnNSdWxlW107XG4gICAgLyoqXG4gICAgICogRGVzdGluYXRpb24gYnVja2V0IGZvciB0aGUgc2VydmVyIGFjY2VzcyBsb2dzLlxuICAgICAqIEBkZWZhdWx0IC0gSWYgXCJzZXJ2ZXJBY2Nlc3NMb2dzUHJlZml4XCIgdW5kZWZpbmVkIC0gYWNjZXNzIGxvZ3MgZGlzYWJsZWQsIG90aGVyd2lzZSAtIGxvZyB0byBjdXJyZW50IGJ1Y2tldC5cbiAgICAgKi9cbiAgICByZWFkb25seSBzZXJ2ZXJBY2Nlc3NMb2dzQnVja2V0PzogSUJ1Y2tldDtcbiAgICAvKipcbiAgICAgKiBPcHRpb25hbCBsb2cgZmlsZSBwcmVmaXggdG8gdXNlIGZvciB0aGUgYnVja2V0J3MgYWNjZXNzIGxvZ3MuXG4gICAgICogSWYgZGVmaW5lZCB3aXRob3V0IFwic2VydmVyQWNjZXNzTG9nc0J1Y2tldFwiLCBlbmFibGVzIGFjY2VzcyBsb2dzIHRvIGN1cnJlbnQgYnVja2V0IHdpdGggdGhpcyBwcmVmaXguXG4gICAgICogQGRlZmF1bHQgLSBObyBsb2cgZmlsZSBwcmVmaXhcbiAgICAgKi9cbiAgICByZWFkb25seSBzZXJ2ZXJBY2Nlc3NMb2dzUHJlZml4Pzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBpbnZlbnRvcnkgY29uZmlndXJhdGlvbiBvZiB0aGUgYnVja2V0LlxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9zdG9yYWdlLWludmVudG9yeS5odG1sXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGludmVudG9yeSBjb25maWd1cmF0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW52ZW50b3JpZXM/OiBJbnZlbnRvcnlbXTtcbn1cbi8qKlxuICogQW4gUzMgYnVja2V0IHdpdGggYXNzb2NpYXRlZCBwb2xpY3kgb2JqZWN0c1xuICpcbiAqIFRoaXMgYnVja2V0IGRvZXMgbm90IHlldCBoYXZlIGFsbCBmZWF0dXJlcyB0aGF0IGV4cG9zZWQgYnkgdGhlIHVuZGVybHlpbmdcbiAqIEJ1Y2tldFJlc291cmNlLlxuICovXG5leHBvcnQgY2xhc3MgQnVja2V0IGV4dGVuZHMgQnVja2V0QmFzZSB7XG4gICAgcHVibGljIHN0YXRpYyBmcm9tQnVja2V0QXJuKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGJ1Y2tldEFybjogc3RyaW5nKTogSUJ1Y2tldCB7XG4gICAgICAgIHJldHVybiBCdWNrZXQuZnJvbUJ1Y2tldEF0dHJpYnV0ZXMoc2NvcGUsIGlkLCB7IGJ1Y2tldEFybiB9KTtcbiAgICB9XG4gICAgcHVibGljIHN0YXRpYyBmcm9tQnVja2V0TmFtZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBidWNrZXROYW1lOiBzdHJpbmcpOiBJQnVja2V0IHtcbiAgICAgICAgcmV0dXJuIEJ1Y2tldC5mcm9tQnVja2V0QXR0cmlidXRlcyhzY29wZSwgaWQsIHsgYnVja2V0TmFtZSB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ3JlYXRlcyBhIEJ1Y2tldCBjb25zdHJ1Y3QgdGhhdCByZXByZXNlbnRzIGFuIGV4dGVybmFsIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBzY29wZSBUaGUgcGFyZW50IGNyZWF0aW5nIGNvbnN0cnVjdCAodXN1YWxseSBgdGhpc2ApLlxuICAgICAqIEBwYXJhbSBpZCBUaGUgY29uc3RydWN0J3MgbmFtZS5cbiAgICAgKiBAcGFyYW0gYXR0cnMgQSBgQnVja2V0QXR0cmlidXRlc2Agb2JqZWN0LiBDYW4gYmUgb2J0YWluZWQgZnJvbSBhIGNhbGwgdG9cbiAgICAgKiBgYnVja2V0LmV4cG9ydCgpYCBvciBtYW51YWxseSBjcmVhdGVkLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbUJ1Y2tldEF0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IEJ1Y2tldEF0dHJpYnV0ZXMpOiBJQnVja2V0IHtcbiAgICAgICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZihzY29wZSk7XG4gICAgICAgIGNvbnN0IHJlZ2lvbiA9IGF0dHJzLnJlZ2lvbiA/PyBzdGFjay5yZWdpb247XG4gICAgICAgIGNvbnN0IHVybFN1ZmZpeCA9IHN0YWNrLnVybFN1ZmZpeDtcbiAgICAgICAgY29uc3QgYnVja2V0TmFtZSA9IHBhcnNlQnVja2V0TmFtZShzY29wZSwgYXR0cnMpO1xuICAgICAgICBpZiAoIWJ1Y2tldE5hbWUpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQnVja2V0IG5hbWUgaXMgcmVxdWlyZWQnKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBuZXdVcmxGb3JtYXQgPSBhdHRycy5idWNrZXRXZWJzaXRlTmV3VXJsRm9ybWF0ID09PSB1bmRlZmluZWRcbiAgICAgICAgICAgID8gZmFsc2VcbiAgICAgICAgICAgIDogYXR0cnMuYnVja2V0V2Vic2l0ZU5ld1VybEZvcm1hdDtcbiAgICAgICAgY29uc3Qgd2Vic2l0ZURvbWFpbiA9IG5ld1VybEZvcm1hdFxuICAgICAgICAgICAgPyBgJHtidWNrZXROYW1lfS5zMy13ZWJzaXRlLiR7cmVnaW9ufS4ke3VybFN1ZmZpeH1gXG4gICAgICAgICAgICA6IGAke2J1Y2tldE5hbWV9LnMzLXdlYnNpdGUtJHtyZWdpb259LiR7dXJsU3VmZml4fWA7XG4gICAgICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIEJ1Y2tldEJhc2Uge1xuICAgICAgICAgICAgcHVibGljIHJlYWRvbmx5IGJ1Y2tldE5hbWUgPSBidWNrZXROYW1lITtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBidWNrZXRBcm4gPSBwYXJzZUJ1Y2tldEFybihzY29wZSwgYXR0cnMpO1xuICAgICAgICAgICAgcHVibGljIHJlYWRvbmx5IGJ1Y2tldERvbWFpbk5hbWUgPSBhdHRycy5idWNrZXREb21haW5OYW1lIHx8IGAke2J1Y2tldE5hbWV9LnMzLiR7dXJsU3VmZml4fWA7XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgYnVja2V0V2Vic2l0ZVVybCA9IGF0dHJzLmJ1Y2tldFdlYnNpdGVVcmwgfHwgYGh0dHA6Ly8ke3dlYnNpdGVEb21haW59YDtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBidWNrZXRXZWJzaXRlRG9tYWluTmFtZSA9IGF0dHJzLmJ1Y2tldFdlYnNpdGVVcmwgPyBGbi5zZWxlY3QoMiwgRm4uc3BsaXQoJy8nLCBhdHRycy5idWNrZXRXZWJzaXRlVXJsKSkgOiB3ZWJzaXRlRG9tYWluO1xuICAgICAgICAgICAgcHVibGljIHJlYWRvbmx5IGJ1Y2tldFJlZ2lvbmFsRG9tYWluTmFtZSA9IGF0dHJzLmJ1Y2tldFJlZ2lvbmFsRG9tYWluTmFtZSB8fCBgJHtidWNrZXROYW1lfS5zMy4ke3JlZ2lvbn0uJHt1cmxTdWZmaXh9YDtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBidWNrZXREdWFsU3RhY2tEb21haW5OYW1lID0gYXR0cnMuYnVja2V0RHVhbFN0YWNrRG9tYWluTmFtZSB8fCBgJHtidWNrZXROYW1lfS5zMy5kdWFsc3RhY2suJHtyZWdpb259LiR7dXJsU3VmZml4fWA7XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgYnVja2V0V2Vic2l0ZU5ld1VybEZvcm1hdCA9IG5ld1VybEZvcm1hdDtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBlbmNyeXB0aW9uS2V5ID0gYXR0cnMuZW5jcnlwdGlvbktleTtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBpc1dlYnNpdGUgPSBhdHRycy5pc1dlYnNpdGUgPz8gZmFsc2U7XG4gICAgICAgICAgICBwdWJsaWMgcG9saWN5PzogQnVja2V0UG9saWN5ID0gdW5kZWZpbmVkO1xuICAgICAgICAgICAgcHJvdGVjdGVkIGF1dG9DcmVhdGVQb2xpY3kgPSBmYWxzZTtcbiAgICAgICAgICAgIHByb3RlY3RlZCBkaXNhbGxvd1B1YmxpY0FjY2VzcyA9IGZhbHNlO1xuICAgICAgICAgICAgLyoqXG4gICAgICAgICAgICAgKiBFeHBvcnRzIHRoaXMgYnVja2V0IGZyb20gdGhlIHN0YWNrLlxuICAgICAgICAgICAgICovXG4gICAgICAgICAgICBwdWJsaWMgZXhwb3J0KCkge1xuICAgICAgICAgICAgICAgIHJldHVybiBhdHRycztcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQsIHtcbiAgICAgICAgICAgIGFjY291bnQ6IGF0dHJzLmFjY291bnQsXG4gICAgICAgICAgICByZWdpb246IGF0dHJzLnJlZ2lvbixcbiAgICAgICAgfSk7XG4gICAgfVxuICAgIHB1YmxpYyByZWFkb25seSBidWNrZXRBcm46IHN0cmluZztcbiAgICBwdWJsaWMgcmVhZG9ubHkgYnVja2V0TmFtZTogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSBidWNrZXREb21haW5OYW1lOiBzdHJpbmc7XG4gICAgcHVibGljIHJlYWRvbmx5IGJ1Y2tldFdlYnNpdGVVcmw6IHN0cmluZztcbiAgICBwdWJsaWMgcmVhZG9ubHkgYnVja2V0V2Vic2l0ZURvbWFpbk5hbWU6IHN0cmluZztcbiAgICBwdWJsaWMgcmVhZG9ubHkgYnVja2V0RHVhbFN0YWNrRG9tYWluTmFtZTogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSBidWNrZXRSZWdpb25hbERvbWFpbk5hbWU6IHN0cmluZztcbiAgICBwdWJsaWMgcmVhZG9ubHkgZW5jcnlwdGlvbktleT86IGttcy5JS2V5O1xuICAgIHB1YmxpYyByZWFkb25seSBpc1dlYnNpdGU/OiBib29sZWFuO1xuICAgIHB1YmxpYyBwb2xpY3k/OiBCdWNrZXRQb2xpY3k7XG4gICAgcHJvdGVjdGVkIGF1dG9DcmVhdGVQb2xpY3kgPSB0cnVlO1xuICAgIHByb3RlY3RlZCBkaXNhbGxvd1B1YmxpY0FjY2Vzcz86IGJvb2xlYW47XG4gICAgcHJpdmF0ZSBhY2Nlc3NDb250cm9sPzogQnVja2V0QWNjZXNzQ29udHJvbDtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGxpZmVjeWNsZVJ1bGVzOiBMaWZlY3ljbGVSdWxlW10gPSBbXTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IHZlcnNpb25lZD86IGJvb2xlYW47XG4gICAgcHJpdmF0ZSByZWFkb25seSBub3RpZmljYXRpb25zOiBCdWNrZXROb3RpZmljYXRpb25zO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgbWV0cmljczogQnVja2V0TWV0cmljc1tdID0gW107XG4gICAgcHJpdmF0ZSByZWFkb25seSBjb3JzOiBDb3JzUnVsZVtdID0gW107XG4gICAgcHJpdmF0ZSByZWFkb25seSBpbnZlbnRvcmllczogSW52ZW50b3J5W10gPSBbXTtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQnVja2V0UHJvcHMgPSB7fSkge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgICAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuYnVja2V0TmFtZSxcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IHsgYnVja2V0RW5jcnlwdGlvbiwgZW5jcnlwdGlvbktleSB9ID0gdGhpcy5wYXJzZUVuY3J5cHRpb24ocHJvcHMpO1xuICAgICAgICB0aGlzLnZhbGlkYXRlQnVja2V0TmFtZSh0aGlzLnBoeXNpY2FsTmFtZSk7XG4gICAgICAgIGNvbnN0IHdlYnNpdGVDb25maWd1cmF0aW9uID0gdGhpcy5yZW5kZXJXZWJzaXRlQ29uZmlndXJhdGlvbihwcm9wcyk7XG4gICAgICAgIHRoaXMuaXNXZWJzaXRlID0gKHdlYnNpdGVDb25maWd1cmF0aW9uICE9PSB1bmRlZmluZWQpO1xuICAgICAgICBjb25zdCByZXNvdXJjZSA9IG5ldyBDZm5CdWNrZXQodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgICAgICAgYnVja2V0TmFtZTogdGhpcy5waHlzaWNhbE5hbWUsXG4gICAgICAgICAgICBidWNrZXRFbmNyeXB0aW9uLFxuICAgICAgICAgICAgdmVyc2lvbmluZ0NvbmZpZ3VyYXRpb246IHByb3BzLnZlcnNpb25lZCA/IHsgc3RhdHVzOiAnRW5hYmxlZCcgfSA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIGxpZmVjeWNsZUNvbmZpZ3VyYXRpb246IExhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLnBhcnNlTGlmZWN5Y2xlQ29uZmlndXJhdGlvbigpIH0pLFxuICAgICAgICAgICAgd2Vic2l0ZUNvbmZpZ3VyYXRpb24sXG4gICAgICAgICAgICBwdWJsaWNBY2Nlc3NCbG9ja0NvbmZpZ3VyYXRpb246IHByb3BzLmJsb2NrUHVibGljQWNjZXNzLFxuICAgICAgICAgICAgbWV0cmljc0NvbmZpZ3VyYXRpb25zOiBMYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5wYXJzZU1ldHJpY0NvbmZpZ3VyYXRpb24oKSB9KSxcbiAgICAgICAgICAgIGNvcnNDb25maWd1cmF0aW9uOiBMYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5wYXJzZUNvcnNDb25maWd1cmF0aW9uKCkgfSksXG4gICAgICAgICAgICBhY2Nlc3NDb250cm9sOiBMYXp5LnN0cmluZ1ZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5hY2Nlc3NDb250cm9sIH0pLFxuICAgICAgICAgICAgbG9nZ2luZ0NvbmZpZ3VyYXRpb246IHRoaXMucGFyc2VTZXJ2ZXJBY2Nlc3NMb2dzKHByb3BzKSxcbiAgICAgICAgICAgIGludmVudG9yeUNvbmZpZ3VyYXRpb25zOiBMYXp5LmFueVZhbHVlKHsgcHJvZHVjZTogKCkgPT4gdGhpcy5wYXJzZUludmVudG9yeUNvbmZpZ3VyYXRpb24oKSB9KSxcbiAgICAgICAgfSk7XG4gICAgICAgIHJlc291cmNlLmFwcGx5UmVtb3ZhbFBvbGljeShwcm9wcy5yZW1vdmFsUG9saWN5KTtcbiAgICAgICAgdGhpcy52ZXJzaW9uZWQgPSBwcm9wcy52ZXJzaW9uZWQ7XG4gICAgICAgIHRoaXMuZW5jcnlwdGlvbktleSA9IGVuY3J5cHRpb25LZXk7XG4gICAgICAgIHRoaXMuYnVja2V0TmFtZSA9IHRoaXMuZ2V0UmVzb3VyY2VOYW1lQXR0cmlidXRlKHJlc291cmNlLnJlZik7XG4gICAgICAgIHRoaXMuYnVja2V0QXJuID0gdGhpcy5nZXRSZXNvdXJjZUFybkF0dHJpYnV0ZShyZXNvdXJjZS5hdHRyQXJuLCB7XG4gICAgICAgICAgICByZWdpb246ICcnLFxuICAgICAgICAgICAgYWNjb3VudDogJycsXG4gICAgICAgICAgICBzZXJ2aWNlOiAnczMnLFxuICAgICAgICAgICAgcmVzb3VyY2U6IHRoaXMucGh5c2ljYWxOYW1lLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5idWNrZXREb21haW5OYW1lID0gcmVzb3VyY2UuYXR0ckRvbWFpbk5hbWU7XG4gICAgICAgIHRoaXMuYnVja2V0V2Vic2l0ZVVybCA9IHJlc291cmNlLmF0dHJXZWJzaXRlVXJsO1xuICAgICAgICB0aGlzLmJ1Y2tldFdlYnNpdGVEb21haW5OYW1lID0gRm4uc2VsZWN0KDIsIEZuLnNwbGl0KCcvJywgdGhpcy5idWNrZXRXZWJzaXRlVXJsKSk7XG4gICAgICAgIHRoaXMuYnVja2V0RHVhbFN0YWNrRG9tYWluTmFtZSA9IHJlc291cmNlLmF0dHJEdWFsU3RhY2tEb21haW5OYW1lO1xuICAgICAgICB0aGlzLmJ1Y2tldFJlZ2lvbmFsRG9tYWluTmFtZSA9IHJlc291cmNlLmF0dHJSZWdpb25hbERvbWFpbk5hbWU7XG4gICAgICAgIHRoaXMuZGlzYWxsb3dQdWJsaWNBY2Nlc3MgPSBwcm9wcy5ibG9ja1B1YmxpY0FjY2VzcyAmJiBwcm9wcy5ibG9ja1B1YmxpY0FjY2Vzcy5ibG9ja1B1YmxpY1BvbGljeTtcbiAgICAgICAgdGhpcy5hY2Nlc3NDb250cm9sID0gcHJvcHMuYWNjZXNzQ29udHJvbDtcbiAgICAgICAgaWYgKHByb3BzLnNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQgaW5zdGFuY2VvZiBCdWNrZXQpIHtcbiAgICAgICAgICAgIHByb3BzLnNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQuYWxsb3dMb2dEZWxpdmVyeSgpO1xuICAgICAgICB9XG4gICAgICAgIGZvciAoY29uc3QgaW52ZW50b3J5IG9mIHByb3BzLmludmVudG9yaWVzID8/IFtdKSB7XG4gICAgICAgICAgICB0aGlzLmFkZEludmVudG9yeShpbnZlbnRvcnkpO1xuICAgICAgICB9XG4gICAgICAgIC8vIEFkZCBhbGwgYnVja2V0IG1ldHJpYyBjb25maWd1cmF0aW9ucyBydWxlc1xuICAgICAgICAocHJvcHMubWV0cmljcyB8fCBbXSkuZm9yRWFjaCh0aGlzLmFkZE1ldHJpYy5iaW5kKHRoaXMpKTtcbiAgICAgICAgLy8gQWRkIGFsbCBjb3JzIGNvbmZpZ3VyYXRpb24gcnVsZXNcbiAgICAgICAgKHByb3BzLmNvcnMgfHwgW10pLmZvckVhY2godGhpcy5hZGRDb3JzUnVsZS5iaW5kKHRoaXMpKTtcbiAgICAgICAgLy8gQWRkIGFsbCBsaWZlY3ljbGUgcnVsZXNcbiAgICAgICAgKHByb3BzLmxpZmVjeWNsZVJ1bGVzIHx8IFtdKS5mb3JFYWNoKHRoaXMuYWRkTGlmZWN5Y2xlUnVsZS5iaW5kKHRoaXMpKTtcbiAgICAgICAgLy8gZGVmaW5lcyBhIEJ1Y2tldE5vdGlmaWNhdGlvbnMgY29uc3RydWN0LiBOb3RpY2UgdGhhdCBhbiBhY3R1YWwgcmVzb3VyY2Ugd2lsbCBvbmx5XG4gICAgICAgIC8vIGJlIGFkZGVkIGlmIHRoZXJlIGFyZSBub3RpZmljYXRpb25zIGFkZGVkLCBzbyB3ZSBkb24ndCBuZWVkIHRvIGNvbmRpdGlvbiB0aGlzLlxuICAgICAgICB0aGlzLm5vdGlmaWNhdGlvbnMgPSBuZXcgQnVja2V0Tm90aWZpY2F0aW9ucyh0aGlzLCAnTm90aWZpY2F0aW9ucycsIHsgYnVja2V0OiB0aGlzIH0pO1xuICAgICAgICBpZiAocHJvcHMucHVibGljUmVhZEFjY2Vzcykge1xuICAgICAgICAgICAgdGhpcy5ncmFudFB1YmxpY0FjY2VzcygpO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBhIGxpZmVjeWNsZSBydWxlIHRvIHRoZSBidWNrZXRcbiAgICAgKlxuICAgICAqIEBwYXJhbSBydWxlIFRoZSBydWxlIHRvIGFkZFxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRMaWZlY3ljbGVSdWxlKHJ1bGU6IExpZmVjeWNsZVJ1bGUpIHtcbiAgICAgICAgaWYgKChydWxlLm5vbmN1cnJlbnRWZXJzaW9uRXhwaXJhdGlvbiAhPT0gdW5kZWZpbmVkXG4gICAgICAgICAgICB8fCAocnVsZS5ub25jdXJyZW50VmVyc2lvblRyYW5zaXRpb25zICYmIHJ1bGUubm9uY3VycmVudFZlcnNpb25UcmFuc2l0aW9ucy5sZW5ndGggPiAwKSlcbiAgICAgICAgICAgICYmICF0aGlzLnZlcnNpb25lZCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiQ2Fubm90IHVzZSAnbm9uY3VycmVudCcgcnVsZXMgb24gYSBub252ZXJzaW9uZWQgYnVja2V0XCIpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMubGlmZWN5Y2xlUnVsZXMucHVzaChydWxlKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkcyBhIG1ldHJpY3MgY29uZmlndXJhdGlvbiBmb3IgdGhlIENsb3VkV2F0Y2ggcmVxdWVzdCBtZXRyaWNzIGZyb20gdGhlIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBtZXRyaWMgVGhlIG1ldHJpYyBjb25maWd1cmF0aW9uIHRvIGFkZFxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRNZXRyaWMobWV0cmljOiBCdWNrZXRNZXRyaWNzKSB7XG4gICAgICAgIHRoaXMubWV0cmljcy5wdXNoKG1ldHJpYyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZHMgYSBjcm9zcy1vcmlnaW4gYWNjZXNzIGNvbmZpZ3VyYXRpb24gZm9yIG9iamVjdHMgaW4gYW4gQW1hem9uIFMzIGJ1Y2tldFxuICAgICAqXG4gICAgICogQHBhcmFtIHJ1bGUgVGhlIENPUlMgY29uZmlndXJhdGlvbiBydWxlIHRvIGFkZFxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRDb3JzUnVsZShydWxlOiBDb3JzUnVsZSkge1xuICAgICAgICB0aGlzLmNvcnMucHVzaChydWxlKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkcyBhIGJ1Y2tldCBub3RpZmljYXRpb24gZXZlbnQgZGVzdGluYXRpb24uXG4gICAgICogQHBhcmFtIGV2ZW50IFRoZSBldmVudCB0byB0cmlnZ2VyIHRoZSBub3RpZmljYXRpb25cbiAgICAgKiBAcGFyYW0gZGVzdCBUaGUgbm90aWZpY2F0aW9uIGRlc3RpbmF0aW9uIChMYW1iZGEsIFNOUyBUb3BpYyBvciBTUVMgUXVldWUpXG4gICAgICpcbiAgICAgKiBAcGFyYW0gZmlsdGVycyBTMyBvYmplY3Qga2V5IGZpbHRlciBydWxlcyB0byBkZXRlcm1pbmUgd2hpY2ggb2JqZWN0c1xuICAgICAqIHRyaWdnZXIgdGhpcyBldmVudC4gRWFjaCBmaWx0ZXIgbXVzdCBpbmNsdWRlIGEgYHByZWZpeGAgYW5kL29yIGBzdWZmaXhgXG4gICAgICogdGhhdCB3aWxsIGJlIG1hdGNoZWQgYWdhaW5zdCB0aGUgczMgb2JqZWN0IGtleS4gUmVmZXIgdG8gdGhlIFMzIERldmVsb3BlciBHdWlkZVxuICAgICAqIGZvciBkZXRhaWxzIGFib3V0IGFsbG93ZWQgZmlsdGVyIHJ1bGVzLlxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Ob3RpZmljYXRpb25Ib3dUby5odG1sI25vdGlmaWNhdGlvbi1ob3ctdG8tZmlsdGVyaW5nXG4gICAgICpcbiAgICAgKiBAZXhhbXBsZVxuICAgICAqXG4gICAgICogICAgYnVja2V0LmFkZEV2ZW50Tm90aWZpY2F0aW9uKEV2ZW50VHlwZS5Pbk9iamVjdENyZWF0ZWQsIG15TGFtYmRhLCAnaG9tZS9teXVzZXJuYW1lLyonKVxuICAgICAqXG4gICAgICogQHNlZVxuICAgICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L05vdGlmaWNhdGlvbkhvd1RvLmh0bWxcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkRXZlbnROb3RpZmljYXRpb24oZXZlbnQ6IEV2ZW50VHlwZSwgZGVzdDogSUJ1Y2tldE5vdGlmaWNhdGlvbkRlc3RpbmF0aW9uLCAuLi5maWx0ZXJzOiBOb3RpZmljYXRpb25LZXlGaWx0ZXJbXSkge1xuICAgICAgICB0aGlzLm5vdGlmaWNhdGlvbnMuYWRkTm90aWZpY2F0aW9uKGV2ZW50LCBkZXN0LCAuLi5maWx0ZXJzKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogU3Vic2NyaWJlcyBhIGRlc3RpbmF0aW9uIHRvIHJlY2VpdmUgbm90aWZpY2F0aW5zIHdoZW4gYW4gb2JqZWN0IGlzXG4gICAgICogY3JlYXRlZCBpbiB0aGUgYnVja2V0LiBUaGlzIGlzIGlkZW50aWNhbCB0byBjYWxsaW5nXG4gICAgICogYG9uRXZlbnQoRXZlbnRUeXBlLk9iamVjdENyZWF0ZWQpYC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBkZXN0IFRoZSBub3RpZmljYXRpb24gZGVzdGluYXRpb24gKHNlZSBvbkV2ZW50KVxuICAgICAqIEBwYXJhbSBmaWx0ZXJzIEZpbHRlcnMgKHNlZSBvbkV2ZW50KVxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRPYmplY3RDcmVhdGVkTm90aWZpY2F0aW9uKGRlc3Q6IElCdWNrZXROb3RpZmljYXRpb25EZXN0aW5hdGlvbiwgLi4uZmlsdGVyczogTm90aWZpY2F0aW9uS2V5RmlsdGVyW10pIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuYWRkRXZlbnROb3RpZmljYXRpb24oRXZlbnRUeXBlLk9CSkVDVF9DUkVBVEVELCBkZXN0LCAuLi5maWx0ZXJzKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogU3Vic2NyaWJlcyBhIGRlc3RpbmF0aW9uIHRvIHJlY2VpdmUgbm90aWZpY2F0aW5zIHdoZW4gYW4gb2JqZWN0IGlzXG4gICAgICogcmVtb3ZlZCBmcm9tIHRoZSBidWNrZXQuIFRoaXMgaXMgaWRlbnRpY2FsIHRvIGNhbGxpbmdcbiAgICAgKiBgb25FdmVudChFdmVudFR5cGUuT2JqZWN0UmVtb3ZlZClgLlxuICAgICAqXG4gICAgICogQHBhcmFtIGRlc3QgVGhlIG5vdGlmaWNhdGlvbiBkZXN0aW5hdGlvbiAoc2VlIG9uRXZlbnQpXG4gICAgICogQHBhcmFtIGZpbHRlcnMgRmlsdGVycyAoc2VlIG9uRXZlbnQpXG4gICAgICovXG4gICAgcHVibGljIGFkZE9iamVjdFJlbW92ZWROb3RpZmljYXRpb24oZGVzdDogSUJ1Y2tldE5vdGlmaWNhdGlvbkRlc3RpbmF0aW9uLCAuLi5maWx0ZXJzOiBOb3RpZmljYXRpb25LZXlGaWx0ZXJbXSkge1xuICAgICAgICByZXR1cm4gdGhpcy5hZGRFdmVudE5vdGlmaWNhdGlvbihFdmVudFR5cGUuT0JKRUNUX1JFTU9WRUQsIGRlc3QsIC4uLmZpbHRlcnMpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYW4gaW52ZW50b3J5IGNvbmZpZ3VyYXRpb24uXG4gICAgICpcbiAgICAgKiBAcGFyYW0gaW52ZW50b3J5IGNvbmZpZ3VyYXRpb24gdG8gYWRkXG4gICAgICovXG4gICAgcHVibGljIGFkZEludmVudG9yeShpbnZlbnRvcnk6IEludmVudG9yeSk6IHZvaWQge1xuICAgICAgICB0aGlzLmludmVudG9yaWVzLnB1c2goaW52ZW50b3J5KTtcbiAgICB9XG4gICAgcHJpdmF0ZSB2YWxpZGF0ZUJ1Y2tldE5hbWUocGh5c2ljYWxOYW1lOiBzdHJpbmcpOiB2b2lkIHtcbiAgICAgICAgY29uc3QgYnVja2V0TmFtZSA9IHBoeXNpY2FsTmFtZTtcbiAgICAgICAgaWYgKCFidWNrZXROYW1lIHx8IFRva2VuLmlzVW5yZXNvbHZlZChidWNrZXROYW1lKSkge1xuICAgICAgICAgICAgLy8gdGhlIG5hbWUgaXMgYSBsYXRlLWJvdW5kIHZhbHVlLCBub3QgYSBkZWZpbmVkIHN0cmluZyxcbiAgICAgICAgICAgIC8vIHNvIHNraXAgdmFsaWRhdGlvblxuICAgICAgICAgICAgcmV0dXJuO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGVycm9yczogc3RyaW5nW10gPSBbXTtcbiAgICAgICAgLy8gUnVsZXMgY29kaWZpZWQgZnJvbSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9CdWNrZXRSZXN0cmljdGlvbnMuaHRtbFxuICAgICAgICBpZiAoYnVja2V0TmFtZS5sZW5ndGggPCAzIHx8IGJ1Y2tldE5hbWUubGVuZ3RoID4gNjMpIHtcbiAgICAgICAgICAgIGVycm9ycy5wdXNoKCdCdWNrZXQgbmFtZSBtdXN0IGJlIGF0IGxlYXN0IDMgYW5kIG5vIG1vcmUgdGhhbiA2MyBjaGFyYWN0ZXJzJyk7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgY2hhcnNldE1hdGNoID0gYnVja2V0TmFtZS5tYXRjaCgvW15hLXowLTkuLV0vKTtcbiAgICAgICAgaWYgKGNoYXJzZXRNYXRjaCkge1xuICAgICAgICAgICAgZXJyb3JzLnB1c2goJ0J1Y2tldCBuYW1lIG11c3Qgb25seSBjb250YWluIGxvd2VyY2FzZSBjaGFyYWN0ZXJzIGFuZCB0aGUgc3ltYm9scywgcGVyaW9kICguKSBhbmQgZGFzaCAoLSkgJ1xuICAgICAgICAgICAgICAgICsgYChvZmZzZXQ6ICR7Y2hhcnNldE1hdGNoLmluZGV4fSlgKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoIS9bYS16MC05XS8udGVzdChidWNrZXROYW1lLmNoYXJBdCgwKSkpIHtcbiAgICAgICAgICAgIGVycm9ycy5wdXNoKCdCdWNrZXQgbmFtZSBtdXN0IHN0YXJ0IGFuZCBlbmQgd2l0aCBhIGxvd2VyY2FzZSBjaGFyYWN0ZXIgb3IgbnVtYmVyICdcbiAgICAgICAgICAgICAgICArICcob2Zmc2V0OiAwKScpO1xuICAgICAgICB9XG4gICAgICAgIGlmICghL1thLXowLTldLy50ZXN0KGJ1Y2tldE5hbWUuY2hhckF0KGJ1Y2tldE5hbWUubGVuZ3RoIC0gMSkpKSB7XG4gICAgICAgICAgICBlcnJvcnMucHVzaCgnQnVja2V0IG5hbWUgbXVzdCBzdGFydCBhbmQgZW5kIHdpdGggYSBsb3dlcmNhc2UgY2hhcmFjdGVyIG9yIG51bWJlciAnXG4gICAgICAgICAgICAgICAgKyBgKG9mZnNldDogJHtidWNrZXROYW1lLmxlbmd0aCAtIDF9KWApO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGNvbnNlY1N5bWJvbE1hdGNoID0gYnVja2V0TmFtZS5tYXRjaCgvXFwuLXwtXFwufFxcLlxcLi8pO1xuICAgICAgICBpZiAoY29uc2VjU3ltYm9sTWF0Y2gpIHtcbiAgICAgICAgICAgIGVycm9ycy5wdXNoKCdCdWNrZXQgbmFtZSBtdXN0IG5vdCBoYXZlIGRhc2ggbmV4dCB0byBwZXJpb2QsIG9yIHBlcmlvZCBuZXh0IHRvIGRhc2gsIG9yIGNvbnNlY3V0aXZlIHBlcmlvZHMgJ1xuICAgICAgICAgICAgICAgICsgYChvZmZzZXQ6ICR7Y29uc2VjU3ltYm9sTWF0Y2guaW5kZXh9KWApO1xuICAgICAgICB9XG4gICAgICAgIGlmICgvXlxcZHsxLDN9XFwuXFxkezEsM31cXC5cXGR7MSwzfVxcLlxcZHsxLDN9JC8udGVzdChidWNrZXROYW1lKSkge1xuICAgICAgICAgICAgZXJyb3JzLnB1c2goJ0J1Y2tldCBuYW1lIG11c3Qgbm90IHJlc2VtYmxlIGFuIElQIGFkZHJlc3MnKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoZXJyb3JzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgSW52YWxpZCBTMyBidWNrZXQgbmFtZSAodmFsdWU6ICR7YnVja2V0TmFtZX0pJHtFT0x9JHtlcnJvcnMuam9pbihFT0wpfWApO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFNldCB1cCBrZXkgcHJvcGVydGllcyBhbmQgcmV0dXJuIHRoZSBCdWNrZXQgZW5jcnlwdGlvbiBwcm9wZXJ0eSBmcm9tIHRoZVxuICAgICAqIHVzZXIncyBjb25maWd1cmF0aW9uLlxuICAgICAqL1xuICAgIHByaXZhdGUgcGFyc2VFbmNyeXB0aW9uKHByb3BzOiBCdWNrZXRQcm9wcyk6IHtcbiAgICAgICAgYnVja2V0RW5jcnlwdGlvbj86IENmbkJ1Y2tldC5CdWNrZXRFbmNyeXB0aW9uUHJvcGVydHk7XG4gICAgICAgIGVuY3J5cHRpb25LZXk/OiBrbXMuSUtleTtcbiAgICB9IHtcbiAgICAgICAgLy8gZGVmYXVsdCBiYXNlZCBvbiB3aGV0aGVyIGVuY3J5cHRpb25LZXkgaXMgc3BlY2lmaWVkXG4gICAgICAgIGxldCBlbmNyeXB0aW9uVHlwZSA9IHByb3BzLmVuY3J5cHRpb247XG4gICAgICAgIGlmIChlbmNyeXB0aW9uVHlwZSA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICBlbmNyeXB0aW9uVHlwZSA9IHByb3BzLmVuY3J5cHRpb25LZXkgPyBCdWNrZXRFbmNyeXB0aW9uLktNUyA6IEJ1Y2tldEVuY3J5cHRpb24uVU5FTkNSWVBURUQ7XG4gICAgICAgIH1cbiAgICAgICAgLy8gaWYgZW5jcnlwdGlvbiBrZXkgaXMgc2V0LCBlbmNyeXB0aW9uIG11c3QgYmUgc2V0IHRvIEtNUy5cbiAgICAgICAgaWYgKGVuY3J5cHRpb25UeXBlICE9PSBCdWNrZXRFbmNyeXB0aW9uLktNUyAmJiBwcm9wcy5lbmNyeXB0aW9uS2V5KSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYGVuY3J5cHRpb25LZXkgaXMgc3BlY2lmaWVkLCBzbyAnZW5jcnlwdGlvbicgbXVzdCBiZSBzZXQgdG8gS01TICh2YWx1ZTogJHtlbmNyeXB0aW9uVHlwZX0pYCk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKGVuY3J5cHRpb25UeXBlID09PSBCdWNrZXRFbmNyeXB0aW9uLlVORU5DUllQVEVEKSB7XG4gICAgICAgICAgICByZXR1cm4geyBidWNrZXRFbmNyeXB0aW9uOiB1bmRlZmluZWQsIGVuY3J5cHRpb25LZXk6IHVuZGVmaW5lZCB9O1xuICAgICAgICB9XG4gICAgICAgIGlmIChlbmNyeXB0aW9uVHlwZSA9PT0gQnVja2V0RW5jcnlwdGlvbi5LTVMpIHtcbiAgICAgICAgICAgIGNvbnN0IGVuY3J5cHRpb25LZXkgPSBwcm9wcy5lbmNyeXB0aW9uS2V5IHx8IG5ldyBrbXMuS2V5KHRoaXMsICdLZXknLCB7XG4gICAgICAgICAgICAgICAgZGVzY3JpcHRpb246IGBDcmVhdGVkIGJ5ICR7dGhpcy5ub2RlLnBhdGh9YCxcbiAgICAgICAgICAgIH0pO1xuICAgICAgICAgICAgY29uc3QgYnVja2V0RW5jcnlwdGlvbiA9IHtcbiAgICAgICAgICAgICAgICBzZXJ2ZXJTaWRlRW5jcnlwdGlvbkNvbmZpZ3VyYXRpb246IFtcbiAgICAgICAgICAgICAgICAgICAge1xuICAgICAgICAgICAgICAgICAgICAgICAgc2VydmVyU2lkZUVuY3J5cHRpb25CeURlZmF1bHQ6IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBzc2VBbGdvcml0aG06ICdhd3M6a21zJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICBrbXNNYXN0ZXJLZXlJZDogZW5jcnlwdGlvbktleS5rZXlBcm4sXG4gICAgICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9O1xuICAgICAgICAgICAgcmV0dXJuIHsgZW5jcnlwdGlvbktleSwgYnVja2V0RW5jcnlwdGlvbiB9O1xuICAgICAgICB9XG4gICAgICAgIGlmIChlbmNyeXB0aW9uVHlwZSA9PT0gQnVja2V0RW5jcnlwdGlvbi5TM19NQU5BR0VEKSB7XG4gICAgICAgICAgICBjb25zdCBidWNrZXRFbmNyeXB0aW9uID0ge1xuICAgICAgICAgICAgICAgIHNlcnZlclNpZGVFbmNyeXB0aW9uQ29uZmlndXJhdGlvbjogW1xuICAgICAgICAgICAgICAgICAgICB7IHNlcnZlclNpZGVFbmNyeXB0aW9uQnlEZWZhdWx0OiB7IHNzZUFsZ29yaXRobTogJ0FFUzI1NicgfSB9LFxuICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICB9O1xuICAgICAgICAgICAgcmV0dXJuIHsgYnVja2V0RW5jcnlwdGlvbiB9O1xuICAgICAgICB9XG4gICAgICAgIGlmIChlbmNyeXB0aW9uVHlwZSA9PT0gQnVja2V0RW5jcnlwdGlvbi5LTVNfTUFOQUdFRCkge1xuICAgICAgICAgICAgY29uc3QgYnVja2V0RW5jcnlwdGlvbiA9IHtcbiAgICAgICAgICAgICAgICBzZXJ2ZXJTaWRlRW5jcnlwdGlvbkNvbmZpZ3VyYXRpb246IFtcbiAgICAgICAgICAgICAgICAgICAgeyBzZXJ2ZXJTaWRlRW5jcnlwdGlvbkJ5RGVmYXVsdDogeyBzc2VBbGdvcml0aG06ICdhd3M6a21zJyB9IH0sXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIH07XG4gICAgICAgICAgICByZXR1cm4geyBidWNrZXRFbmNyeXB0aW9uIH07XG4gICAgICAgIH1cbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBVbmV4cGVjdGVkICdlbmNyeXB0aW9uVHlwZSc6ICR7ZW5jcnlwdGlvblR5cGV9YCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFBhcnNlIHRoZSBsaWZlY3ljbGUgY29uZmlndXJhdGlvbiBvdXQgb2YgdGhlIHV1Y2tldCBwcm9wc1xuICAgICAqIEBwYXJhbSBwcm9wcyBQYXJcbiAgICAgKi9cbiAgICBwcml2YXRlIHBhcnNlTGlmZWN5Y2xlQ29uZmlndXJhdGlvbigpOiBDZm5CdWNrZXQuTGlmZWN5Y2xlQ29uZmlndXJhdGlvblByb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICAgICAgaWYgKCF0aGlzLmxpZmVjeWNsZVJ1bGVzIHx8IHRoaXMubGlmZWN5Y2xlUnVsZXMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHNlbGYgPSB0aGlzO1xuICAgICAgICByZXR1cm4geyBydWxlczogdGhpcy5saWZlY3ljbGVSdWxlcy5tYXAocGFyc2VMaWZlY3ljbGVSdWxlKSB9O1xuICAgICAgICBmdW5jdGlvbiBwYXJzZUxpZmVjeWNsZVJ1bGUocnVsZTogTGlmZWN5Y2xlUnVsZSk6IENmbkJ1Y2tldC5SdWxlUHJvcGVydHkge1xuICAgICAgICAgICAgY29uc3QgZW5hYmxlZCA9IHJ1bGUuZW5hYmxlZCAhPT0gdW5kZWZpbmVkID8gcnVsZS5lbmFibGVkIDogdHJ1ZTtcbiAgICAgICAgICAgIGNvbnN0IHg6IENmbkJ1Y2tldC5SdWxlUHJvcGVydHkgPSB7XG4gICAgICAgICAgICAgICAgLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIG1heC1sZW5cbiAgICAgICAgICAgICAgICBhYm9ydEluY29tcGxldGVNdWx0aXBhcnRVcGxvYWQ6IHJ1bGUuYWJvcnRJbmNvbXBsZXRlTXVsdGlwYXJ0VXBsb2FkQWZ0ZXIgIT09IHVuZGVmaW5lZCA/IHsgZGF5c0FmdGVySW5pdGlhdGlvbjogcnVsZS5hYm9ydEluY29tcGxldGVNdWx0aXBhcnRVcGxvYWRBZnRlci50b0RheXMoKSB9IDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgICAgIGV4cGlyYXRpb25EYXRlOiBydWxlLmV4cGlyYXRpb25EYXRlLFxuICAgICAgICAgICAgICAgIGV4cGlyYXRpb25JbkRheXM6IHJ1bGUuZXhwaXJhdGlvbiAmJiBydWxlLmV4cGlyYXRpb24udG9EYXlzKCksXG4gICAgICAgICAgICAgICAgaWQ6IHJ1bGUuaWQsXG4gICAgICAgICAgICAgICAgbm9uY3VycmVudFZlcnNpb25FeHBpcmF0aW9uSW5EYXlzOiBydWxlLm5vbmN1cnJlbnRWZXJzaW9uRXhwaXJhdGlvbiAmJiBydWxlLm5vbmN1cnJlbnRWZXJzaW9uRXhwaXJhdGlvbi50b0RheXMoKSxcbiAgICAgICAgICAgICAgICBub25jdXJyZW50VmVyc2lvblRyYW5zaXRpb25zOiBtYXBPclVuZGVmaW5lZChydWxlLm5vbmN1cnJlbnRWZXJzaW9uVHJhbnNpdGlvbnMsIHQgPT4gKHtcbiAgICAgICAgICAgICAgICAgICAgc3RvcmFnZUNsYXNzOiB0LnN0b3JhZ2VDbGFzcy52YWx1ZSxcbiAgICAgICAgICAgICAgICAgICAgdHJhbnNpdGlvbkluRGF5czogdC50cmFuc2l0aW9uQWZ0ZXIudG9EYXlzKCksXG4gICAgICAgICAgICAgICAgfSkpLFxuICAgICAgICAgICAgICAgIHByZWZpeDogcnVsZS5wcmVmaXgsXG4gICAgICAgICAgICAgICAgc3RhdHVzOiBlbmFibGVkID8gJ0VuYWJsZWQnIDogJ0Rpc2FibGVkJyxcbiAgICAgICAgICAgICAgICB0cmFuc2l0aW9uczogbWFwT3JVbmRlZmluZWQocnVsZS50cmFuc2l0aW9ucywgdCA9PiAoe1xuICAgICAgICAgICAgICAgICAgICBzdG9yYWdlQ2xhc3M6IHQuc3RvcmFnZUNsYXNzLnZhbHVlLFxuICAgICAgICAgICAgICAgICAgICB0cmFuc2l0aW9uRGF0ZTogdC50cmFuc2l0aW9uRGF0ZSxcbiAgICAgICAgICAgICAgICAgICAgdHJhbnNpdGlvbkluRGF5czogdC50cmFuc2l0aW9uQWZ0ZXIgJiYgdC50cmFuc2l0aW9uQWZ0ZXIudG9EYXlzKCksXG4gICAgICAgICAgICAgICAgfSkpLFxuICAgICAgICAgICAgICAgIHRhZ0ZpbHRlcnM6IHNlbGYucGFyc2VUYWdGaWx0ZXJzKHJ1bGUudGFnRmlsdGVycyksXG4gICAgICAgICAgICB9O1xuICAgICAgICAgICAgcmV0dXJuIHg7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHJpdmF0ZSBwYXJzZVNlcnZlckFjY2Vzc0xvZ3MocHJvcHM6IEJ1Y2tldFByb3BzKTogQ2ZuQnVja2V0LkxvZ2dpbmdDb25maWd1cmF0aW9uUHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgICAgICBpZiAoIXByb3BzLnNlcnZlckFjY2Vzc0xvZ3NCdWNrZXQgJiYgIXByb3BzLnNlcnZlckFjY2Vzc0xvZ3NQcmVmaXgpIHtcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIGRlc3RpbmF0aW9uQnVja2V0TmFtZTogcHJvcHMuc2VydmVyQWNjZXNzTG9nc0J1Y2tldD8uYnVja2V0TmFtZSxcbiAgICAgICAgICAgIGxvZ0ZpbGVQcmVmaXg6IHByb3BzLnNlcnZlckFjY2Vzc0xvZ3NQcmVmaXgsXG4gICAgICAgIH07XG4gICAgfVxuICAgIHByaXZhdGUgcGFyc2VNZXRyaWNDb25maWd1cmF0aW9uKCk6IENmbkJ1Y2tldC5NZXRyaWNzQ29uZmlndXJhdGlvblByb3BlcnR5W10gfCB1bmRlZmluZWQge1xuICAgICAgICBpZiAoIXRoaXMubWV0cmljcyB8fCB0aGlzLm1ldHJpY3MubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHNlbGYgPSB0aGlzO1xuICAgICAgICByZXR1cm4gdGhpcy5tZXRyaWNzLm1hcChwYXJzZU1ldHJpYyk7XG4gICAgICAgIGZ1bmN0aW9uIHBhcnNlTWV0cmljKG1ldHJpYzogQnVja2V0TWV0cmljcyk6IENmbkJ1Y2tldC5NZXRyaWNzQ29uZmlndXJhdGlvblByb3BlcnR5IHtcbiAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgaWQ6IG1ldHJpYy5pZCxcbiAgICAgICAgICAgICAgICBwcmVmaXg6IG1ldHJpYy5wcmVmaXgsXG4gICAgICAgICAgICAgICAgdGFnRmlsdGVyczogc2VsZi5wYXJzZVRhZ0ZpbHRlcnMobWV0cmljLnRhZ0ZpbHRlcnMpLFxuICAgICAgICAgICAgfTtcbiAgICAgICAgfVxuICAgIH1cbiAgICBwcml2YXRlIHBhcnNlQ29yc0NvbmZpZ3VyYXRpb24oKTogQ2ZuQnVja2V0LkNvcnNDb25maWd1cmF0aW9uUHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgICAgICBpZiAoIXRoaXMuY29ycyB8fCB0aGlzLmNvcnMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB7IGNvcnNSdWxlczogdGhpcy5jb3JzLm1hcChwYXJzZUNvcnMpIH07XG4gICAgICAgIGZ1bmN0aW9uIHBhcnNlQ29ycyhydWxlOiBDb3JzUnVsZSk6IENmbkJ1Y2tldC5Db3JzUnVsZVByb3BlcnR5IHtcbiAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgaWQ6IHJ1bGUuaWQsXG4gICAgICAgICAgICAgICAgbWF4QWdlOiBydWxlLm1heEFnZSxcbiAgICAgICAgICAgICAgICBhbGxvd2VkSGVhZGVyczogcnVsZS5hbGxvd2VkSGVhZGVycyxcbiAgICAgICAgICAgICAgICBhbGxvd2VkTWV0aG9kczogcnVsZS5hbGxvd2VkTWV0aG9kcyxcbiAgICAgICAgICAgICAgICBhbGxvd2VkT3JpZ2luczogcnVsZS5hbGxvd2VkT3JpZ2lucyxcbiAgICAgICAgICAgICAgICBleHBvc2VkSGVhZGVyczogcnVsZS5leHBvc2VkSGVhZGVycyxcbiAgICAgICAgICAgIH07XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHJpdmF0ZSBwYXJzZVRhZ0ZpbHRlcnModGFnRmlsdGVycz86IHtcbiAgICAgICAgW3RhZzogc3RyaW5nXTogYW55O1xuICAgIH0pIHtcbiAgICAgICAgaWYgKCF0YWdGaWx0ZXJzIHx8IHRhZ0ZpbHRlcnMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBPYmplY3Qua2V5cyh0YWdGaWx0ZXJzKS5tYXAodGFnID0+ICh7XG4gICAgICAgICAgICBrZXk6IHRhZyxcbiAgICAgICAgICAgIHZhbHVlOiB0YWdGaWx0ZXJzW3RhZ10sXG4gICAgICAgIH0pKTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZW5kZXJXZWJzaXRlQ29uZmlndXJhdGlvbihwcm9wczogQnVja2V0UHJvcHMpOiBDZm5CdWNrZXQuV2Vic2l0ZUNvbmZpZ3VyYXRpb25Qcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGlmICghcHJvcHMud2Vic2l0ZUVycm9yRG9jdW1lbnQgJiYgIXByb3BzLndlYnNpdGVJbmRleERvY3VtZW50ICYmICFwcm9wcy53ZWJzaXRlUmVkaXJlY3QgJiYgIXByb3BzLndlYnNpdGVSb3V0aW5nUnVsZXMpIHtcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLndlYnNpdGVFcnJvckRvY3VtZW50ICYmICFwcm9wcy53ZWJzaXRlSW5kZXhEb2N1bWVudCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdcIndlYnNpdGVJbmRleERvY3VtZW50XCIgaXMgcmVxdWlyZWQgaWYgXCJ3ZWJzaXRlRXJyb3JEb2N1bWVudFwiIGlzIHNldCcpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy53ZWJzaXRlUmVkaXJlY3QgJiYgKHByb3BzLndlYnNpdGVFcnJvckRvY3VtZW50IHx8IHByb3BzLndlYnNpdGVJbmRleERvY3VtZW50IHx8IHByb3BzLndlYnNpdGVSb3V0aW5nUnVsZXMpKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1wid2Vic2l0ZUluZGV4RG9jdW1lbnRcIiwgXCJ3ZWJzaXRlRXJyb3JEb2N1bWVudFwiIGFuZCwgXCJ3ZWJzaXRlUm91dGluZ1J1bGVzXCIgY2Fubm90IGJlIHNldCBpZiBcIndlYnNpdGVSZWRpcmVjdFwiIGlzIHVzZWQnKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCByb3V0aW5nUnVsZXMgPSBwcm9wcy53ZWJzaXRlUm91dGluZ1J1bGVzID8gcHJvcHMud2Vic2l0ZVJvdXRpbmdSdWxlcy5tYXA8Q2ZuQnVja2V0LlJvdXRpbmdSdWxlUHJvcGVydHk+KChydWxlKSA9PiB7XG4gICAgICAgICAgICBpZiAocnVsZS5jb25kaXRpb24gJiYgIXJ1bGUuY29uZGl0aW9uLmh0dHBFcnJvckNvZGVSZXR1cm5lZEVxdWFscyAmJiAhcnVsZS5jb25kaXRpb24ua2V5UHJlZml4RXF1YWxzKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdUaGUgY29uZGl0aW9uIHByb3BlcnR5IGNhbm5vdCBiZSBhbiBlbXB0eSBvYmplY3QnKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgcmVkaXJlY3RSdWxlOiB7XG4gICAgICAgICAgICAgICAgICAgIGhvc3ROYW1lOiBydWxlLmhvc3ROYW1lLFxuICAgICAgICAgICAgICAgICAgICBodHRwUmVkaXJlY3RDb2RlOiBydWxlLmh0dHBSZWRpcmVjdENvZGUsXG4gICAgICAgICAgICAgICAgICAgIHByb3RvY29sOiBydWxlLnByb3RvY29sLFxuICAgICAgICAgICAgICAgICAgICByZXBsYWNlS2V5V2l0aDogcnVsZS5yZXBsYWNlS2V5ICYmIHJ1bGUucmVwbGFjZUtleS53aXRoS2V5LFxuICAgICAgICAgICAgICAgICAgICByZXBsYWNlS2V5UHJlZml4V2l0aDogcnVsZS5yZXBsYWNlS2V5ICYmIHJ1bGUucmVwbGFjZUtleS5wcmVmaXhXaXRoS2V5LFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgcm91dGluZ1J1bGVDb25kaXRpb246IHJ1bGUuY29uZGl0aW9uLFxuICAgICAgICAgICAgfTtcbiAgICAgICAgfSkgOiB1bmRlZmluZWQ7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBpbmRleERvY3VtZW50OiBwcm9wcy53ZWJzaXRlSW5kZXhEb2N1bWVudCxcbiAgICAgICAgICAgIGVycm9yRG9jdW1lbnQ6IHByb3BzLndlYnNpdGVFcnJvckRvY3VtZW50LFxuICAgICAgICAgICAgcmVkaXJlY3RBbGxSZXF1ZXN0c1RvOiBwcm9wcy53ZWJzaXRlUmVkaXJlY3QsXG4gICAgICAgICAgICByb3V0aW5nUnVsZXMsXG4gICAgICAgIH07XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFsbG93cyB0aGUgTG9nRGVsaXZlcnkgZ3JvdXAgdG8gd3JpdGUsIGZhaWxzIGlmIEFDTCB3YXMgc2V0IGRpZmZlcmVudGx5LlxuICAgICAqXG4gICAgICogQHNlZVxuICAgICAqIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2FjbC1vdmVydmlldy5odG1sI2Nhbm5lZC1hY2xcbiAgICAgKi9cbiAgICBwcml2YXRlIGFsbG93TG9nRGVsaXZlcnkoKSB7XG4gICAgICAgIGlmICh0aGlzLmFjY2Vzc0NvbnRyb2wgJiYgdGhpcy5hY2Nlc3NDb250cm9sICE9PSBCdWNrZXRBY2Nlc3NDb250cm9sLkxPR19ERUxJVkVSWV9XUklURSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwiQ2Fubm90IGVuYWJsZSBsb2cgZGVsaXZlcnkgdG8gdGhpcyBidWNrZXQgYmVjYXVzZSB0aGUgYnVja2V0J3MgQUNMIGhhcyBiZWVuIHNldCBhbmQgY2FuJ3QgYmUgY2hhbmdlZFwiKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLmFjY2Vzc0NvbnRyb2wgPSBCdWNrZXRBY2Nlc3NDb250cm9sLkxPR19ERUxJVkVSWV9XUklURTtcbiAgICB9XG4gICAgcHJpdmF0ZSBwYXJzZUludmVudG9yeUNvbmZpZ3VyYXRpb24oKTogQ2ZuQnVja2V0LkludmVudG9yeUNvbmZpZ3VyYXRpb25Qcm9wZXJ0eVtdIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgaWYgKCF0aGlzLmludmVudG9yaWVzIHx8IHRoaXMuaW52ZW50b3JpZXMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0aGlzLmludmVudG9yaWVzLm1hcCgoaW52ZW50b3J5LCBpbmRleCkgPT4ge1xuICAgICAgICAgICAgY29uc3QgZm9ybWF0ID0gaW52ZW50b3J5LmZvcm1hdCA/PyBJbnZlbnRvcnlGb3JtYXQuQ1NWO1xuICAgICAgICAgICAgY29uc3QgZnJlcXVlbmN5ID0gaW52ZW50b3J5LmZyZXF1ZW5jeSA/PyBJbnZlbnRvcnlGcmVxdWVuY3kuV0VFS0xZO1xuICAgICAgICAgICAgY29uc3QgaWQgPSBpbnZlbnRvcnkuaW52ZW50b3J5SWQgPz8gYCR7dGhpcy5ub2RlLmlkfUludmVudG9yeSR7aW5kZXh9YDtcbiAgICAgICAgICAgIGlmIChpbnZlbnRvcnkuZGVzdGluYXRpb24uYnVja2V0IGluc3RhbmNlb2YgQnVja2V0KSB7XG4gICAgICAgICAgICAgICAgaW52ZW50b3J5LmRlc3RpbmF0aW9uLmJ1Y2tldC5hZGRUb1Jlc291cmNlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICAgICAgICBhY3Rpb25zOiBbJ3MzOlB1dE9iamVjdCddLFxuICAgICAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFtcbiAgICAgICAgICAgICAgICAgICAgICAgIGludmVudG9yeS5kZXN0aW5hdGlvbi5idWNrZXQuYnVja2V0QXJuLFxuICAgICAgICAgICAgICAgICAgICAgICAgaW52ZW50b3J5LmRlc3RpbmF0aW9uLmJ1Y2tldC5hcm5Gb3JPYmplY3RzKGAke2ludmVudG9yeS5kZXN0aW5hdGlvbi5wcmVmaXggPz8gJyd9KmApLFxuICAgICAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICAgICAgICBwcmluY2lwYWxzOiBbbmV3IGlhbS5TZXJ2aWNlUHJpbmNpcGFsKCdzMy5hbWF6b25hd3MuY29tJyldLFxuICAgICAgICAgICAgICAgICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICBBcm5MaWtlOiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgJ2F3czpTb3VyY2VBcm4nOiB0aGlzLmJ1Y2tldEFybixcbiAgICAgICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgICAgfSkpO1xuICAgICAgICAgICAgfVxuICAgICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgICAgICBpZCxcbiAgICAgICAgICAgICAgICBkZXN0aW5hdGlvbjoge1xuICAgICAgICAgICAgICAgICAgICBidWNrZXRBcm46IGludmVudG9yeS5kZXN0aW5hdGlvbi5idWNrZXQuYnVja2V0QXJuLFxuICAgICAgICAgICAgICAgICAgICBidWNrZXRBY2NvdW50SWQ6IGludmVudG9yeS5kZXN0aW5hdGlvbi5idWNrZXRPd25lcixcbiAgICAgICAgICAgICAgICAgICAgcHJlZml4OiBpbnZlbnRvcnkuZGVzdGluYXRpb24ucHJlZml4LFxuICAgICAgICAgICAgICAgICAgICBmb3JtYXQsXG4gICAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgICBlbmFibGVkOiBpbnZlbnRvcnkuZW5hYmxlZCA/PyB0cnVlLFxuICAgICAgICAgICAgICAgIGluY2x1ZGVkT2JqZWN0VmVyc2lvbnM6IGludmVudG9yeS5pbmNsdWRlT2JqZWN0VmVyc2lvbnMgPz8gSW52ZW50b3J5T2JqZWN0VmVyc2lvbi5BTEwsXG4gICAgICAgICAgICAgICAgc2NoZWR1bGVGcmVxdWVuY3k6IGZyZXF1ZW5jeSxcbiAgICAgICAgICAgICAgICBvcHRpb25hbEZpZWxkczogaW52ZW50b3J5Lm9wdGlvbmFsRmllbGRzLFxuICAgICAgICAgICAgICAgIHByZWZpeDogaW52ZW50b3J5Lm9iamVjdHNQcmVmaXgsXG4gICAgICAgICAgICB9O1xuICAgICAgICB9KTtcbiAgICB9XG59XG4vKipcbiAqIFdoYXQga2luZCBvZiBzZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIHRvIGFwcGx5IHRvIHRoaXMgYnVja2V0XG4gKi9cbmV4cG9ydCBlbnVtIEJ1Y2tldEVuY3J5cHRpb24ge1xuICAgIC8qKlxuICAgICAqIE9iamVjdHMgaW4gdGhlIGJ1Y2tldCBhcmUgbm90IGVuY3J5cHRlZC5cbiAgICAgKi9cbiAgICBVTkVOQ1JZUFRFRCA9ICdOT05FJyxcbiAgICAvKipcbiAgICAgKiBTZXJ2ZXItc2lkZSBLTVMgZW5jcnlwdGlvbiB3aXRoIGEgbWFzdGVyIGtleSBtYW5hZ2VkIGJ5IEtNUy5cbiAgICAgKi9cbiAgICBLTVNfTUFOQUdFRCA9ICdNQU5BR0VEJyxcbiAgICAvKipcbiAgICAgKiBTZXJ2ZXItc2lkZSBlbmNyeXB0aW9uIHdpdGggYSBtYXN0ZXIga2V5IG1hbmFnZWQgYnkgUzMuXG4gICAgICovXG4gICAgUzNfTUFOQUdFRCA9ICdTM01BTkFHRUQnLFxuICAgIC8qKlxuICAgICAqIFNlcnZlci1zaWRlIGVuY3J5cHRpb24gd2l0aCBhIEtNUyBrZXkgbWFuYWdlZCBieSB0aGUgdXNlci5cbiAgICAgKiBJZiBgZW5jcnlwdGlvbktleWAgaXMgc3BlY2lmaWVkLCB0aGlzIGtleSB3aWxsIGJlIHVzZWQsIG90aGVyd2lzZSwgb25lIHdpbGwgYmUgZGVmaW5lZC5cbiAgICAgKi9cbiAgICBLTVMgPSAnS01TJ1xufVxuLyoqXG4gKiBOb3RpZmljYXRpb24gZXZlbnQgdHlwZXMuXG4gKi9cbmV4cG9ydCBlbnVtIEV2ZW50VHlwZSB7XG4gICAgLyoqXG4gICAgICogQW1hem9uIFMzIEFQSXMgc3VjaCBhcyBQVVQsIFBPU1QsIGFuZCBDT1BZIGNhbiBjcmVhdGUgYW4gb2JqZWN0LiBVc2luZ1xuICAgICAqIHRoZXNlIGV2ZW50IHR5cGVzLCB5b3UgY2FuIGVuYWJsZSBub3RpZmljYXRpb24gd2hlbiBhbiBvYmplY3QgaXMgY3JlYXRlZFxuICAgICAqIHVzaW5nIGEgc3BlY2lmaWMgQVBJLCBvciB5b3UgY2FuIHVzZSB0aGUgczM6T2JqZWN0Q3JlYXRlZDoqIGV2ZW50IHR5cGUgdG9cbiAgICAgKiByZXF1ZXN0IG5vdGlmaWNhdGlvbiByZWdhcmRsZXNzIG9mIHRoZSBBUEkgdGhhdCB3YXMgdXNlZCB0byBjcmVhdGUgYW5cbiAgICAgKiBvYmplY3QuXG4gICAgICovXG4gICAgT0JKRUNUX0NSRUFURUQgPSAnczM6T2JqZWN0Q3JlYXRlZDoqJyxcbiAgICAvKipcbiAgICAgKiBBbWF6b24gUzMgQVBJcyBzdWNoIGFzIFBVVCwgUE9TVCwgYW5kIENPUFkgY2FuIGNyZWF0ZSBhbiBvYmplY3QuIFVzaW5nXG4gICAgICogdGhlc2UgZXZlbnQgdHlwZXMsIHlvdSBjYW4gZW5hYmxlIG5vdGlmaWNhdGlvbiB3aGVuIGFuIG9iamVjdCBpcyBjcmVhdGVkXG4gICAgICogdXNpbmcgYSBzcGVjaWZpYyBBUEksIG9yIHlvdSBjYW4gdXNlIHRoZSBzMzpPYmplY3RDcmVhdGVkOiogZXZlbnQgdHlwZSB0b1xuICAgICAqIHJlcXVlc3Qgbm90aWZpY2F0aW9uIHJlZ2FyZGxlc3Mgb2YgdGhlIEFQSSB0aGF0IHdhcyB1c2VkIHRvIGNyZWF0ZSBhblxuICAgICAqIG9iamVjdC5cbiAgICAgKi9cbiAgICBPQkpFQ1RfQ1JFQVRFRF9QVVQgPSAnczM6T2JqZWN0Q3JlYXRlZDpQdXQnLFxuICAgIC8qKlxuICAgICAqIEFtYXpvbiBTMyBBUElzIHN1Y2ggYXMgUFVULCBQT1NULCBhbmQgQ09QWSBjYW4gY3JlYXRlIGFuIG9iamVjdC4gVXNpbmdcbiAgICAgKiB0aGVzZSBldmVudCB0eXBlcywgeW91IGNhbiBlbmFibGUgbm90aWZpY2F0aW9uIHdoZW4gYW4gb2JqZWN0IGlzIGNyZWF0ZWRcbiAgICAgKiB1c2luZyBhIHNwZWNpZmljIEFQSSwgb3IgeW91IGNhbiB1c2UgdGhlIHMzOk9iamVjdENyZWF0ZWQ6KiBldmVudCB0eXBlIHRvXG4gICAgICogcmVxdWVzdCBub3RpZmljYXRpb24gcmVnYXJkbGVzcyBvZiB0aGUgQVBJIHRoYXQgd2FzIHVzZWQgdG8gY3JlYXRlIGFuXG4gICAgICogb2JqZWN0LlxuICAgICAqL1xuICAgIE9CSkVDVF9DUkVBVEVEX1BPU1QgPSAnczM6T2JqZWN0Q3JlYXRlZDpQb3N0JyxcbiAgICAvKipcbiAgICAgKiBBbWF6b24gUzMgQVBJcyBzdWNoIGFzIFBVVCwgUE9TVCwgYW5kIENPUFkgY2FuIGNyZWF0ZSBhbiBvYmplY3QuIFVzaW5nXG4gICAgICogdGhlc2UgZXZlbnQgdHlwZXMsIHlvdSBjYW4gZW5hYmxlIG5vdGlmaWNhdGlvbiB3aGVuIGFuIG9iamVjdCBpcyBjcmVhdGVkXG4gICAgICogdXNpbmcgYSBzcGVjaWZpYyBBUEksIG9yIHlvdSBjYW4gdXNlIHRoZSBzMzpPYmplY3RDcmVhdGVkOiogZXZlbnQgdHlwZSB0b1xuICAgICAqIHJlcXVlc3Qgbm90aWZpY2F0aW9uIHJlZ2FyZGxlc3Mgb2YgdGhlIEFQSSB0aGF0IHdhcyB1c2VkIHRvIGNyZWF0ZSBhblxuICAgICAqIG9iamVjdC5cbiAgICAgKi9cbiAgICBPQkpFQ1RfQ1JFQVRFRF9DT1BZID0gJ3MzOk9iamVjdENyZWF0ZWQ6Q29weScsXG4gICAgLyoqXG4gICAgICogQW1hem9uIFMzIEFQSXMgc3VjaCBhcyBQVVQsIFBPU1QsIGFuZCBDT1BZIGNhbiBjcmVhdGUgYW4gb2JqZWN0LiBVc2luZ1xuICAgICAqIHRoZXNlIGV2ZW50IHR5cGVzLCB5b3UgY2FuIGVuYWJsZSBub3RpZmljYXRpb24gd2hlbiBhbiBvYmplY3QgaXMgY3JlYXRlZFxuICAgICAqIHVzaW5nIGEgc3BlY2lmaWMgQVBJLCBvciB5b3UgY2FuIHVzZSB0aGUgczM6T2JqZWN0Q3JlYXRlZDoqIGV2ZW50IHR5cGUgdG9cbiAgICAgKiByZXF1ZXN0IG5vdGlmaWNhdGlvbiByZWdhcmRsZXNzIG9mIHRoZSBBUEkgdGhhdCB3YXMgdXNlZCB0byBjcmVhdGUgYW5cbiAgICAgKiBvYmplY3QuXG4gICAgICovXG4gICAgT0JKRUNUX0NSRUFURURfQ09NUExFVEVfTVVMVElQQVJUX1VQTE9BRCA9ICdzMzpPYmplY3RDcmVhdGVkOkNvbXBsZXRlTXVsdGlwYXJ0VXBsb2FkJyxcbiAgICAvKipcbiAgICAgKiBCeSB1c2luZyB0aGUgT2JqZWN0UmVtb3ZlZCBldmVudCB0eXBlcywgeW91IGNhbiBlbmFibGUgbm90aWZpY2F0aW9uIHdoZW5cbiAgICAgKiBhbiBvYmplY3Qgb3IgYSBiYXRjaCBvZiBvYmplY3RzIGlzIHJlbW92ZWQgZnJvbSBhIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIFlvdSBjYW4gcmVxdWVzdCBub3RpZmljYXRpb24gd2hlbiBhbiBvYmplY3QgaXMgZGVsZXRlZCBvciBhIHZlcnNpb25lZFxuICAgICAqIG9iamVjdCBpcyBwZXJtYW5lbnRseSBkZWxldGVkIGJ5IHVzaW5nIHRoZSBzMzpPYmplY3RSZW1vdmVkOkRlbGV0ZSBldmVudFxuICAgICAqIHR5cGUuIE9yIHlvdSBjYW4gcmVxdWVzdCBub3RpZmljYXRpb24gd2hlbiBhIGRlbGV0ZSBtYXJrZXIgaXMgY3JlYXRlZCBmb3JcbiAgICAgKiBhIHZlcnNpb25lZCBvYmplY3QgYnkgdXNpbmcgczM6T2JqZWN0UmVtb3ZlZDpEZWxldGVNYXJrZXJDcmVhdGVkLiBGb3JcbiAgICAgKiBpbmZvcm1hdGlvbiBhYm91dCBkZWxldGluZyB2ZXJzaW9uZWQgb2JqZWN0cywgc2VlIERlbGV0aW5nIE9iamVjdFxuICAgICAqIFZlcnNpb25zLiBZb3UgY2FuIGFsc28gdXNlIGEgd2lsZGNhcmQgczM6T2JqZWN0UmVtb3ZlZDoqIHRvIHJlcXVlc3RcbiAgICAgKiBub3RpZmljYXRpb24gYW55dGltZSBhbiBvYmplY3QgaXMgZGVsZXRlZC5cbiAgICAgKlxuICAgICAqIFlvdSB3aWxsIG5vdCByZWNlaXZlIGV2ZW50IG5vdGlmaWNhdGlvbnMgZnJvbSBhdXRvbWF0aWMgZGVsZXRlcyBmcm9tXG4gICAgICogbGlmZWN5Y2xlIHBvbGljaWVzIG9yIGZyb20gZmFpbGVkIG9wZXJhdGlvbnMuXG4gICAgICovXG4gICAgT0JKRUNUX1JFTU9WRUQgPSAnczM6T2JqZWN0UmVtb3ZlZDoqJyxcbiAgICAvKipcbiAgICAgKiBCeSB1c2luZyB0aGUgT2JqZWN0UmVtb3ZlZCBldmVudCB0eXBlcywgeW91IGNhbiBlbmFibGUgbm90aWZpY2F0aW9uIHdoZW5cbiAgICAgKiBhbiBvYmplY3Qgb3IgYSBiYXRjaCBvZiBvYmplY3RzIGlzIHJlbW92ZWQgZnJvbSBhIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIFlvdSBjYW4gcmVxdWVzdCBub3RpZmljYXRpb24gd2hlbiBhbiBvYmplY3QgaXMgZGVsZXRlZCBvciBhIHZlcnNpb25lZFxuICAgICAqIG9iamVjdCBpcyBwZXJtYW5lbnRseSBkZWxldGVkIGJ5IHVzaW5nIHRoZSBzMzpPYmplY3RSZW1vdmVkOkRlbGV0ZSBldmVudFxuICAgICAqIHR5cGUuIE9yIHlvdSBjYW4gcmVxdWVzdCBub3RpZmljYXRpb24gd2hlbiBhIGRlbGV0ZSBtYXJrZXIgaXMgY3JlYXRlZCBmb3JcbiAgICAgKiBhIHZlcnNpb25lZCBvYmplY3QgYnkgdXNpbmcgczM6T2JqZWN0UmVtb3ZlZDpEZWxldGVNYXJrZXJDcmVhdGVkLiBGb3JcbiAgICAgKiBpbmZvcm1hdGlvbiBhYm91dCBkZWxldGluZyB2ZXJzaW9uZWQgb2JqZWN0cywgc2VlIERlbGV0aW5nIE9iamVjdFxuICAgICAqIFZlcnNpb25zLiBZb3UgY2FuIGFsc28gdXNlIGEgd2lsZGNhcmQgczM6T2JqZWN0UmVtb3ZlZDoqIHRvIHJlcXVlc3RcbiAgICAgKiBub3RpZmljYXRpb24gYW55dGltZSBhbiBvYmplY3QgaXMgZGVsZXRlZC5cbiAgICAgKlxuICAgICAqIFlvdSB3aWxsIG5vdCByZWNlaXZlIGV2ZW50IG5vdGlmaWNhdGlvbnMgZnJvbSBhdXRvbWF0aWMgZGVsZXRlcyBmcm9tXG4gICAgICogbGlmZWN5Y2xlIHBvbGljaWVzIG9yIGZyb20gZmFpbGVkIG9wZXJhdGlvbnMuXG4gICAgICovXG4gICAgT0JKRUNUX1JFTU9WRURfREVMRVRFID0gJ3MzOk9iamVjdFJlbW92ZWQ6RGVsZXRlJyxcbiAgICAvKipcbiAgICAgKiBCeSB1c2luZyB0aGUgT2JqZWN0UmVtb3ZlZCBldmVudCB0eXBlcywgeW91IGNhbiBlbmFibGUgbm90aWZpY2F0aW9uIHdoZW5cbiAgICAgKiBhbiBvYmplY3Qgb3IgYSBiYXRjaCBvZiBvYmplY3RzIGlzIHJlbW92ZWQgZnJvbSBhIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIFlvdSBjYW4gcmVxdWVzdCBub3RpZmljYXRpb24gd2hlbiBhbiBvYmplY3QgaXMgZGVsZXRlZCBvciBhIHZlcnNpb25lZFxuICAgICAqIG9iamVjdCBpcyBwZXJtYW5lbnRseSBkZWxldGVkIGJ5IHVzaW5nIHRoZSBzMzpPYmplY3RSZW1vdmVkOkRlbGV0ZSBldmVudFxuICAgICAqIHR5cGUuIE9yIHlvdSBjYW4gcmVxdWVzdCBub3RpZmljYXRpb24gd2hlbiBhIGRlbGV0ZSBtYXJrZXIgaXMgY3JlYXRlZCBmb3JcbiAgICAgKiBhIHZlcnNpb25lZCBvYmplY3QgYnkgdXNpbmcgczM6T2JqZWN0UmVtb3ZlZDpEZWxldGVNYXJrZXJDcmVhdGVkLiBGb3JcbiAgICAgKiBpbmZvcm1hdGlvbiBhYm91dCBkZWxldGluZyB2ZXJzaW9uZWQgb2JqZWN0cywgc2VlIERlbGV0aW5nIE9iamVjdFxuICAgICAqIFZlcnNpb25zLiBZb3UgY2FuIGFsc28gdXNlIGEgd2lsZGNhcmQgczM6T2JqZWN0UmVtb3ZlZDoqIHRvIHJlcXVlc3RcbiAgICAgKiBub3RpZmljYXRpb24gYW55dGltZSBhbiBvYmplY3QgaXMgZGVsZXRlZC5cbiAgICAgKlxuICAgICAqIFlvdSB3aWxsIG5vdCByZWNlaXZlIGV2ZW50IG5vdGlmaWNhdGlvbnMgZnJvbSBhdXRvbWF0aWMgZGVsZXRlcyBmcm9tXG4gICAgICogbGlmZWN5Y2xlIHBvbGljaWVzIG9yIGZyb20gZmFpbGVkIG9wZXJhdGlvbnMuXG4gICAgICovXG4gICAgT0JKRUNUX1JFTU9WRURfREVMRVRFX01BUktFUl9DUkVBVEVEID0gJ3MzOk9iamVjdFJlbW92ZWQ6RGVsZXRlTWFya2VyQ3JlYXRlZCcsXG4gICAgLyoqXG4gICAgICogWW91IGNhbiB1c2UgdGhpcyBldmVudCB0eXBlIHRvIHJlcXVlc3QgQW1hem9uIFMzIHRvIHNlbmQgYSBub3RpZmljYXRpb25cbiAgICAgKiBtZXNzYWdlIHdoZW4gQW1hem9uIFMzIGRldGVjdHMgdGhhdCBhbiBvYmplY3Qgb2YgdGhlIFJSUyBzdG9yYWdlIGNsYXNzIGlzXG4gICAgICogbG9zdC5cbiAgICAgKi9cbiAgICBSRURVQ0VEX1JFRFVOREFOQ1lfTE9TVF9PQkpFQ1QgPSAnczM6UmVkdWNlZFJlZHVuZGFuY3lMb3N0T2JqZWN0J1xufVxuZXhwb3J0IGludGVyZmFjZSBOb3RpZmljYXRpb25LZXlGaWx0ZXIge1xuICAgIC8qKlxuICAgICAqIFMzIGtleXMgbXVzdCBoYXZlIHRoZSBzcGVjaWZpZWQgcHJlZml4LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByZWZpeD86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBTMyBrZXlzIG11c3QgaGF2ZSB0aGUgc3BlY2lmaWVkIHN1ZmZpeC5cbiAgICAgKi9cbiAgICByZWFkb25seSBzdWZmaXg/OiBzdHJpbmc7XG59XG4vKipcbiAqIE9wdGlvbnMgZm9yIHRoZSBvbkNsb3VkVHJhaWxQdXRPYmplY3QgbWV0aG9kXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgT25DbG91ZFRyYWlsQnVja2V0RXZlbnRPcHRpb25zIGV4dGVuZHMgZXZlbnRzLk9uRXZlbnRPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBPbmx5IHdhdGNoIGNoYW5nZXMgdG8gdGhlc2Ugb2JqZWN0IHBhdGhzXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFdhdGNoIGNoYW5nZXMgdG8gYWxsIG9iamVjdHNcbiAgICAgKi9cbiAgICByZWFkb25seSBwYXRocz86IHN0cmluZ1tdO1xufVxuLyoqXG4gKiBEZWZhdWx0IGJ1Y2tldCBhY2Nlc3MgY29udHJvbCB0eXBlcy5cbiAqXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L2FjbC1vdmVydmlldy5odG1sXG4gKi9cbmV4cG9ydCBlbnVtIEJ1Y2tldEFjY2Vzc0NvbnRyb2wge1xuICAgIC8qKlxuICAgICAqIE93bmVyIGdldHMgRlVMTF9DT05UUk9MLiBObyBvbmUgZWxzZSBoYXMgYWNjZXNzIHJpZ2h0cy5cbiAgICAgKi9cbiAgICBQUklWQVRFID0gJ1ByaXZhdGUnLFxuICAgIC8qKlxuICAgICAqIE93bmVyIGdldHMgRlVMTF9DT05UUk9MLiBUaGUgQWxsVXNlcnMgZ3JvdXAgZ2V0cyBSRUFEIGFjY2Vzcy5cbiAgICAgKi9cbiAgICBQVUJMSUNfUkVBRCA9ICdQdWJsaWNSZWFkJyxcbiAgICAvKipcbiAgICAgKiBPd25lciBnZXRzIEZVTExfQ09OVFJPTC4gVGhlIEFsbFVzZXJzIGdyb3VwIGdldHMgUkVBRCBhbmQgV1JJVEUgYWNjZXNzLlxuICAgICAqIEdyYW50aW5nIHRoaXMgb24gYSBidWNrZXQgaXMgZ2VuZXJhbGx5IG5vdCByZWNvbW1lbmRlZC5cbiAgICAgKi9cbiAgICBQVUJMSUNfUkVBRF9XUklURSA9ICdQdWJsaWNSZWFkV3JpdGUnLFxuICAgIC8qKlxuICAgICAqIE93bmVyIGdldHMgRlVMTF9DT05UUk9MLiBUaGUgQXV0aGVudGljYXRlZFVzZXJzIGdyb3VwIGdldHMgUkVBRCBhY2Nlc3MuXG4gICAgICovXG4gICAgQVVUSEVOVElDQVRFRF9SRUFEID0gJ0F1dGhlbnRpY2F0ZWRSZWFkJyxcbiAgICAvKipcbiAgICAgKiBUaGUgTG9nRGVsaXZlcnkgZ3JvdXAgZ2V0cyBXUklURSBhbmQgUkVBRF9BQ1AgcGVybWlzc2lvbnMgb24gdGhlIGJ1Y2tldC5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1NlcnZlckxvZ3MuaHRtbFxuICAgICAqL1xuICAgIExPR19ERUxJVkVSWV9XUklURSA9ICdMb2dEZWxpdmVyeVdyaXRlJyxcbiAgICAvKipcbiAgICAgKiBPYmplY3Qgb3duZXIgZ2V0cyBGVUxMX0NPTlRST0wuIEJ1Y2tldCBvd25lciBnZXRzIFJFQUQgYWNjZXNzLlxuICAgICAqIElmIHlvdSBzcGVjaWZ5IHRoaXMgY2FubmVkIEFDTCB3aGVuIGNyZWF0aW5nIGEgYnVja2V0LCBBbWF6b24gUzMgaWdub3JlcyBpdC5cbiAgICAgKi9cbiAgICBCVUNLRVRfT1dORVJfUkVBRCA9ICdCdWNrZXRPd25lclJlYWQnLFxuICAgIC8qKlxuICAgICAqIEJvdGggdGhlIG9iamVjdCBvd25lciBhbmQgdGhlIGJ1Y2tldCBvd25lciBnZXQgRlVMTF9DT05UUk9MIG92ZXIgdGhlIG9iamVjdC5cbiAgICAgKiBJZiB5b3Ugc3BlY2lmeSB0aGlzIGNhbm5lZCBBQ0wgd2hlbiBjcmVhdGluZyBhIGJ1Y2tldCwgQW1hem9uIFMzIGlnbm9yZXMgaXQuXG4gICAgICovXG4gICAgQlVDS0VUX09XTkVSX0ZVTExfQ09OVFJPTCA9ICdCdWNrZXRPd25lckZ1bGxDb250cm9sJyxcbiAgICAvKipcbiAgICAgKiBPd25lciBnZXRzIEZVTExfQ09OVFJPTC4gQW1hem9uIEVDMiBnZXRzIFJFQUQgYWNjZXNzIHRvIEdFVCBhbiBBbWF6b24gTWFjaGluZSBJbWFnZSAoQU1JKSBidW5kbGUgZnJvbSBBbWF6b24gUzMuXG4gICAgICovXG4gICAgQVdTX0VYRUNfUkVBRCA9ICdBd3NFeGVjUmVhZCdcbn1cbmV4cG9ydCBpbnRlcmZhY2UgUm91dGluZ1J1bGVDb25kaXRpb24ge1xuICAgIC8qKlxuICAgICAqIFRoZSBIVFRQIGVycm9yIGNvZGUgd2hlbiB0aGUgcmVkaXJlY3QgaXMgYXBwbGllZFxuICAgICAqXG4gICAgICogSW4gdGhlIGV2ZW50IG9mIGFuIGVycm9yLCBpZiB0aGUgZXJyb3IgY29kZSBlcXVhbHMgdGhpcyB2YWx1ZSwgdGhlbiB0aGUgc3BlY2lmaWVkIHJlZGlyZWN0IGlzIGFwcGxpZWQuXG4gICAgICpcbiAgICAgKiBJZiBib3RoIGNvbmRpdGlvbiBwcm9wZXJ0aWVzIGFyZSBzcGVjaWZpZWQsIGJvdGggbXVzdCBiZSB0cnVlIGZvciB0aGUgcmVkaXJlY3QgdG8gYmUgYXBwbGllZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gVGhlIEhUVFAgZXJyb3IgY29kZSB3aWxsIG5vdCBiZSB2ZXJpZmllZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGh0dHBFcnJvckNvZGVSZXR1cm5lZEVxdWFscz86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgb2JqZWN0IGtleSBuYW1lIHByZWZpeCB3aGVuIHRoZSByZWRpcmVjdCBpcyBhcHBsaWVkXG4gICAgICpcbiAgICAgKiBJZiBib3RoIGNvbmRpdGlvbiBwcm9wZXJ0aWVzIGFyZSBzcGVjaWZpZWQsIGJvdGggbXVzdCBiZSB0cnVlIGZvciB0aGUgcmVkaXJlY3QgdG8gYmUgYXBwbGllZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gVGhlIG9iamVjdCBrZXkgbmFtZSB3aWxsIG5vdCBiZSB2ZXJpZmllZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IGtleVByZWZpeEVxdWFscz86IHN0cmluZztcbn1cbmV4cG9ydCBjbGFzcyBSZXBsYWNlS2V5IHtcbiAgICAvKipcbiAgICAgKiBUaGUgc3BlY2lmaWMgb2JqZWN0IGtleSB0byB1c2UgaW4gdGhlIHJlZGlyZWN0IHJlcXVlc3RcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIHdpdGgoa2V5UmVwbGFjZW1lbnQ6IHN0cmluZykge1xuICAgICAgICByZXR1cm4gbmV3IHRoaXMoa2V5UmVwbGFjZW1lbnQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgb2JqZWN0IGtleSBwcmVmaXggdG8gdXNlIGluIHRoZSByZWRpcmVjdCByZXF1ZXN0XG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyBwcmVmaXhXaXRoKGtleVJlcGxhY2VtZW50OiBzdHJpbmcpIHtcbiAgICAgICAgcmV0dXJuIG5ldyB0aGlzKHVuZGVmaW5lZCwga2V5UmVwbGFjZW1lbnQpO1xuICAgIH1cbiAgICBwcml2YXRlIGNvbnN0cnVjdG9yKHB1YmxpYyByZWFkb25seSB3aXRoS2V5Pzogc3RyaW5nLCBwdWJsaWMgcmVhZG9ubHkgcHJlZml4V2l0aEtleT86IHN0cmluZykge1xuICAgIH1cbn1cbi8qKlxuICogUnVsZSB0aGF0IGRlZmluZSB3aGVuIGEgcmVkaXJlY3QgaXMgYXBwbGllZCBhbmQgdGhlIHJlZGlyZWN0IGJlaGF2aW9yLlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvaG93LXRvLXBhZ2UtcmVkaXJlY3QuaHRtbFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJvdXRpbmdSdWxlIHtcbiAgICAvKipcbiAgICAgKiBUaGUgaG9zdCBuYW1lIHRvIHVzZSBpbiB0aGUgcmVkaXJlY3QgcmVxdWVzdFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBUaGUgaG9zdCBuYW1lIHVzZWQgaW4gdGhlIG9yaWdpbmFsIHJlcXVlc3QuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaG9zdE5hbWU/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIEhUVFAgcmVkaXJlY3QgY29kZSB0byB1c2Ugb24gdGhlIHJlc3BvbnNlXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBcIjMwMVwiIC0gTW92ZWQgUGVybWFuZW50bHlcbiAgICAgKi9cbiAgICByZWFkb25seSBodHRwUmVkaXJlY3RDb2RlPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFByb3RvY29sIHRvIHVzZSB3aGVuIHJlZGlyZWN0aW5nIHJlcXVlc3RzXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFRoZSBwcm90b2NvbCB1c2VkIGluIHRoZSBvcmlnaW5hbCByZXF1ZXN0LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByb3RvY29sPzogUmVkaXJlY3RQcm90b2NvbDtcbiAgICAvKipcbiAgICAgKiBTcGVjaWZpZXMgdGhlIG9iamVjdCBrZXkgcHJlZml4IHRvIHVzZSBpbiB0aGUgcmVkaXJlY3QgcmVxdWVzdFxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBUaGUga2V5IHdpbGwgbm90IGJlIHJlcGxhY2VkXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVwbGFjZUtleT86IFJlcGxhY2VLZXk7XG4gICAgLyoqXG4gICAgICogU3BlY2lmaWVzIGEgY29uZGl0aW9uIHRoYXQgbXVzdCBiZSBtZXQgZm9yIHRoZSBzcGVjaWZpZWQgcmVkaXJlY3QgdG8gYXBwbHkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGNvbmRpdGlvblxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbmRpdGlvbj86IFJvdXRpbmdSdWxlQ29uZGl0aW9uO1xufVxuZnVuY3Rpb24gbWFwT3JVbmRlZmluZWQ8VCwgVT4obGlzdDogVFtdIHwgdW5kZWZpbmVkLCBjYWxsYmFjazogKGVsZW1lbnQ6IFQpID0+IFUpOiBVW10gfCB1bmRlZmluZWQge1xuICAgIGlmICghbGlzdCB8fCBsaXN0Lmxlbmd0aCA9PT0gMCkge1xuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICByZXR1cm4gbGlzdC5tYXAoY2FsbGJhY2spO1xufVxuIl19