"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DatabaseClusterEngine = exports.AuroraPostgresEngineVersion = exports.AuroraMysqlEngineVersion = exports.AuroraEngineVersion = void 0;
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const parameter_group_1 = require("./parameter-group");
class ClusterEngineBase {
    constructor(props) {
        this.engineType = props.engineType;
        this.singleUserRotationApplication = props.singleUserRotationApplication;
        this.multiUserRotationApplication = props.multiUserRotationApplication;
        this.defaultPort = props.defaultPort;
        this.engineVersion = props.engineVersion;
        this.parameterGroupFamily = this.engineVersion ? `${this.engineType}${this.engineVersion.majorVersion}` : undefined;
    }
    bindToCluster(scope, options) {
        var _a;
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : this.defaultParameterGroup(scope);
        return {
            parameterGroup,
            port: this.defaultPort,
        };
    }
}
class MySqlClusterEngineBase extends ClusterEngineBase {
    constructor(props) {
        super({
            ...props,
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER,
            engineVersion: props.engineVersion ? props.engineVersion : { majorVersion: props.defaultMajorVersion },
        });
        this.supportedLogTypes = ['error', 'general', 'slowquery', 'audit'];
    }
    bindToCluster(scope, options) {
        var _a;
        const config = super.bindToCluster(scope, options);
        const parameterGroup = (_a = options.parameterGroup) !== null && _a !== void 0 ? _a : (options.s3ImportRole || options.s3ExportRole
            ? new parameter_group_1.ParameterGroup(scope, 'ClusterParameterGroup', {
                engine: this,
            })
            : config.parameterGroup);
        if (options.s3ImportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_load_from_s3_role', options.s3ImportRole.roleArn);
        }
        if (options.s3ExportRole) {
            parameterGroup === null || parameterGroup === void 0 ? void 0 : parameterGroup.addParameter('aurora_select_into_s3_role', options.s3ExportRole.roleArn);
        }
        return {
            ...config,
            parameterGroup,
        };
    }
}
/**
 * The versions for the Aurora cluster engine
 * (those returned by {@link DatabaseClusterEngine.aurora}).
 */
class AuroraEngineVersion {
    constructor(auroraFullVersion, auroraMajorVersion = '5.6') {
        this.auroraFullVersion = auroraFullVersion;
        this.auroraMajorVersion = auroraMajorVersion;
    }
    /**
     * Create a new AuroraEngineVersion with an arbitrary version.
     *
     * @param auroraFullVersion the full version string,
     *   for example "5.6.mysql_aurora.1.78.3.6"
     * @param auroraMajorVersion the major version of the engine,
     *   defaults to "5.6"
     */
    static of(auroraFullVersion, auroraMajorVersion) {
        return new AuroraEngineVersion(auroraFullVersion, auroraMajorVersion);
    }
    static builtIn_5_6(minorVersion, addStandardPrefix = true) {
        return new AuroraEngineVersion(`5.6.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraEngineVersion = AuroraEngineVersion;
/** Version "5.6.10a". */
AuroraEngineVersion.VER_10A = AuroraEngineVersion.builtIn_5_6('10a', false);
/** Version "5.6.mysql_aurora.1.17.9". */
AuroraEngineVersion.VER_1_17_9 = AuroraEngineVersion.builtIn_5_6('1.17.9');
/** Version "5.6.mysql_aurora.1.19.0". */
AuroraEngineVersion.VER_1_19_0 = AuroraEngineVersion.builtIn_5_6('1.19.0');
/** Version "5.6.mysql_aurora.1.19.1". */
AuroraEngineVersion.VER_1_19_1 = AuroraEngineVersion.builtIn_5_6('1.19.1');
/** Version "5.6.mysql_aurora.1.19.2". */
AuroraEngineVersion.VER_1_19_2 = AuroraEngineVersion.builtIn_5_6('1.19.2');
/** Version "5.6.mysql_aurora.1.19.5". */
AuroraEngineVersion.VER_1_19_5 = AuroraEngineVersion.builtIn_5_6('1.19.5');
/** Version "5.6.mysql_aurora.1.19.6". */
AuroraEngineVersion.VER_1_19_6 = AuroraEngineVersion.builtIn_5_6('1.19.6');
/** Version "5.6.mysql_aurora.1.20.0". */
AuroraEngineVersion.VER_1_20_0 = AuroraEngineVersion.builtIn_5_6('1.20.0');
/** Version "5.6.mysql_aurora.1.20.1". */
AuroraEngineVersion.VER_1_20_1 = AuroraEngineVersion.builtIn_5_6('1.20.1');
/** Version "5.6.mysql_aurora.1.21.0". */
AuroraEngineVersion.VER_1_21_0 = AuroraEngineVersion.builtIn_5_6('1.21.0');
/** Version "5.6.mysql_aurora.1.22.0". */
AuroraEngineVersion.VER_1_22_0 = AuroraEngineVersion.builtIn_5_6('1.22.0');
/** Version "5.6.mysql_aurora.1.22.1". */
AuroraEngineVersion.VER_1_22_1 = AuroraEngineVersion.builtIn_5_6('1.22.1');
/** Version "5.6.mysql_aurora.1.22.1.3". */
AuroraEngineVersion.VER_1_22_1_3 = AuroraEngineVersion.builtIn_5_6('1.22.1.3');
/** Version "5.6.mysql_aurora.1.22.2". */
AuroraEngineVersion.VER_1_22_2 = AuroraEngineVersion.builtIn_5_6('1.22.2');
class AuroraClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora',
            engineVersion: version
                ? {
                    fullVersion: version.auroraFullVersion,
                    majorVersion: version.auroraMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.6',
        });
    }
    defaultParameterGroup(_scope) {
        // the default.aurora5.6 ParameterGroup is actually the default,
        // so just return undefined in this case
        return undefined;
    }
}
/**
 * The versions for the Aurora MySQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraMysql}).
 */
class AuroraMysqlEngineVersion {
    constructor(auroraMysqlFullVersion, auroraMysqlMajorVersion = '5.7') {
        this.auroraMysqlFullVersion = auroraMysqlFullVersion;
        this.auroraMysqlMajorVersion = auroraMysqlMajorVersion;
    }
    /**
     * Create a new AuroraMysqlEngineVersion with an arbitrary version.
     *
     * @param auroraMysqlFullVersion the full version string,
     *   for example "5.7.mysql_aurora.2.78.3.6"
     * @param auroraMysqlMajorVersion the major version of the engine,
     *   defaults to "5.7"
     */
    static of(auroraMysqlFullVersion, auroraMysqlMajorVersion) {
        return new AuroraMysqlEngineVersion(auroraMysqlFullVersion, auroraMysqlMajorVersion);
    }
    static builtIn_5_7(minorVersion, addStandardPrefix = true) {
        return new AuroraMysqlEngineVersion(`5.7.${addStandardPrefix ? 'mysql_aurora.' : ''}${minorVersion}`);
    }
}
exports.AuroraMysqlEngineVersion = AuroraMysqlEngineVersion;
/** Version "5.7.12". */
AuroraMysqlEngineVersion.VER_5_7_12 = AuroraMysqlEngineVersion.builtIn_5_7('12', false);
/** Version "5.7.mysql_aurora.2.03.2". */
AuroraMysqlEngineVersion.VER_2_03_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.2');
/** Version "5.7.mysql_aurora.2.03.3". */
AuroraMysqlEngineVersion.VER_2_03_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.3');
/** Version "5.7.mysql_aurora.2.03.4". */
AuroraMysqlEngineVersion.VER_2_03_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.03.4');
/** Version "5.7.mysql_aurora.2.04.0". */
AuroraMysqlEngineVersion.VER_2_04_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.0');
/** Version "5.7.mysql_aurora.2.04.1". */
AuroraMysqlEngineVersion.VER_2_04_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.1');
/** Version "5.7.mysql_aurora.2.04.2". */
AuroraMysqlEngineVersion.VER_2_04_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.2');
/** Version "5.7.mysql_aurora.2.04.3". */
AuroraMysqlEngineVersion.VER_2_04_3 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.3');
/** Version "5.7.mysql_aurora.2.04.4". */
AuroraMysqlEngineVersion.VER_2_04_4 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.4');
/** Version "5.7.mysql_aurora.2.04.5". */
AuroraMysqlEngineVersion.VER_2_04_5 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.5');
/** Version "5.7.mysql_aurora.2.04.6". */
AuroraMysqlEngineVersion.VER_2_04_6 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.6');
/** Version "5.7.mysql_aurora.2.04.7". */
AuroraMysqlEngineVersion.VER_2_04_7 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.7');
/** Version "5.7.mysql_aurora.2.04.8". */
AuroraMysqlEngineVersion.VER_2_04_8 = AuroraMysqlEngineVersion.builtIn_5_7('2.04.8');
/** Version "5.7.mysql_aurora.2.05.0". */
AuroraMysqlEngineVersion.VER_2_05_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.05.0');
/** Version "5.7.mysql_aurora.2.06.0". */
AuroraMysqlEngineVersion.VER_2_06_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.06.0');
/** Version "5.7.mysql_aurora.2.07.0". */
AuroraMysqlEngineVersion.VER_2_07_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.0');
/** Version "5.7.mysql_aurora.2.07.1". */
AuroraMysqlEngineVersion.VER_2_07_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.1');
/** Version "5.7.mysql_aurora.2.07.2". */
AuroraMysqlEngineVersion.VER_2_07_2 = AuroraMysqlEngineVersion.builtIn_5_7('2.07.2');
/** Version "5.7.mysql_aurora.2.08.0". */
AuroraMysqlEngineVersion.VER_2_08_0 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.0');
/** Version "5.7.mysql_aurora.2.08.1". */
AuroraMysqlEngineVersion.VER_2_08_1 = AuroraMysqlEngineVersion.builtIn_5_7('2.08.1');
class AuroraMysqlClusterEngine extends MySqlClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-mysql',
            engineVersion: version
                ? {
                    fullVersion: version.auroraMysqlFullVersion,
                    majorVersion: version.auroraMysqlMajorVersion,
                }
                : undefined,
            defaultMajorVersion: '5.7',
        });
    }
    defaultParameterGroup(scope) {
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraMySqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * The versions for the Aurora PostgreSQL cluster engine
 * (those returned by {@link DatabaseClusterEngine.auroraPostgres}).
 */
class AuroraPostgresEngineVersion {
    constructor(auroraPostgresFullVersion, auroraPostgresMajorVersion) {
        this.auroraPostgresFullVersion = auroraPostgresFullVersion;
        this.auroraPostgresMajorVersion = auroraPostgresMajorVersion;
    }
    /**
     * Create a new AuroraPostgresEngineVersion with an arbitrary version.
     *
     * @param auroraPostgresFullVersion the full version string,
     *   for example "9.6.25.1"
     * @param auroraPostgresMajorVersion the major version of the engine,
     *   for example "9.6"
     */
    static of(auroraPostgresFullVersion, auroraPostgresMajorVersion) {
        return new AuroraPostgresEngineVersion(auroraPostgresFullVersion, auroraPostgresMajorVersion);
    }
}
exports.AuroraPostgresEngineVersion = AuroraPostgresEngineVersion;
/** Version "9.6.8". */
AuroraPostgresEngineVersion.VER_9_6_8 = AuroraPostgresEngineVersion.of('9.6.8', '9.6');
/** Version "9.6.9". */
AuroraPostgresEngineVersion.VER_9_6_9 = AuroraPostgresEngineVersion.of('9.6.9', '9.6');
/** Version "9.6.11". */
AuroraPostgresEngineVersion.VER_9_6_11 = AuroraPostgresEngineVersion.of('9.6.11', '9.6');
/** Version "9.6.12". */
AuroraPostgresEngineVersion.VER_9_6_12 = AuroraPostgresEngineVersion.of('9.6.12', '9.6');
/** Version "9.6.16". */
AuroraPostgresEngineVersion.VER_9_6_16 = AuroraPostgresEngineVersion.of('9.6.16', '9.6');
/** Version "9.6.17". */
AuroraPostgresEngineVersion.VER_9_6_17 = AuroraPostgresEngineVersion.of('9.6.17', '9.6');
/** Version "10.4". */
AuroraPostgresEngineVersion.VER_10_4 = AuroraPostgresEngineVersion.of('10.4', '10');
/** Version "10.5". */
AuroraPostgresEngineVersion.VER_10_5 = AuroraPostgresEngineVersion.of('10.5', '10');
/** Version "10.6". */
AuroraPostgresEngineVersion.VER_10_6 = AuroraPostgresEngineVersion.of('10.6', '10');
/** Version "10.7". */
AuroraPostgresEngineVersion.VER_10_7 = AuroraPostgresEngineVersion.of('10.7', '10');
/** Version "10.11". */
AuroraPostgresEngineVersion.VER_10_11 = AuroraPostgresEngineVersion.of('10.11', '10');
/** Version "10.12". */
AuroraPostgresEngineVersion.VER_10_12 = AuroraPostgresEngineVersion.of('10.12', '10');
/** Version "11.4". */
AuroraPostgresEngineVersion.VER_11_4 = AuroraPostgresEngineVersion.of('11.4', '11');
/** Version "11.6". */
AuroraPostgresEngineVersion.VER_11_6 = AuroraPostgresEngineVersion.of('11.6', '11');
/** Version "11.7". */
AuroraPostgresEngineVersion.VER_11_7 = AuroraPostgresEngineVersion.of('11.7', '11');
class AuroraPostgresClusterEngine extends ClusterEngineBase {
    constructor(version) {
        super({
            engineType: 'aurora-postgresql',
            singleUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER,
            multiUserRotationApplication: secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER,
            defaultPort: 5432,
            engineVersion: version
                ? {
                    fullVersion: version.auroraPostgresFullVersion,
                    majorVersion: version.auroraPostgresMajorVersion,
                }
                : undefined,
        });
        this.supportedLogTypes = ['postgresql'];
    }
    defaultParameterGroup(scope) {
        if (!this.parameterGroupFamily) {
            throw new Error('Could not create a new ParameterGroup for an unversioned aurora-postgresql cluster engine. ' +
                'Please either use a versioned engine, or pass an explicit ParameterGroup when creating the cluster');
        }
        return parameter_group_1.ParameterGroup.fromParameterGroupName(scope, 'AuroraPostgreSqlDatabaseClusterEngineDefaultParameterGroup', `default.${this.parameterGroupFamily}`);
    }
}
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    /** Creates a new plain Aurora database cluster engine. */
    static aurora(props) {
        return new AuroraClusterEngine(props.version);
    }
    /** Creates a new Aurora MySQL database cluster engine. */
    static auroraMysql(props) {
        return new AuroraMysqlClusterEngine(props.version);
    }
    /** Creates a new Aurora PostgreSQL database cluster engine. */
    static auroraPostgres(props) {
        return new AuroraPostgresClusterEngine(props.version);
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/**
 * The unversioned 'aurora' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link aurora()} method
 */
DatabaseClusterEngine.AURORA = new AuroraClusterEngine();
/**
 * The unversioned 'aurora-msql' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link auroraMysql()} method
 */
DatabaseClusterEngine.AURORA_MYSQL = new AuroraMysqlClusterEngine();
/**
 * The unversioned 'aurora-postgresql' cluster engine.
 *
 * @deprecated using unversioned engines is an availability risk.
 *   We recommend using versioned engines created using the {@link auroraPostgres()} method
 */
DatabaseClusterEngine.AURORA_POSTGRESQL = new AuroraPostgresClusterEngine();
//# sourceMappingURL=data:application/json;base64,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