"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const bundling_1 = require("./bundling");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using Parcel
 */
class NodejsFunction extends lambda.Function {
    constructor(scope, id, props = {}) {
        var _a, _b, _c;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        const handler = (_a = props.handler) !== null && _a !== void 0 ? _a : 'handler';
        const defaultRunTime = util_1.nodeMajorVersion() >= 12
            ? lambda.Runtime.NODEJS_12_X
            : lambda.Runtime.NODEJS_10_X;
        const runtime = (_b = props.runtime) !== null && _b !== void 0 ? _b : defaultRunTime;
        // Look for the closest package.json starting in the directory of the entry
        // file. We need to restore it after bundling.
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(entry));
        try {
            super(scope, id, {
                ...props,
                runtime,
                code: bundling_1.Bundling.parcel({
                    ...props,
                    entry,
                    runtime,
                }),
                handler: `index.${handler}`,
            });
            // Enable connection reuse for aws-sdk
            if ((_c = props.awsSdkConnectionReuse) !== null && _c !== void 0 ? _c : true) {
                this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
            }
        }
        finally {
            // We can only restore after the code has been bound to the function
            packageJsonManager.restore();
        }
    }
}
exports.NodejsFunction = NodejsFunction;
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    const stackTrace = util_1.parseStackTrace();
    const functionIndex = stackTrace.findIndex(s => /NodejsFunction/.test(s.methodName || ''));
    if (functionIndex === -1 || !stackTrace[functionIndex + 1]) {
        throw new Error('Cannot find defining file.');
    }
    return stackTrace[functionIndex + 1].file;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZnVuY3Rpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJmdW5jdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLDJDQUEyQyxDQUFDLHNEQUFzRDtBQUVsRyx5Q0FBeUQ7QUFDekQsaUVBQTREO0FBQzVELGlDQUEyRDtBQXlDM0Q7O0dBRUc7QUFDSCxNQUFhLGNBQWUsU0FBUSxNQUFNLENBQUMsUUFBUTtJQUMvQyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLFFBQTZCLEVBQUU7O1FBQ3pFLElBQUksS0FBSyxDQUFDLE9BQU8sSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sS0FBSyxNQUFNLENBQUMsYUFBYSxDQUFDLE1BQU0sRUFBRTtZQUN2RSxNQUFNLElBQUksS0FBSyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7U0FDNUQ7UUFDRCxxQkFBcUI7UUFDckIsTUFBTSxLQUFLLEdBQUcsSUFBSSxDQUFDLE9BQU8sQ0FBQyxTQUFTLENBQUMsRUFBRSxFQUFFLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1FBQ3ZELE1BQU0sT0FBTyxTQUFHLEtBQUssQ0FBQyxPQUFPLG1DQUFJLFNBQVMsQ0FBQztRQUMzQyxNQUFNLGNBQWMsR0FBRyx1QkFBZ0IsRUFBRSxJQUFJLEVBQUU7WUFDM0MsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztZQUM1QixDQUFDLENBQUMsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUM7UUFDakMsTUFBTSxPQUFPLFNBQUcsS0FBSyxDQUFDLE9BQU8sbUNBQUksY0FBYyxDQUFDO1FBQ2hELDJFQUEyRTtRQUMzRSw4Q0FBOEM7UUFDOUMsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLHlDQUFrQixDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztRQUN2RSxJQUFJO1lBQ0EsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7Z0JBQ2IsR0FBRyxLQUFLO2dCQUNSLE9BQU87Z0JBQ1AsSUFBSSxFQUFFLG1CQUFRLENBQUMsTUFBTSxDQUFDO29CQUNsQixHQUFHLEtBQUs7b0JBQ1IsS0FBSztvQkFDTCxPQUFPO2lCQUNWLENBQUM7Z0JBQ0YsT0FBTyxFQUFFLFNBQVMsT0FBTyxFQUFFO2FBQzlCLENBQUMsQ0FBQztZQUNILHNDQUFzQztZQUN0QyxVQUFJLEtBQUssQ0FBQyxxQkFBcUIsbUNBQUksSUFBSSxFQUFFO2dCQUNyQyxJQUFJLENBQUMsY0FBYyxDQUFDLHFDQUFxQyxFQUFFLEdBQUcsRUFBRSxFQUFFLFlBQVksRUFBRSxJQUFJLEVBQUUsQ0FBQyxDQUFDO2FBQzNGO1NBQ0o7Z0JBQ087WUFDSixvRUFBb0U7WUFDcEUsa0JBQWtCLENBQUMsT0FBTyxFQUFFLENBQUM7U0FDaEM7SUFDTCxDQUFDO0NBQ0o7QUFwQ0Qsd0NBb0NDO0FBQ0Q7Ozs7O0dBS0c7QUFDSCxTQUFTLFNBQVMsQ0FBQyxFQUFVLEVBQUUsS0FBYztJQUN6QyxJQUFJLEtBQUssRUFBRTtRQUNQLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDL0IsTUFBTSxJQUFJLEtBQUssQ0FBQywwREFBMEQsQ0FBQyxDQUFDO1NBQy9FO1FBQ0QsSUFBSSxDQUFDLEVBQUUsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyw2QkFBNkIsS0FBSyxFQUFFLENBQUMsQ0FBQztTQUN6RDtRQUNELE9BQU8sS0FBSyxDQUFDO0tBQ2hCO0lBQ0QsTUFBTSxZQUFZLEdBQUcsZ0JBQWdCLEVBQUUsQ0FBQztJQUN4QyxNQUFNLE9BQU8sR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQzNDLE1BQU0sYUFBYSxHQUFHLFlBQVksQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsR0FBRyxPQUFPLEdBQUcsQ0FBQyxFQUFFLElBQUksRUFBRSxLQUFLLENBQUMsQ0FBQztJQUNuRixJQUFJLEVBQUUsQ0FBQyxVQUFVLENBQUMsYUFBYSxDQUFDLEVBQUU7UUFDOUIsT0FBTyxhQUFhLENBQUM7S0FDeEI7SUFDRCxNQUFNLGFBQWEsR0FBRyxZQUFZLENBQUMsT0FBTyxDQUFDLElBQUksTUFBTSxDQUFDLEdBQUcsT0FBTyxHQUFHLENBQUMsRUFBRSxJQUFJLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDbkYsSUFBSSxFQUFFLENBQUMsVUFBVSxDQUFDLGFBQWEsQ0FBQyxFQUFFO1FBQzlCLE9BQU8sYUFBYSxDQUFDO0tBQ3hCO0lBQ0QsTUFBTSxJQUFJLEtBQUssQ0FBQyx5QkFBeUIsQ0FBQyxDQUFDO0FBQy9DLENBQUM7QUFDRDs7R0FFRztBQUNILFNBQVMsZ0JBQWdCO0lBQ3JCLE1BQU0sVUFBVSxHQUFHLHNCQUFlLEVBQUUsQ0FBQztJQUNyQyxNQUFNLGFBQWEsR0FBRyxVQUFVLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsZ0JBQWdCLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxVQUFVLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQztJQUMzRixJQUFJLGFBQWEsS0FBSyxDQUFDLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxhQUFhLEdBQUcsQ0FBQyxDQUFDLEVBQUU7UUFDeEQsTUFBTSxJQUFJLEtBQUssQ0FBQyw0QkFBNEIsQ0FBQyxDQUFDO0tBQ2pEO0lBQ0QsT0FBTyxVQUFVLENBQUMsYUFBYSxHQUFHLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQztBQUM5QyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tIFwiLi4vLi4vYXdzLWxhbWJkYVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSdcbmltcG9ydCAqIGFzIGNkayBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IEJ1bmRsaW5nLCBQYXJjZWxCYXNlT3B0aW9ucyB9IGZyb20gJy4vYnVuZGxpbmcnO1xuaW1wb3J0IHsgUGFja2FnZUpzb25NYW5hZ2VyIH0gZnJvbSAnLi9wYWNrYWdlLWpzb24tbWFuYWdlcic7XG5pbXBvcnQgeyBub2RlTWFqb3JWZXJzaW9uLCBwYXJzZVN0YWNrVHJhY2UgfSBmcm9tICcuL3V0aWwnO1xuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhIE5vZGVqc0Z1bmN0aW9uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTm9kZWpzRnVuY3Rpb25Qcm9wcyBleHRlbmRzIGxhbWJkYS5GdW5jdGlvbk9wdGlvbnMsIFBhcmNlbEJhc2VPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBQYXRoIHRvIHRoZSBlbnRyeSBmaWxlIChKYXZhU2NyaXB0IG9yIFR5cGVTY3JpcHQpLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBEZXJpdmVkIGZyb20gdGhlIG5hbWUgb2YgdGhlIGRlZmluaW5nIGZpbGUgYW5kIHRoZSBjb25zdHJ1Y3QncyBpZC5cbiAgICAgKiBJZiB0aGUgYE5vZGVqc0Z1bmN0aW9uYCBpcyBkZWZpbmVkIGluIGBzdGFjay50c2Agd2l0aCBgbXktaGFuZGxlcmAgYXMgaWRcbiAgICAgKiAoYG5ldyBOb2RlanNGdW5jdGlvbih0aGlzLCAnbXktaGFuZGxlcicpYCksIHRoZSBjb25zdHJ1Y3Qgd2lsbCBsb29rIGF0IGBzdGFjay5teS1oYW5kbGVyLnRzYFxuICAgICAqIGFuZCBgc3RhY2subXktaGFuZGxlci5qc2AuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZW50cnk/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIG5hbWUgb2YgdGhlIGV4cG9ydGVkIGhhbmRsZXIgaW4gdGhlIGVudHJ5IGZpbGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBoYW5kbGVyXG4gICAgICovXG4gICAgcmVhZG9ubHkgaGFuZGxlcj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgcnVudGltZSBlbnZpcm9ubWVudC4gT25seSBydW50aW1lcyBvZiB0aGUgTm9kZS5qcyBmYW1pbHkgYXJlXG4gICAgICogc3VwcG9ydGVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBgTk9ERUpTXzEyX1hgIGlmIGBwcm9jZXNzLnZlcnNpb25zLm5vZGVgID49ICcxMi4wLjAnLFxuICAgICAqIGBOT0RFSlNfMTBfWGAgb3RoZXJ3aXNlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJ1bnRpbWU/OiBsYW1iZGEuUnVudGltZTtcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIHRvIGF1dG9tYXRpY2FsbHkgcmV1c2UgVENQIGNvbm5lY3Rpb25zIHdoZW4gd29ya2luZyB3aXRoIHRoZSBBV1NcbiAgICAgKiBTREsgZm9yIEphdmFTY3JpcHQuXG4gICAgICpcbiAgICAgKiBUaGlzIHNldHMgdGhlIGBBV1NfTk9ERUpTX0NPTk5FQ1RJT05fUkVVU0VfRU5BQkxFRGAgZW52aXJvbm1lbnQgdmFyaWFibGVcbiAgICAgKiB0byBgMWAuXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zZGstZm9yLWphdmFzY3JpcHQvdjIvZGV2ZWxvcGVyLWd1aWRlL25vZGUtcmV1c2luZy1jb25uZWN0aW9ucy5odG1sXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCB0cnVlXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXdzU2RrQ29ubmVjdGlvblJldXNlPzogYm9vbGVhbjtcbn1cbi8qKlxuICogQSBOb2RlLmpzIExhbWJkYSBmdW5jdGlvbiBidW5kbGVkIHVzaW5nIFBhcmNlbFxuICovXG5leHBvcnQgY2xhc3MgTm9kZWpzRnVuY3Rpb24gZXh0ZW5kcyBsYW1iZGEuRnVuY3Rpb24ge1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogTm9kZWpzRnVuY3Rpb25Qcm9wcyA9IHt9KSB7XG4gICAgICAgIGlmIChwcm9wcy5ydW50aW1lICYmIHByb3BzLnJ1bnRpbWUuZmFtaWx5ICE9PSBsYW1iZGEuUnVudGltZUZhbWlseS5OT0RFSlMpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignT25seSBgTk9ERUpTYCBydW50aW1lcyBhcmUgc3VwcG9ydGVkLicpO1xuICAgICAgICB9XG4gICAgICAgIC8vIEVudHJ5IGFuZCBkZWZhdWx0c1xuICAgICAgICBjb25zdCBlbnRyeSA9IHBhdGgucmVzb2x2ZShmaW5kRW50cnkoaWQsIHByb3BzLmVudHJ5KSk7XG4gICAgICAgIGNvbnN0IGhhbmRsZXIgPSBwcm9wcy5oYW5kbGVyID8/ICdoYW5kbGVyJztcbiAgICAgICAgY29uc3QgZGVmYXVsdFJ1blRpbWUgPSBub2RlTWFqb3JWZXJzaW9uKCkgPj0gMTJcbiAgICAgICAgICAgID8gbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1hcbiAgICAgICAgICAgIDogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1g7XG4gICAgICAgIGNvbnN0IHJ1bnRpbWUgPSBwcm9wcy5ydW50aW1lID8/IGRlZmF1bHRSdW5UaW1lO1xuICAgICAgICAvLyBMb29rIGZvciB0aGUgY2xvc2VzdCBwYWNrYWdlLmpzb24gc3RhcnRpbmcgaW4gdGhlIGRpcmVjdG9yeSBvZiB0aGUgZW50cnlcbiAgICAgICAgLy8gZmlsZS4gV2UgbmVlZCB0byByZXN0b3JlIGl0IGFmdGVyIGJ1bmRsaW5nLlxuICAgICAgICBjb25zdCBwYWNrYWdlSnNvbk1hbmFnZXIgPSBuZXcgUGFja2FnZUpzb25NYW5hZ2VyKHBhdGguZGlybmFtZShlbnRyeSkpO1xuICAgICAgICB0cnkge1xuICAgICAgICAgICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgICAgICAgICAgcnVudGltZSxcbiAgICAgICAgICAgICAgICBjb2RlOiBCdW5kbGluZy5wYXJjZWwoe1xuICAgICAgICAgICAgICAgICAgICAuLi5wcm9wcyxcbiAgICAgICAgICAgICAgICAgICAgZW50cnksXG4gICAgICAgICAgICAgICAgICAgIHJ1bnRpbWUsXG4gICAgICAgICAgICAgICAgfSksXG4gICAgICAgICAgICAgICAgaGFuZGxlcjogYGluZGV4LiR7aGFuZGxlcn1gLFxuICAgICAgICAgICAgfSk7XG4gICAgICAgICAgICAvLyBFbmFibGUgY29ubmVjdGlvbiByZXVzZSBmb3IgYXdzLXNka1xuICAgICAgICAgICAgaWYgKHByb3BzLmF3c1Nka0Nvbm5lY3Rpb25SZXVzZSA/PyB0cnVlKSB7XG4gICAgICAgICAgICAgICAgdGhpcy5hZGRFbnZpcm9ubWVudCgnQVdTX05PREVKU19DT05ORUNUSU9OX1JFVVNFX0VOQUJMRUQnLCAnMScsIHsgcmVtb3ZlSW5FZGdlOiB0cnVlIH0pO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGZpbmFsbHkge1xuICAgICAgICAgICAgLy8gV2UgY2FuIG9ubHkgcmVzdG9yZSBhZnRlciB0aGUgY29kZSBoYXMgYmVlbiBib3VuZCB0byB0aGUgZnVuY3Rpb25cbiAgICAgICAgICAgIHBhY2thZ2VKc29uTWFuYWdlci5yZXN0b3JlKCk7XG4gICAgICAgIH1cbiAgICB9XG59XG4vKipcbiAqIFNlYXJjaGVzIGZvciBhbiBlbnRyeSBmaWxlLiBQcmVmZXJlbmNlIG9yZGVyIGlzIHRoZSBmb2xsb3dpbmc6XG4gKiAxLiBHaXZlbiBlbnRyeSBmaWxlXG4gKiAyLiBBIC50cyBmaWxlIG5hbWVkIGFzIHRoZSBkZWZpbmluZyBmaWxlIHdpdGggaWQgYXMgc3VmZml4IChkZWZpbmluZy1maWxlLmlkLnRzKVxuICogMy4gQSAuanMgZmlsZSBuYW1lIGFzIHRoZSBkZWZpbmluZyBmaWxlIHdpdGggaWQgYXMgc3VmZml4IChkZWZpbmluZy1maWxlLmlkLmpzKVxuICovXG5mdW5jdGlvbiBmaW5kRW50cnkoaWQ6IHN0cmluZywgZW50cnk/OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgIGlmIChlbnRyeSkge1xuICAgICAgICBpZiAoIS9cXC4oanN4P3x0c3g/KSQvLnRlc3QoZW50cnkpKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ09ubHkgSmF2YVNjcmlwdCBvciBUeXBlU2NyaXB0IGVudHJ5IGZpbGVzIGFyZSBzdXBwb3J0ZWQuJyk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKCFmcy5leGlzdHNTeW5jKGVudHJ5KSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3QgZmluZCBlbnRyeSBmaWxlIGF0ICR7ZW50cnl9YCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIGVudHJ5O1xuICAgIH1cbiAgICBjb25zdCBkZWZpbmluZ0ZpbGUgPSBmaW5kRGVmaW5pbmdGaWxlKCk7XG4gICAgY29uc3QgZXh0bmFtZSA9IHBhdGguZXh0bmFtZShkZWZpbmluZ0ZpbGUpO1xuICAgIGNvbnN0IHRzSGFuZGxlckZpbGUgPSBkZWZpbmluZ0ZpbGUucmVwbGFjZShuZXcgUmVnRXhwKGAke2V4dG5hbWV9JGApLCBgLiR7aWR9LnRzYCk7XG4gICAgaWYgKGZzLmV4aXN0c1N5bmModHNIYW5kbGVyRmlsZSkpIHtcbiAgICAgICAgcmV0dXJuIHRzSGFuZGxlckZpbGU7XG4gICAgfVxuICAgIGNvbnN0IGpzSGFuZGxlckZpbGUgPSBkZWZpbmluZ0ZpbGUucmVwbGFjZShuZXcgUmVnRXhwKGAke2V4dG5hbWV9JGApLCBgLiR7aWR9LmpzYCk7XG4gICAgaWYgKGZzLmV4aXN0c1N5bmMoanNIYW5kbGVyRmlsZSkpIHtcbiAgICAgICAgcmV0dXJuIGpzSGFuZGxlckZpbGU7XG4gICAgfVxuICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGZpbmQgZW50cnkgZmlsZS4nKTtcbn1cbi8qKlxuICogRmluZHMgdGhlIG5hbWUgb2YgdGhlIGZpbGUgd2hlcmUgdGhlIGBOb2RlanNGdW5jdGlvbmAgaXMgZGVmaW5lZFxuICovXG5mdW5jdGlvbiBmaW5kRGVmaW5pbmdGaWxlKCk6IHN0cmluZyB7XG4gICAgY29uc3Qgc3RhY2tUcmFjZSA9IHBhcnNlU3RhY2tUcmFjZSgpO1xuICAgIGNvbnN0IGZ1bmN0aW9uSW5kZXggPSBzdGFja1RyYWNlLmZpbmRJbmRleChzID0+IC9Ob2RlanNGdW5jdGlvbi8udGVzdChzLm1ldGhvZE5hbWUgfHwgJycpKTtcbiAgICBpZiAoZnVuY3Rpb25JbmRleCA9PT0gLTEgfHwgIXN0YWNrVHJhY2VbZnVuY3Rpb25JbmRleCArIDFdKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IGZpbmQgZGVmaW5pbmcgZmlsZS4nKTtcbiAgICB9XG4gICAgcmV0dXJuIHN0YWNrVHJhY2VbZnVuY3Rpb25JbmRleCArIDFdLmZpbGU7XG59XG4iXX0=