"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const bundlers_1 = require("./bundlers");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        // Find project root
        const projectRoot = (_d = (_c = (_b = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`)) !== null && _b !== void 0 ? _b : util_1.findUp(bundlers_1.LockFile.YARN)) !== null && _c !== void 0 ? _c : util_1.findUp(bundlers_1.LockFile.NPM)) !== null && _d !== void 0 ? _d : util_1.findUp('package.json');
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        const relativeEntryPath = path.relative(projectRoot, path.resolve(options.entry));
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_e = options.externalModules) !== null && _e !== void 0 ? _e : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_f = options.nodeModules) !== null && _f !== void 0 ? _f : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        let installer = bundlers_1.Installer.NPM;
        let lockFile;
        if (dependencies) {
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.YARN))) {
                installer = bundlers_1.Installer.YARN;
                lockFile = bundlers_1.LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.NPM))) {
                lockFile = bundlers_1.LockFile.NPM;
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            targets: {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_g = options.sourceMaps) !== null && _g !== void 0 ? _g : false,
                    minify: (_h = options.minify) !== null && _h !== void 0 ? _h : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Local
        let localBundler;
        if (!options.forceDockerBundling) {
            localBundler = new bundlers_1.LocalBundler({
                projectRoot,
                relativeEntryPath,
                cacheDir: options.cacheDir,
                environment: options.parcelEnvironment,
                dependencies,
                installer,
                lockFile,
            });
        }
        // Docker
        const dockerBundler = new bundlers_1.DockerBundler({
            runtime: options.runtime,
            relativeEntryPath,
            cacheDir: options.cacheDir,
            environment: options.parcelEnvironment,
            buildImage: !bundlers_1.LocalBundler.runsLocally || options.forceDockerBundling,
            buildArgs: options.buildArgs,
            parcelVersion: options.parcelVersion,
            dependencies,
            installer,
            lockFile,
        });
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                local: localBundler,
                ...dockerBundler.bundlingOptions,
            },
        });
    }
}
exports.Bundling = Bundling;
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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