"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LockFile = exports.Installer = exports.DockerBundler = exports.LocalBundler = void 0;
const child_process_1 = require("child_process");
const os = require("os");
const path = require("path");
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const util_1 = require("./util");
const PARCEL_VERSION = '2.0.0-beta.1';
/**
 * Local Parcel bundler
 */
class LocalBundler {
    constructor(props) {
        this.props = props;
    }
    static get runsLocally() {
        if (LocalBundler._runsLocally !== undefined) {
            return LocalBundler._runsLocally;
        }
        if (os.platform() === 'win32') { // TODO: add Windows support
            return false;
        }
        try {
            const parcel = child_process_1.spawnSync(require.resolve('parcel'), ['--version']);
            const version = parcel.stdout.toString().trim();
            LocalBundler._runsLocally = new RegExp(`^${PARCEL_VERSION}`).test(version); // Cache result to avoid unnecessary spawns
            if (!LocalBundler._runsLocally) {
                process.stderr.write(`Incorrect parcel version detected: ${version} <> ${PARCEL_VERSION}. Switching to Docker bundling.\n`);
            }
            return LocalBundler._runsLocally;
        }
        catch (_a) {
            return false;
        }
    }
    tryBundle(outputDir) {
        var _a;
        if (!LocalBundler.runsLocally) {
            return false;
        }
        const localCommand = createBundlingCommand({
            projectRoot: this.props.projectRoot,
            relativeEntryPath: this.props.relativeEntryPath,
            cacheDir: this.props.cacheDir,
            outputDir,
            dependencies: this.props.dependencies,
            installer: this.props.installer,
            lockFile: this.props.lockFile,
        });
        util_1.exec('bash', ['-c', localCommand], {
            env: { ...process.env, ...(_a = this.props.environment) !== null && _a !== void 0 ? _a : {} },
            stdio: [
                'ignore',
                process.stderr,
                'inherit',
            ],
            cwd: path.dirname(path.join(this.props.projectRoot, this.props.relativeEntryPath)),
        });
        return true;
    }
}
exports.LocalBundler = LocalBundler;
/**
 * Docker bundler
 */
class DockerBundler {
    constructor(props) {
        var _a, _b;
        const image = props.buildImage
            ? cdk.BundlingDockerImage.fromAsset(path.join(__dirname, '../parcel'), {
                buildArgs: {
                    ...(_a = props.buildArgs) !== null && _a !== void 0 ? _a : {},
                    IMAGE: props.runtime.bundlingDockerImage.image,
                    PARCEL_VERSION: (_b = props.parcelVersion) !== null && _b !== void 0 ? _b : PARCEL_VERSION,
                },
            })
            : cdk.BundlingDockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const command = createBundlingCommand({
            projectRoot: cdk.AssetStaging.BUNDLING_INPUT_DIR,
            relativeEntryPath: props.relativeEntryPath,
            cacheDir: props.cacheDir,
            outputDir: cdk.AssetStaging.BUNDLING_OUTPUT_DIR,
            installer: props.installer,
            lockFile: props.lockFile,
            dependencies: props.dependencies,
        });
        this.bundlingOptions = {
            image,
            command: ['bash', '-c', command],
            environment: props.environment,
            workingDirectory: path.dirname(path.join(cdk.AssetStaging.BUNDLING_INPUT_DIR, props.relativeEntryPath)).replace(/\\/g, '/'),
        };
    }
}
exports.DockerBundler = DockerBundler;
/**
 * Generates bundling command
 */
function createBundlingCommand(options) {
    const entryPath = path.join(options.projectRoot, options.relativeEntryPath);
    const distFile = path.basename(options.relativeEntryPath).replace(/\.(jsx|tsx?)$/, '.js');
    const parcelCommand = chain([
        [
            '$(node -p "require.resolve(\'parcel\')")',
            'build', entryPath.replace(/\\/g, '/'),
            '--target', 'cdk-lambda',
            '--dist-dir', options.outputDir,
            '--no-autoinstall',
            '--no-scope-hoist',
            ...options.cacheDir
                ? ['--cache-dir', path.join(options.projectRoot, options.cacheDir)]
                : [],
        ].join(' '),
        // Always rename dist file to index.js because Lambda doesn't support filenames
        // with multiple dots and we can end up with multiple dots when using automatic
        // entry lookup
        distFile !== 'index.js' ? `mv ${options.outputDir}/${distFile} ${options.outputDir}/index.js` : '',
    ]);
    let depsCommand = '';
    if (options.dependencies) {
        // create dummy package.json, copy lock file if any and then install
        depsCommand = chain([
            `echo '${JSON.stringify({ dependencies: options.dependencies })}' > ${options.outputDir}/package.json`,
            options.lockFile ? `cp ${options.projectRoot}/${options.lockFile} ${options.outputDir}/${options.lockFile}` : '',
            `cd ${options.outputDir}`,
            `${options.installer} install`,
        ]);
    }
    return chain([parcelCommand, depsCommand]);
}
var Installer;
(function (Installer) {
    Installer["NPM"] = "npm";
    Installer["YARN"] = "yarn";
})(Installer = exports.Installer || (exports.Installer = {}));
var LockFile;
(function (LockFile) {
    LockFile["NPM"] = "package-lock.json";
    LockFile["YARN"] = "yarn.lock";
})(LockFile = exports.LockFile || (exports.LockFile = {}));
function chain(commands) {
    return commands.filter(c => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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