"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnknownPrincipal = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A principal for use in resources that need to have a role but it's unknown
 *
 * Some resources have roles associated with them which they assume, such as
 * Lambda Functions, CodeBuild projects, StepFunctions machines, etc.
 *
 * When those resources are imported, their actual roles are not always
 * imported with them. When that happens, we use an instance of this class
 * instead, which will add user warnings when statements are attempted to be
 * added to it.
 */
class UnknownPrincipal {
    constructor(props) {
        this.assumeRoleAction = 'sts:AssumeRole';
        this.resource = props.resource;
        this.grantPrincipal = this;
    }
    get policyFragment() {
        throw new Error(`Cannot get policy fragment of ${this.resource.node.path}, resource imported without a role`);
    }
    addToPrincipalPolicy(statement) {
        const stack = core_1.Stack.of(this.resource);
        const repr = JSON.stringify(stack.resolve(statement));
        core_1.Annotations.of(this.resource).addWarning(`Add statement to this resource's role: ${repr}`);
        // Pretend we did the work. The human will do it for us, eventually.
        return { statementAdded: true, policyDependable: new core_1.ConcreteDependable() };
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
}
exports.UnknownPrincipal = UnknownPrincipal;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidW5rbm93bi1wcmluY2lwYWwuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ1bmtub3duLXByaW5jaXBhbC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxQ0FBZ0YsQ0FBQyxnREFBZ0Q7QUFZakk7Ozs7Ozs7Ozs7R0FVRztBQUNILE1BQWEsZ0JBQWdCO0lBSXpCLFlBQVksS0FBNEI7UUFIeEIscUJBQWdCLEdBQVcsZ0JBQWdCLENBQUM7UUFJeEQsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDO1FBQy9CLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDO0lBQy9CLENBQUM7SUFDRCxJQUFXLGNBQWM7UUFDckIsTUFBTSxJQUFJLEtBQUssQ0FBQyxpQ0FBaUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsSUFBSSxvQ0FBb0MsQ0FBQyxDQUFDO0lBQ2xILENBQUM7SUFDTSxvQkFBb0IsQ0FBQyxTQUEwQjtRQUNsRCxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUN0QyxNQUFNLElBQUksR0FBRyxJQUFJLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztRQUN0RCxrQkFBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsVUFBVSxDQUFDLDBDQUEwQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQzNGLG9FQUFvRTtRQUNwRSxPQUFPLEVBQUUsY0FBYyxFQUFFLElBQUksRUFBRSxnQkFBZ0IsRUFBRSxJQUFJLHlCQUFrQixFQUFFLEVBQUUsQ0FBQztJQUNoRixDQUFDO0lBQ00sV0FBVyxDQUFDLFNBQTBCO1FBQ3pDLE9BQU8sSUFBSSxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxDQUFDLGNBQWMsQ0FBQztJQUMvRCxDQUFDO0NBQ0o7QUFyQkQsNENBcUJDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQW5ub3RhdGlvbnMsIENvbmNyZXRlRGVwZW5kYWJsZSwgSUNvbnN0cnVjdCwgU3RhY2sgfSBmcm9tIFwiLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gJy4vcG9saWN5LXN0YXRlbWVudCc7XG5pbXBvcnQgeyBBZGRUb1ByaW5jaXBhbFBvbGljeVJlc3VsdCwgSVByaW5jaXBhbCwgUHJpbmNpcGFsUG9saWN5RnJhZ21lbnQgfSBmcm9tICcuL3ByaW5jaXBhbHMnO1xuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBhbiBVbmtub3duUHJpbmNpcGFsXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVW5rbm93blByaW5jaXBhbFByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgcmVzb3VyY2UgdGhlIHJvbGUgcHJveHkgaXMgZm9yXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVzb3VyY2U6IElDb25zdHJ1Y3Q7XG59XG4vKipcbiAqIEEgcHJpbmNpcGFsIGZvciB1c2UgaW4gcmVzb3VyY2VzIHRoYXQgbmVlZCB0byBoYXZlIGEgcm9sZSBidXQgaXQncyB1bmtub3duXG4gKlxuICogU29tZSByZXNvdXJjZXMgaGF2ZSByb2xlcyBhc3NvY2lhdGVkIHdpdGggdGhlbSB3aGljaCB0aGV5IGFzc3VtZSwgc3VjaCBhc1xuICogTGFtYmRhIEZ1bmN0aW9ucywgQ29kZUJ1aWxkIHByb2plY3RzLCBTdGVwRnVuY3Rpb25zIG1hY2hpbmVzLCBldGMuXG4gKlxuICogV2hlbiB0aG9zZSByZXNvdXJjZXMgYXJlIGltcG9ydGVkLCB0aGVpciBhY3R1YWwgcm9sZXMgYXJlIG5vdCBhbHdheXNcbiAqIGltcG9ydGVkIHdpdGggdGhlbS4gV2hlbiB0aGF0IGhhcHBlbnMsIHdlIHVzZSBhbiBpbnN0YW5jZSBvZiB0aGlzIGNsYXNzXG4gKiBpbnN0ZWFkLCB3aGljaCB3aWxsIGFkZCB1c2VyIHdhcm5pbmdzIHdoZW4gc3RhdGVtZW50cyBhcmUgYXR0ZW1wdGVkIHRvIGJlXG4gKiBhZGRlZCB0byBpdC5cbiAqL1xuZXhwb3J0IGNsYXNzIFVua25vd25QcmluY2lwYWwgaW1wbGVtZW50cyBJUHJpbmNpcGFsIHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgYXNzdW1lUm9sZUFjdGlvbjogc3RyaW5nID0gJ3N0czpBc3N1bWVSb2xlJztcbiAgICBwdWJsaWMgcmVhZG9ubHkgZ3JhbnRQcmluY2lwYWw6IElQcmluY2lwYWw7XG4gICAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZTogSUNvbnN0cnVjdDtcbiAgICBjb25zdHJ1Y3Rvcihwcm9wczogVW5rbm93blByaW5jaXBhbFByb3BzKSB7XG4gICAgICAgIHRoaXMucmVzb3VyY2UgPSBwcm9wcy5yZXNvdXJjZTtcbiAgICAgICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IHRoaXM7XG4gICAgfVxuICAgIHB1YmxpYyBnZXQgcG9saWN5RnJhZ21lbnQoKTogUHJpbmNpcGFsUG9saWN5RnJhZ21lbnQge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBnZXQgcG9saWN5IGZyYWdtZW50IG9mICR7dGhpcy5yZXNvdXJjZS5ub2RlLnBhdGh9LCByZXNvdXJjZSBpbXBvcnRlZCB3aXRob3V0IGEgcm9sZWApO1xuICAgIH1cbiAgICBwdWJsaWMgYWRkVG9QcmluY2lwYWxQb2xpY3koc3RhdGVtZW50OiBQb2xpY3lTdGF0ZW1lbnQpOiBBZGRUb1ByaW5jaXBhbFBvbGljeVJlc3VsdCB7XG4gICAgICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2YodGhpcy5yZXNvdXJjZSk7XG4gICAgICAgIGNvbnN0IHJlcHIgPSBKU09OLnN0cmluZ2lmeShzdGFjay5yZXNvbHZlKHN0YXRlbWVudCkpO1xuICAgICAgICBBbm5vdGF0aW9ucy5vZih0aGlzLnJlc291cmNlKS5hZGRXYXJuaW5nKGBBZGQgc3RhdGVtZW50IHRvIHRoaXMgcmVzb3VyY2UncyByb2xlOiAke3JlcHJ9YCk7XG4gICAgICAgIC8vIFByZXRlbmQgd2UgZGlkIHRoZSB3b3JrLiBUaGUgaHVtYW4gd2lsbCBkbyBpdCBmb3IgdXMsIGV2ZW50dWFsbHkuXG4gICAgICAgIHJldHVybiB7IHN0YXRlbWVudEFkZGVkOiB0cnVlLCBwb2xpY3lEZXBlbmRhYmxlOiBuZXcgQ29uY3JldGVEZXBlbmRhYmxlKCkgfTtcbiAgICB9XG4gICAgcHVibGljIGFkZFRvUG9saWN5KHN0YXRlbWVudDogUG9saWN5U3RhdGVtZW50KTogYm9vbGVhbiB7XG4gICAgICAgIHJldHVybiB0aGlzLmFkZFRvUHJpbmNpcGFsUG9saWN5KHN0YXRlbWVudCkuc3RhdGVtZW50QWRkZWQ7XG4gICAgfVxufVxuIl19