"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Role = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const grant_1 = require("./grant");
const iam_generated_1 = require("./iam.generated");
const policy_1 = require("./policy");
const policy_document_1 = require("./policy-document");
const policy_statement_1 = require("./policy-statement");
const principals_1 = require("./principals");
const immutable_role_1 = require("./private/immutable-role");
const util_1 = require("./util");
/**
 * IAM Role
 *
 * Defines an IAM role. The role is created with an assume policy document associated with
 * the specified AWS service principal defined in `serviceAssumeRole`.
 */
class Role extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.roleName,
        });
        this.grantPrincipal = this;
        this.principalAccount = this.env.account;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.managedPolicies = [];
        this.attachedPolicies = new util_1.AttachedPolicies();
        const externalIds = props.externalIds || [];
        if (props.externalId) {
            externalIds.push(props.externalId);
        }
        this.assumeRolePolicy = createAssumeRolePolicy(props.assumedBy, externalIds);
        this.managedPolicies.push(...props.managedPolicies || []);
        this.inlinePolicies = props.inlinePolicies || {};
        this.permissionsBoundary = props.permissionsBoundary;
        const maxSessionDuration = props.maxSessionDuration && props.maxSessionDuration.toSeconds();
        validateMaxSessionDuration(maxSessionDuration);
        const description = (props.description && ((_a = props.description) === null || _a === void 0 ? void 0 : _a.length) > 0) ? props.description : undefined;
        if (description && description.length > 1000) {
            throw new Error('Role description must be no longer than 1000 characters.');
        }
        const role = new iam_generated_1.CfnRole(this, 'Resource', {
            assumeRolePolicyDocument: this.assumeRolePolicy,
            managedPolicyArns: core_1.Lazy.listValue({ produce: () => this.managedPolicies.map(p => p.managedPolicyArn) }, { omitEmpty: true }),
            policies: _flatten(this.inlinePolicies),
            path: props.path,
            permissionsBoundary: this.permissionsBoundary ? this.permissionsBoundary.managedPolicyArn : undefined,
            roleName: this.physicalName,
            maxSessionDuration,
            description,
        });
        this.roleId = role.attrRoleId;
        this.roleArn = this.getResourceArnAttribute(role.attrArn, {
            region: '',
            service: 'iam',
            resource: 'role',
            resourceName: this.physicalName,
        });
        this.roleName = this.getResourceNameAttribute(role.ref);
        this.policyFragment = new principals_1.ArnPrincipal(this.roleArn).policyFragment;
        function _flatten(policies) {
            if (policies == null || Object.keys(policies).length === 0) {
                return undefined;
            }
            const result = new Array();
            for (const policyName of Object.keys(policies)) {
                const policyDocument = policies[policyName];
                result.push({ policyName, policyDocument });
            }
            return result;
        }
    }
    /**
     * Import an external role by ARN.
     *
     * If the imported Role ARN is a Token (such as a
     * `CfnParameter.valueAsString` or a `Fn.importValue()`) *and* the referenced
     * role has a `path` (like `arn:...:role/AdminRoles/Alice`), the
     * `roleName` property will not resolve to the correct value. Instead it
     * will resolve to the first path component. We unfortunately cannot express
     * the correct calculation of the full path name as a CloudFormation
     * expression. In this scenario the Role ARN should be supplied without the
     * `path` in order to resolve the correct role resource.
     *
     * @param scope construct scope
     * @param id construct id
     * @param roleArn the ARN of the role to import
     * @param options allow customizing the behavior of the returned role
     */
    static fromRoleArn(scope, id, roleArn, options = {}) {
        const scopeStack = core_1.Stack.of(scope);
        const parsedArn = scopeStack.parseArn(roleArn);
        const resourceName = parsedArn.resourceName;
        const roleAccount = parsedArn.account;
        // service roles have an ARN like 'arn:aws:iam::<account>:role/service-role/<roleName>'
        // or 'arn:aws:iam::<account>:role/service-role/servicename.amazonaws.com/service-role/<roleName>'
        // we want to support these as well, so we just use the element after the last slash as role name
        const roleName = resourceName.split('/').pop();
        class Import extends core_1.Resource {
            constructor(_scope, _id) {
                super(_scope, _id, {
                    account: roleAccount,
                });
                this.grantPrincipal = this;
                this.principalAccount = roleAccount;
                this.assumeRoleAction = 'sts:AssumeRole';
                this.policyFragment = new principals_1.ArnPrincipal(roleArn).policyFragment;
                this.roleArn = roleArn;
                this.roleName = roleName;
                this.attachedPolicies = new util_1.AttachedPolicies();
            }
            addToPolicy(statement) {
                return this.addToPrincipalPolicy(statement).statementAdded;
            }
            addToPrincipalPolicy(statement) {
                if (!this.defaultPolicy) {
                    this.defaultPolicy = new policy_1.Policy(this, 'Policy');
                    this.attachInlinePolicy(this.defaultPolicy);
                }
                this.defaultPolicy.addStatements(statement);
                return { statementAdded: true, policyDependable: this.defaultPolicy };
            }
            attachInlinePolicy(policy) {
                const thisAndPolicyAccountComparison = core_1.Token.compareStrings(this.env.account, policy.env.account);
                const equalOrAnyUnresolved = thisAndPolicyAccountComparison === core_1.TokenComparison.SAME ||
                    thisAndPolicyAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED ||
                    thisAndPolicyAccountComparison === core_1.TokenComparison.ONE_UNRESOLVED;
                if (equalOrAnyUnresolved) {
                    this.attachedPolicies.attach(policy);
                    policy.attachToRole(this);
                }
            }
            addManagedPolicy(_policy) {
                // FIXME: Add warning that we're ignoring this
            }
            /**
             * Grant permissions to the given principal to pass this role.
             */
            grantPassRole(identity) {
                return this.grant(identity, 'iam:PassRole');
            }
            /**
             * Grant the actions defined in actions to the identity Principal on this resource.
             */
            grant(grantee, ...actions) {
                return grant_1.Grant.addToPrincipal({
                    grantee,
                    actions,
                    resourceArns: [this.roleArn],
                    scope: this,
                });
            }
        }
        const importedRole = new Import(scope, id);
        const roleArnAndScopeStackAccountComparison = core_1.Token.compareStrings(importedRole.env.account, scopeStack.account);
        const equalOrAnyUnresolved = roleArnAndScopeStackAccountComparison === core_1.TokenComparison.SAME ||
            roleArnAndScopeStackAccountComparison === core_1.TokenComparison.BOTH_UNRESOLVED ||
            roleArnAndScopeStackAccountComparison === core_1.TokenComparison.ONE_UNRESOLVED;
        // we only return an immutable Role if both accounts were explicitly provided, and different
        return options.mutable !== false && equalOrAnyUnresolved
            ? importedRole
            : new immutable_role_1.ImmutableRole(scope, `ImmutableRole${id}`, importedRole);
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPrincipalPolicy(statement) {
        if (!this.defaultPolicy) {
            this.defaultPolicy = new policy_1.Policy(this, 'DefaultPolicy');
            this.attachInlinePolicy(this.defaultPolicy);
        }
        this.defaultPolicy.addStatements(statement);
        return { statementAdded: true, policyDependable: this.defaultPolicy };
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    /**
     * Attaches a managed policy to this role.
     * @param policy The the managed policy to attach.
     */
    addManagedPolicy(policy) {
        if (this.managedPolicies.find(mp => mp === policy)) {
            return;
        }
        this.managedPolicies.push(policy);
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        this.attachedPolicies.attach(policy);
        policy.attachToRole(this);
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(grantee, ...actions) {
        return grant_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.roleArn],
            scope: this,
        });
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        return this.grant(identity, 'iam:PassRole');
    }
    /**
     * Return a copy of this Role object whose Policies will not be updated
     *
     * Use the object returned by this method if you want this Role to be used by
     * a construct without it automatically updating the Role's Policies.
     *
     * If you do, you are responsible for adding the correct statements to the
     * Role's policies yourself.
     */
    withoutPolicyUpdates() {
        if (!this.immutableRole) {
            this.immutableRole = new immutable_role_1.ImmutableRole(this.node.scope, `ImmutableRole${this.node.id}`, this);
        }
        return this.immutableRole;
    }
    validate() {
        var _a;
        const errors = super.validate();
        errors.push(...((_a = this.assumeRolePolicy) === null || _a === void 0 ? void 0 : _a.validateForResourcePolicy()) || []);
        for (const policy of Object.values(this.inlinePolicies)) {
            errors.push(...policy.validateForIdentityPolicy());
        }
        return errors;
    }
}
exports.Role = Role;
function createAssumeRolePolicy(principal, externalIds) {
    const statement = new AwsStarStatement();
    statement.addPrincipals(principal);
    statement.addActions(principal.assumeRoleAction);
    if (externalIds.length) {
        statement.addCondition('StringEquals', { 'sts:ExternalId': externalIds.length === 1 ? externalIds[0] : externalIds });
    }
    const doc = new policy_document_1.PolicyDocument();
    doc.addStatements(statement);
    return doc;
}
function validateMaxSessionDuration(duration) {
    if (duration === undefined) {
        return;
    }
    if (duration < 3600 || duration > 43200) {
        throw new Error(`maxSessionDuration is set to ${duration}, but must be >= 3600sec (1hr) and <= 43200sec (12hrs)`);
    }
}
/**
 * A PolicyStatement that normalizes its Principal field differently
 *
 * Normally, "anyone" is normalized to "Principal: *", but this statement
 * normalizes to "Principal: { AWS: * }".
 */
class AwsStarStatement extends policy_statement_1.PolicyStatement {
    toStatementJson() {
        const stat = super.toStatementJson();
        if (stat.Principal === '*') {
            stat.Principal = { AWS: '*' };
        }
        return stat;
    }
}
//# sourceMappingURL=data:application/json;base64,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