"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenIdConnectProvider = void 0;
const path = require("path");
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const RESOURCE_TYPE = 'Custom::AWSCDKOpenIdConnectProvider';
/**
 * IAM OIDC identity providers are entities in IAM that describe an external
 * identity provider (IdP) service that supports the OpenID Connect (OIDC)
 * standard, such as Google or Salesforce. You use an IAM OIDC identity provider
 * when you want to establish trust between an OIDC-compatible IdP and your AWS
 * account. This is useful when creating a mobile app or web application that
 * requires access to AWS resources, but you don't want to create custom sign-in
 * code or manage your own user identities.
 *
 * @see http://openid.net/connect
 * @see https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_providers_oidc.html
 *
 * @experimental
 * @resource AWS::CloudFormation::CustomResource
 */
class OpenIdConnectProvider extends core_1.Resource {
    /**
     * Defines an OpenID Connect provider.
     * @param scope The definition scope
     * @param id Construct ID
     * @param props Initialization properties
     */
    constructor(scope, id, props) {
        super(scope, id);
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: RESOURCE_TYPE,
            serviceToken: this.getOrCreateProvider(),
            properties: {
                ClientIDList: props.clientIds,
                ThumbprintList: props.thumbprints,
                Url: props.url,
            },
        });
        this.openIdConnectProviderArn = core_1.Token.asString(resource.ref);
    }
    /**
     * Imports an Open ID connect provider from an ARN.
     * @param scope The definition scope
     * @param id ID of the construct
     * @param openIdConnectProviderArn the ARN to import
     */
    static fromOpenIdConnectProviderArn(scope, id, openIdConnectProviderArn) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.openIdConnectProviderArn = openIdConnectProviderArn;
            }
        }
        return new Import(scope, id);
    }
    getOrCreateProvider() {
        return core_1.CustomResourceProvider.getOrCreate(this, RESOURCE_TYPE, {
            codeDirectory: path.join(__dirname, 'oidc-provider'),
            runtime: core_1.CustomResourceProviderRuntime.NODEJS_12,
            policyStatements: [
                {
                    Effect: 'Allow',
                    Resource: '*',
                    Action: [
                        'iam:CreateOpenIDConnectProvider',
                        'iam:DeleteOpenIDConnectProvider',
                        'iam:UpdateOpenIDConnectProviderThumbprint',
                        'iam:AddClientIDToOpenIDConnectProvider',
                        'iam:RemoveClientIDFromOpenIDConnectProvider',
                    ],
                },
            ],
        });
    }
}
exports.OpenIdConnectProvider = OpenIdConnectProvider;
//# sourceMappingURL=data:application/json;base64,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