"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LazyRole = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const role_1 = require("./role");
/**
 * An IAM role that only gets attached to the construct tree once it gets used, not before
 *
 * This construct can be used to simplify logic in other constructs
 * which need to create a role but only if certain configurations occur
 * (such as when AutoScaling is configured). The role can be configured in one
 * place, but if it never gets used it doesn't get instantiated and will
 * not be synthesized or deployed.
 *
 * @resource AWS::IAM::Role
 */
class LazyRole extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.grantPrincipal = this;
        this.principalAccount = this.env.account;
        this.assumeRoleAction = 'sts:AssumeRole';
        this.statements = new Array();
        this.policies = new Array();
        this.managedPolicies = new Array();
    }
    /**
     * Adds a permission to the role's default policy document.
     * If there is no default policy attached to this role, it will be created.
     * @param statement The permission statement to add to the policy document
     */
    addToPrincipalPolicy(statement) {
        if (this.role) {
            return this.role.addToPrincipalPolicy(statement);
        }
        else {
            this.statements.push(statement);
            return { statementAdded: true, policyDependable: this };
        }
    }
    addToPolicy(statement) {
        return this.addToPrincipalPolicy(statement).statementAdded;
    }
    /**
     * Attaches a policy to this role.
     * @param policy The policy to attach
     */
    attachInlinePolicy(policy) {
        if (this.role) {
            this.role.attachInlinePolicy(policy);
        }
        else {
            this.policies.push(policy);
        }
    }
    /**
     * Attaches a managed policy to this role.
     * @param policy The managed policy to attach.
     */
    addManagedPolicy(policy) {
        if (this.role) {
            this.role.addManagedPolicy(policy);
        }
        else {
            this.managedPolicies.push(policy);
        }
    }
    /**
     * Returns the ARN of this role.
     */
    get roleArn() {
        return this.instantiate().roleArn;
    }
    /**
     * Returns the stable and unique string identifying the role (i.e. AIDAJQABLZS4A3QDU576Q)
     *
     * @attribute
     */
    get roleId() {
        return this.instantiate().roleId;
    }
    get roleName() {
        return this.instantiate().roleName;
    }
    get policyFragment() {
        return this.instantiate().policyFragment;
    }
    /**
     * Grant the actions defined in actions to the identity Principal on this resource.
     */
    grant(identity, ...actions) {
        return this.instantiate().grant(identity, ...actions);
    }
    /**
     * Grant permissions to the given principal to pass this role.
     */
    grantPassRole(identity) {
        return this.instantiate().grantPassRole(identity);
    }
    instantiate() {
        if (!this.role) {
            const role = new role_1.Role(this, 'Default', this.props);
            this.statements.forEach(role.addToPolicy.bind(role));
            this.policies.forEach(role.attachInlinePolicy.bind(role));
            this.managedPolicies.forEach(role.addManagedPolicy.bind(role));
            this.role = role;
        }
        return this.role;
    }
}
exports.LazyRole = LazyRole;
//# sourceMappingURL=data:application/json;base64,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