"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcsTask = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const ecs = require("../../aws-ecs"); // Automatically re-written from '@aws-cdk/aws-ecs'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const util_1 = require("./util");
/**
 * Start a task on an ECS cluster
 */
class EcsTask {
    constructor(props) {
        this.props = props;
        if (props.securityGroup !== undefined && props.securityGroups !== undefined) {
            throw new Error('Only one of SecurityGroup or SecurityGroups can be populated.');
        }
        this.cluster = props.cluster;
        this.taskDefinition = props.taskDefinition;
        this.taskCount = props.taskCount !== undefined ? props.taskCount : 1;
        if (props.role) {
            const role = props.role;
            this.createEventRolePolicyStatements().forEach(role.addToPolicy.bind(role));
            this.role = role;
        }
        else {
            this.role = util_1.singletonEventRole(this.taskDefinition, this.createEventRolePolicyStatements());
        }
        // Security groups are only configurable with the "awsvpc" network mode.
        if (this.taskDefinition.networkMode !== ecs.NetworkMode.AWS_VPC) {
            if (props.securityGroup !== undefined || props.securityGroups !== undefined) {
                cdk.Annotations.of(this.taskDefinition).addWarning('security groups are ignored when network mode is not awsvpc');
            }
            return;
        }
        if (props.securityGroups) {
            this.securityGroups = props.securityGroups;
            return;
        }
        let securityGroup = props.securityGroup || this.taskDefinition.node.tryFindChild('SecurityGroup');
        securityGroup = securityGroup || new ec2.SecurityGroup(this.taskDefinition, 'SecurityGroup', { vpc: this.props.cluster.vpc });
        this.securityGroup = securityGroup; // Maintain backwards-compatibility for customers that read the generated security group.
        this.securityGroups = [securityGroup];
    }
    /**
     * Allows using tasks as target of EventBridge events
     */
    bind(_rule, _id) {
        const arn = this.cluster.clusterArn;
        const role = this.role;
        const containerOverrides = this.props.containerOverrides && this.props.containerOverrides
            .map(({ containerName, ...overrides }) => ({ name: containerName, ...overrides }));
        const input = { containerOverrides };
        const taskCount = this.taskCount;
        const taskDefinitionArn = this.taskDefinition.taskDefinitionArn;
        const subnetSelection = this.props.subnetSelection || { subnetType: ec2.SubnetType.PRIVATE };
        const assignPublicIp = subnetSelection.subnetType === ec2.SubnetType.PUBLIC ? 'ENABLED' : 'DISABLED';
        const baseEcsParameters = { taskCount, taskDefinitionArn };
        const ecsParameters = this.taskDefinition.networkMode === ecs.NetworkMode.AWS_VPC
            ? {
                ...baseEcsParameters,
                launchType: this.taskDefinition.isEc2Compatible ? 'EC2' : 'FARGATE',
                networkConfiguration: {
                    awsVpcConfiguration: {
                        subnets: this.props.cluster.vpc.selectSubnets(subnetSelection).subnetIds,
                        assignPublicIp,
                        securityGroups: this.securityGroups && this.securityGroups.map(sg => sg.securityGroupId),
                    },
                },
            }
            : baseEcsParameters;
        return {
            id: '',
            arn,
            role,
            ecsParameters,
            input: events.RuleTargetInput.fromObject(input),
            targetResource: this.taskDefinition,
        };
    }
    createEventRolePolicyStatements() {
        const policyStatements = [new iam.PolicyStatement({
                actions: ['ecs:RunTask'],
                resources: [this.taskDefinition.taskDefinitionArn],
                conditions: {
                    ArnEquals: { 'ecs:cluster': this.cluster.clusterArn },
                },
            })];
        // If it so happens that a Task Execution Role was created for the TaskDefinition,
        // then the EventBridge Role must have permissions to pass it (otherwise it doesn't).
        if (this.taskDefinition.executionRole !== undefined) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.executionRole.roleArn],
            }));
        }
        // For Fargate task we need permission to pass the task role.
        if (this.taskDefinition.isFargateCompatible) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.taskDefinition.taskRole.roleArn],
            }));
        }
        return policyStatements;
    }
}
exports.EcsTask = EcsTask;
//# sourceMappingURL=data:application/json;base64,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