"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.loadBalancerNameFromListenerArn = exports.TargetGroupBase = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const enums_1 = require("./enums");
const util_1 = require("./util");
/**
 * Define the target of a load balancer
 */
class TargetGroupBase extends cdk.Construct {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id);
        /**
         * Configurable dependable with all resources that lead to load balancer attachment
         */
        this.loadBalancerAttachedDependencies = new cdk.ConcreteDependable();
        /**
         * Attributes of this target group
         */
        this.attributes = {};
        /**
         * The JSON objects returned by the directly registered members of this target group
         */
        this.targetsJson = new Array();
        if (baseProps.deregistrationDelay !== undefined) {
            this.setAttribute('deregistration_delay.timeout_seconds', baseProps.deregistrationDelay.toSeconds().toString());
        }
        this.healthCheck = baseProps.healthCheck || {};
        this.vpc = baseProps.vpc;
        this.targetType = baseProps.targetType;
        this.resource = new elasticloadbalancingv2_generated_1.CfnTargetGroup(this, 'Resource', {
            name: baseProps.targetGroupName,
            targetGroupAttributes: cdk.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            targetType: cdk.Lazy.stringValue({ produce: () => this.targetType }),
            targets: cdk.Lazy.anyValue({ produce: () => this.targetsJson }, { omitEmptyArray: true }),
            vpcId: cdk.Lazy.stringValue({ produce: () => this.vpc && this.targetType !== enums_1.TargetType.LAMBDA ? this.vpc.vpcId : undefined }),
            // HEALTH CHECK
            healthCheckEnabled: cdk.Lazy.anyValue({ produce: () => this.healthCheck && this.healthCheck.enabled }),
            healthCheckIntervalSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.interval && this.healthCheck.interval.toSeconds(),
            }),
            healthCheckPath: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.path }),
            healthCheckPort: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.port }),
            healthCheckProtocol: cdk.Lazy.stringValue({ produce: () => this.healthCheck && this.healthCheck.protocol }),
            healthCheckTimeoutSeconds: cdk.Lazy.numberValue({
                produce: () => this.healthCheck && this.healthCheck.timeout && this.healthCheck.timeout.toSeconds(),
            }),
            healthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.healthyThresholdCount }),
            unhealthyThresholdCount: cdk.Lazy.numberValue({ produce: () => this.healthCheck && this.healthCheck.unhealthyThresholdCount }),
            matcher: cdk.Lazy.anyValue({
                produce: () => this.healthCheck && this.healthCheck.healthyHttpCodes !== undefined ? {
                    httpCode: this.healthCheck.healthyHttpCodes,
                } : undefined,
            }),
            ...additionalProps,
        });
        this.targetGroupLoadBalancerArns = this.resource.attrLoadBalancerArns;
        this.targetGroupArn = this.resource.ref;
        this.targetGroupFullName = this.resource.attrTargetGroupFullName;
        this.loadBalancerArns = this.resource.attrLoadBalancerArns.toString();
        this.targetGroupName = this.resource.attrTargetGroupName;
        this.defaultPort = additionalProps.port;
    }
    /**
     * List of constructs that need to be depended on to ensure the TargetGroup is associated to a load balancer
     */
    get loadBalancerAttached() {
        return this.loadBalancerAttachedDependencies;
    }
    /**
     * Set/replace the target group's health check
     */
    configureHealthCheck(healthCheck) {
        this.healthCheck = healthCheck;
    }
    /**
     * Set a non-standard attribute on the target group
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-target-groups.html#target-group-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Register the given load balancing target as part of this group
     */
    addLoadBalancerTarget(props) {
        if (this.targetType !== undefined && this.targetType !== props.targetType) {
            throw new Error(`Already have a of type '${this.targetType}', adding '${props.targetType}'; make all targets the same type.`);
        }
        this.targetType = props.targetType;
        if (this.targetType === enums_1.TargetType.LAMBDA && this.targetsJson.length >= 1) {
            throw new Error('TargetGroup can only contain one LAMBDA target. Create a new TargetGroup.');
        }
        if (props.targetJson) {
            this.targetsJson.push(props.targetJson);
        }
    }
    validate() {
        const ret = super.validate();
        if (this.targetType === undefined && this.targetsJson.length === 0) {
            cdk.Annotations.of(this).addWarning("When creating an empty TargetGroup, you should specify a 'targetType' (this warning may become an error in the future).");
        }
        if (this.targetType !== enums_1.TargetType.LAMBDA && this.vpc === undefined) {
            ret.push("'vpc' is required for a non-Lambda TargetGroup");
        }
        return ret;
    }
}
exports.TargetGroupBase = TargetGroupBase;
/**
 * Extract the full load balancer name (used for metrics) from the listener ARN:
 *
 * Turns
 *
 *     arn:aws:elasticloadbalancing:us-west-2:123456789012:listener/app/my-load-balancer/50dc6c495c0c9188/f2f7dc8efc522ab2
 *
 * Into
 *
 *     app/my-load-balancer/50dc6c495c0c9188
 */
function loadBalancerNameFromListenerArn(listenerArn) {
    const arnParts = cdk.Fn.split('/', listenerArn);
    return `${cdk.Fn.select(1, arnParts)}/${cdk.Fn.select(2, arnParts)}/${cdk.Fn.select(3, arnParts)}`;
}
exports.loadBalancerNameFromListenerArn = loadBalancerNameFromListenerArn;
//# sourceMappingURL=data:application/json;base64,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