"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseLoadBalancer = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const region_info_1 = require("../../../region-info"); // Automatically re-written from '@aws-cdk/region-info'
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for both Application and Network Load Balancers
 */
class BaseLoadBalancer extends core_1.Resource {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id, {
            physicalName: baseProps.loadBalancerName,
        });
        /**
         * Attributes set on this load balancer
         */
        this.attributes = {};
        const internetFacing = util_1.ifUndefined(baseProps.internetFacing, false);
        const vpcSubnets = util_1.ifUndefined(baseProps.vpcSubnets, (internetFacing ? { subnetType: ec2.SubnetType.PUBLIC } : {}));
        const { subnetIds, internetConnectivityEstablished } = baseProps.vpc.selectSubnets(vpcSubnets);
        this.vpc = baseProps.vpc;
        const resource = new elasticloadbalancingv2_generated_1.CfnLoadBalancer(this, 'Resource', {
            name: this.physicalName,
            subnets: subnetIds,
            scheme: internetFacing ? 'internet-facing' : 'internal',
            loadBalancerAttributes: core_1.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (internetFacing) {
            resource.node.addDependency(internetConnectivityEstablished);
        }
        this.setAttribute('deletion_protection.enabled', baseProps.deletionProtection ? 'true' : 'false');
        this.loadBalancerCanonicalHostedZoneId = resource.attrCanonicalHostedZoneId;
        this.loadBalancerDnsName = resource.attrDnsName;
        this.loadBalancerFullName = resource.attrLoadBalancerFullName;
        this.loadBalancerName = resource.attrLoadBalancerName;
        this.loadBalancerArn = resource.ref;
        this.loadBalancerSecurityGroups = resource.attrSecurityGroups;
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    logAccessLogs(bucket, prefix) {
        this.setAttribute('access_logs.s3.enabled', 'true');
        this.setAttribute('access_logs.s3.bucket', bucket.bucketName.toString());
        this.setAttribute('access_logs.s3.prefix', prefix);
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Region is required to enable ELBv2 access logging');
        }
        const account = region_info_1.RegionInfo.get(region).elbv2Account;
        if (!account) {
            throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${region}`);
        }
        prefix = prefix || '';
        bucket.grantPut(new iam.AccountPrincipal(account), `${(prefix ? prefix + '/' : '')}AWSLogs/${core_1.Stack.of(this).account}/*`);
        // make sure the bucket's policy is created before the ALB (see https://github.com/aws/aws-cdk/issues/1633)
        this.node.addDependency(bucket);
    }
    /**
     * Set a non-standard attribute on the load balancer
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Remove an attribute from the load balancer
     */
    removeAttribute(key) {
        this.setAttribute(key, undefined);
    }
}
exports.BaseLoadBalancer = BaseLoadBalancer;
//# sourceMappingURL=data:application/json;base64,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