"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NetworkTargetGroup = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define a Network Target Group
 */
class NetworkTargetGroup extends base_target_group_1.TargetGroupBase {
    constructor(scope, id, props) {
        const proto = props.protocol || enums_1.Protocol.TCP;
        util_1.validateNetworkProtocol(proto);
        super(scope, id, props, {
            protocol: proto,
            port: props.port,
        });
        this.listeners = [];
        if (props.proxyProtocolV2 != null) {
            this.setAttribute('proxy_protocol_v2.enabled', props.proxyProtocolV2 ? 'true' : 'false');
        }
        this.addTarget(...(props.targets || []));
    }
    /**
     * Import an existing target group
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedNetworkTargetGroup(scope, id, attrs);
    }
    /**
     * Import an existing listener
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return NetworkTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener) {
        this.loadBalancerAttachedDependencies.add(listener);
        this.listeners.push(listener);
    }
    /**
     * Full name of first load balancer
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    validate() {
        const ret = super.validate();
        const healthCheck = this.healthCheck || {};
        const allowedIntervals = [10, 30];
        if (healthCheck.interval) {
            const seconds = healthCheck.interval.toSeconds();
            if (!cdk.Token.isUnresolved(seconds) && !allowedIntervals.includes(seconds)) {
                ret.push(`Health check interval '${seconds}' not supported. Must be one of the following values '${allowedIntervals.join(',')}'.`);
            }
        }
        if (healthCheck.healthyThresholdCount) {
            const thresholdCount = healthCheck.healthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Healthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.unhealthyThresholdCount) {
            const thresholdCount = healthCheck.unhealthyThresholdCount;
            if (thresholdCount < 2 || thresholdCount > 10) {
                ret.push(`Unhealthy Threshold Count '${thresholdCount}' not supported. Must be a number between 2 and 10.`);
            }
        }
        if (healthCheck.healthyThresholdCount && healthCheck.unhealthyThresholdCount &&
            healthCheck.healthyThresholdCount !== healthCheck.unhealthyThresholdCount) {
            ret.push([
                `Healthy and Unhealthy Threshold Counts must be the same: ${healthCheck.healthyThresholdCount}`,
                `is not equal to ${healthCheck.unhealthyThresholdCount}.`,
            ].join(' '));
        }
        if (!healthCheck.protocol) {
            return ret;
        }
        if (!NLB_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push(`Health check protocol '${healthCheck.protocol}' is not supported. Must be one of [${NLB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`);
        }
        if (healthCheck.path && !NLB_PATH_HEALTH_CHECK_PROTOCOLS.includes(healthCheck.protocol)) {
            ret.push([
                `'${healthCheck.protocol}' health checks do not support the path property.`,
                `Must be one of [${NLB_PATH_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(' '));
        }
        if (healthCheck.timeout && healthCheck.timeout.toSeconds() !== NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]) {
            ret.push([
                'Custom health check timeouts are not supported for Network Load Balancer health checks.',
                `Expected ${NLB_HEALTH_CHECK_TIMEOUTS[healthCheck.protocol]} seconds for ${healthCheck.protocol}, got ${healthCheck.timeout.toSeconds()}`,
            ].join(' '));
        }
        return ret;
    }
}
exports.NetworkTargetGroup = NetworkTargetGroup;
/**
 * An imported network target group
 */
class ImportedNetworkTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener) {
        // Nothing to do, we know nothing of our members
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToNetworkTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const NLB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS, enums_1.Protocol.TCP];
const NLB_PATH_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
const NLB_HEALTH_CHECK_TIMEOUTS = {
    [enums_1.Protocol.HTTP]: 6,
    [enums_1.Protocol.HTTPS]: 10,
    [enums_1.Protocol.TCP]: 10,
};
//# sourceMappingURL=data:application/json;base64,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