"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApplicationTargetGroup = void 0;
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_target_group_1 = require("../shared/base-target-group");
const enums_1 = require("../shared/enums");
const imported_1 = require("../shared/imported");
const util_1 = require("../shared/util");
/**
 * Define an Application Target Group
 */
class ApplicationTargetGroup extends base_target_group_1.TargetGroupBase {
    constructor(scope, id, props = {}) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        super(scope, id, { ...props }, {
            protocol,
            port,
        });
        this.protocol = protocol;
        this.port = port;
        this.connectableMembers = [];
        this.listeners = [];
        if (props) {
            if (props.slowStart !== undefined) {
                this.setAttribute('slow_start.duration_seconds', props.slowStart.toSeconds().toString());
            }
            if (props.stickinessCookieDuration !== undefined) {
                this.enableCookieStickiness(props.stickinessCookieDuration);
            }
            this.addTarget(...(props.targets || []));
        }
    }
    /**
     * Import an existing target group
     */
    static fromTargetGroupAttributes(scope, id, attrs) {
        return new ImportedApplicationTargetGroup(scope, id, attrs);
    }
    /**
     * Import an existing target group
     *
     * @deprecated Use `fromTargetGroupAttributes` instead
     */
    static import(scope, id, props) {
        return ApplicationTargetGroup.fromTargetGroupAttributes(scope, id, props);
    }
    /**
     * Add a load balancing target to this target group
     */
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            this.addLoadBalancerTarget(result);
        }
    }
    /**
     * Enable sticky routing via a cookie to members of this target group
     */
    enableCookieStickiness(duration) {
        this.setAttribute('stickiness.enabled', 'true');
        this.setAttribute('stickiness.type', 'lb_cookie');
        this.setAttribute('stickiness.lb_cookie.duration_seconds', duration.toSeconds().toString());
    }
    /**
     * Register a connectable as a member of this target group.
     *
     * Don't call this directly. It will be called by load balancing targets.
     */
    registerConnectable(connectable, portRange) {
        portRange = portRange || ec2.Port.tcp(this.defaultPort);
        // Notify all listeners that we already know about of this new connectable.
        // Then remember for new listeners that might get added later.
        this.connectableMembers.push({ connectable, portRange });
        for (const listener of this.listeners) {
            listener.registerConnectable(connectable, portRange);
        }
    }
    /**
     * Register a listener that is load balancing to this target group.
     *
     * Don't call this directly. It will be called by listeners.
     */
    registerListener(listener, associatingConstruct) {
        // Notify this listener of all connectables that we know about.
        // Then remember for new connectables that might get added later.
        for (const member of this.connectableMembers) {
            listener.registerConnectable(member.connectable, member.portRange);
        }
        this.listeners.push(listener);
        this.loadBalancerAttachedDependencies.add(associatingConstruct || listener);
    }
    /**
     * Full name of first load balancer
     */
    get firstLoadBalancerFullName() {
        if (this.listeners.length === 0) {
            throw new Error('The TargetGroup needs to be attached to a LoadBalancer before you can call this method');
        }
        return base_target_group_1.loadBalancerNameFromListenerArn(this.listeners[0].listenerArn);
    }
    /**
     * Return the given named metric for this Application Load Balancer Target Group
     *
     * Returns the metric for this target group from the point of view of the first
     * load balancer load balancing to it. If you have multiple load balancers load
     * sending traffic to the same target group, you will have to override the dimensions
     * on this metric.
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensions: {
                TargetGroup: this.targetGroupFullName,
                LoadBalancer: this.firstLoadBalancerFullName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of IPv6 requests received by the target group
     *
     * @default Sum over 5 minutes
     */
    metricIpv6RequestCount(props) {
        return this.metric('IPv6RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCount(props) {
        return this.metric('RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of healthy hosts in the target group
     *
     * @default Average over 5 minutes
     */
    metricHealthyHostCount(props) {
        return this.metric('HealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of unhealthy hosts in the target group
     *
     * @default Average over 5 minutes
     */
    metricUnhealthyHostCount(props) {
        return this.metric('UnHealthyHostCount', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets in this target group.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeTarget(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The average number of requests received by each target in a target group.
     *
     * The only valid statistic is Sum. Note that this represents the average not the sum.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCountPerTarget(props) {
        return this.metric('RequestCountPerTarget', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     */
    metricTargetConnectionErrorCount(props) {
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     */
    metricTargetResponseTime(props) {
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricTargetTLSNegotiationErrorCount(props) {
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    validate() {
        const ret = super.validate();
        if (this.targetType !== undefined && this.targetType !== enums_1.TargetType.LAMBDA
            && (this.protocol === undefined || this.port === undefined)) {
            ret.push('At least one of \'port\' or \'protocol\' is required for a non-Lambda TargetGroup');
        }
        if (this.healthCheck && this.healthCheck.protocol && !ALB_HEALTH_CHECK_PROTOCOLS.includes(this.healthCheck.protocol)) {
            ret.push([
                `Health check protocol '${this.healthCheck.protocol}' is not supported. `,
                `Must be one of [${ALB_HEALTH_CHECK_PROTOCOLS.join(', ')}]`,
            ].join(''));
        }
        return ret;
    }
}
exports.ApplicationTargetGroup = ApplicationTargetGroup;
/**
 * An imported application target group
 */
class ImportedApplicationTargetGroup extends imported_1.ImportedTargetGroupBase {
    registerListener(_listener, _associatingConstruct) {
        // Nothing to do, we know nothing of our members
        core_1.Annotations.of(this).addWarning('Cannot register listener on imported target group -- security groups might need to be updated manually');
    }
    registerConnectable(_connectable, _portRange) {
        core_1.Annotations.of(this).addWarning('Cannot register connectable on imported target group -- security groups might need to be updated manually');
    }
    addTarget(...targets) {
        for (const target of targets) {
            const result = target.attachToApplicationTargetGroup(this);
            if (result.targetJson !== undefined) {
                throw new Error('Cannot add a non-self registering target to an imported TargetGroup. Create a new TargetGroup instead.');
            }
        }
    }
}
const ALB_HEALTH_CHECK_PROTOCOLS = [enums_1.Protocol.HTTP, enums_1.Protocol.HTTPS];
//# sourceMappingURL=data:application/json;base64,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