"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpCodeTarget = exports.HttpCodeElb = exports.ApplicationLoadBalancer = void 0;
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const base_load_balancer_1 = require("../shared/base-load-balancer");
const enums_1 = require("../shared/enums");
const application_listener_1 = require("./application-listener");
const application_listener_action_1 = require("./application-listener-action");
/**
 * Define an Application Load Balancer
 *
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class ApplicationLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props, {
            type: 'application',
            securityGroups: core_1.Lazy.listValue({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
            ipAddressType: props.ipAddressType,
        });
        this.ipAddressType = (_a = props.ipAddressType) !== null && _a !== void 0 ? _a : enums_1.IpAddressType.IPV4;
        const securityGroups = [props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: `Automatically created Security Group for ELB ${this.node.uniqueId}`,
                allowAllOutbound: false,
            })];
        this.connections = new ec2.Connections({ securityGroups });
        if (props.http2Enabled === false) {
            this.setAttribute('routing.http2.enabled', 'false');
        }
        if (props.idleTimeout !== undefined) {
            this.setAttribute('idle_timeout.timeout_seconds', props.idleTimeout.toSeconds().toString());
        }
    }
    /**
     * Import an existing Application Load Balancer
     */
    static fromApplicationLoadBalancerAttributes(scope, id, attrs) {
        return new ImportedApplicationLoadBalancer(scope, id, attrs);
    }
    /**
     * Add a new listener to this load balancer
     */
    addListener(id, props) {
        return new application_listener_1.ApplicationListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    /**
     * Add a redirection listener to this load balancer
     */
    addRedirect(props = {}) {
        var _a, _b, _c, _d;
        const sourcePort = (_a = props.sourcePort) !== null && _a !== void 0 ? _a : 80;
        const targetPort = ((_b = props.targetPort) !== null && _b !== void 0 ? _b : 443).toString();
        return this.addListener(`Redirect${sourcePort}To${targetPort}`, {
            protocol: (_c = props.sourceProtocol) !== null && _c !== void 0 ? _c : enums_1.ApplicationProtocol.HTTP,
            port: sourcePort,
            open: true,
            defaultAction: application_listener_action_1.ListenerAction.redirect({
                port: targetPort,
                protocol: (_d = props.targetProtocol) !== null && _d !== void 0 ? _d : enums_1.ApplicationProtocol.HTTPS,
                permanent: true,
            }),
        });
    }
    /**
     * Add a security group to this load balancer
     */
    addSecurityGroup(securityGroup) {
        this.connections.addSecurityGroup(securityGroup);
    }
    /**
     * Return the given named metric for this Application Load Balancer
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props,
        });
    }
    /**
     * The total number of concurrent TCP connections active from clients to the
     * load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     */
    metricActiveConnectionCount(props) {
        return this.metric('ActiveConnectionCount', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the client that did not
     * establish a session with the load balancer. Possible causes include a
     * mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricClientTlsNegotiationErrorCount(props) {
        return this.metric('ClientTLSNegotiationErrorCount', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricConsumedLCUs(props) {
        return this.metric('ConsumedLCUs', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of fixed-response actions that were successful.
     *
     * @default Sum over 5 minutes
     */
    metricHttpFixedResponseCount(props) {
        return this.metric('HTTP_Fixed_Response_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of redirect actions that were successful.
     *
     * @default Sum over 5 minutes
     */
    metricHttpRedirectCount(props) {
        return this.metric('HTTP_Redirect_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of redirect actions that couldn't be completed because the URL
     * in the response location header is larger than 8K.
     *
     * @default Sum over 5 minutes
     */
    metricHttpRedirectUrlLimitExceededCount(props) {
        return this.metric('HTTP_Redirect_Url_Limit_Exceeded_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of HTTP 3xx/4xx/5xx codes that originate from the load balancer.
     *
     * This does not include any response codes generated by the targets.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeElb(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets
     * in the load balancer.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeTarget(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of bytes processed by the load balancer over IPv6.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6ProcessedBytes(props) {
        return this.metric('IPv6ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of IPv6 requests received by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6RequestCount(props) {
        return this.metric('IPv6RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of new TCP connections established from clients to the
     * load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     */
    metricNewConnectionCount(props) {
        return this.metric('NewConnectionCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of bytes processed by the load balancer over IPv4 and IPv6.
     *
     * @default Sum over 5 minutes
     */
    metricProcessedBytes(props) {
        return this.metric('ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were rejected because the load balancer had
     * reached its maximum number of connections.
     *
     * @default Sum over 5 minutes
     */
    metricRejectedConnectionCount(props) {
        return this.metric('RejectedConnectionCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCount(props) {
        return this.metric('RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of rules processed by the load balancer given a request rate averaged over an hour.
     *
     * @default Sum over 5 minutes
     */
    metricRuleEvaluations(props) {
        return this.metric('RuleEvaluations', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     */
    metricTargetConnectionErrorCount(props) {
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     */
    metricTargetResponseTime(props) {
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricTargetTLSNegotiationErrorCount(props) {
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of user authentications that could not be completed
     *
     * Because an authenticate action was misconfigured, the load balancer
     * couldn't establish a connection with the IdP, or the load balancer
     * couldn't complete the authentication flow due to an internal error.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthError(props) {
        return this.metric('ELBAuthError', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of user authentications that could not be completed because the
     * IdP denied access to the user or an authorization code was used more than
     * once.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthFailure(props) {
        return this.metric('ELBAuthFailure', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in milliseconds, to query the IdP for the ID token and user info.
     *
     * If one or more of these operations fail, this is the time to failure.
     *
     * @default Average over 5 minutes
     */
    metricElbAuthLatency(props) {
        return this.metric('ELBAuthLatency', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of authenticate actions that were successful.
     *
     * This metric is incremented at the end of the authentication workflow,
     * after the load balancer has retrieved the user claims from the IdP.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthSuccess(props) {
        return this.metric('ELBAuthSuccess', {
            statistic: 'Sum',
            ...props,
        });
    }
}
exports.ApplicationLoadBalancer = ApplicationLoadBalancer;
/**
 * Count of HTTP status originating from the load balancer
 *
 * This count does not include any response codes generated by the targets.
 */
var HttpCodeElb;
(function (HttpCodeElb) {
    /**
     * The number of HTTP 3XX redirection codes that originate from the load balancer.
     */
    HttpCodeElb["ELB_3XX_COUNT"] = "HTTPCode_ELB_3XX_Count";
    /**
     * The number of HTTP 4XX client error codes that originate from the load balancer.
     *
     * Client errors are generated when requests are malformed or incomplete.
     * These requests have not been received by the target. This count does not
     * include any response codes generated by the targets.
     */
    HttpCodeElb["ELB_4XX_COUNT"] = "HTTPCode_ELB_4XX_Count";
    /**
     * The number of HTTP 5XX server error codes that originate from the load balancer.
     */
    HttpCodeElb["ELB_5XX_COUNT"] = "HTTPCode_ELB_5XX_Count";
})(HttpCodeElb = exports.HttpCodeElb || (exports.HttpCodeElb = {}));
/**
 * Count of HTTP status originating from the targets
 */
var HttpCodeTarget;
(function (HttpCodeTarget) {
    /**
     * The number of 2xx response codes from targets
     */
    HttpCodeTarget["TARGET_2XX_COUNT"] = "HTTPCode_Target_2XX_Count";
    /**
     * The number of 3xx response codes from targets
     */
    HttpCodeTarget["TARGET_3XX_COUNT"] = "HTTPCode_Target_3XX_Count";
    /**
     * The number of 4xx response codes from targets
     */
    HttpCodeTarget["TARGET_4XX_COUNT"] = "HTTPCode_Target_4XX_Count";
    /**
     * The number of 5xx response codes from targets
     */
    HttpCodeTarget["TARGET_5XX_COUNT"] = "HTTPCode_Target_5XX_Count";
})(HttpCodeTarget = exports.HttpCodeTarget || (exports.HttpCodeTarget = {}));
/**
 * An ApplicationLoadBalancer that has been defined elsewhere
 */
class ImportedApplicationLoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.vpc = props.vpc;
        this.loadBalancerArn = props.loadBalancerArn;
        this.connections = new ec2.Connections({
            securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', props.securityGroupId, {
                    allowAllOutbound: props.securityGroupAllowsAllOutbound,
                })],
        });
    }
    addListener(id, props) {
        return new application_listener_1.ApplicationListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    get loadBalancerCanonicalHostedZoneId() {
        if (this.props.loadBalancerCanonicalHostedZoneId) {
            return this.props.loadBalancerCanonicalHostedZoneId;
        }
        // eslint-disable-next-line max-len
        throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Application Load Balancer ${this.node.path} from attributes`);
    }
    get loadBalancerDnsName() {
        if (this.props.loadBalancerDnsName) {
            return this.props.loadBalancerDnsName;
        }
        // eslint-disable-next-line max-len
        throw new Error(`'loadBalancerDnsName' was not provided when constructing Application Load Balancer ${this.node.path} from attributes`);
    }
}
//# sourceMappingURL=data:application/json;base64,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