"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateRedirectResponse = exports.validateFixedResponse = exports.ApplicationListenerRule = exports.ContentType = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const application_listener_action_1 = require("./application-listener-action");
/**
 * The content type for a fixed response
 */
var ContentType;
(function (ContentType) {
    ContentType["TEXT_PLAIN"] = "text/plain";
    ContentType["TEXT_CSS"] = "text/css";
    ContentType["TEXT_HTML"] = "text/html";
    ContentType["APPLICATION_JAVASCRIPT"] = "application/javascript";
    ContentType["APPLICATION_JSON"] = "application/json";
})(ContentType = exports.ContentType || (exports.ContentType = {}));
/**
 * Define a new listener rule
 */
class ApplicationListenerRule extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.legacyConditions = {};
        this.conditions = props.conditions || [];
        const hasPathPatterns = props.pathPatterns || props.pathPattern;
        if (this.conditions.length === 0 && !props.hostHeader && !hasPathPatterns) {
            throw new Error('At least one of \'conditions\', \'hostHeader\', \'pathPattern\' or \'pathPatterns\' is required when defining a load balancing rule.');
        }
        const possibleActions = ['action', 'targetGroups', 'fixedResponse', 'redirectResponse'];
        const providedActions = possibleActions.filter(action => props[action] !== undefined);
        if (providedActions.length > 1) {
            throw new Error(`'${providedActions}' specified together, specify only one`);
        }
        if (props.priority <= 0) {
            throw new Error('Priority must have value greater than or equal to 1');
        }
        this.listener = props.listener;
        const resource = new elasticloadbalancingv2_generated_1.CfnListenerRule(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            priority: props.priority,
            conditions: cdk.Lazy.anyValue({ produce: () => this.renderConditions() }),
            actions: cdk.Lazy.anyValue({ produce: () => this.action ? this.action.renderActions() : [] }),
        });
        if (props.hostHeader) {
            this.setCondition('host-header', [props.hostHeader]);
        }
        if (hasPathPatterns) {
            if (props.pathPattern && props.pathPatterns) {
                throw new Error('Both `pathPatterns` and `pathPattern` are specified, specify only one');
            }
            const pathPattern = props.pathPattern ? [props.pathPattern] : props.pathPatterns;
            this.setCondition('path-pattern', pathPattern);
        }
        if (props.action) {
            this.configureAction(props.action);
        }
        (props.targetGroups || []).forEach(this.addTargetGroup.bind(this));
        if (props.fixedResponse) {
            this.addFixedResponse(props.fixedResponse);
        }
        else if (props.redirectResponse) {
            this.addRedirectResponse(props.redirectResponse);
        }
        this.listenerRuleArn = resource.ref;
    }
    /**
     * Add a non-standard condition to this rule
     *
     * If the condition conflicts with an already set condition, it will be overwritten by the one you specified.
     *
     * @deprecated use `addCondition` instead.
     */
    setCondition(field, values) {
        if (values === undefined) {
            delete this.legacyConditions[field];
            return;
        }
        this.legacyConditions[field] = values;
    }
    /**
     * Add a non-standard condition to this rule
     */
    addCondition(condition) {
        this.conditions.push(condition);
    }
    /**
     * Configure the action to perform for this rule
     */
    configureAction(action) {
        // It might make sense to 'throw' here.
        //
        // However, programs may already exist out there which configured an action twice,
        // in which case the second action accidentally overwrite the initial action, and in some
        // way ended up with a program that did what the author intended. If we were to add throw now,
        // the previously working program would be broken.
        //
        // Instead, signal this through a warning.
        // @deprecate: upon the next major version bump, replace this with a `throw`
        if (this.action) {
            cdk.Annotations.of(this).addWarning('An Action already existed on this ListenerRule and was replaced. Configure exactly one default Action.');
        }
        action.bind(this, this.listener, this);
        this.action = action;
    }
    /**
     * Add a TargetGroup to load balance to
     *
     * @deprecated Use configureAction instead
     */
    addTargetGroup(targetGroup) {
        this.configureAction(application_listener_action_1.ListenerAction.forward([targetGroup]));
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use configureAction instead
     */
    addFixedResponse(fixedResponse) {
        validateFixedResponse(fixedResponse);
        this.configureAction(application_listener_action_1.ListenerAction.fixedResponse(cdk.Token.asNumber(fixedResponse.statusCode), {
            contentType: fixedResponse.contentType,
            messageBody: fixedResponse.messageBody,
        }));
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use configureAction instead
     */
    addRedirectResponse(redirectResponse) {
        validateRedirectResponse(redirectResponse);
        this.configureAction(application_listener_action_1.ListenerAction.redirect({
            host: redirectResponse.host,
            path: redirectResponse.path,
            permanent: redirectResponse.statusCode === 'HTTP_301',
            port: redirectResponse.port,
            protocol: redirectResponse.protocol,
            query: redirectResponse.query,
        }));
    }
    /**
     * Validate the rule
     */
    validate() {
        if (this.action === undefined) {
            return ['Listener rule needs at least one action'];
        }
        const legacyConditionFields = Object.keys(this.legacyConditions);
        if (legacyConditionFields.length === 0 && this.conditions.length === 0) {
            return ['Listener rule needs at least one condition'];
        }
        return [];
    }
    /**
     * Render the conditions for this rule
     */
    renderConditions() {
        const legacyConditions = Object.entries(this.legacyConditions).map(([field, values]) => {
            return { field, values };
        });
        const conditions = this.conditions.map(condition => condition.renderRawCondition());
        return [
            ...legacyConditions,
            ...conditions,
        ];
    }
}
exports.ApplicationListenerRule = ApplicationListenerRule;
/**
 * Validate the status code and message body of a fixed response
 *
 * @internal
 */
function validateFixedResponse(fixedResponse) {
    if (fixedResponse.statusCode && !/^(2|4|5)\d\d$/.test(fixedResponse.statusCode)) {
        throw new Error('`statusCode` must be 2XX, 4XX or 5XX.');
    }
    if (fixedResponse.messageBody && fixedResponse.messageBody.length > 1024) {
        throw new Error('`messageBody` cannot have more than 1024 characters.');
    }
}
exports.validateFixedResponse = validateFixedResponse;
/**
 * Validate the status code and message body of a redirect response
 *
 * @internal
 */
function validateRedirectResponse(redirectResponse) {
    if (redirectResponse.protocol && !/^(HTTPS?|#\{protocol\})$/i.test(redirectResponse.protocol)) {
        throw new Error('`protocol` must be HTTP, HTTPS, or #{protocol}.');
    }
    if (!redirectResponse.statusCode || !/^HTTP_30[12]$/.test(redirectResponse.statusCode)) {
        throw new Error('`statusCode` must be HTTP_301 or HTTP_302.');
    }
}
exports.validateRedirectResponse = validateRedirectResponse;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbGlzdGVuZXItcnVsZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFwcGxpY2F0aW9uLWxpc3RlbmVyLXJ1bGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEscUNBQXFDLENBQUMsZ0RBQWdEO0FBQ3RGLDBGQUFzRTtBQUd0RSwrRUFBK0Q7QUFrRy9EOztHQUVHO0FBQ0gsSUFBWSxXQU1YO0FBTkQsV0FBWSxXQUFXO0lBQ25CLHdDQUF5QixDQUFBO0lBQ3pCLG9DQUFxQixDQUFBO0lBQ3JCLHNDQUF1QixDQUFBO0lBQ3ZCLGdFQUFpRCxDQUFBO0lBQ2pELG9EQUFxQyxDQUFBO0FBQ3pDLENBQUMsRUFOVyxXQUFXLEdBQVgsbUJBQVcsS0FBWCxtQkFBVyxRQU10QjtBQWlFRDs7R0FFRztBQUNILE1BQWEsdUJBQXdCLFNBQVEsR0FBRyxDQUFDLFNBQVM7SUFXdEQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUFtQztRQUM3RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBTkoscUJBQWdCLEdBRTdCLEVBQUUsQ0FBQztRQUtILElBQUksQ0FBQyxVQUFVLEdBQUcsS0FBSyxDQUFDLFVBQVUsSUFBSSxFQUFFLENBQUM7UUFDekMsTUFBTSxlQUFlLEdBQUcsS0FBSyxDQUFDLFlBQVksSUFBSSxLQUFLLENBQUMsV0FBVyxDQUFDO1FBQ2hFLElBQUksSUFBSSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFVBQVUsSUFBSSxDQUFDLGVBQWUsRUFBRTtZQUN2RSxNQUFNLElBQUksS0FBSyxDQUFDLHNJQUFzSSxDQUFDLENBQUM7U0FDM0o7UUFDRCxNQUFNLGVBQWUsR0FBOEMsQ0FBQyxRQUFRLEVBQUUsY0FBYyxFQUFFLGVBQWUsRUFBRSxrQkFBa0IsQ0FBQyxDQUFDO1FBQ25JLE1BQU0sZUFBZSxHQUFHLGVBQWUsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEtBQUssU0FBUyxDQUFDLENBQUM7UUFDdEYsSUFBSSxlQUFlLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLElBQUksZUFBZSx3Q0FBd0MsQ0FBQyxDQUFDO1NBQ2hGO1FBQ0QsSUFBSSxLQUFLLENBQUMsUUFBUSxJQUFJLENBQUMsRUFBRTtZQUNyQixNQUFNLElBQUksS0FBSyxDQUFDLHFEQUFxRCxDQUFDLENBQUM7U0FDMUU7UUFDRCxJQUFJLENBQUMsUUFBUSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDL0IsTUFBTSxRQUFRLEdBQUcsSUFBSSxrREFBZSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDbkQsV0FBVyxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsV0FBVztZQUN2QyxRQUFRLEVBQUUsS0FBSyxDQUFDLFFBQVE7WUFDeEIsVUFBVSxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxnQkFBZ0IsRUFBRSxFQUFFLENBQUM7WUFDekUsT0FBTyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsYUFBYSxFQUFFLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDO1NBQ2hHLENBQUMsQ0FBQztRQUNILElBQUksS0FBSyxDQUFDLFVBQVUsRUFBRTtZQUNsQixJQUFJLENBQUMsWUFBWSxDQUFDLGFBQWEsRUFBRSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDO1NBQ3hEO1FBQ0QsSUFBSSxlQUFlLEVBQUU7WUFDakIsSUFBSSxLQUFLLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7Z0JBQ3pDLE1BQU0sSUFBSSxLQUFLLENBQUMsdUVBQXVFLENBQUMsQ0FBQzthQUM1RjtZQUNELE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDO1lBQ2pGLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1NBQ2xEO1FBQ0QsSUFBSSxLQUFLLENBQUMsTUFBTSxFQUFFO1lBQ2QsSUFBSSxDQUFDLGVBQWUsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUM7U0FDdEM7UUFDRCxDQUFDLEtBQUssQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxjQUFjLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDbkUsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQ3JCLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUM7U0FDOUM7YUFDSSxJQUFJLEtBQUssQ0FBQyxnQkFBZ0IsRUFBRTtZQUM3QixJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUM7U0FDcEQ7UUFDRCxJQUFJLENBQUMsZUFBZSxHQUFHLFFBQVEsQ0FBQyxHQUFHLENBQUM7SUFDeEMsQ0FBQztJQUNEOzs7Ozs7T0FNRztJQUNJLFlBQVksQ0FBQyxLQUFhLEVBQUUsTUFBNEI7UUFDM0QsSUFBSSxNQUFNLEtBQUssU0FBUyxFQUFFO1lBQ3RCLE9BQU8sSUFBSSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDO1lBQ3BDLE9BQU87U0FDVjtRQUNELElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLENBQUMsR0FBRyxNQUFNLENBQUM7SUFDMUMsQ0FBQztJQUNEOztPQUVHO0lBQ0ksWUFBWSxDQUFDLFNBQTRCO1FBQzVDLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0lBQ3BDLENBQUM7SUFDRDs7T0FFRztJQUNJLGVBQWUsQ0FBQyxNQUFzQjtRQUN6Qyx1Q0FBdUM7UUFDdkMsRUFBRTtRQUNGLGtGQUFrRjtRQUNsRix5RkFBeUY7UUFDekYsOEZBQThGO1FBQzlGLGtEQUFrRDtRQUNsRCxFQUFFO1FBQ0YsMENBQTBDO1FBQzFDLDRFQUE0RTtRQUM1RSxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDYixHQUFHLENBQUMsV0FBVyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMsd0dBQXdHLENBQUMsQ0FBQztTQUNqSjtRQUNELE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLElBQUksQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLENBQUM7UUFDdkMsSUFBSSxDQUFDLE1BQU0sR0FBRyxNQUFNLENBQUM7SUFDekIsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxjQUFjLENBQUMsV0FBb0M7UUFDdEQsSUFBSSxDQUFDLGVBQWUsQ0FBQyw0Q0FBYyxDQUFDLE9BQU8sQ0FBQyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUNoRSxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLGdCQUFnQixDQUFDLGFBQTRCO1FBQ2hELHFCQUFxQixDQUFDLGFBQWEsQ0FBQyxDQUFDO1FBQ3JDLElBQUksQ0FBQyxlQUFlLENBQUMsNENBQWMsQ0FBQyxhQUFhLENBQUMsR0FBRyxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsYUFBYSxDQUFDLFVBQVUsQ0FBQyxFQUFFO1lBQzVGLFdBQVcsRUFBRSxhQUFhLENBQUMsV0FBVztZQUN0QyxXQUFXLEVBQUUsYUFBYSxDQUFDLFdBQVc7U0FDekMsQ0FBQyxDQUFDLENBQUM7SUFDUixDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLG1CQUFtQixDQUFDLGdCQUFrQztRQUN6RCx3QkFBd0IsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzNDLElBQUksQ0FBQyxlQUFlLENBQUMsNENBQWMsQ0FBQyxRQUFRLENBQUM7WUFDekMsSUFBSSxFQUFFLGdCQUFnQixDQUFDLElBQUk7WUFDM0IsSUFBSSxFQUFFLGdCQUFnQixDQUFDLElBQUk7WUFDM0IsU0FBUyxFQUFFLGdCQUFnQixDQUFDLFVBQVUsS0FBSyxVQUFVO1lBQ3JELElBQUksRUFBRSxnQkFBZ0IsQ0FBQyxJQUFJO1lBQzNCLFFBQVEsRUFBRSxnQkFBZ0IsQ0FBQyxRQUFRO1lBQ25DLEtBQUssRUFBRSxnQkFBZ0IsQ0FBQyxLQUFLO1NBQ2hDLENBQUMsQ0FBQyxDQUFDO0lBQ1IsQ0FBQztJQUNEOztPQUVHO0lBQ08sUUFBUTtRQUNkLElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxTQUFTLEVBQUU7WUFDM0IsT0FBTyxDQUFDLHlDQUF5QyxDQUFDLENBQUM7U0FDdEQ7UUFDRCxNQUFNLHFCQUFxQixHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFDakUsSUFBSSxxQkFBcUIsQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNwRSxPQUFPLENBQUMsNENBQTRDLENBQUMsQ0FBQztTQUN6RDtRQUNELE9BQU8sRUFBRSxDQUFDO0lBQ2QsQ0FBQztJQUNEOztPQUVHO0lBQ0ssZ0JBQWdCO1FBQ3BCLE1BQU0sZ0JBQWdCLEdBQUcsTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEtBQUssRUFBRSxNQUFNLENBQUMsRUFBRSxFQUFFO1lBQ25GLE9BQU8sRUFBRSxLQUFLLEVBQUUsTUFBTSxFQUFFLENBQUM7UUFDN0IsQ0FBQyxDQUFDLENBQUM7UUFDSCxNQUFNLFVBQVUsR0FBRyxJQUFJLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsRUFBRSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsRUFBRSxDQUFDLENBQUM7UUFDcEYsT0FBTztZQUNILEdBQUcsZ0JBQWdCO1lBQ25CLEdBQUcsVUFBVTtTQUNoQixDQUFDO0lBQ04sQ0FBQztDQUNKO0FBNUpELDBEQTRKQztBQUNEOzs7O0dBSUc7QUFDSCxTQUFnQixxQkFBcUIsQ0FBQyxhQUE0QjtJQUM5RCxJQUFJLGFBQWEsQ0FBQyxVQUFVLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxVQUFVLENBQUMsRUFBRTtRQUM3RSxNQUFNLElBQUksS0FBSyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7S0FDNUQ7SUFDRCxJQUFJLGFBQWEsQ0FBQyxXQUFXLElBQUksYUFBYSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEdBQUcsSUFBSSxFQUFFO1FBQ3RFLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELENBQUMsQ0FBQztLQUMzRTtBQUNMLENBQUM7QUFQRCxzREFPQztBQUNEOzs7O0dBSUc7QUFDSCxTQUFnQix3QkFBd0IsQ0FBQyxnQkFBa0M7SUFDdkUsSUFBSSxnQkFBZ0IsQ0FBQyxRQUFRLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLEVBQUU7UUFDM0YsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO0tBQ3RFO0lBQ0QsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLEVBQUU7UUFDcEYsTUFBTSxJQUFJLEtBQUssQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO0tBQ2pFO0FBQ0wsQ0FBQztBQVBELDREQU9DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gXCIuLi8uLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgQ2ZuTGlzdGVuZXJSdWxlIH0gZnJvbSAnLi4vZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mi5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSUxpc3RlbmVyQWN0aW9uIH0gZnJvbSAnLi4vc2hhcmVkL2xpc3RlbmVyLWFjdGlvbic7XG5pbXBvcnQgeyBJQXBwbGljYXRpb25MaXN0ZW5lciB9IGZyb20gJy4vYXBwbGljYXRpb24tbGlzdGVuZXInO1xuaW1wb3J0IHsgTGlzdGVuZXJBY3Rpb24gfSBmcm9tICcuL2FwcGxpY2F0aW9uLWxpc3RlbmVyLWFjdGlvbic7XG5pbXBvcnQgeyBJQXBwbGljYXRpb25UYXJnZXRHcm91cCB9IGZyb20gJy4vYXBwbGljYXRpb24tdGFyZ2V0LWdyb3VwJztcbmltcG9ydCB7IExpc3RlbmVyQ29uZGl0aW9uIH0gZnJvbSAnLi9jb25kaXRpb25zJztcbi8qKlxuICogQmFzaWMgcHJvcGVydGllcyBmb3IgZGVmaW5pbmcgYSBydWxlIG9uIGEgbGlzdGVuZXJcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBCYXNlQXBwbGljYXRpb25MaXN0ZW5lclJ1bGVQcm9wcyB7XG4gICAgLyoqXG4gICAgICogUHJpb3JpdHkgb2YgdGhlIHJ1bGVcbiAgICAgKlxuICAgICAqIFRoZSBydWxlIHdpdGggdGhlIGxvd2VzdCBwcmlvcml0eSB3aWxsIGJlIHVzZWQgZm9yIGV2ZXJ5IHJlcXVlc3QuXG4gICAgICpcbiAgICAgKiBQcmlvcml0aWVzIG11c3QgYmUgdW5pcXVlLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByaW9yaXR5OiBudW1iZXI7XG4gICAgLyoqXG4gICAgICogVGFyZ2V0IGdyb3VwcyB0byBmb3J3YXJkIHJlcXVlc3RzIHRvLlxuICAgICAqXG4gICAgICogT25seSBvbmUgb2YgYGFjdGlvbmAsIGBmaXhlZFJlc3BvbnNlYCwgYHJlZGlyZWN0UmVzcG9uc2VgIG9yIGB0YXJnZXRHcm91cHNgIGNhbiBiZSBzcGVjaWZpZWQuXG4gICAgICpcbiAgICAgKiBJbXBsaWVzIGEgYGZvcndhcmRgIGFjdGlvbi5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gdGFyZ2V0IGdyb3Vwcy5cbiAgICAgKi9cbiAgICByZWFkb25seSB0YXJnZXRHcm91cHM/OiBJQXBwbGljYXRpb25UYXJnZXRHcm91cFtdO1xuICAgIC8qKlxuICAgICAqIEFjdGlvbiB0byBwZXJmb3JtIHdoZW4gcmVxdWVzdHMgYXJlIHJlY2VpdmVkXG4gICAgICpcbiAgICAgKiBPbmx5IG9uZSBvZiBgYWN0aW9uYCwgYGZpeGVkUmVzcG9uc2VgLCBgcmVkaXJlY3RSZXNwb25zZWAgb3IgYHRhcmdldEdyb3Vwc2AgY2FuIGJlIHNwZWNpZmllZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gYWN0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWN0aW9uPzogTGlzdGVuZXJBY3Rpb247XG4gICAgLyoqXG4gICAgICogRml4ZWQgcmVzcG9uc2UgdG8gcmV0dXJuLlxuICAgICAqXG4gICAgICogT25seSBvbmUgb2YgYGFjdGlvbmAsIGBmaXhlZFJlc3BvbnNlYCwgYHJlZGlyZWN0UmVzcG9uc2VgIG9yIGB0YXJnZXRHcm91cHNgIGNhbiBiZSBzcGVjaWZpZWQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGZpeGVkIHJlc3BvbnNlLlxuICAgICAqIEBkZXByZWNhdGVkIFVzZSBgYWN0aW9uYCBpbnN0ZWFkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGZpeGVkUmVzcG9uc2U/OiBGaXhlZFJlc3BvbnNlO1xuICAgIC8qKlxuICAgICAqIFJlZGlyZWN0IHJlc3BvbnNlIHRvIHJldHVybi5cbiAgICAgKlxuICAgICAqIE9ubHkgb25lIG9mIGBhY3Rpb25gLCBgZml4ZWRSZXNwb25zZWAsIGByZWRpcmVjdFJlc3BvbnNlYCBvciBgdGFyZ2V0R3JvdXBzYCBjYW4gYmUgc3BlY2lmaWVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyByZWRpcmVjdCByZXNwb25zZS5cbiAgICAgKiBAZGVwcmVjYXRlZCBVc2UgYGFjdGlvbmAgaW5zdGVhZC5cbiAgICAgKi9cbiAgICByZWFkb25seSByZWRpcmVjdFJlc3BvbnNlPzogUmVkaXJlY3RSZXNwb25zZTtcbiAgICAvKipcbiAgICAgKiBSdWxlIGFwcGxpZXMgaWYgbWF0Y2hlcyB0aGUgY29uZGl0aW9ucy5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNsb2FkYmFsYW5jaW5nL2xhdGVzdC9hcHBsaWNhdGlvbi9sb2FkLWJhbGFuY2VyLWxpc3RlbmVycy5odG1sXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGNvbmRpdGlvbnMuXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29uZGl0aW9ucz86IExpc3RlbmVyQ29uZGl0aW9uW107XG4gICAgLyoqXG4gICAgICogUnVsZSBhcHBsaWVzIGlmIHRoZSByZXF1ZXN0ZWQgaG9zdCBtYXRjaGVzIHRoZSBpbmRpY2F0ZWQgaG9zdFxuICAgICAqXG4gICAgICogTWF5IGNvbnRhaW4gdXAgdG8gdGhyZWUgJyonIHdpbGRjYXJkcy5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNsb2FkYmFsYW5jaW5nL2xhdGVzdC9hcHBsaWNhdGlvbi9sb2FkLWJhbGFuY2VyLWxpc3RlbmVycy5odG1sI2hvc3QtY29uZGl0aW9uc1xuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBob3N0IGNvbmRpdGlvbi5cbiAgICAgKiBAZGVwcmVjYXRlZCBVc2UgYGNvbmRpdGlvbnNgIGluc3RlYWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgaG9zdEhlYWRlcj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBSdWxlIGFwcGxpZXMgaWYgdGhlIHJlcXVlc3RlZCBwYXRoIG1hdGNoZXMgdGhlIGdpdmVuIHBhdGggcGF0dGVyblxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItbGlzdGVuZXJzLmh0bWwjcGF0aC1jb25kaXRpb25zXG4gICAgICogQGRlZmF1bHQgLSBObyBwYXRoIGNvbmRpdGlvbi5cbiAgICAgKiBAZGVwcmVjYXRlZCBVc2UgYGNvbmRpdGlvbnNgIGluc3RlYWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGF0aFBhdHRlcm4/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogUnVsZSBhcHBsaWVzIGlmIHRoZSByZXF1ZXN0ZWQgcGF0aCBtYXRjaGVzIGFueSBvZiB0aGUgZ2l2ZW4gcGF0dGVybnMuXG4gICAgICpcbiAgICAgKiBQYXRocyBtYXkgY29udGFpbiB1cCB0byB0aHJlZSAnKicgd2lsZGNhcmRzLlxuICAgICAqXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItbGlzdGVuZXJzLmh0bWwjcGF0aC1jb25kaXRpb25zXG4gICAgICogQGRlZmF1bHQgLSBObyBwYXRoIGNvbmRpdGlvbnMuXG4gICAgICogQGRlcHJlY2F0ZWQgVXNlIGBjb25kaXRpb25zYCBpbnN0ZWFkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHBhdGhQYXR0ZXJucz86IHN0cmluZ1tdO1xufVxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBkZWZpbmluZyBhIGxpc3RlbmVyIHJ1bGVcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBcHBsaWNhdGlvbkxpc3RlbmVyUnVsZVByb3BzIGV4dGVuZHMgQmFzZUFwcGxpY2F0aW9uTGlzdGVuZXJSdWxlUHJvcHMge1xuICAgIC8qKlxuICAgICAqIFRoZSBsaXN0ZW5lciB0byBhdHRhY2ggdGhlIHJ1bGUgdG9cbiAgICAgKi9cbiAgICByZWFkb25seSBsaXN0ZW5lcjogSUFwcGxpY2F0aW9uTGlzdGVuZXI7XG59XG4vKipcbiAqIFRoZSBjb250ZW50IHR5cGUgZm9yIGEgZml4ZWQgcmVzcG9uc2VcbiAqL1xuZXhwb3J0IGVudW0gQ29udGVudFR5cGUge1xuICAgIFRFWFRfUExBSU4gPSAndGV4dC9wbGFpbicsXG4gICAgVEVYVF9DU1MgPSAndGV4dC9jc3MnLFxuICAgIFRFWFRfSFRNTCA9ICd0ZXh0L2h0bWwnLFxuICAgIEFQUExJQ0FUSU9OX0pBVkFTQ1JJUFQgPSAnYXBwbGljYXRpb24vamF2YXNjcmlwdCcsXG4gICAgQVBQTElDQVRJT05fSlNPTiA9ICdhcHBsaWNhdGlvbi9qc29uJ1xufVxuLyoqXG4gKiBBIGZpeGVkIHJlc3BvbnNlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRml4ZWRSZXNwb25zZSB7XG4gICAgLyoqXG4gICAgICogVGhlIEhUVFAgcmVzcG9uc2UgY29kZSAoMlhYLCA0WFggb3IgNVhYKVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHN0YXR1c0NvZGU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBUaGUgY29udGVudCB0eXBlXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCB0ZXh0L3BsYWluXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29udGVudFR5cGU/OiBDb250ZW50VHlwZTtcbiAgICAvKipcbiAgICAgKiBUaGUgbWVzc2FnZVxuICAgICAqXG4gICAgICogQGRlZmF1bHQgbm8gbWVzc2FnZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IG1lc3NhZ2VCb2R5Pzogc3RyaW5nO1xufVxuLyoqXG4gKiBBIHJlZGlyZWN0IHJlc3BvbnNlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVkaXJlY3RSZXNwb25zZSB7XG4gICAgLyoqXG4gICAgICogVGhlIGhvc3RuYW1lLiBUaGlzIGNvbXBvbmVudCBpcyBub3QgcGVyY2VudC1lbmNvZGVkLiBUaGUgaG9zdG5hbWUgY2FuIGNvbnRhaW4gI3tob3N0fS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IG9yaWdpbiBob3N0IG9mIHJlcXVlc3RcbiAgICAgKi9cbiAgICByZWFkb25seSBob3N0Pzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBhYnNvbHV0ZSBwYXRoLCBzdGFydGluZyB3aXRoIHRoZSBsZWFkaW5nIFwiL1wiLiBUaGlzIGNvbXBvbmVudCBpcyBub3QgcGVyY2VudC1lbmNvZGVkLlxuICAgICAqIFRoZSBwYXRoIGNhbiBjb250YWluICN7aG9zdH0sICN7cGF0aH0sIGFuZCAje3BvcnR9LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgb3JpZ2luIHBhdGggb2YgcmVxdWVzdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHBhdGg/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHBvcnQuIFlvdSBjYW4gc3BlY2lmeSBhIHZhbHVlIGZyb20gMSB0byA2NTUzNSBvciAje3BvcnR9LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgb3JpZ2luIHBvcnQgb2YgcmVxdWVzdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHBvcnQ/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHByb3RvY29sLiBZb3UgY2FuIHNwZWNpZnkgSFRUUCwgSFRUUFMsIG9yICN7cHJvdG9jb2x9LiBZb3UgY2FuIHJlZGlyZWN0IEhUVFAgdG8gSFRUUCxcbiAgICAgKiBIVFRQIHRvIEhUVFBTLCBhbmQgSFRUUFMgdG8gSFRUUFMuIFlvdSBjYW5ub3QgcmVkaXJlY3QgSFRUUFMgdG8gSFRUUC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IG9yaWdpbiBwcm90b2NvbCBvZiByZXF1ZXN0XG4gICAgICovXG4gICAgcmVhZG9ubHkgcHJvdG9jb2w/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHF1ZXJ5IHBhcmFtZXRlcnMsIFVSTC1lbmNvZGVkIHdoZW4gbmVjZXNzYXJ5LCBidXQgbm90IHBlcmNlbnQtZW5jb2RlZC5cbiAgICAgKiBEbyBub3QgaW5jbHVkZSB0aGUgbGVhZGluZyBcIj9cIiwgYXMgaXQgaXMgYXV0b21hdGljYWxseSBhZGRlZC5cbiAgICAgKiBZb3UgY2FuIHNwZWNpZnkgYW55IG9mIHRoZSByZXNlcnZlZCBrZXl3b3Jkcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IG9yaWdpbiBxdWVyeSBzdHJpbmcgb2YgcmVxdWVzdFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHF1ZXJ5Pzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBIVFRQIHJlZGlyZWN0IGNvZGUgKEhUVFBfMzAxIG9yIEhUVFBfMzAyKVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHN0YXR1c0NvZGU6IHN0cmluZztcbn1cbi8qKlxuICogRGVmaW5lIGEgbmV3IGxpc3RlbmVyIHJ1bGVcbiAqL1xuZXhwb3J0IGNsYXNzIEFwcGxpY2F0aW9uTGlzdGVuZXJSdWxlIGV4dGVuZHMgY2RrLkNvbnN0cnVjdCB7XG4gICAgLyoqXG4gICAgICogVGhlIEFSTiBvZiB0aGlzIHJ1bGVcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgbGlzdGVuZXJSdWxlQXJuOiBzdHJpbmc7XG4gICAgcHJpdmF0ZSByZWFkb25seSBjb25kaXRpb25zOiBMaXN0ZW5lckNvbmRpdGlvbltdO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgbGVnYWN5Q29uZGl0aW9uczoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBzdHJpbmdbXTtcbiAgICB9ID0ge307XG4gICAgcHJpdmF0ZSByZWFkb25seSBsaXN0ZW5lcjogSUFwcGxpY2F0aW9uTGlzdGVuZXI7XG4gICAgcHJpdmF0ZSBhY3Rpb24/OiBJTGlzdGVuZXJBY3Rpb247XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBBcHBsaWNhdGlvbkxpc3RlbmVyUnVsZVByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCk7XG4gICAgICAgIHRoaXMuY29uZGl0aW9ucyA9IHByb3BzLmNvbmRpdGlvbnMgfHwgW107XG4gICAgICAgIGNvbnN0IGhhc1BhdGhQYXR0ZXJucyA9IHByb3BzLnBhdGhQYXR0ZXJucyB8fCBwcm9wcy5wYXRoUGF0dGVybjtcbiAgICAgICAgaWYgKHRoaXMuY29uZGl0aW9ucy5sZW5ndGggPT09IDAgJiYgIXByb3BzLmhvc3RIZWFkZXIgJiYgIWhhc1BhdGhQYXR0ZXJucykge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdBdCBsZWFzdCBvbmUgb2YgXFwnY29uZGl0aW9uc1xcJywgXFwnaG9zdEhlYWRlclxcJywgXFwncGF0aFBhdHRlcm5cXCcgb3IgXFwncGF0aFBhdHRlcm5zXFwnIGlzIHJlcXVpcmVkIHdoZW4gZGVmaW5pbmcgYSBsb2FkIGJhbGFuY2luZyBydWxlLicpO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHBvc3NpYmxlQWN0aW9uczogQXJyYXk8a2V5b2YgQXBwbGljYXRpb25MaXN0ZW5lclJ1bGVQcm9wcz4gPSBbJ2FjdGlvbicsICd0YXJnZXRHcm91cHMnLCAnZml4ZWRSZXNwb25zZScsICdyZWRpcmVjdFJlc3BvbnNlJ107XG4gICAgICAgIGNvbnN0IHByb3ZpZGVkQWN0aW9ucyA9IHBvc3NpYmxlQWN0aW9ucy5maWx0ZXIoYWN0aW9uID0+IHByb3BzW2FjdGlvbl0gIT09IHVuZGVmaW5lZCk7XG4gICAgICAgIGlmIChwcm92aWRlZEFjdGlvbnMubGVuZ3RoID4gMSkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGAnJHtwcm92aWRlZEFjdGlvbnN9JyBzcGVjaWZpZWQgdG9nZXRoZXIsIHNwZWNpZnkgb25seSBvbmVgKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAocHJvcHMucHJpb3JpdHkgPD0gMCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdQcmlvcml0eSBtdXN0IGhhdmUgdmFsdWUgZ3JlYXRlciB0aGFuIG9yIGVxdWFsIHRvIDEnKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLmxpc3RlbmVyID0gcHJvcHMubGlzdGVuZXI7XG4gICAgICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmbkxpc3RlbmVyUnVsZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICAgICAgICBsaXN0ZW5lckFybjogcHJvcHMubGlzdGVuZXIubGlzdGVuZXJBcm4sXG4gICAgICAgICAgICBwcmlvcml0eTogcHJvcHMucHJpb3JpdHksXG4gICAgICAgICAgICBjb25kaXRpb25zOiBjZGsuTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyQ29uZGl0aW9ucygpIH0pLFxuICAgICAgICAgICAgYWN0aW9uczogY2RrLkxhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmFjdGlvbiA/IHRoaXMuYWN0aW9uLnJlbmRlckFjdGlvbnMoKSA6IFtdIH0pLFxuICAgICAgICB9KTtcbiAgICAgICAgaWYgKHByb3BzLmhvc3RIZWFkZXIpIHtcbiAgICAgICAgICAgIHRoaXMuc2V0Q29uZGl0aW9uKCdob3N0LWhlYWRlcicsIFtwcm9wcy5ob3N0SGVhZGVyXSk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKGhhc1BhdGhQYXR0ZXJucykge1xuICAgICAgICAgICAgaWYgKHByb3BzLnBhdGhQYXR0ZXJuICYmIHByb3BzLnBhdGhQYXR0ZXJucykge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQm90aCBgcGF0aFBhdHRlcm5zYCBhbmQgYHBhdGhQYXR0ZXJuYCBhcmUgc3BlY2lmaWVkLCBzcGVjaWZ5IG9ubHkgb25lJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBjb25zdCBwYXRoUGF0dGVybiA9IHByb3BzLnBhdGhQYXR0ZXJuID8gW3Byb3BzLnBhdGhQYXR0ZXJuXSA6IHByb3BzLnBhdGhQYXR0ZXJucztcbiAgICAgICAgICAgIHRoaXMuc2V0Q29uZGl0aW9uKCdwYXRoLXBhdHRlcm4nLCBwYXRoUGF0dGVybik7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKHByb3BzLmFjdGlvbikge1xuICAgICAgICAgICAgdGhpcy5jb25maWd1cmVBY3Rpb24ocHJvcHMuYWN0aW9uKTtcbiAgICAgICAgfVxuICAgICAgICAocHJvcHMudGFyZ2V0R3JvdXBzIHx8IFtdKS5mb3JFYWNoKHRoaXMuYWRkVGFyZ2V0R3JvdXAuYmluZCh0aGlzKSk7XG4gICAgICAgIGlmIChwcm9wcy5maXhlZFJlc3BvbnNlKSB7XG4gICAgICAgICAgICB0aGlzLmFkZEZpeGVkUmVzcG9uc2UocHJvcHMuZml4ZWRSZXNwb25zZSk7XG4gICAgICAgIH1cbiAgICAgICAgZWxzZSBpZiAocHJvcHMucmVkaXJlY3RSZXNwb25zZSkge1xuICAgICAgICAgICAgdGhpcy5hZGRSZWRpcmVjdFJlc3BvbnNlKHByb3BzLnJlZGlyZWN0UmVzcG9uc2UpO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMubGlzdGVuZXJSdWxlQXJuID0gcmVzb3VyY2UucmVmO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSBub24tc3RhbmRhcmQgY29uZGl0aW9uIHRvIHRoaXMgcnVsZVxuICAgICAqXG4gICAgICogSWYgdGhlIGNvbmRpdGlvbiBjb25mbGljdHMgd2l0aCBhbiBhbHJlYWR5IHNldCBjb25kaXRpb24sIGl0IHdpbGwgYmUgb3ZlcndyaXR0ZW4gYnkgdGhlIG9uZSB5b3Ugc3BlY2lmaWVkLlxuICAgICAqXG4gICAgICogQGRlcHJlY2F0ZWQgdXNlIGBhZGRDb25kaXRpb25gIGluc3RlYWQuXG4gICAgICovXG4gICAgcHVibGljIHNldENvbmRpdGlvbihmaWVsZDogc3RyaW5nLCB2YWx1ZXM6IHN0cmluZ1tdIHwgdW5kZWZpbmVkKSB7XG4gICAgICAgIGlmICh2YWx1ZXMgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgZGVsZXRlIHRoaXMubGVnYWN5Q29uZGl0aW9uc1tmaWVsZF07XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5sZWdhY3lDb25kaXRpb25zW2ZpZWxkXSA9IHZhbHVlcztcbiAgICB9XG4gICAgLyoqXG4gICAgICogQWRkIGEgbm9uLXN0YW5kYXJkIGNvbmRpdGlvbiB0byB0aGlzIHJ1bGVcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkQ29uZGl0aW9uKGNvbmRpdGlvbjogTGlzdGVuZXJDb25kaXRpb24pIHtcbiAgICAgICAgdGhpcy5jb25kaXRpb25zLnB1c2goY29uZGl0aW9uKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ29uZmlndXJlIHRoZSBhY3Rpb24gdG8gcGVyZm9ybSBmb3IgdGhpcyBydWxlXG4gICAgICovXG4gICAgcHVibGljIGNvbmZpZ3VyZUFjdGlvbihhY3Rpb246IExpc3RlbmVyQWN0aW9uKSB7XG4gICAgICAgIC8vIEl0IG1pZ2h0IG1ha2Ugc2Vuc2UgdG8gJ3Rocm93JyBoZXJlLlxuICAgICAgICAvL1xuICAgICAgICAvLyBIb3dldmVyLCBwcm9ncmFtcyBtYXkgYWxyZWFkeSBleGlzdCBvdXQgdGhlcmUgd2hpY2ggY29uZmlndXJlZCBhbiBhY3Rpb24gdHdpY2UsXG4gICAgICAgIC8vIGluIHdoaWNoIGNhc2UgdGhlIHNlY29uZCBhY3Rpb24gYWNjaWRlbnRhbGx5IG92ZXJ3cml0ZSB0aGUgaW5pdGlhbCBhY3Rpb24sIGFuZCBpbiBzb21lXG4gICAgICAgIC8vIHdheSBlbmRlZCB1cCB3aXRoIGEgcHJvZ3JhbSB0aGF0IGRpZCB3aGF0IHRoZSBhdXRob3IgaW50ZW5kZWQuIElmIHdlIHdlcmUgdG8gYWRkIHRocm93IG5vdyxcbiAgICAgICAgLy8gdGhlIHByZXZpb3VzbHkgd29ya2luZyBwcm9ncmFtIHdvdWxkIGJlIGJyb2tlbi5cbiAgICAgICAgLy9cbiAgICAgICAgLy8gSW5zdGVhZCwgc2lnbmFsIHRoaXMgdGhyb3VnaCBhIHdhcm5pbmcuXG4gICAgICAgIC8vIEBkZXByZWNhdGU6IHVwb24gdGhlIG5leHQgbWFqb3IgdmVyc2lvbiBidW1wLCByZXBsYWNlIHRoaXMgd2l0aCBhIGB0aHJvd2BcbiAgICAgICAgaWYgKHRoaXMuYWN0aW9uKSB7XG4gICAgICAgICAgICBjZGsuQW5ub3RhdGlvbnMub2YodGhpcykuYWRkV2FybmluZygnQW4gQWN0aW9uIGFscmVhZHkgZXhpc3RlZCBvbiB0aGlzIExpc3RlbmVyUnVsZSBhbmQgd2FzIHJlcGxhY2VkLiBDb25maWd1cmUgZXhhY3RseSBvbmUgZGVmYXVsdCBBY3Rpb24uJyk7XG4gICAgICAgIH1cbiAgICAgICAgYWN0aW9uLmJpbmQodGhpcywgdGhpcy5saXN0ZW5lciwgdGhpcyk7XG4gICAgICAgIHRoaXMuYWN0aW9uID0gYWN0aW9uO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSBUYXJnZXRHcm91cCB0byBsb2FkIGJhbGFuY2UgdG9cbiAgICAgKlxuICAgICAqIEBkZXByZWNhdGVkIFVzZSBjb25maWd1cmVBY3Rpb24gaW5zdGVhZFxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRUYXJnZXRHcm91cCh0YXJnZXRHcm91cDogSUFwcGxpY2F0aW9uVGFyZ2V0R3JvdXApIHtcbiAgICAgICAgdGhpcy5jb25maWd1cmVBY3Rpb24oTGlzdGVuZXJBY3Rpb24uZm9yd2FyZChbdGFyZ2V0R3JvdXBdKSk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEFkZCBhIGZpeGVkIHJlc3BvbnNlXG4gICAgICpcbiAgICAgKiBAZGVwcmVjYXRlZCBVc2UgY29uZmlndXJlQWN0aW9uIGluc3RlYWRcbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkRml4ZWRSZXNwb25zZShmaXhlZFJlc3BvbnNlOiBGaXhlZFJlc3BvbnNlKSB7XG4gICAgICAgIHZhbGlkYXRlRml4ZWRSZXNwb25zZShmaXhlZFJlc3BvbnNlKTtcbiAgICAgICAgdGhpcy5jb25maWd1cmVBY3Rpb24oTGlzdGVuZXJBY3Rpb24uZml4ZWRSZXNwb25zZShjZGsuVG9rZW4uYXNOdW1iZXIoZml4ZWRSZXNwb25zZS5zdGF0dXNDb2RlKSwge1xuICAgICAgICAgICAgY29udGVudFR5cGU6IGZpeGVkUmVzcG9uc2UuY29udGVudFR5cGUsXG4gICAgICAgICAgICBtZXNzYWdlQm9keTogZml4ZWRSZXNwb25zZS5tZXNzYWdlQm9keSxcbiAgICAgICAgfSkpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGQgYSByZWRpcmVjdCByZXNwb25zZVxuICAgICAqXG4gICAgICogQGRlcHJlY2F0ZWQgVXNlIGNvbmZpZ3VyZUFjdGlvbiBpbnN0ZWFkXG4gICAgICovXG4gICAgcHVibGljIGFkZFJlZGlyZWN0UmVzcG9uc2UocmVkaXJlY3RSZXNwb25zZTogUmVkaXJlY3RSZXNwb25zZSkge1xuICAgICAgICB2YWxpZGF0ZVJlZGlyZWN0UmVzcG9uc2UocmVkaXJlY3RSZXNwb25zZSk7XG4gICAgICAgIHRoaXMuY29uZmlndXJlQWN0aW9uKExpc3RlbmVyQWN0aW9uLnJlZGlyZWN0KHtcbiAgICAgICAgICAgIGhvc3Q6IHJlZGlyZWN0UmVzcG9uc2UuaG9zdCxcbiAgICAgICAgICAgIHBhdGg6IHJlZGlyZWN0UmVzcG9uc2UucGF0aCxcbiAgICAgICAgICAgIHBlcm1hbmVudDogcmVkaXJlY3RSZXNwb25zZS5zdGF0dXNDb2RlID09PSAnSFRUUF8zMDEnLFxuICAgICAgICAgICAgcG9ydDogcmVkaXJlY3RSZXNwb25zZS5wb3J0LFxuICAgICAgICAgICAgcHJvdG9jb2w6IHJlZGlyZWN0UmVzcG9uc2UucHJvdG9jb2wsXG4gICAgICAgICAgICBxdWVyeTogcmVkaXJlY3RSZXNwb25zZS5xdWVyeSxcbiAgICAgICAgfSkpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBWYWxpZGF0ZSB0aGUgcnVsZVxuICAgICAqL1xuICAgIHByb3RlY3RlZCB2YWxpZGF0ZSgpIHtcbiAgICAgICAgaWYgKHRoaXMuYWN0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICAgICAgICAgIHJldHVybiBbJ0xpc3RlbmVyIHJ1bGUgbmVlZHMgYXQgbGVhc3Qgb25lIGFjdGlvbiddO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IGxlZ2FjeUNvbmRpdGlvbkZpZWxkcyA9IE9iamVjdC5rZXlzKHRoaXMubGVnYWN5Q29uZGl0aW9ucyk7XG4gICAgICAgIGlmIChsZWdhY3lDb25kaXRpb25GaWVsZHMubGVuZ3RoID09PSAwICYmIHRoaXMuY29uZGl0aW9ucy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIHJldHVybiBbJ0xpc3RlbmVyIHJ1bGUgbmVlZHMgYXQgbGVhc3Qgb25lIGNvbmRpdGlvbiddO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiBbXTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogUmVuZGVyIHRoZSBjb25kaXRpb25zIGZvciB0aGlzIHJ1bGVcbiAgICAgKi9cbiAgICBwcml2YXRlIHJlbmRlckNvbmRpdGlvbnMoKTogYW55IHtcbiAgICAgICAgY29uc3QgbGVnYWN5Q29uZGl0aW9ucyA9IE9iamVjdC5lbnRyaWVzKHRoaXMubGVnYWN5Q29uZGl0aW9ucykubWFwKChbZmllbGQsIHZhbHVlc10pID0+IHtcbiAgICAgICAgICAgIHJldHVybiB7IGZpZWxkLCB2YWx1ZXMgfTtcbiAgICAgICAgfSk7XG4gICAgICAgIGNvbnN0IGNvbmRpdGlvbnMgPSB0aGlzLmNvbmRpdGlvbnMubWFwKGNvbmRpdGlvbiA9PiBjb25kaXRpb24ucmVuZGVyUmF3Q29uZGl0aW9uKCkpO1xuICAgICAgICByZXR1cm4gW1xuICAgICAgICAgICAgLi4ubGVnYWN5Q29uZGl0aW9ucyxcbiAgICAgICAgICAgIC4uLmNvbmRpdGlvbnMsXG4gICAgICAgIF07XG4gICAgfVxufVxuLyoqXG4gKiBWYWxpZGF0ZSB0aGUgc3RhdHVzIGNvZGUgYW5kIG1lc3NhZ2UgYm9keSBvZiBhIGZpeGVkIHJlc3BvbnNlXG4gKlxuICogQGludGVybmFsXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiB2YWxpZGF0ZUZpeGVkUmVzcG9uc2UoZml4ZWRSZXNwb25zZTogRml4ZWRSZXNwb25zZSkge1xuICAgIGlmIChmaXhlZFJlc3BvbnNlLnN0YXR1c0NvZGUgJiYgIS9eKDJ8NHw1KVxcZFxcZCQvLnRlc3QoZml4ZWRSZXNwb25zZS5zdGF0dXNDb2RlKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ2BzdGF0dXNDb2RlYCBtdXN0IGJlIDJYWCwgNFhYIG9yIDVYWC4nKTtcbiAgICB9XG4gICAgaWYgKGZpeGVkUmVzcG9uc2UubWVzc2FnZUJvZHkgJiYgZml4ZWRSZXNwb25zZS5tZXNzYWdlQm9keS5sZW5ndGggPiAxMDI0KSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignYG1lc3NhZ2VCb2R5YCBjYW5ub3QgaGF2ZSBtb3JlIHRoYW4gMTAyNCBjaGFyYWN0ZXJzLicpO1xuICAgIH1cbn1cbi8qKlxuICogVmFsaWRhdGUgdGhlIHN0YXR1cyBjb2RlIGFuZCBtZXNzYWdlIGJvZHkgb2YgYSByZWRpcmVjdCByZXNwb25zZVxuICpcbiAqIEBpbnRlcm5hbFxuICovXG5leHBvcnQgZnVuY3Rpb24gdmFsaWRhdGVSZWRpcmVjdFJlc3BvbnNlKHJlZGlyZWN0UmVzcG9uc2U6IFJlZGlyZWN0UmVzcG9uc2UpIHtcbiAgICBpZiAocmVkaXJlY3RSZXNwb25zZS5wcm90b2NvbCAmJiAhL14oSFRUUFM/fCNcXHtwcm90b2NvbFxcfSkkL2kudGVzdChyZWRpcmVjdFJlc3BvbnNlLnByb3RvY29sKSkge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ2Bwcm90b2NvbGAgbXVzdCBiZSBIVFRQLCBIVFRQUywgb3IgI3twcm90b2NvbH0uJyk7XG4gICAgfVxuICAgIGlmICghcmVkaXJlY3RSZXNwb25zZS5zdGF0dXNDb2RlIHx8ICEvXkhUVFBfMzBbMTJdJC8udGVzdChyZWRpcmVjdFJlc3BvbnNlLnN0YXR1c0NvZGUpKSB7XG4gICAgICAgIHRocm93IG5ldyBFcnJvcignYHN0YXR1c0NvZGVgIG11c3QgYmUgSFRUUF8zMDEgb3IgSFRUUF8zMDIuJyk7XG4gICAgfVxufVxuIl19