"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IpTarget = void 0;
const elbv2 = require("../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
/**
 * An IP address that is a target for load balancing.
 *
 * Specify IP addresses from the subnets of the virtual private cloud (VPC) for
 * the target group, the RFC 1918 range (10.0.0.0/8, 172.16.0.0/12, and
 * 192.168.0.0/16), and the RFC 6598 range (100.64.0.0/10). You can't specify
 * publicly routable IP addresses.
 *
 * If you register a target of this type, you are responsible for making
 * sure the load balancer's security group can send packets to the IP address.
 */
class IpTarget {
    /**
     * Create a new IPAddress target
     *
     * The availabilityZone parameter determines whether the target receives
     * traffic from the load balancer nodes in the specified Availability Zone
     * or from all enabled Availability Zones for the load balancer.
     *
     * This parameter is not supported if the target type of the target group
     * is instance. If the IP address is in a subnet of the VPC for the target
     * group, the Availability Zone is automatically detected and this
     * parameter is optional. If the IP address is outside the VPC, this
     * parameter is required.
     *
     * With an Application Load Balancer, if the IP address is outside the VPC
     * for the target group, the only supported value is all.
     *
     * Default is automatic.
     *
     * @param ipAddress The IP Address to load balance to
     * @param port Override the group's default port
     * @param availabilityZone Availability zone to send traffic from
     */
    constructor(ipAddress, port, availabilityZone) {
        this.ipAddress = ipAddress;
        this.port = port;
        this.availabilityZone = availabilityZone;
    }
    /**
     * Register this instance target with a load balancer
     *
     * Don't call this, it is called automatically when you add the target to a
     * load balancer.
     */
    attachToApplicationTargetGroup(targetGroup) {
        return this.attach(targetGroup);
    }
    /**
     * Register this instance target with a load balancer
     *
     * Don't call this, it is called automatically when you add the target to a
     * load balancer.
     */
    attachToNetworkTargetGroup(targetGroup) {
        return this.attach(targetGroup);
    }
    attach(_targetGroup) {
        return {
            targetType: elbv2.TargetType.IP,
            targetJson: { id: this.ipAddress, port: this.port, availabilityZone: this.availabilityZone },
        };
    }
}
exports.IpTarget = IpTarget;
//# sourceMappingURL=data:application/json;base64,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