"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ListenerPort = exports.LoadBalancer = exports.LoadBalancingProtocol = void 0;
const aws_ec2_1 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const elasticloadbalancing_generated_1 = require("./elasticloadbalancing.generated");
var LoadBalancingProtocol;
(function (LoadBalancingProtocol) {
    LoadBalancingProtocol["TCP"] = "tcp";
    LoadBalancingProtocol["SSL"] = "ssl";
    LoadBalancingProtocol["HTTP"] = "http";
    LoadBalancingProtocol["HTTPS"] = "https";
})(LoadBalancingProtocol = exports.LoadBalancingProtocol || (exports.LoadBalancingProtocol = {}));
/**
 * A load balancer with a single listener
 *
 * Routes to a fleet of of instances in a VPC.
 */
class LoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * An object controlling specifically the connections for each listener added to this load balancer
         */
        this.listenerPorts = [];
        this.listeners = [];
        this.instancePorts = [];
        this.targets = [];
        this.securityGroup = new aws_ec2_1.SecurityGroup(this, 'SecurityGroup', { vpc: props.vpc, allowAllOutbound: false });
        this.connections = new aws_ec2_1.Connections({ securityGroups: [this.securityGroup] });
        // Depending on whether the ELB has public or internal IPs, pick the right backend subnets
        const selectedSubnets = loadBalancerSubnets(props);
        this.elb = new elasticloadbalancing_generated_1.CfnLoadBalancer(this, 'Resource', {
            securityGroups: [this.securityGroup.securityGroupId],
            subnets: selectedSubnets.subnetIds,
            listeners: core_1.Lazy.anyValue({ produce: () => this.listeners }),
            scheme: props.internetFacing ? 'internet-facing' : 'internal',
            healthCheck: props.healthCheck && healthCheckToJSON(props.healthCheck),
            crossZone: (props.crossZone === undefined || props.crossZone) ? true : false,
        });
        if (props.internetFacing) {
            this.elb.node.addDependency(selectedSubnets.internetConnectivityEstablished);
        }
        ifUndefined(props.listeners, []).forEach(b => this.addListener(b));
        ifUndefined(props.targets, []).forEach(t => this.addTarget(t));
    }
    /**
     * Add a backend to the load balancer
     *
     * @returns A ListenerPort object that controls connections to the listener port
     */
    addListener(listener) {
        const protocol = ifUndefinedLazy(listener.externalProtocol, () => wellKnownProtocol(listener.externalPort));
        const instancePort = listener.internalPort || listener.externalPort;
        const instanceProtocol = ifUndefined(listener.internalProtocol, ifUndefined(tryWellKnownProtocol(instancePort), isHttpProtocol(protocol) ? LoadBalancingProtocol.HTTP : LoadBalancingProtocol.TCP));
        this.listeners.push({
            loadBalancerPort: listener.externalPort.toString(),
            protocol,
            instancePort: instancePort.toString(),
            instanceProtocol,
            sslCertificateId: listener.sslCertificateId,
            policyNames: listener.policyNames,
        });
        const port = new ListenerPort(this.securityGroup, aws_ec2_1.Port.tcp(listener.externalPort));
        // Allow connections on the public port for all supplied peers (default: everyone)
        ifUndefined(listener.allowConnectionsFrom, [aws_ec2_1.Peer.anyIpv4()]).forEach(peer => {
            port.connections.allowDefaultPortFrom(peer, `Default rule allow on ${listener.externalPort}`);
        });
        this.newInstancePort(instancePort);
        // Keep track using array so user can get to them even if they were all supplied in the constructor
        this.listenerPorts.push(port);
        return port;
    }
    addTarget(target) {
        target.attachToClassicLB(this);
        this.newTarget(target);
    }
    /**
     * @attribute
     */
    get loadBalancerName() {
        return this.elb.ref;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneNameId() {
        return this.elb.attrCanonicalHostedZoneNameId;
    }
    /**
     * @attribute
     */
    get loadBalancerCanonicalHostedZoneName() {
        return this.elb.attrCanonicalHostedZoneName;
    }
    /**
     * @attribute
     */
    get loadBalancerDnsName() {
        return this.elb.attrDnsName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupGroupName() {
        return this.elb.attrSourceSecurityGroupGroupName;
    }
    /**
     * @attribute
     */
    get loadBalancerSourceSecurityGroupOwnerAlias() {
        return this.elb.attrSourceSecurityGroupOwnerAlias;
    }
    /**
     * Allow connections to all existing targets on new instance port
     */
    newInstancePort(instancePort) {
        this.targets.forEach(t => this.allowTargetConnection(instancePort, t));
        // Keep track of port for future targets
        this.instancePorts.push(instancePort);
    }
    /**
     * Allow connections to target on all existing instance ports
     */
    newTarget(target) {
        this.instancePorts.forEach(p => this.allowTargetConnection(p, target));
        // Keep track of target for future listeners.
        this.targets.push(target);
    }
    /**
     * Allow connections for a single (port, target) pair
     */
    allowTargetConnection(instancePort, target) {
        this.connections.allowTo(target, aws_ec2_1.Port.tcp(instancePort), `Port ${instancePort} LB to fleet`);
    }
}
exports.LoadBalancer = LoadBalancer;
/**
 * Reference to a listener's port just created.
 *
 * This implements IConnectable with a default port (the port that an ELB
 * listener was just created on) for a given security group so that it can be
 * conveniently used just like any Connectable. E.g:
 *
 *    const listener = elb.addListener(...);
 *
 *    listener.connections.allowDefaultPortFromAnyIPv4();
 *    // or
 *    instance.connections.allowToDefaultPort(listener);
 */
class ListenerPort {
    constructor(securityGroup, defaultPort) {
        this.connections = new aws_ec2_1.Connections({ securityGroups: [securityGroup], defaultPort });
    }
}
exports.ListenerPort = ListenerPort;
function wellKnownProtocol(port) {
    const proto = tryWellKnownProtocol(port);
    if (!proto) {
        throw new Error(`Please supply protocol to go with port ${port}`);
    }
    return proto;
}
function tryWellKnownProtocol(port) {
    if (port === 80) {
        return LoadBalancingProtocol.HTTP;
    }
    if (port === 443) {
        return LoadBalancingProtocol.HTTPS;
    }
    return undefined;
}
function isHttpProtocol(proto) {
    return proto === LoadBalancingProtocol.HTTPS || proto === LoadBalancingProtocol.HTTP;
}
function ifUndefined(x, def) {
    return x != null ? x : def;
}
function ifUndefinedLazy(x, def) {
    return x != null ? x : def();
}
/**
 * Turn health check parameters into a parameter blob for the LB
 */
function healthCheckToJSON(healthCheck) {
    const protocol = ifUndefined(healthCheck.protocol, ifUndefined(tryWellKnownProtocol(healthCheck.port), LoadBalancingProtocol.TCP));
    const path = protocol === LoadBalancingProtocol.HTTP || protocol === LoadBalancingProtocol.HTTPS ? ifUndefined(healthCheck.path, '/') : '';
    const target = `${protocol.toUpperCase()}:${healthCheck.port}${path}`;
    return {
        healthyThreshold: ifUndefined(healthCheck.healthyThreshold, 2).toString(),
        interval: (healthCheck.interval || core_1.Duration.seconds(30)).toSeconds().toString(),
        target,
        timeout: (healthCheck.timeout || core_1.Duration.seconds(5)).toSeconds().toString(),
        unhealthyThreshold: ifUndefined(healthCheck.unhealthyThreshold, 5).toString(),
    };
}
function loadBalancerSubnets(props) {
    if (props.subnetSelection !== undefined) {
        return props.vpc.selectSubnets(props.subnetSelection);
    }
    else if (props.internetFacing) {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PUBLIC,
        });
    }
    else {
        return props.vpc.selectSubnets({
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
        });
    }
}
//# sourceMappingURL=data:application/json;base64,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