"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceAccount = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const k8s_manifest_1 = require("./k8s-manifest");
/**
 * Service Account
 */
class ServiceAccount extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const { cluster } = props;
        this.serviceAccountName = (_a = props.name) !== null && _a !== void 0 ? _a : this.node.uniqueId.toLowerCase();
        this.serviceAccountNamespace = (_b = props.namespace) !== null && _b !== void 0 ? _b : 'default';
        /* Add conditions to the role to improve security. This prevents other pods in the same namespace to assume the role.
        * See documentation: https://docs.aws.amazon.com/eks/latest/userguide/create-service-account-iam-policy-and-role.html
        */
        const conditions = new core_1.CfnJson(this, 'ConditionJson', {
            value: {
                [`${cluster.clusterOpenIdConnectIssuer}:aud`]: 'sts.amazonaws.com',
                [`${cluster.clusterOpenIdConnectIssuer}:sub`]: `system:serviceaccount:${this.serviceAccountNamespace}:${this.serviceAccountName}`,
            },
        });
        const principal = new aws_iam_1.OpenIdConnectPrincipal(cluster.openIdConnectProvider).withConditions({
            StringEquals: conditions,
        });
        this.role = new aws_iam_1.Role(this, 'Role', { assumedBy: principal });
        this.assumeRoleAction = this.role.assumeRoleAction;
        this.grantPrincipal = this.role.grantPrincipal;
        this.policyFragment = this.role.policyFragment;
        // Note that we cannot use `cluster.addManifest` here because that would create the manifest
        // constrct in the scope of the cluster stack, which might be a different stack than this one.
        // This means that the cluster stack would depend on this stack because of the role,
        // and since this stack inherintely depends on the cluster stack, we will have a circular dependency.
        new k8s_manifest_1.KubernetesManifest(this, `manifest-${id}ServiceAccountResource`, {
            cluster,
            manifest: [{
                    apiVersion: 'v1',
                    kind: 'ServiceAccount',
                    metadata: {
                        name: this.serviceAccountName,
                        namespace: this.serviceAccountNamespace,
                        labels: {
                            'app.kubernetes.io/name': this.serviceAccountName,
                        },
                        annotations: {
                            'eks.amazonaws.com/role-arn': this.role.roleArn,
                        },
                    },
                }],
        });
    }
    addToPolicy(statement) {
        return this.role.addToPolicy(statement);
    }
    addToPrincipalPolicy(statement) {
        return this.role.addToPrincipalPolicy(statement);
    }
}
exports.ServiceAccount = ServiceAccount;
//# sourceMappingURL=data:application/json;base64,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