"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.getOrCreateKubectlLayer = exports.KubectlProvider = void 0;
const path = require("path");
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cr = require("../../custom-resources"); // Automatically re-written from '@aws-cdk/custom-resources'
const cluster_1 = require("./cluster");
const kubectl_layer_1 = require("./kubectl-layer");
class KubectlProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const cluster = props.cluster;
        if (!cluster.kubectlRole) {
            throw new Error('"kubectlRole" is not defined, cannot issue kubectl commands against this cluster');
        }
        if (cluster.kubectlPrivateSubnets && !cluster.kubectlSecurityGroup) {
            throw new Error('"kubectlSecurityGroup" is required if "kubectlSubnets" is specified');
        }
        const layer = (_a = cluster.kubectlLayer) !== null && _a !== void 0 ? _a : getOrCreateKubectlLayer(this);
        const handler = new lambda.Function(this, 'Handler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'kubectl-handler')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            description: 'onEvent handler for EKS kubectl resource provider',
            layers: [layer],
            memorySize: 256,
            environment: cluster.kubectlEnvironment,
            // defined only when using private access
            vpc: cluster.kubectlPrivateSubnets ? cluster.vpc : undefined,
            securityGroups: cluster.kubectlSecurityGroup ? [cluster.kubectlSecurityGroup] : undefined,
            vpcSubnets: cluster.kubectlPrivateSubnets ? { subnets: cluster.kubectlPrivateSubnets } : undefined,
        });
        this.handlerRole = handler.role;
        this.handlerRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeCluster'],
            resources: [cluster.clusterArn],
        }));
        // allow this handler to assume the kubectl role
        cluster.kubectlRole.grant(this.handlerRole, 'sts:AssumeRole');
        const provider = new cr.Provider(this, 'Provider', {
            onEventHandler: handler,
        });
        this.serviceToken = provider.serviceToken;
        this.roleArn = cluster.kubectlRole.roleArn;
    }
    static getOrCreate(scope, cluster) {
        // if this is an "owned" cluster, it has a provider associated with it
        if (cluster instanceof cluster_1.Cluster) {
            return cluster._attachKubectlResourceScope(scope);
        }
        // if this is an imported cluster, we need to provision a custom resource provider in this stack
        // we will define one per stack for each cluster based on the cluster uniqueid
        const uid = `${cluster.node.uniqueId}-KubectlProvider`;
        const stack = core_1.Stack.of(scope);
        let provider = stack.node.tryFindChild(uid);
        if (!provider) {
            provider = new KubectlProvider(stack, uid, { cluster });
        }
        return provider;
    }
}
exports.KubectlProvider = KubectlProvider;
/**
 * Gets or create a singleton instance of KubectlLayer.
 *
 * (exported for unit tests).
 */
function getOrCreateKubectlLayer(scope, props = {}) {
    const stack = core_1.Stack.of(scope);
    const id = 'kubectl-layer-' + (props.version ? props.version : '8C2542BC-BF2B-4DFE-B765-E181FD30A9A0');
    const exists = stack.node.tryFindChild(id);
    if (exists) {
        return exists;
    }
    return new kubectl_layer_1.KubectlLayer(stack, id, props);
}
exports.getOrCreateKubectlLayer = getOrCreateKubectlLayer;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia3ViZWN0bC1wcm92aWRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImt1YmVjdGwtcHJvdmlkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBQzdCLHFDQUFxQyxDQUFDLG1EQUFtRDtBQUN6RiwyQ0FBMkMsQ0FBQyxzREFBc0Q7QUFDbEcscUNBQXFFLENBQUMsZ0RBQWdEO0FBQ3RILDZDQUE2QyxDQUFDLDREQUE0RDtBQUMxRyx1Q0FBOEM7QUFDOUMsbURBQWtFO0FBT2xFLE1BQWEsZUFBZ0IsU0FBUSxrQkFBVztJQTRCNUMsWUFBbUIsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBMkI7O1FBQ3hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFDakIsTUFBTSxPQUFPLEdBQUcsS0FBSyxDQUFDLE9BQU8sQ0FBQztRQUM5QixJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsRUFBRTtZQUN0QixNQUFNLElBQUksS0FBSyxDQUFDLGtGQUFrRixDQUFDLENBQUM7U0FDdkc7UUFDRCxJQUFJLE9BQU8sQ0FBQyxxQkFBcUIsSUFBSSxDQUFDLE9BQU8sQ0FBQyxvQkFBb0IsRUFBRTtZQUNoRSxNQUFNLElBQUksS0FBSyxDQUFDLHFFQUFxRSxDQUFDLENBQUM7U0FDMUY7UUFDRCxNQUFNLEtBQUssU0FBRyxPQUFPLENBQUMsWUFBWSxtQ0FBSSx1QkFBdUIsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNwRSxNQUFNLE9BQU8sR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUNqRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsaUJBQWlCLENBQUMsQ0FBQztZQUNwRSxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVO1lBQ2xDLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixXQUFXLEVBQUUsbURBQW1EO1lBQ2hFLE1BQU0sRUFBRSxDQUFDLEtBQUssQ0FBQztZQUNmLFVBQVUsRUFBRSxHQUFHO1lBQ2YsV0FBVyxFQUFFLE9BQU8sQ0FBQyxrQkFBa0I7WUFDdkMseUNBQXlDO1lBQ3pDLEdBQUcsRUFBRSxPQUFPLENBQUMscUJBQXFCLENBQUMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDNUQsY0FBYyxFQUFFLE9BQU8sQ0FBQyxvQkFBb0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN6RixVQUFVLEVBQUUsT0FBTyxDQUFDLHFCQUFxQixDQUFDLENBQUMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxPQUFPLENBQUMscUJBQXFCLEVBQUUsQ0FBQyxDQUFDLENBQUMsU0FBUztTQUNyRyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsV0FBVyxHQUFHLE9BQU8sQ0FBQyxJQUFLLENBQUM7UUFDakMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQ2pELE9BQU8sRUFBRSxDQUFDLHFCQUFxQixDQUFDO1lBQ2hDLFNBQVMsRUFBRSxDQUFDLE9BQU8sQ0FBQyxVQUFVLENBQUM7U0FDbEMsQ0FBQyxDQUFDLENBQUM7UUFDSixnREFBZ0Q7UUFDaEQsT0FBTyxDQUFDLFdBQVcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO1FBQzlELE1BQU0sUUFBUSxHQUFHLElBQUksRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQy9DLGNBQWMsRUFBRSxPQUFPO1NBQzFCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxZQUFZLEdBQUcsUUFBUSxDQUFDLFlBQVksQ0FBQztRQUMxQyxJQUFJLENBQUMsT0FBTyxHQUFHLE9BQU8sQ0FBQyxXQUFXLENBQUMsT0FBTyxDQUFDO0lBQy9DLENBQUM7SUEvRE0sTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFnQixFQUFFLE9BQWlCO1FBQ3pELHNFQUFzRTtRQUN0RSxJQUFJLE9BQU8sWUFBWSxpQkFBTyxFQUFFO1lBQzVCLE9BQU8sT0FBTyxDQUFDLDJCQUEyQixDQUFDLEtBQUssQ0FBQyxDQUFDO1NBQ3JEO1FBQ0QsZ0dBQWdHO1FBQ2hHLDhFQUE4RTtRQUM5RSxNQUFNLEdBQUcsR0FBRyxHQUFHLE9BQU8sQ0FBQyxJQUFJLENBQUMsUUFBUSxrQkFBa0IsQ0FBQztRQUN2RCxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLElBQUksUUFBUSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEdBQUcsQ0FBb0IsQ0FBQztRQUMvRCxJQUFJLENBQUMsUUFBUSxFQUFFO1lBQ1gsUUFBUSxHQUFHLElBQUksZUFBZSxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO1NBQzNEO1FBQ0QsT0FBTyxRQUFRLENBQUM7SUFDcEIsQ0FBQztDQWtESjtBQWpFRCwwQ0FpRUM7QUFDRDs7OztHQUlHO0FBQ0gsU0FBZ0IsdUJBQXVCLENBQUMsS0FBZ0IsRUFBRSxRQUEyQixFQUFFO0lBQ25GLE1BQU0sS0FBSyxHQUFHLFlBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDOUIsTUFBTSxFQUFFLEdBQUcsZ0JBQWdCLEdBQUcsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxzQ0FBc0MsQ0FBQyxDQUFDO0lBQ3ZHLE1BQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBaUIsQ0FBQztJQUMzRCxJQUFJLE1BQU0sRUFBRTtRQUNSLE9BQU8sTUFBTSxDQUFDO0tBQ2pCO0lBQ0QsT0FBTyxJQUFJLDRCQUFZLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztBQUM5QyxDQUFDO0FBUkQsMERBUUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gXCIuLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gXCIuLi8uLi9hd3MtbGFtYmRhXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJ1xuaW1wb3J0IHsgQ29uc3RydWN0LCBEdXJhdGlvbiwgU3RhY2ssIE5lc3RlZFN0YWNrIH0gZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgKiBhcyBjciBmcm9tIFwiLi4vLi4vY3VzdG9tLXJlc291cmNlc1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY3VzdG9tLXJlc291cmNlcydcbmltcG9ydCB7IElDbHVzdGVyLCBDbHVzdGVyIH0gZnJvbSAnLi9jbHVzdGVyJztcbmltcG9ydCB7IEt1YmVjdGxMYXllciwgS3ViZWN0bExheWVyUHJvcHMgfSBmcm9tICcuL2t1YmVjdGwtbGF5ZXInO1xuZXhwb3J0IGludGVyZmFjZSBLdWJlY3RsUHJvdmlkZXJQcm9wcyB7XG4gICAgLyoqXG4gICAgICogVGhlIGNsdXN0ZXIgdG8gY29udHJvbC5cbiAgICAgKi9cbiAgICByZWFkb25seSBjbHVzdGVyOiBJQ2x1c3Rlcjtcbn1cbmV4cG9ydCBjbGFzcyBLdWJlY3RsUHJvdmlkZXIgZXh0ZW5kcyBOZXN0ZWRTdGFjayB7XG4gICAgcHVibGljIHN0YXRpYyBnZXRPckNyZWF0ZShzY29wZTogQ29uc3RydWN0LCBjbHVzdGVyOiBJQ2x1c3Rlcikge1xuICAgICAgICAvLyBpZiB0aGlzIGlzIGFuIFwib3duZWRcIiBjbHVzdGVyLCBpdCBoYXMgYSBwcm92aWRlciBhc3NvY2lhdGVkIHdpdGggaXRcbiAgICAgICAgaWYgKGNsdXN0ZXIgaW5zdGFuY2VvZiBDbHVzdGVyKSB7XG4gICAgICAgICAgICByZXR1cm4gY2x1c3Rlci5fYXR0YWNoS3ViZWN0bFJlc291cmNlU2NvcGUoc2NvcGUpO1xuICAgICAgICB9XG4gICAgICAgIC8vIGlmIHRoaXMgaXMgYW4gaW1wb3J0ZWQgY2x1c3Rlciwgd2UgbmVlZCB0byBwcm92aXNpb24gYSBjdXN0b20gcmVzb3VyY2UgcHJvdmlkZXIgaW4gdGhpcyBzdGFja1xuICAgICAgICAvLyB3ZSB3aWxsIGRlZmluZSBvbmUgcGVyIHN0YWNrIGZvciBlYWNoIGNsdXN0ZXIgYmFzZWQgb24gdGhlIGNsdXN0ZXIgdW5pcXVlaWRcbiAgICAgICAgY29uc3QgdWlkID0gYCR7Y2x1c3Rlci5ub2RlLnVuaXF1ZUlkfS1LdWJlY3RsUHJvdmlkZXJgO1xuICAgICAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHNjb3BlKTtcbiAgICAgICAgbGV0IHByb3ZpZGVyID0gc3RhY2subm9kZS50cnlGaW5kQ2hpbGQodWlkKSBhcyBLdWJlY3RsUHJvdmlkZXI7XG4gICAgICAgIGlmICghcHJvdmlkZXIpIHtcbiAgICAgICAgICAgIHByb3ZpZGVyID0gbmV3IEt1YmVjdGxQcm92aWRlcihzdGFjaywgdWlkLCB7IGNsdXN0ZXIgfSk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHByb3ZpZGVyO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBUaGUgY3VzdG9tIHJlc291cmNlIHByb3ZpZGVyJ3Mgc2VydmljZSB0b2tlbi5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgc2VydmljZVRva2VuOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIElBTSByb2xlIHRvIGFzc3VtZSBpbiBvcmRlciB0byBwZXJmb3JtIGt1YmVjdGwgb3BlcmF0aW9ucyBhZ2FpbnN0IHRoaXMgY2x1c3Rlci5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgcm9sZUFybjogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBJQU0gZXhlY3V0aW9uIHJvbGUgb2YgdGhlIGhhbmRsZXIuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGhhbmRsZXJSb2xlOiBpYW0uSVJvbGU7XG4gICAgcHVibGljIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBLdWJlY3RsUHJvdmlkZXJQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgICAgICBjb25zdCBjbHVzdGVyID0gcHJvcHMuY2x1c3RlcjtcbiAgICAgICAgaWYgKCFjbHVzdGVyLmt1YmVjdGxSb2xlKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ1wia3ViZWN0bFJvbGVcIiBpcyBub3QgZGVmaW5lZCwgY2Fubm90IGlzc3VlIGt1YmVjdGwgY29tbWFuZHMgYWdhaW5zdCB0aGlzIGNsdXN0ZXInKTtcbiAgICAgICAgfVxuICAgICAgICBpZiAoY2x1c3Rlci5rdWJlY3RsUHJpdmF0ZVN1Ym5ldHMgJiYgIWNsdXN0ZXIua3ViZWN0bFNlY3VyaXR5R3JvdXApIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignXCJrdWJlY3RsU2VjdXJpdHlHcm91cFwiIGlzIHJlcXVpcmVkIGlmIFwia3ViZWN0bFN1Ym5ldHNcIiBpcyBzcGVjaWZpZWQnKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBsYXllciA9IGNsdXN0ZXIua3ViZWN0bExheWVyID8/IGdldE9yQ3JlYXRlS3ViZWN0bExheWVyKHRoaXMpO1xuICAgICAgICBjb25zdCBoYW5kbGVyID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCAnSGFuZGxlcicsIHtcbiAgICAgICAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAna3ViZWN0bC1oYW5kbGVyJykpLFxuICAgICAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuUFlUSE9OXzNfNyxcbiAgICAgICAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgICAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTUpLFxuICAgICAgICAgICAgZGVzY3JpcHRpb246ICdvbkV2ZW50IGhhbmRsZXIgZm9yIEVLUyBrdWJlY3RsIHJlc291cmNlIHByb3ZpZGVyJyxcbiAgICAgICAgICAgIGxheWVyczogW2xheWVyXSxcbiAgICAgICAgICAgIG1lbW9yeVNpemU6IDI1NixcbiAgICAgICAgICAgIGVudmlyb25tZW50OiBjbHVzdGVyLmt1YmVjdGxFbnZpcm9ubWVudCxcbiAgICAgICAgICAgIC8vIGRlZmluZWQgb25seSB3aGVuIHVzaW5nIHByaXZhdGUgYWNjZXNzXG4gICAgICAgICAgICB2cGM6IGNsdXN0ZXIua3ViZWN0bFByaXZhdGVTdWJuZXRzID8gY2x1c3Rlci52cGMgOiB1bmRlZmluZWQsXG4gICAgICAgICAgICBzZWN1cml0eUdyb3VwczogY2x1c3Rlci5rdWJlY3RsU2VjdXJpdHlHcm91cCA/IFtjbHVzdGVyLmt1YmVjdGxTZWN1cml0eUdyb3VwXSA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgIHZwY1N1Ym5ldHM6IGNsdXN0ZXIua3ViZWN0bFByaXZhdGVTdWJuZXRzID8geyBzdWJuZXRzOiBjbHVzdGVyLmt1YmVjdGxQcml2YXRlU3VibmV0cyB9IDogdW5kZWZpbmVkLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5oYW5kbGVyUm9sZSA9IGhhbmRsZXIucm9sZSE7XG4gICAgICAgIHRoaXMuaGFuZGxlclJvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgYWN0aW9uczogWydla3M6RGVzY3JpYmVDbHVzdGVyJ10sXG4gICAgICAgICAgICByZXNvdXJjZXM6IFtjbHVzdGVyLmNsdXN0ZXJBcm5dLFxuICAgICAgICB9KSk7XG4gICAgICAgIC8vIGFsbG93IHRoaXMgaGFuZGxlciB0byBhc3N1bWUgdGhlIGt1YmVjdGwgcm9sZVxuICAgICAgICBjbHVzdGVyLmt1YmVjdGxSb2xlLmdyYW50KHRoaXMuaGFuZGxlclJvbGUsICdzdHM6QXNzdW1lUm9sZScpO1xuICAgICAgICBjb25zdCBwcm92aWRlciA9IG5ldyBjci5Qcm92aWRlcih0aGlzLCAnUHJvdmlkZXInLCB7XG4gICAgICAgICAgICBvbkV2ZW50SGFuZGxlcjogaGFuZGxlcixcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuc2VydmljZVRva2VuID0gcHJvdmlkZXIuc2VydmljZVRva2VuO1xuICAgICAgICB0aGlzLnJvbGVBcm4gPSBjbHVzdGVyLmt1YmVjdGxSb2xlLnJvbGVBcm47XG4gICAgfVxufVxuLyoqXG4gKiBHZXRzIG9yIGNyZWF0ZSBhIHNpbmdsZXRvbiBpbnN0YW5jZSBvZiBLdWJlY3RsTGF5ZXIuXG4gKlxuICogKGV4cG9ydGVkIGZvciB1bml0IHRlc3RzKS5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldE9yQ3JlYXRlS3ViZWN0bExheWVyKHNjb3BlOiBDb25zdHJ1Y3QsIHByb3BzOiBLdWJlY3RsTGF5ZXJQcm9wcyA9IHt9KTogS3ViZWN0bExheWVyIHtcbiAgICBjb25zdCBzdGFjayA9IFN0YWNrLm9mKHNjb3BlKTtcbiAgICBjb25zdCBpZCA9ICdrdWJlY3RsLWxheWVyLScgKyAocHJvcHMudmVyc2lvbiA/IHByb3BzLnZlcnNpb24gOiAnOEMyNTQyQkMtQkYyQi00REZFLUI3NjUtRTE4MUZEMzBBOUEwJyk7XG4gICAgY29uc3QgZXhpc3RzID0gc3RhY2subm9kZS50cnlGaW5kQ2hpbGQoaWQpIGFzIEt1YmVjdGxMYXllcjtcbiAgICBpZiAoZXhpc3RzKSB7XG4gICAgICAgIHJldHVybiBleGlzdHM7XG4gICAgfVxuICAgIHJldHVybiBuZXcgS3ViZWN0bExheWVyKHN0YWNrLCBpZCwgcHJvcHMpO1xufVxuIl19