"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FargateProfile = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * Fargate profiles allows an administrator to declare which pods run on
 * Fargate. This declaration is done through the profile’s selectors. Each
 * profile can have up to five selectors that contain a namespace and optional
 * labels. You must define a namespace for every selector. The label field
 * consists of multiple optional key-value pairs. Pods that match a selector (by
 * matching a namespace for the selector and all of the labels specified in the
 * selector) are scheduled on Fargate. If a namespace selector is defined
 * without any labels, Amazon EKS will attempt to schedule all pods that run in
 * that namespace onto Fargate using the profile. If a to-be-scheduled pod
 * matches any of the selectors in the Fargate profile, then that pod is
 * scheduled on Fargate.
 *
 * If a pod matches multiple Fargate profiles, Amazon EKS picks one of the
 * matches at random. In this case, you can specify which profile a pod should
 * use by adding the following Kubernetes label to the pod specification:
 * eks.amazonaws.com/fargate-profile: profile_name. However, the pod must still
 * match a selector in that profile in order to be scheduled onto Fargate.
 */
class FargateProfile extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this, {
            adminRole: props.cluster.adminRole,
        });
        this.podExecutionRole = (_a = props.podExecutionRole) !== null && _a !== void 0 ? _a : new iam.Role(this, 'PodExecutionRole', {
            assumedBy: new iam.ServicePrincipal('eks-fargate-pods.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSFargatePodExecutionRolePolicy')],
        });
        this.podExecutionRole.grantPassRole(props.cluster.adminRole);
        let subnets;
        if (props.vpc) {
            const selection = (_b = props.subnetSelection) !== null && _b !== void 0 ? _b : { subnetType: ec2.SubnetType.PRIVATE };
            subnets = props.vpc.selectSubnets(selection).subnetIds;
        }
        if (props.selectors.length < 1) {
            throw new Error('Fargate profile requires at least one selector');
        }
        if (props.selectors.length > 5) {
            throw new Error('Fargate profile supports up to five selectors');
        }
        this.tags = new core_1.TagManager(core_1.TagType.MAP, 'AWS::EKS::FargateProfile');
        const resource = new core_1.CustomResource(this, 'Resource', {
            serviceToken: provider.serviceToken,
            resourceType: consts_1.FARGATE_PROFILE_RESOURCE_TYPE,
            properties: {
                AssumeRoleArn: props.cluster.adminRole.roleArn,
                Config: {
                    clusterName: props.cluster.clusterName,
                    fargateProfileName: props.fargateProfileName,
                    podExecutionRoleArn: this.podExecutionRole.roleArn,
                    selectors: props.selectors,
                    subnets,
                    tags: core_1.Lazy.anyValue({ produce: () => this.tags.renderTags() }),
                },
            },
        });
        this.fargateProfileArn = resource.getAttString('fargateProfileArn');
        this.fargateProfileName = resource.ref;
        // Fargate profiles must be created sequentially. If other profile(s) already
        // exist on the same cluster, create a dependency to force sequential creation.
        const clusterFargateProfiles = props.cluster._attachFargateProfile(this);
        if (clusterFargateProfiles.length > 1) {
            const previousProfile = clusterFargateProfiles[clusterFargateProfiles.length - 2];
            resource.node.addDependency(previousProfile);
        }
        // map the fargate pod execution role to the relevant groups in rbac
        // see https://github.com/aws/aws-cdk/issues/7981
        props.cluster.awsAuth.addRoleMapping(this.podExecutionRole, {
            username: 'system:node:{{SessionName}}',
            groups: [
                'system:bootstrappers',
                'system:nodes',
                'system:node-proxier',
            ],
        });
    }
}
exports.FargateProfile = FargateProfile;
//# sourceMappingURL=data:application/json;base64,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