"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.nodeTypeForInstanceType = exports.NodeType = exports.EksOptimizedImage = exports.Cluster = void 0;
const path = require("path");
const autoscaling = require("../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const aws_auth_1 = require("./aws-auth");
const cluster_resource_1 = require("./cluster-resource");
const eks_generated_1 = require("./eks.generated");
const helm_chart_1 = require("./helm-chart");
const k8s_resource_1 = require("./k8s-resource");
const kubectl_layer_1 = require("./kubectl-layer");
const spot_interrupt_handler_1 = require("./spot-interrupt-handler");
const user_data_1 = require("./user-data");
// defaults are based on https://eksctl.io
const DEFAULT_CAPACITY_COUNT = 2;
const DEFAULT_CAPACITY_TYPE = ec2.InstanceType.of(ec2.InstanceClass.M5, ec2.InstanceSize.LARGE);
/**
 * A Cluster represents a managed Kubernetes Service (EKS)
 *
 * This is a fully managed cluster of API Servers (control-plane)
 * The user is still required to create the worker nodes.
 *
 * @resource AWS::EKS::Cluster
 */
class Cluster extends core_1.Resource {
    /**
     * Initiates an EKS Cluster with the supplied arguments
     *
     * @param scope a Construct, most likely a cdk.Stack created
     * @param name the name of the Construct to create
     * @param props properties in the IClusterProps interface
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        core_1.Annotations.of(this).addWarning('The @aws-cdk/aws-eks-legacy module will no longer be released as part of the AWS CDK starting March 1st, 2020. Please refer to https://github.com/aws/aws-cdk/issues/5544 for upgrade instructions');
        const stack = core_1.Stack.of(this);
        this.vpc = props.vpc || new ec2.Vpc(this, 'DefaultVpc');
        this.version = props.version;
        this.tagSubnets();
        this.role = props.role || new iam.Role(this, 'ClusterRole', {
            assumedBy: new iam.ServicePrincipal('eks.amazonaws.com'),
            managedPolicies: [
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSClusterPolicy'),
                iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSServicePolicy'),
            ],
        });
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'ControlPlaneSecurityGroup', {
            vpc: this.vpc,
            description: 'EKS Control Plane Security Group',
        });
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort: ec2.Port.tcp(443),
        });
        // Get subnetIds for all selected subnets
        const placements = props.vpcSubnets || [{ subnetType: ec2.SubnetType.PUBLIC }, { subnetType: ec2.SubnetType.PRIVATE }];
        const subnetIds = [...new Set(Array().concat(...placements.map(s => this.vpc.selectSubnets(s).subnetIds)))];
        const clusterProps = {
            name: this.physicalName,
            roleArn: this.role.roleArn,
            version: props.version,
            resourcesVpcConfig: {
                securityGroupIds: [securityGroup.securityGroupId],
                subnetIds,
            },
        };
        let resource;
        this.kubectlEnabled = props.kubectlEnabled === undefined ? true : props.kubectlEnabled;
        if (this.kubectlEnabled) {
            resource = new cluster_resource_1.ClusterResource(this, 'Resource', clusterProps);
            this._defaultMastersRole = resource.creationRole;
        }
        else {
            resource = new eks_generated_1.CfnCluster(this, 'Resource', clusterProps);
        }
        this.clusterName = this.getResourceNameAttribute(resource.ref);
        this.clusterArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'eks',
            resource: 'cluster',
            resourceName: this.physicalName,
        });
        this.clusterEndpoint = resource.attrEndpoint;
        this.clusterCertificateAuthorityData = resource.attrCertificateAuthorityData;
        const updateConfigCommandPrefix = `aws eks update-kubeconfig --name ${this.clusterName}`;
        const getTokenCommandPrefix = `aws eks get-token --cluster-name ${this.clusterName}`;
        const commonCommandOptions = [`--region ${stack.region}`];
        if (props.outputClusterName) {
            new core_1.CfnOutput(this, 'ClusterName', { value: this.clusterName });
        }
        // we maintain a single manifest custom resource handler per cluster since
        // permissions and role are scoped. This will return `undefined` if kubectl
        // is not enabled for this cluster.
        this._k8sResourceHandler = this.createKubernetesResourceHandler();
        // map the IAM role to the `system:masters` group.
        if (props.mastersRole) {
            if (!this.kubectlEnabled) {
                throw new Error('Cannot specify a "masters" role if kubectl is disabled');
            }
            this.awsAuth.addMastersRole(props.mastersRole);
            if (props.outputMastersRoleArn) {
                new core_1.CfnOutput(this, 'MastersRoleArn', { value: props.mastersRole.roleArn });
            }
            commonCommandOptions.push(`--role-arn ${props.mastersRole.roleArn}`);
        }
        // allocate default capacity if non-zero (or default).
        const desiredCapacity = props.defaultCapacity === undefined ? DEFAULT_CAPACITY_COUNT : props.defaultCapacity;
        if (desiredCapacity > 0) {
            const instanceType = props.defaultCapacityInstance || DEFAULT_CAPACITY_TYPE;
            this.defaultCapacity = this.addCapacity('DefaultCapacity', { instanceType, desiredCapacity });
        }
        const outputConfigCommand = props.outputConfigCommand === undefined ? true : props.outputConfigCommand;
        if (outputConfigCommand) {
            const postfix = commonCommandOptions.join(' ');
            new core_1.CfnOutput(this, 'ConfigCommand', { value: `${updateConfigCommandPrefix} ${postfix}` });
            new core_1.CfnOutput(this, 'GetTokenCommand', { value: `${getTokenCommandPrefix} ${postfix}` });
        }
    }
    /**
     * Import an existing cluster
     *
     * @param scope the construct scope, in most cases 'this'
     * @param id the id or name to import as
     * @param attrs the cluster properties to use for importing information
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add nodes to this EKS cluster
     *
     * The nodes will automatically be configured with the right VPC and AMI
     * for the instance type and Kubernetes version.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     */
    addCapacity(id, options) {
        const asg = new autoscaling.AutoScalingGroup(this, id, {
            ...options,
            vpc: this.vpc,
            machineImage: new EksOptimizedImage({
                nodeType: nodeTypeForInstanceType(options.instanceType),
                kubernetesVersion: this.version,
            }),
            updateType: options.updateType || autoscaling.UpdateType.ROLLING_UPDATE,
            instanceType: options.instanceType,
        });
        this.addAutoScalingGroup(asg, {
            mapRole: options.mapRole,
            bootstrapOptions: options.bootstrapOptions,
            bootstrapEnabled: options.bootstrapEnabled,
        });
        return asg;
    }
    /**
     * Add compute capacity to this EKS cluster in the form of an AutoScalingGroup
     *
     * The AutoScalingGroup must be running an EKS-optimized AMI containing the
     * /etc/eks/bootstrap.sh script. This method will configure Security Groups,
     * add the right policies to the instance role, apply the right tags, and add
     * the required user data to the instance's launch configuration.
     *
     * Spot instances will be labeled `lifecycle=Ec2Spot` and tainted with `PreferNoSchedule`.
     * If kubectl is enabled, the
     * [spot interrupt handler](https://github.com/awslabs/ec2-spot-labs/tree/master/ec2-spot-eks-solution/spot-termination-handler)
     * daemon will be installed on all spot instances to handle
     * [EC2 Spot Instance Termination Notices](https://aws.amazon.com/blogs/aws/new-ec2-spot-instance-termination-notices/).
     *
     * Prefer to use `addCapacity` if possible.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/launch-workers.html
     * @param autoScalingGroup [disable-awslint:ref-via-interface]
     * @param options options for adding auto scaling groups, like customizing the bootstrap script
     */
    addAutoScalingGroup(autoScalingGroup, options) {
        // self rules
        autoScalingGroup.connections.allowInternally(ec2.Port.allTraffic());
        // Cluster to:nodes rules
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcp(443));
        autoScalingGroup.connections.allowFrom(this, ec2.Port.tcpRange(1025, 65535));
        // Allow HTTPS from Nodes to Cluster
        autoScalingGroup.connections.allowTo(this, ec2.Port.tcp(443));
        // Allow all node outbound traffic
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allTcp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allUdp());
        autoScalingGroup.connections.allowToAnyIpv4(ec2.Port.allIcmp());
        const bootstrapEnabled = options.bootstrapEnabled !== undefined ? options.bootstrapEnabled : true;
        if (options.bootstrapOptions && !bootstrapEnabled) {
            throw new Error('Cannot specify "bootstrapOptions" if "bootstrapEnabled" is false');
        }
        if (bootstrapEnabled) {
            const userData = user_data_1.renderUserData(this.clusterName, autoScalingGroup, options.bootstrapOptions);
            autoScalingGroup.addUserData(...userData);
        }
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKSWorkerNodePolicy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEKS_CNI_Policy'));
        autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonEC2ContainerRegistryReadOnly'));
        // EKS Required Tags
        core_1.Tags.of(autoScalingGroup).add(`kubernetes.io/cluster/${this.clusterName}`, 'owned', {
            applyToLaunchedInstances: true,
        });
        if (options.mapRole === true && !this.kubectlEnabled) {
            throw new Error('Cannot map instance IAM role to RBAC if kubectl is disabled for the cluster');
        }
        // do not attempt to map the role if `kubectl` is not enabled for this
        // cluster or if `mapRole` is set to false. By default this should happen.
        const mapRole = options.mapRole === undefined ? true : options.mapRole;
        if (mapRole && this.kubectlEnabled) {
            // see https://docs.aws.amazon.com/en_us/eks/latest/userguide/add-user-role.html
            this.awsAuth.addRoleMapping(autoScalingGroup.role, {
                username: 'system:node:{{EC2PrivateDNSName}}',
                groups: [
                    'system:bootstrappers',
                    'system:nodes',
                ],
            });
        }
        else {
            // since we are not mapping the instance role to RBAC, synthesize an
            // output so it can be pasted into `aws-auth-cm.yaml`
            new core_1.CfnOutput(autoScalingGroup, 'InstanceRoleARN', {
                value: autoScalingGroup.role.roleArn,
            });
        }
        // if this is an ASG with spot instances, install the spot interrupt handler (only if kubectl is enabled).
        if (autoScalingGroup.spotPrice && this.kubectlEnabled) {
            this.addResource('spot-interrupt-handler', ...spot_interrupt_handler_1.spotInterruptHandler());
        }
    }
    /**
     * Lazily creates the AwsAuth resource, which manages AWS authentication mapping.
     */
    get awsAuth() {
        if (!this.kubectlEnabled) {
            throw new Error('Cannot define aws-auth mappings if kubectl is disabled');
        }
        if (!this._awsAuth) {
            this._awsAuth = new aws_auth_1.AwsAuth(this, 'AwsAuth', { cluster: this });
        }
        return this._awsAuth;
    }
    /**
     * Defines a Kubernetes resource in this cluster.
     *
     * The manifest will be applied/deleted using kubectl as needed.
     *
     * @param id logical id of this manifest
     * @param manifest a list of Kubernetes resource specifications
     * @returns a `KubernetesResource` object.
     * @throws If `kubectlEnabled` is `false`
     */
    addResource(id, ...manifest) {
        return new k8s_resource_1.KubernetesResource(this, `manifest-${id}`, { cluster: this, manifest });
    }
    /**
     * Defines a Helm chart in this cluster.
     *
     * @param id logical id of this chart.
     * @param options options of this chart.
     * @returns a `HelmChart` object
     * @throws If `kubectlEnabled` is `false`
     */
    addChart(id, options) {
        return new helm_chart_1.HelmChart(this, `chart-${id}`, { cluster: this, ...options });
    }
    createKubernetesResourceHandler() {
        if (!this.kubectlEnabled) {
            return undefined;
        }
        return new lambda.Function(this, 'KubernetesResourceHandler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'k8s-resource')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            layers: [kubectl_layer_1.KubectlLayer.getOrCreate(this)],
            memorySize: 256,
            environment: {
                CLUSTER_NAME: this.clusterName,
            },
            // NOTE: we must use the default IAM role that's mapped to "system:masters"
            // as the execution role of this custom resource handler. This is the only
            // way to be able to interact with the cluster after it's been created.
            role: this._defaultMastersRole,
        });
    }
    /**
     * Opportunistically tag subnets with the required tags.
     *
     * If no subnets could be found (because this is an imported VPC), add a warning.
     *
     * @see https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
     */
    tagSubnets() {
        const tagAllSubnets = (type, subnets, tag) => {
            for (const subnet of subnets) {
                // if this is not a concrete subnet, attach a construct warning
                if (!ec2.Subnet.isVpcSubnet(subnet)) {
                    // message (if token): "could not auto-tag public/private subnet with tag..."
                    // message (if not token): "count not auto-tag public/private subnet xxxxx with tag..."
                    const subnetID = core_1.Token.isUnresolved(subnet.subnetId) ? '' : ` ${subnet.subnetId}`;
                    core_1.Annotations.of(this).addWarning(`Could not auto-tag ${type} subnet${subnetID} with "${tag}=1", please remember to do this manually`);
                    continue;
                }
                core_1.Tags.of(subnet).add(tag, '1');
            }
        };
        // https://docs.aws.amazon.com/eks/latest/userguide/network_reqs.html
        tagAllSubnets('private', this.vpc.privateSubnets, 'kubernetes.io/role/internal-elb');
        tagAllSubnets('public', this.vpc.publicSubnets, 'kubernetes.io/role/elb');
    }
}
exports.Cluster = Cluster;
/**
 * Import a cluster to use in another stack
 */
class ImportedCluster extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.connections = new ec2.Connections();
        this.vpc = ec2.Vpc.fromVpcAttributes(this, 'VPC', props.vpc);
        this.clusterName = props.clusterName;
        this.clusterEndpoint = props.clusterEndpoint;
        this.clusterArn = props.clusterArn;
        this.clusterCertificateAuthorityData = props.clusterCertificateAuthorityData;
        let i = 1;
        for (const sgProps of props.securityGroups) {
            this.connections.addSecurityGroup(ec2.SecurityGroup.fromSecurityGroupId(this, `SecurityGroup${i}`, sgProps.securityGroupId));
            i++;
        }
    }
}
/**
 * Construct an Amazon Linux 2 image from the latest EKS Optimized AMI published in SSM
 */
class EksOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.nodeType = props && props.nodeType;
        this.kubernetesVersion = props && props.kubernetesVersion || LATEST_KUBERNETES_VERSION;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/eks/optimized-ami/${this.kubernetesVersion}/`
            + (this.nodeType === NodeType.STANDARD ? 'amazon-linux-2/' : '')
            + (this.nodeType === NodeType.GPU ? 'amazon-linux2-gpu/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.forLinux(),
        };
    }
}
exports.EksOptimizedImage = EksOptimizedImage;
// MAINTAINERS: use ./scripts/kube_bump.sh to update LATEST_KUBERNETES_VERSION
const LATEST_KUBERNETES_VERSION = '1.14';
/**
 * Whether the worker nodes should support GPU or just standard instances
 */
var NodeType;
(function (NodeType) {
    /**
     * Standard instances
     */
    NodeType["STANDARD"] = "Standard";
    /**
     * GPU instances
     */
    NodeType["GPU"] = "GPU";
})(NodeType = exports.NodeType || (exports.NodeType = {}));
const GPU_INSTANCETYPES = ['p2', 'p3', 'g4'];
function nodeTypeForInstanceType(instanceType) {
    return GPU_INSTANCETYPES.includes(instanceType.toString().substring(0, 2)) ? NodeType.GPU : NodeType.STANDARD;
}
exports.nodeTypeForInstanceType = nodeTypeForInstanceType;
//# sourceMappingURL=data:application/json;base64,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