"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterResource = void 0;
const path = require("path");
const cfn = require("../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const kubectl_layer_1 = require("./kubectl-layer");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // each cluster resource will have it's own lambda handler since permissions
        // are scoped to this cluster and related resources like it's role
        const handler = new lambda.Function(this, 'ResourceHandler', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'cluster-resource')),
            runtime: lambda.Runtime.PYTHON_3_7,
            handler: 'index.handler',
            timeout: core_1.Duration.minutes(15),
            memorySize: 512,
            layers: [kubectl_layer_1.KubectlLayer.getOrCreate(this)],
        });
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        // since we don't know the cluster name at this point, we must give this role star resource permissions
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['eks:CreateCluster', 'eks:DescribeCluster', 'eks:DeleteCluster', 'eks:UpdateClusterVersion'],
            resources: ['*'],
        }));
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        handler.addToRolePolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        const resource = new cfn.CustomResource(this, 'Resource', {
            resourceType: ClusterResource.RESOURCE_TYPE,
            provider: cfn.CustomResourceProvider.lambda(handler),
            properties: {
                Config: props,
            },
        });
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.creationRole = handler.role;
    }
}
exports.ClusterResource = ClusterResource;
/**
 * The AWS CloudFormation resource type used for this resource.
 */
ClusterResource.RESOURCE_TYPE = 'Custom::AWSCDK-EKS-Cluster';
//# sourceMappingURL=data:application/json;base64,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