"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PlacementConstraint = exports.PlacementStrategy = exports.BinPackResource = void 0;
const ec2_service_1 = require("./ec2/ec2-service");
/**
 * Instance resource used for bin packing
 */
var BinPackResource;
(function (BinPackResource) {
    /**
     * Fill up hosts' CPU allocations first
     */
    BinPackResource["CPU"] = "cpu";
    /**
     * Fill up hosts' memory allocations first
     */
    BinPackResource["MEMORY"] = "memory";
})(BinPackResource = exports.BinPackResource || (exports.BinPackResource = {}));
/**
 * The placement strategies to use for tasks in the service. For more information, see
 * [Amazon ECS Task Placement Strategies](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-strategies.html).
 *
 * Tasks will preferentially be placed on instances that match these rules.
 */
class PlacementStrategy {
    /**
     * Constructs a new instance of the PlacementStrategy class.
     */
    constructor(json) {
        this.json = json;
    }
    /**
     * Places tasks evenly across all container instances in the cluster.
     */
    static spreadAcrossInstances() {
        return new PlacementStrategy([{ type: 'spread', field: ec2_service_1.BuiltInAttributes.INSTANCE_ID }]);
    }
    /**
     * Places tasks evenly based on the specified value.
     *
     * You can use one of the built-in attributes found on `BuiltInAttributes`
     * or supply your own custom instance attributes. If more than one attribute
     * is supplied, spreading is done in order.
     *
     * @default attributes instanceId
     */
    static spreadAcross(...fields) {
        if (fields.length === 0) {
            throw new Error('spreadAcross: give at least one field to spread by');
        }
        return new PlacementStrategy(fields.map(field => ({ type: 'spread', field })));
    }
    /**
     * Places tasks on container instances with the least available amount of CPU capacity.
     *
     * This minimizes the number of instances in use.
     */
    static packedByCpu() {
        return PlacementStrategy.packedBy(BinPackResource.CPU);
    }
    /**
     * Places tasks on container instances with the least available amount of memory capacity.
     *
     * This minimizes the number of instances in use.
     */
    static packedByMemory() {
        return PlacementStrategy.packedBy(BinPackResource.MEMORY);
    }
    /**
     * Places tasks on the container instances with the least available capacity of the specified resource.
     */
    static packedBy(resource) {
        return new PlacementStrategy([{ type: 'binpack', field: resource }]);
    }
    /**
     * Places tasks randomly.
     */
    static randomly() {
        return new PlacementStrategy([{ type: 'random' }]);
    }
    /**
     * Return the placement JSON
     */
    toJson() {
        return this.json;
    }
}
exports.PlacementStrategy = PlacementStrategy;
/**
 * The placement constraints to use for tasks in the service. For more information, see
 * [Amazon ECS Task Placement Constraints](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/task-placement-constraints.html).
 *
 * Tasks will only be placed on instances that match these rules.
 */
class PlacementConstraint {
    /**
     * Constructs a new instance of the PlacementConstraint class.
     */
    constructor(json) {
        this.json = json;
    }
    /**
     * Use distinctInstance to ensure that each task in a particular group is running on a different container instance.
     */
    static distinctInstances() {
        return new PlacementConstraint([{ type: 'distinctInstance' }]);
    }
    /**
     * Use memberOf to restrict the selection to a group of valid candidates specified by a query expression.
     *
     * Multiple expressions can be specified. For more information, see
     * [Cluster Query Language](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html).
     *
     * You can specify multiple expressions in one call. The tasks will only be placed on instances matching all expressions.
     *
     * @see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/cluster-query-language.html
     */
    static memberOf(...expressions) {
        return new PlacementConstraint(expressions.map(expression => ({ type: 'memberOf', expression })));
    }
    /**
     * Return the placement JSON
     */
    toJson() {
        return this.json;
    }
}
exports.PlacementConstraint = PlacementConstraint;
//# sourceMappingURL=data:application/json;base64,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