"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RepositoryImage = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const container_image_1 = require("../container-image");
/**
 * Regex pattern to check if it is an ECR image URL.
 *
 * @experimental
 */
const ECR_IMAGE_REGEX = /(^[a-zA-Z0-9][a-zA-Z0-9-_]*).dkr.ecr.([a-zA-Z0-9][a-zA-Z0-9-_]*).amazonaws.com(.cn)?\/.*/;
/**
 * An image hosted in a public or private repository. For images hosted in Amazon ECR, see
 * [EcrImage](https://docs.aws.amazon.com/AmazonECR/latest/userguide/images.html).
 */
class RepositoryImage extends container_image_1.ContainerImage {
    /**
     * Constructs a new instance of the RepositoryImage class.
     */
    constructor(imageName, props = {}) {
        super();
        this.imageName = imageName;
        this.props = props;
    }
    bind(scope, containerDefinition) {
        // name could be a Token - in that case, skip validation altogether
        if (!core_1.Token.isUnresolved(this.imageName) && ECR_IMAGE_REGEX.test(this.imageName)) {
            core_1.Annotations.of(scope).addWarning("Proper policies need to be attached before pulling from ECR repository, or use 'fromEcrRepository'.");
        }
        if (this.props.credentials) {
            this.props.credentials.grantRead(containerDefinition.taskDefinition.obtainExecutionRole());
        }
        return {
            imageName: this.imageName,
            repositoryCredentials: this.props.credentials && {
                credentialsParameter: this.props.credentials.secretArn,
            },
        };
    }
}
exports.RepositoryImage = RepositoryImage;
//# sourceMappingURL=data:application/json;base64,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