"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ContainerImage = void 0;
/**
 * Constructs for types of container images
 */
class ContainerImage {
    /**
     * Reference an image on DockerHub or another online registry
     */
    static fromRegistry(name, props = {}) {
        return new repository_1.RepositoryImage(name, props);
    }
    /**
     * Reference an image in an ECR repository
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new ecr_1.EcrImage(repository, tag);
    }
    /**
     * Reference an image that's constructed directly from sources on disk.
     *
     * If you already have a `DockerImageAsset` instance, you can use the
     * `ContainerImage.fromDockerImageAsset` method instead.
     *
     * @param directory The directory containing the Dockerfile
     */
    static fromAsset(directory, props = {}) {
        return new asset_image_1.AssetImage(directory, props);
    }
    /**
     * Use an existing `DockerImageAsset` for this container image.
     *
     * @param asset The `DockerImageAsset` to use for this container definition.
     */
    static fromDockerImageAsset(asset) {
        return {
            bind(_scope, containerDefinition) {
                asset.repository.grantPull(containerDefinition.taskDefinition.obtainExecutionRole());
                return {
                    imageName: asset.imageUri,
                };
            },
        };
    }
}
exports.ContainerImage = ContainerImage;
const asset_image_1 = require("./images/asset-image");
const ecr_1 = require("./images/ecr");
const repository_1 = require("./images/repository");
//# sourceMappingURL=data:application/json;base64,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