"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AmiHardwareType = exports.BottleRocketImage = exports.BottlerocketEcsVariant = exports.EcsOptimizedImage = exports.EcsOptimizedAmi = exports.WindowsOptimizedVersion = exports.Cluster = exports.MachineImageType = void 0;
const autoscaling = require("../../aws-autoscaling"); // Automatically re-written from '@aws-cdk/aws-autoscaling'
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cloudmap = require("../../aws-servicediscovery"); // Automatically re-written from '@aws-cdk/aws-servicediscovery'
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const instance_drain_hook_1 = require("./drain-hook/instance-drain-hook");
const ecs_generated_1 = require("./ecs.generated");
/**
 * The machine image type
 */
var MachineImageType;
(function (MachineImageType) {
    /**
     * Amazon ECS-optimized Amazon Linux 2 AMI
     */
    MachineImageType[MachineImageType["AMAZON_LINUX_2"] = 0] = "AMAZON_LINUX_2";
    /**
     * Bottlerocket AMI
     */
    MachineImageType[MachineImageType["BOTTLEROCKET"] = 1] = "BOTTLEROCKET";
})(MachineImageType = exports.MachineImageType || (exports.MachineImageType = {}));
/**
 * A regional grouping of one or more container instances on which you can run tasks and services.
 */
class Cluster extends core_1.Resource {
    /**
     * Constructs a new instance of the Cluster class.
     */
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.clusterName,
        });
        /**
         * Manage the allowed network connections for the cluster with Security Groups.
         */
        this.connections = new ec2.Connections();
        /**
         * Specifies whether the cluster has EC2 instance capacity.
         */
        this._hasEc2Capacity = false;
        const containerInsights = props.containerInsights !== undefined ? props.containerInsights : false;
        const clusterSettings = containerInsights ? [{ name: 'containerInsights', value: 'enabled' }] : undefined;
        const cluster = new ecs_generated_1.CfnCluster(this, 'Resource', {
            clusterName: this.physicalName,
            clusterSettings,
        });
        this.clusterArn = this.getResourceArnAttribute(cluster.attrArn, {
            service: 'ecs',
            resource: 'cluster',
            resourceName: this.physicalName,
        });
        this.clusterName = this.getResourceNameAttribute(cluster.ref);
        this.vpc = props.vpc || new ec2.Vpc(this, 'Vpc', { maxAzs: 2 });
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace !== undefined
            ? this.addDefaultCloudMapNamespace(props.defaultCloudMapNamespace)
            : undefined;
        this._autoscalingGroup = props.capacity !== undefined
            ? this.addCapacity('DefaultAutoScalingGroup', props.capacity)
            : undefined;
    }
    /**
     * Import an existing cluster to the stack from its attributes.
     */
    static fromClusterAttributes(scope, id, attrs) {
        return new ImportedCluster(scope, id, attrs);
    }
    /**
     * Add an AWS Cloud Map DNS namespace for this cluster.
     * NOTE: HttpNamespaces are not supported, as ECS always requires a DNSConfig when registering an instance to a Cloud
     * Map service.
     */
    addDefaultCloudMapNamespace(options) {
        if (this._defaultCloudMapNamespace !== undefined) {
            throw new Error('Can only add default namespace once.');
        }
        const namespaceType = options.type !== undefined
            ? options.type
            : cloudmap.NamespaceType.DNS_PRIVATE;
        const sdNamespace = namespaceType === cloudmap.NamespaceType.DNS_PRIVATE ?
            new cloudmap.PrivateDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
                vpc: this.vpc,
            }) :
            new cloudmap.PublicDnsNamespace(this, 'DefaultServiceDiscoveryNamespace', {
                name: options.name,
            });
        this._defaultCloudMapNamespace = sdNamespace;
        return sdNamespace;
    }
    /**
     * Getter for namespace added to cluster
     */
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
    /**
     * This method adds compute capacity to a cluster by creating an AutoScalingGroup with the specified options.
     *
     * Returns the AutoScalingGroup so you can add autoscaling settings to it.
     */
    addCapacity(id, options) {
        var _a;
        if (options.machineImage && options.machineImageType) {
            throw new Error('You can only specify either machineImage or machineImageType, not both.');
        }
        const machineImage = ((_a = options.machineImage) !== null && _a !== void 0 ? _a : options.machineImageType === MachineImageType.BOTTLEROCKET) ?
            new BottleRocketImage() : new EcsOptimizedAmi();
        const autoScalingGroup = new autoscaling.AutoScalingGroup(this, id, {
            vpc: this.vpc,
            machineImage,
            updateType: options.updateType || autoscaling.UpdateType.REPLACING_UPDATE,
            ...options,
        });
        this.addAutoScalingGroup(autoScalingGroup, {
            machineImageType: options.machineImageType,
            ...options,
        });
        return autoScalingGroup;
    }
    /**
     * This method adds compute capacity to a cluster using the specified AutoScalingGroup.
     *
     * @param autoScalingGroup the ASG to add to this cluster.
     * [disable-awslint:ref-via-interface] is needed in order to install the ECS
     * agent by updating the ASGs user data.
     */
    addAutoScalingGroup(autoScalingGroup, options = {}) {
        this._hasEc2Capacity = true;
        this.connections.connections.addSecurityGroup(...autoScalingGroup.connections.securityGroups);
        // Tie instances to cluster
        switch (options.machineImageType) {
            // Bottlerocket AMI
            case MachineImageType.BOTTLEROCKET: {
                autoScalingGroup.addUserData(
                // Connect to the cluster
                // Source: https://github.com/bottlerocket-os/bottlerocket/blob/develop/QUICKSTART-ECS.md#connecting-to-your-cluster
                '[settings.ecs]', `cluster = "${this.clusterName}"`);
                // Enabling SSM
                // Source: https://github.com/bottlerocket-os/bottlerocket/blob/develop/QUICKSTART-ECS.md#enabling-ssm
                autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'));
                // required managed policy
                autoScalingGroup.role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonEC2ContainerServiceforEC2Role'));
                break;
            }
            default:
                // Amazon ECS-optimized AMI for Amazon Linux 2
                autoScalingGroup.addUserData(`echo ECS_CLUSTER=${this.clusterName} >> /etc/ecs/ecs.config`);
                if (!options.canContainersAccessInstanceRole) {
                    // Deny containers access to instance metadata service
                    // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
                    autoScalingGroup.addUserData('sudo iptables --insert FORWARD 1 --in-interface docker+ --destination 169.254.169.254/32 --jump DROP');
                    autoScalingGroup.addUserData('sudo service iptables save');
                    // The following is only for AwsVpc networking mode, but doesn't hurt for the other modes.
                    autoScalingGroup.addUserData('echo ECS_AWSVPC_BLOCK_IMDS=true >> /etc/ecs/ecs.config');
                }
                if (autoScalingGroup.spotPrice && options.spotInstanceDraining) {
                    autoScalingGroup.addUserData('echo ECS_ENABLE_SPOT_INSTANCE_DRAINING=true >> /etc/ecs/ecs.config');
                }
        }
        // ECS instances must be able to do these things
        // Source: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/instance_IAM_role.html
        // But, scoped down to minimal permissions required.
        //  Notes:
        //   - 'ecs:CreateCluster' removed. The cluster already exists.
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'ecs:DeregisterContainerInstance',
                'ecs:RegisterContainerInstance',
                'ecs:Submit*',
            ],
            resources: [
                this.clusterArn,
            ],
        }));
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                // These act on a cluster instance, and the instance doesn't exist until the service starts.
                // Thus, scope to the cluster using a condition.
                // See: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonelasticcontainerservice.html
                'ecs:Poll',
                'ecs:StartTelemetrySession',
            ],
            resources: ['*'],
            conditions: {
                ArnEquals: { 'ecs:cluster': this.clusterArn },
            },
        }));
        autoScalingGroup.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                // These do not support resource constraints, and must be resource '*'
                'ecs:DiscoverPollEndpoint',
                'ecr:GetAuthorizationToken',
                // Preserved for backwards compatibility.
                // Users are able to enable cloudwatch agent using CDK. Existing
                // customers might be installing CW agent as part of user-data so if we
                // remove these permissions we will break that customer use cases.
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: ['*'],
        }));
        // 0 disables, otherwise forward to underlying implementation which picks the sane default
        if (!options.taskDrainTime || options.taskDrainTime.toSeconds() !== 0) {
            new instance_drain_hook_1.InstanceDrainHook(autoScalingGroup, 'DrainECSHook', {
                autoScalingGroup,
                cluster: this,
                drainTime: options.taskDrainTime,
                topicEncryptionKey: options.topicEncryptionKey,
            });
        }
    }
    /**
     * Getter for autoscaling group added to cluster
     */
    get autoscalingGroup() {
        return this._autoscalingGroup;
    }
    /**
     * Whether the cluster has EC2 capacity associated with it
     */
    get hasEc2Capacity() {
        return this._hasEc2Capacity;
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU reservation.
     *
     * @default average over 5 minutes
     */
    metricCpuReservation(props) {
        return this.metric('CPUReservation', props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory reservation.
     *
     * @default average over 5 minutes
     */
    metricMemoryReservation(props) {
        return this.metric('MemoryReservation', props);
    }
    /**
     * This method returns the specifed CloudWatch metric for this cluster.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensions: { ClusterName: this.clusterName },
            ...props,
        }).attachTo(this);
    }
}
exports.Cluster = Cluster;
/**
 * ECS-optimized Windows version list
 */
var WindowsOptimizedVersion;
(function (WindowsOptimizedVersion) {
    WindowsOptimizedVersion["SERVER_2019"] = "2019";
    WindowsOptimizedVersion["SERVER_2016"] = "2016";
})(WindowsOptimizedVersion = exports.WindowsOptimizedVersion || (exports.WindowsOptimizedVersion = {}));
/*
 * TODO:v2.0.0 remove EcsOptimizedAmi
 */
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM
 *
 * @deprecated see {@link EcsOptimizedImage#amazonLinux}, {@link EcsOptimizedImage#amazonLinux} and {@link EcsOptimizedImage#windows}
 */
class EcsOptimizedAmi {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.hwType = (props && props.hardwareType) || AmiHardwareType.STANDARD;
        if (props && props.generation) { // generation defined in the props object
            if (props.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX && this.hwType !== AmiHardwareType.STANDARD) {
                throw new Error('Amazon Linux does not support special hardware type. Use Amazon Linux 2 instead');
            }
            else if (props.windowsVersion) {
                throw new Error('"windowsVersion" and Linux image "generation" cannot be both set');
            }
            else {
                this.generation = props.generation;
            }
        }
        else if (props && props.windowsVersion) {
            if (this.hwType !== AmiHardwareType.STANDARD) {
                throw new Error('Windows Server does not support special hardware type');
            }
            else {
                this.windowsVersion = props.windowsVersion;
            }
        }
        else { // generation not defined in props object
            // always default to Amazon Linux v2 regardless of HW
            this.generation = ec2.AmazonLinuxGeneration.AMAZON_LINUX_2;
        }
        // set the SSM parameter name
        this.amiParameterName = '/aws/service/ecs/optimized-ami/'
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX ? 'amazon-linux/' : '')
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 ? 'amazon-linux-2/' : '')
            + (this.windowsVersion ? `windows_server/${this.windowsVersion}/english/full/` : '')
            + (this.hwType === AmiHardwareType.GPU ? 'gpu/' : '')
            + (this.hwType === AmiHardwareType.ARM ? 'arm64/' : '')
            + 'recommended/image_id';
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.amiParameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        const osType = this.windowsVersion ? ec2.OperatingSystemType.WINDOWS : ec2.OperatingSystemType.LINUX;
        return {
            imageId: ami,
            osType,
            userData: ec2.UserData.forOperatingSystem(osType),
        };
    }
}
exports.EcsOptimizedAmi = EcsOptimizedAmi;
/**
 * Construct a Linux or Windows machine image from the latest ECS Optimized AMI published in SSM
 */
class EcsOptimizedImage {
    /**
     * Constructs a new instance of the EcsOptimizedAmi class.
     */
    constructor(props) {
        this.hwType = props && props.hardwareType;
        if (props.windowsVersion) {
            this.windowsVersion = props.windowsVersion;
        }
        else if (props.generation) {
            this.generation = props.generation;
        }
        else {
            throw new Error('This error should never be thrown');
        }
        // set the SSM parameter name
        this.amiParameterName = '/aws/service/ecs/optimized-ami/'
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX ? 'amazon-linux/' : '')
            + (this.generation === ec2.AmazonLinuxGeneration.AMAZON_LINUX_2 ? 'amazon-linux-2/' : '')
            + (this.windowsVersion ? `windows_server/${this.windowsVersion}/english/full/` : '')
            + (this.hwType === AmiHardwareType.GPU ? 'gpu/' : '')
            + (this.hwType === AmiHardwareType.ARM ? 'arm64/' : '')
            + 'recommended/image_id';
    }
    /**
     * Construct an Amazon Linux 2 image from the latest ECS Optimized AMI published in SSM
     *
     * @param hardwareType ECS-optimized AMI variant to use
     */
    static amazonLinux2(hardwareType = AmiHardwareType.STANDARD) {
        return new EcsOptimizedImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX_2, hardwareType });
    }
    /**
     * Construct an Amazon Linux AMI image from the latest ECS Optimized AMI published in SSM
     */
    static amazonLinux() {
        return new EcsOptimizedImage({ generation: ec2.AmazonLinuxGeneration.AMAZON_LINUX });
    }
    /**
     * Construct a Windows image from the latest ECS Optimized AMI published in SSM
     *
     * @param windowsVersion Windows Version to use
     */
    static windows(windowsVersion) {
        return new EcsOptimizedImage({ windowsVersion });
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, this.amiParameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        const osType = this.windowsVersion ? ec2.OperatingSystemType.WINDOWS : ec2.OperatingSystemType.LINUX;
        return {
            imageId: ami,
            osType,
            userData: ec2.UserData.forOperatingSystem(osType),
        };
    }
}
exports.EcsOptimizedImage = EcsOptimizedImage;
/**
 * Amazon ECS variant
 */
var BottlerocketEcsVariant;
(function (BottlerocketEcsVariant) {
    /**
     * aws-ecs-1 variant
     */
    BottlerocketEcsVariant["AWS_ECS_1"] = "aws-ecs-1";
})(BottlerocketEcsVariant = exports.BottlerocketEcsVariant || (exports.BottlerocketEcsVariant = {}));
/**
 * Construct an Bottlerocket image from the latest AMI published in SSM
 */
class BottleRocketImage {
    /**
     * Constructs a new instance of the BottleRocketImage class.
     */
    constructor(props = {}) {
        var _a;
        this.variant = (_a = props.variant) !== null && _a !== void 0 ? _a : BottlerocketEcsVariant.AWS_ECS_1;
        // set the SSM parameter name
        this.amiParameterName = `/aws/service/bottlerocket/${this.variant}/x86_64/latest/image_id`;
    }
    /**
     * Return the correct image
     */
    getImage(scope) {
        const ami = ssm.StringParameter.valueForStringParameter(scope, this.amiParameterName);
        return {
            imageId: ami,
            osType: ec2.OperatingSystemType.LINUX,
            userData: ec2.UserData.custom(''),
        };
    }
}
exports.BottleRocketImage = BottleRocketImage;
/**
 * An Cluster that has been imported
 */
class ImportedCluster extends core_1.Resource {
    /**
     * Constructs a new instance of the ImportedCluster class.
     */
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Security group of the cluster instances
         */
        this.connections = new ec2.Connections();
        this.clusterName = props.clusterName;
        this.vpc = props.vpc;
        this.hasEc2Capacity = props.hasEc2Capacity !== false;
        this._defaultCloudMapNamespace = props.defaultCloudMapNamespace;
        this.clusterArn = props.clusterArn !== undefined ? props.clusterArn : core_1.Stack.of(this).formatArn({
            service: 'ecs',
            resource: 'cluster',
            resourceName: props.clusterName,
        });
        this.connections = new ec2.Connections({
            securityGroups: props.securityGroups,
        });
    }
    get defaultCloudMapNamespace() {
        return this._defaultCloudMapNamespace;
    }
}
/**
 * The ECS-optimized AMI variant to use. For more information, see
 * [Amazon ECS-optimized AMIs](https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-optimized_AMI.html).
 */
var AmiHardwareType;
(function (AmiHardwareType) {
    /**
     * Use the standard Amazon ECS-optimized AMI.
     */
    AmiHardwareType["STANDARD"] = "Standard";
    /**
     * Use the Amazon ECS GPU-optimized AMI.
     */
    AmiHardwareType["GPU"] = "GPU";
    /**
     * Use the Amazon ECS-optimized Amazon Linux 2 (arm64) AMI.
     */
    AmiHardwareType["ARM"] = "ARM64";
})(AmiHardwareType = exports.AmiHardwareType || (exports.AmiHardwareType = {}));
//# sourceMappingURL=data:application/json;base64,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