"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PropagatedTagSource = exports.DeploymentControllerType = exports.LaunchType = exports.BaseService = exports.ListenerConfig = void 0;
const appscaling = require("../../../aws-applicationautoscaling"); // Automatically re-written from '@aws-cdk/aws-applicationautoscaling'
const cloudwatch = require("../../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cloudmap = require("../../../aws-servicediscovery"); // Automatically re-written from '@aws-cdk/aws-servicediscovery'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_definition_1 = require("../base/task-definition");
const ecs_generated_1 = require("../ecs.generated");
const scalable_task_count_1 = require("./scalable-task-count");
/**
 * Base class for configuring listener when registering targets.
 */
class ListenerConfig {
    /**
     * Create a config for adding target group to ALB listener.
     */
    static applicationListener(listener, props) {
        return new ApplicationListenerConfig(listener, props);
    }
    /**
     * Create a config for adding target group to NLB listener.
     */
    static networkListener(listener, props) {
        return new NetworkListenerConfig(listener, props);
    }
}
exports.ListenerConfig = ListenerConfig;
/**
 * Class for configuring application load balancer listener when registering targets.
 */
class ApplicationListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        const props = this.props || {};
        const protocol = props.protocol;
        const port = props.port !== undefined ? props.port : (protocol === undefined ? 80 :
            (protocol === elbv2.ApplicationProtocol.HTTPS ? 443 : 80));
        this.listener.addTargets(id, {
            ...props,
            targets: [
                service.loadBalancerTarget({
                    ...target,
                }),
            ],
            port,
        });
    }
}
/**
 * Class for configuring network load balancer listener when registering targets.
 */
class NetworkListenerConfig extends ListenerConfig {
    constructor(listener, props) {
        super();
        this.listener = listener;
        this.props = props;
    }
    /**
     * Create and attach a target group to listener.
     */
    addTargets(id, target, service) {
        const port = this.props !== undefined ? this.props.port : 80;
        this.listener.addTargets(id, {
            ...this.props,
            targets: [
                service.loadBalancerTarget({
                    ...target,
                }),
            ],
            port,
        });
    }
}
/**
 * The base class for Ec2Service and FargateService services.
 */
class BaseService extends core_1.Resource {
    /**
     * Constructs a new instance of the BaseService class.
     */
    constructor(scope, id, props, additionalProps, taskDefinition) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.serviceName,
        });
        /**
         * The security groups which manage the allowed network traffic for the service.
         */
        this.connections = new ec2.Connections();
        /**
         * A list of Elastic Load Balancing load balancer objects, containing the load balancer name, the container
         * name (as it appears in a container definition), and the container port to access from the load balancer.
         */
        this.loadBalancers = new Array();
        /**
         * The details of the service discovery registries to assign to this service.
         * For more information, see Service Discovery.
         */
        this.serviceRegistries = new Array();
        this.taskDefinition = taskDefinition;
        this.resource = new ecs_generated_1.CfnService(this, 'Service', {
            desiredCount: props.desiredCount,
            serviceName: this.physicalName,
            loadBalancers: core_1.Lazy.anyValue({ produce: () => this.loadBalancers }, { omitEmptyArray: true }),
            deploymentConfiguration: {
                maximumPercent: props.maxHealthyPercent || 200,
                minimumHealthyPercent: props.minHealthyPercent === undefined ? 50 : props.minHealthyPercent,
            },
            propagateTags: props.propagateTags === PropagatedTagSource.NONE ? undefined : props.propagateTags,
            enableEcsManagedTags: props.enableECSManagedTags === undefined ? false : props.enableECSManagedTags,
            deploymentController: props.deploymentController,
            launchType: ((_a = props.deploymentController) === null || _a === void 0 ? void 0 : _a.type) === DeploymentControllerType.EXTERNAL ? undefined : props.launchType,
            healthCheckGracePeriodSeconds: this.evaluateHealthGracePeriod(props.healthCheckGracePeriod),
            /* role: never specified, supplanted by Service Linked Role */
            networkConfiguration: core_1.Lazy.anyValue({ produce: () => this.networkConfiguration }, { omitEmptyArray: true }),
            serviceRegistries: core_1.Lazy.anyValue({ produce: () => this.serviceRegistries }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (((_b = props.deploymentController) === null || _b === void 0 ? void 0 : _b.type) === DeploymentControllerType.EXTERNAL) {
            core_1.Annotations.of(this).addWarning('taskDefinition and launchType are blanked out when using external deployment controller.');
        }
        this.serviceArn = this.getResourceArnAttribute(this.resource.ref, {
            service: 'ecs',
            resource: 'service',
            resourceName: `${props.cluster.clusterName}/${this.physicalName}`,
        });
        this.serviceName = this.getResourceNameAttribute(this.resource.attrName);
        this.cluster = props.cluster;
        if (props.cloudMapOptions) {
            this.enableCloudMap(props.cloudMapOptions);
        }
    }
    /**
     * The CloudMap service created for this service, if any.
     */
    get cloudMapService() {
        return this.cloudmapService;
    }
    /**
     * This method is called to attach this service to an Application Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToApplicationTargetGroup(targetGroup) {
        return this.defaultLoadBalancerTarget.attachToApplicationTargetGroup(targetGroup);
    }
    /**
     * Registers the service as a target of a Classic Load Balancer (CLB).
     *
     * Don't call this. Call `loadBalancer.addTarget()` instead.
     */
    attachToClassicLB(loadBalancer) {
        return this.defaultLoadBalancerTarget.attachToClassicLB(loadBalancer);
    }
    /**
     * Return a load balancing target for a specific container and port.
     *
     * Use this function to create a load balancer target if you want to load balance to
     * another container than the first essential container or the first mapped port on
     * the container.
     *
     * Use the return value of this function where you would normally use a load balancer
     * target, instead of the `Service` object itself.
     *
     * @example
     *
     * listener.addTargets('ECS', {
     *   port: 80,
     *   targets: [service.loadBalancerTarget({
     *     containerName: 'MyContainer',
     *     containerPort: 1234,
     *   })],
     * });
     */
    loadBalancerTarget(options) {
        const self = this;
        const target = this.taskDefinition._validateTarget(options);
        const connections = self.connections;
        return {
            attachToApplicationTargetGroup(targetGroup) {
                targetGroup.registerConnectable(self, self.taskDefinition._portRangeFromPortMapping(target.portMapping));
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            attachToNetworkTargetGroup(targetGroup) {
                return self.attachToELBv2(targetGroup, target.containerName, target.portMapping.containerPort);
            },
            connections,
            attachToClassicLB(loadBalancer) {
                return self.attachToELB(loadBalancer, target.containerName, target.portMapping.containerPort);
            },
        };
    }
    /**
     * Use this function to create all load balancer targets to be registered in this service, add them to
     * target groups, and attach target groups to listeners accordingly.
     *
     * Alternatively, you can use `listener.addTargets()` to create targets and add them to target groups.
     *
     * @example
     *
     * service.registerLoadBalancerTargets(
     *   {
     *     containerName: 'web',
     *     containerPort: 80,
     *     newTargetGroupId: 'ECS',
     *     listener: ecs.ListenerConfig.applicationListener(listener, {
     *       protocol: elbv2.ApplicationProtocol.HTTPS
     *     }),
     *   },
     * )
     */
    registerLoadBalancerTargets(...targets) {
        for (const target of targets) {
            target.listener.addTargets(target.newTargetGroupId, {
                containerName: target.containerName,
                containerPort: target.containerPort,
                protocol: target.protocol,
            }, this);
        }
    }
    /**
     * This method is called to attach this service to a Network Load Balancer.
     *
     * Don't call this function directly. Instead, call `listener.addTargets()`
     * to add this service to a load balancer.
     */
    attachToNetworkTargetGroup(targetGroup) {
        return this.defaultLoadBalancerTarget.attachToNetworkTargetGroup(targetGroup);
    }
    /**
     * An attribute representing the minimum and maximum task count for an AutoScalingGroup.
     */
    autoScaleTaskCount(props) {
        if (this.scalableTaskCount) {
            throw new Error('AutoScaling of task count already enabled for this service');
        }
        return this.scalableTaskCount = new scalable_task_count_1.ScalableTaskCount(this, 'TaskCount', {
            serviceNamespace: appscaling.ServiceNamespace.ECS,
            resourceId: `service/${this.cluster.clusterName}/${this.serviceName}`,
            dimension: 'ecs:service:DesiredCount',
            role: this.makeAutoScalingRole(),
            ...props,
        });
    }
    /**
     * Enable CloudMap service discovery for the service
     *
     * @returns The created CloudMap service
     */
    enableCloudMap(options) {
        const sdNamespace = options.cloudMapNamespace !== undefined ? options.cloudMapNamespace : this.cluster.defaultCloudMapNamespace;
        if (sdNamespace === undefined) {
            throw new Error('Cannot enable service discovery if a Cloudmap Namespace has not been created in the cluster.');
        }
        // Determine DNS type based on network mode
        const networkMode = this.taskDefinition.networkMode;
        if (networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a service discovery if NetworkMode is None. Use Bridge, Host or AwsVpc instead.');
        }
        // Bridge or host network mode requires SRV records
        let dnsRecordType = options.dnsRecordType;
        if (networkMode === task_definition_1.NetworkMode.BRIDGE || networkMode === task_definition_1.NetworkMode.HOST) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.SRV;
            }
            if (dnsRecordType !== cloudmap.DnsRecordType.SRV) {
                throw new Error('SRV records must be used when network mode is Bridge or Host.');
            }
        }
        // Default DNS record type for AwsVpc network mode is A Records
        if (networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            if (dnsRecordType === undefined) {
                dnsRecordType = cloudmap.DnsRecordType.A;
            }
        }
        // If the task definition that your service task specifies uses the AWSVPC network mode and a type SRV DNS record is
        // used, you must specify a containerName and containerPort combination
        const containerName = dnsRecordType === cloudmap.DnsRecordType.SRV ? this.taskDefinition.defaultContainer.containerName : undefined;
        const containerPort = dnsRecordType === cloudmap.DnsRecordType.SRV ? this.taskDefinition.defaultContainer.containerPort : undefined;
        const cloudmapService = new cloudmap.Service(this, 'CloudmapService', {
            namespace: sdNamespace,
            name: options.name,
            dnsRecordType: dnsRecordType,
            customHealthCheck: { failureThreshold: options.failureThreshold || 1 },
            dnsTtl: options.dnsTtl,
        });
        const serviceArn = cloudmapService.serviceArn;
        // add Cloudmap service to the ECS Service's serviceRegistry
        this.addServiceRegistry({
            arn: serviceArn,
            containerName,
            containerPort,
        });
        this.cloudmapService = cloudmapService;
        return cloudmapService;
    }
    /**
     * This method returns the specified CloudWatch metric name for this service.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ECS',
            metricName,
            dimensions: { ClusterName: this.cluster.clusterName, ServiceName: this.serviceName },
            ...props,
        }).attachTo(this);
    }
    /**
     * This method returns the CloudWatch metric for this clusters memory utilization.
     *
     * @default average over 5 minutes
     */
    metricMemoryUtilization(props) {
        return this.metric('MemoryUtilization', props);
    }
    /**
     * This method returns the CloudWatch metric for this clusters CPU utilization.
     *
     * @default average over 5 minutes
     */
    metricCpuUtilization(props) {
        return this.metric('CPUUtilization', props);
    }
    /**
     * This method is called to create a networkConfiguration.
     * @deprecated use configureAwsVpcNetworkingWithSecurityGroups instead.
     */
    // eslint-disable-next-line max-len
    configureAwsVpcNetworking(vpc, assignPublicIp, vpcSubnets, securityGroup) {
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: ec2.SubnetType.PUBLIC } : {};
        }
        if (securityGroup === undefined) {
            securityGroup = new ec2.SecurityGroup(this, 'SecurityGroup', { vpc });
        }
        this.connections.addSecurityGroup(securityGroup);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: core_1.Lazy.listValue({ produce: () => [securityGroup.securityGroupId] }),
            },
        };
    }
    /**
     * This method is called to create a networkConfiguration.
     */
    // eslint-disable-next-line max-len
    configureAwsVpcNetworkingWithSecurityGroups(vpc, assignPublicIp, vpcSubnets, securityGroups) {
        if (vpcSubnets === undefined) {
            vpcSubnets = assignPublicIp ? { subnetType: ec2.SubnetType.PUBLIC } : {};
        }
        if (securityGroups === undefined || securityGroups.length === 0) {
            securityGroups = [new ec2.SecurityGroup(this, 'SecurityGroup', { vpc })];
        }
        securityGroups.forEach((sg) => { this.connections.addSecurityGroup(sg); }, this);
        this.networkConfiguration = {
            awsvpcConfiguration: {
                assignPublicIp: assignPublicIp ? 'ENABLED' : 'DISABLED',
                subnets: vpc.selectSubnets(vpcSubnets).subnetIds,
                securityGroups: securityGroups.map((sg) => sg.securityGroupId),
            },
        };
    }
    renderServiceRegistry(registry) {
        return {
            registryArn: registry.arn,
            containerName: registry.containerName,
            containerPort: registry.containerPort,
        };
    }
    /**
     * Shared logic for attaching to an ELB
     */
    attachToELB(loadBalancer, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC) {
            throw new Error('Cannot use a Classic Load Balancer if NetworkMode is AwsVpc. Use Host or Bridge instead.');
        }
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a Classic Load Balancer if NetworkMode is None. Use Host or Bridge instead.');
        }
        this.loadBalancers.push({
            loadBalancerName: loadBalancer.loadBalancerName,
            containerName,
            containerPort,
        });
    }
    /**
     * Shared logic for attaching to an ELBv2
     */
    attachToELBv2(targetGroup, containerName, containerPort) {
        if (this.taskDefinition.networkMode === task_definition_1.NetworkMode.NONE) {
            throw new Error('Cannot use a load balancer if NetworkMode is None. Use Bridge, Host or AwsVpc instead.');
        }
        this.loadBalancers.push({
            targetGroupArn: targetGroup.targetGroupArn,
            containerName,
            containerPort,
        });
        // Service creation can only happen after the load balancer has
        // been associated with our target group(s), so add ordering dependency.
        this.resource.node.addDependency(targetGroup.loadBalancerAttached);
        const targetType = this.taskDefinition.networkMode === task_definition_1.NetworkMode.AWS_VPC ? elbv2.TargetType.IP : elbv2.TargetType.INSTANCE;
        return { targetType };
    }
    get defaultLoadBalancerTarget() {
        return this.loadBalancerTarget({
            containerName: this.taskDefinition.defaultContainer.containerName,
        });
    }
    /**
     * Generate the role that will be used for autoscaling this service
     */
    makeAutoScalingRole() {
        // Use a Service Linked Role.
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            region: '',
            service: 'iam',
            resource: 'role/aws-service-role/ecs.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_ECSService',
        }));
    }
    /**
     * Associate Service Discovery (Cloud Map) service
     */
    addServiceRegistry(registry) {
        const sr = this.renderServiceRegistry(registry);
        this.serviceRegistries.push(sr);
    }
    /**
     *  Return the default grace period when load balancers are configured and
     *  healthCheckGracePeriod is not already set
     */
    evaluateHealthGracePeriod(providedHealthCheckGracePeriod) {
        return core_1.Lazy.anyValue({
            produce: () => providedHealthCheckGracePeriod !== undefined ? providedHealthCheckGracePeriod.toSeconds() :
                this.loadBalancers.length > 0 ? 60 :
                    undefined,
        });
    }
}
exports.BaseService = BaseService;
/**
 * The launch type of an ECS service
 */
var LaunchType;
(function (LaunchType) {
    /**
     * The service will be launched using the EC2 launch type
     */
    LaunchType["EC2"] = "EC2";
    /**
     * The service will be launched using the FARGATE launch type
     */
    LaunchType["FARGATE"] = "FARGATE";
})(LaunchType = exports.LaunchType || (exports.LaunchType = {}));
/**
 * The deployment controller type to use for the service.
 */
var DeploymentControllerType;
(function (DeploymentControllerType) {
    /**
     * The rolling update (ECS) deployment type involves replacing the current
     * running version of the container with the latest version.
     */
    DeploymentControllerType["ECS"] = "ECS";
    /**
     * The blue/green (CODE_DEPLOY) deployment type uses the blue/green deployment model powered by AWS CodeDeploy
     */
    DeploymentControllerType["CODE_DEPLOY"] = "CODE_DEPLOY";
    /**
     * The external (EXTERNAL) deployment type enables you to use any third-party deployment controller
     */
    DeploymentControllerType["EXTERNAL"] = "EXTERNAL";
})(DeploymentControllerType = exports.DeploymentControllerType || (exports.DeploymentControllerType = {}));
/**
 * Propagate tags from either service or task definition
 */
var PropagatedTagSource;
(function (PropagatedTagSource) {
    /**
     * Propagate tags from service
     */
    PropagatedTagSource["SERVICE"] = "SERVICE";
    /**
     * Propagate tags from task definition
     */
    PropagatedTagSource["TASK_DEFINITION"] = "TASK_DEFINITION";
    /**
     * Do not propagate
     */
    PropagatedTagSource["NONE"] = "NONE";
})(PropagatedTagSource = exports.PropagatedTagSource || (exports.PropagatedTagSource = {}));
//# sourceMappingURL=data:application/json;base64,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