"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LookupMachineImage = exports.OperatingSystemType = exports.GenericWindowsImage = exports.GenericLinuxImage = exports.AmazonLinuxStorage = exports.AmazonLinuxVirt = exports.AmazonLinuxEdition = exports.AmazonLinuxGeneration = exports.AmazonLinuxImage = exports.AmazonLinuxCpuType = exports.WindowsImage = exports.MachineImage = void 0;
const ssm = require("../../aws-ssm"); // Automatically re-written from '@aws-cdk/aws-ssm'
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const user_data_1 = require("./user-data");
/**
 * Factory functions for standard Amazon Machine Image objects.
 */
class MachineImage {
    /**
     * A Windows image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestWindows(version, props) {
        return new WindowsImage(version, props);
    }
    /**
     * An Amazon Linux image that is automatically kept up-to-date
     *
     * This Machine Image automatically updates to the latest version on every
     * deployment. Be aware this will cause your instances to be replaced when a
     * new version of the image becomes available. Do not store stateful information
     * on the instance if you are using this image.
     */
    static latestAmazonLinux(props) {
        return new AmazonLinuxImage(props);
    }
    /**
     * A Linux image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericLinux(amiMap, props) {
        return new GenericLinuxImage(amiMap, props);
    }
    /**
     * A Windows image where you specify the AMI ID for every region
     *
     * @param amiMap For every region where you are deploying the stack,
     * specify the AMI ID for that region.
     * @param props Customize the image by supplying additional props
     */
    static genericWindows(amiMap, props) {
        return new GenericWindowsImage(amiMap, props);
    }
    /**
     * Look up a shared Machine Image using DescribeImages
     *
     * The most recent, available, launchable image matching the given filter
     * criteria will be used. Looking up AMIs may take a long time; specify
     * as many filter criteria as possible to narrow down the search.
     *
     * The AMI selected will be cached in `cdk.context.json` and the same value
     * will be used on future runs. To refresh the AMI lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static lookup(props) {
        return new LookupMachineImage(props);
    }
}
exports.MachineImage = MachineImage;
/**
 * Select the latest version of the indicated Windows version
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 *
 * https://aws.amazon.com/blogs/mt/query-for-the-latest-windows-ami-using-systems-manager-parameter-store/
 */
class WindowsImage {
    constructor(version, props = {}) {
        this.version = version;
        this.props = props;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        var _a;
        const parameterName = this.imageParameterName();
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forWindows(),
            osType: OperatingSystemType.WINDOWS,
        };
    }
    /**
     * Construct the SSM parameter name for the given Windows image
     */
    imageParameterName() {
        return '/aws/service/ami-windows-latest/' + this.version;
    }
}
exports.WindowsImage = WindowsImage;
/**
 * CPU type
 */
var AmazonLinuxCpuType;
(function (AmazonLinuxCpuType) {
    /**
     * arm64 CPU type
     */
    AmazonLinuxCpuType["ARM_64"] = "arm64";
    /**
     * x86_64 CPU type
     */
    AmazonLinuxCpuType["X86_64"] = "x86_64";
})(AmazonLinuxCpuType = exports.AmazonLinuxCpuType || (exports.AmazonLinuxCpuType = {}));
/**
 * Selects the latest version of Amazon Linux
 *
 * This Machine Image automatically updates to the latest version on every
 * deployment. Be aware this will cause your instances to be replaced when a
 * new version of the image becomes available. Do not store stateful information
 * on the instance if you are using this image.
 *
 * The AMI ID is selected using the values published to the SSM parameter store.
 */
class AmazonLinuxImage {
    constructor(props = {}) {
        this.props = props;
        this.generation = (props && props.generation) || AmazonLinuxGeneration.AMAZON_LINUX;
        this.edition = (props && props.edition) || AmazonLinuxEdition.STANDARD;
        this.virtualization = (props && props.virtualization) || AmazonLinuxVirt.HVM;
        this.storage = (props && props.storage) || AmazonLinuxStorage.GENERAL_PURPOSE;
        this.cpu = (props && props.cpuType) || AmazonLinuxCpuType.X86_64;
    }
    /**
     * Return the image to use in the given context
     */
    getImage(scope) {
        var _a;
        const parts = [
            this.generation,
            'ami',
            this.edition !== AmazonLinuxEdition.STANDARD ? this.edition : undefined,
            this.virtualization,
            this.cpu,
            this.storage,
        ].filter(x => x !== undefined); // Get rid of undefineds
        const parameterName = '/aws/service/ami-amazon-linux-latest/' + parts.join('-');
        const ami = ssm.StringParameter.valueForTypedStringParameter(scope, parameterName, ssm.ParameterType.AWS_EC2_IMAGE_ID);
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forLinux(),
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.AmazonLinuxImage = AmazonLinuxImage;
/**
 * What generation of Amazon Linux to use
 */
var AmazonLinuxGeneration;
(function (AmazonLinuxGeneration) {
    /**
     * Amazon Linux
     */
    AmazonLinuxGeneration["AMAZON_LINUX"] = "amzn";
    /**
     * Amazon Linux 2
     */
    AmazonLinuxGeneration["AMAZON_LINUX_2"] = "amzn2";
})(AmazonLinuxGeneration = exports.AmazonLinuxGeneration || (exports.AmazonLinuxGeneration = {}));
/**
 * Amazon Linux edition
 */
var AmazonLinuxEdition;
(function (AmazonLinuxEdition) {
    /**
     * Standard edition
     */
    AmazonLinuxEdition["STANDARD"] = "standard";
    /**
     * Minimal edition
     */
    AmazonLinuxEdition["MINIMAL"] = "minimal";
})(AmazonLinuxEdition = exports.AmazonLinuxEdition || (exports.AmazonLinuxEdition = {}));
/**
 * Virtualization type for Amazon Linux
 */
var AmazonLinuxVirt;
(function (AmazonLinuxVirt) {
    /**
     * HVM virtualization (recommended)
     */
    AmazonLinuxVirt["HVM"] = "hvm";
    /**
     * PV virtualization
     */
    AmazonLinuxVirt["PV"] = "pv";
})(AmazonLinuxVirt = exports.AmazonLinuxVirt || (exports.AmazonLinuxVirt = {}));
var AmazonLinuxStorage;
(function (AmazonLinuxStorage) {
    /**
     * EBS-backed storage
     */
    AmazonLinuxStorage["EBS"] = "ebs";
    /**
     * S3-backed storage
     */
    AmazonLinuxStorage["S3"] = "ebs";
    /**
     * General Purpose-based storage (recommended)
     */
    AmazonLinuxStorage["GENERAL_PURPOSE"] = "gp2";
})(AmazonLinuxStorage = exports.AmazonLinuxStorage || (exports.AmazonLinuxStorage = {}));
/**
 * Construct a Linux machine image from an AMI map
 *
 * Linux images IDs are not published to SSM parameter store yet, so you'll have to
 * manually specify an AMI map.
 */
class GenericLinuxImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forLinux(),
            osType: OperatingSystemType.LINUX,
        };
    }
}
exports.GenericLinuxImage = GenericLinuxImage;
/**
 * Construct a Windows machine image from an AMI map
 *
 * Allows you to create a generic Windows EC2 , manually specify an AMI map.
 */
class GenericWindowsImage {
    constructor(amiMap, props = {}) {
        this.amiMap = amiMap;
        this.props = props;
    }
    getImage(scope) {
        var _a;
        const region = core_1.Stack.of(scope).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Unable to determine AMI from AMI map since stack is region-agnostic');
        }
        const ami = region !== 'test-region' ? this.amiMap[region] : 'ami-12345';
        if (!ami) {
            throw new Error(`Unable to find AMI in AMI map: no AMI specified for region '${region}'`);
        }
        return {
            imageId: ami,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forWindows(),
            osType: OperatingSystemType.WINDOWS,
        };
    }
}
exports.GenericWindowsImage = GenericWindowsImage;
/**
 * The OS type of a particular image
 */
var OperatingSystemType;
(function (OperatingSystemType) {
    OperatingSystemType[OperatingSystemType["LINUX"] = 0] = "LINUX";
    OperatingSystemType[OperatingSystemType["WINDOWS"] = 1] = "WINDOWS";
    /**
     * Used when the type of the operating system is not known
     * (for example, for imported Auto-Scaling Groups).
     */
    OperatingSystemType[OperatingSystemType["UNKNOWN"] = 2] = "UNKNOWN";
})(OperatingSystemType = exports.OperatingSystemType || (exports.OperatingSystemType = {}));
/**
 * A machine image whose AMI ID will be searched using DescribeImages.
 *
 * The most recent, available, launchable image matching the given filter
 * criteria will be used. Looking up AMIs may take a long time; specify
 * as many filter criteria as possible to narrow down the search.
 *
 * The AMI selected will be cached in `cdk.context.json` and the same value
 * will be used on future runs. To refresh the AMI lookup, you will have to
 * evict the value from the cache using the `cdk context` command. See
 * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
 */
class LookupMachineImage {
    constructor(props) {
        this.props = props;
    }
    getImage(scope) {
        var _a;
        // Need to know 'windows' or not before doing the query to return the right
        // osType for the dummy value, so might as well add it to the filter.
        const filters = {
            'name': [this.props.name],
            'state': ['available'],
            'image-type': ['machine'],
            'platform': this.props.windows ? ['windows'] : undefined,
        };
        Object.assign(filters, this.props.filters);
        const value = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.AMI_PROVIDER,
            props: {
                owners: this.props.owners,
                filters,
            },
            dummyValue: 'ami-1234',
        }).value;
        if (typeof value !== 'string') {
            throw new Error(`Response to AMI lookup invalid, got: ${value}`);
        }
        const osType = this.props.windows ? OperatingSystemType.WINDOWS : OperatingSystemType.LINUX;
        return {
            imageId: value,
            osType,
            userData: (_a = this.props.userData) !== null && _a !== void 0 ? _a : user_data_1.UserData.forOperatingSystem(osType),
        };
    }
}
exports.LookupMachineImage = LookupMachineImage;
//# sourceMappingURL=data:application/json;base64,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