"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateNamespaceName = exports.validateArtifactName = exports.validateName = exports.validateSourceAction = exports.validateArtifactBounds = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("./action");
/**
 * Validation function that checks if the number of artifacts is within the given bounds
 */
function validateArtifactBounds(type, artifacts, min, max, category, provider) {
    const ret = [];
    if (artifacts.length < min) {
        ret.push(`${category}/${provider} must have at least ${min} ${type} artifacts`);
    }
    if (artifacts.length > max) {
        ret.push(`${category}/${provider} cannot have more than ${max} ${type} artifacts`);
    }
    return ret;
}
exports.validateArtifactBounds = validateArtifactBounds;
/**
 * Validation function that guarantees that an action is or is not a source action. This is useful because Source actions can only be
 * in the first stage of a pipeline, and the first stage can only contain source actions.
 */
function validateSourceAction(mustBeSource, category, actionName, stageName) {
    if (mustBeSource !== (category === action_1.ActionCategory.SOURCE)) {
        return [`Action ${actionName} in stage ${stageName}: ` + (mustBeSource ? 'first stage may only contain Source actions'
                : 'Source actions may only occur in first stage')];
    }
    return [];
}
exports.validateSourceAction = validateSourceAction;
/**
 * Regex to validate Pipeline, Stage, Action names
 *
 * https://docs.aws.amazon.com/codepipeline/latest/userguide/limits.html
 */
const VALID_IDENTIFIER_REGEX = /^[a-zA-Z0-9.@_-]{1,100}$/;
/**
 * Validate the given name of a pipeline component. Pipeline component names all have the same restrictions.
 * This can be used to validate the name of all components of a pipeline.
 */
function validateName(thing, name) {
    validateAgainstRegex(VALID_IDENTIFIER_REGEX, thing, name);
}
exports.validateName = validateName;
function validateArtifactName(artifactName) {
    // https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_Artifact.html#CodePipeline-Type-Artifact-name
    validateAgainstRegex(/^[a-zA-Z0-9_-]{1,100}$/, 'Artifact', artifactName);
}
exports.validateArtifactName = validateArtifactName;
function validateNamespaceName(namespaceName) {
    validateAgainstRegex(/^[A-Za-z0-9@_-]{1,100}$/, 'Namespace', namespaceName);
}
exports.validateNamespaceName = validateNamespaceName;
function validateAgainstRegex(regex, thing, name) {
    // name could be a Token - in that case, skip validation altogether
    if (cdk.Token.isUnresolved(name)) {
        return;
    }
    if (name !== undefined && !regex.test(name)) {
        throw new Error(`${thing} name must match regular expression: ${regex.toString()}, got '${name}'`);
    }
}
//# sourceMappingURL=data:application/json;base64,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