"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const validation = require("./validation");
/**
 * A Stage in a Pipeline.
 *
 * Stages are added to a Pipeline by calling {@link Pipeline#addStage},
 * which returns an instance of {@link codepipeline.IStage}.
 *
 * This class is private to the CodePipeline module.
 */
class Stage {
    /**
     * Create a new Stage.
     */
    constructor(props, pipeline) {
        this._actions = new Array();
        validation.validateName('Stage', props.stageName);
        this.stageName = props.stageName;
        this._pipeline = pipeline;
        this.scope = new cdk.Construct(pipeline, this.stageName);
        for (const action of props.actions || []) {
            this.addAction(action);
        }
    }
    /**
     * Get a duplicate of this stage's list of actions.
     */
    get actionDescriptors() {
        return this._actions.slice();
    }
    get actions() {
        return this._actions.map(actionDescriptor => actionDescriptor.action);
    }
    get pipeline() {
        return this._pipeline;
    }
    render() {
        // first, assign names to output Artifacts who don't have one
        for (const action of this._actions) {
            const outputArtifacts = action.outputs;
            const unnamedOutputs = outputArtifacts.filter(o => !o.artifactName);
            for (const outputArtifact of outputArtifacts) {
                if (!outputArtifact.artifactName) {
                    const unsanitizedArtifactName = `Artifact_${this.stageName}_${action.actionName}` + (unnamedOutputs.length === 1
                        ? ''
                        : '_' + (unnamedOutputs.indexOf(outputArtifact) + 1));
                    const artifactName = sanitizeArtifactName(unsanitizedArtifactName);
                    outputArtifact._setName(artifactName);
                }
            }
        }
        return {
            name: this.stageName,
            actions: this._actions.map(action => this.renderAction(action)),
        };
    }
    addAction(action) {
        const actionName = action.actionProperties.actionName;
        // validate the name
        validation.validateName('Action', actionName);
        // check for duplicate Actions and names
        if (this._actions.find(a => a.actionName === actionName)) {
            throw new Error(`Stage ${this.stageName} already contains an action with name '${actionName}'`);
        }
        this._actions.push(this.attachActionToPipeline(action));
    }
    onStateChange(name, target, options) {
        const rule = new events.Rule(this.scope, name, options);
        rule.addTarget(target);
        rule.addEventPattern({
            detailType: ['CodePipeline Stage Execution State Change'],
            source: ['aws.codepipeline'],
            resources: [this.pipeline.pipelineArn],
            detail: {
                stage: [this.stageName],
            },
        });
        return rule;
    }
    validate() {
        return [
            ...this.validateHasActions(),
            ...this.validateActions(),
        ];
    }
    validateHasActions() {
        if (this._actions.length === 0) {
            return [`Stage '${this.stageName}' must have at least one action`];
        }
        return [];
    }
    validateActions() {
        const ret = new Array();
        for (const action of this.actionDescriptors) {
            ret.push(...this.validateAction(action));
        }
        return ret;
    }
    validateAction(action) {
        return validation.validateArtifactBounds('input', action.inputs, action.artifactBounds.minInputs, action.artifactBounds.maxInputs, action.category, action.provider)
            .concat(validation.validateArtifactBounds('output', action.outputs, action.artifactBounds.minOutputs, action.artifactBounds.maxOutputs, action.category, action.provider));
    }
    attachActionToPipeline(action) {
        // notify the Pipeline of the new Action
        const actionScope = new cdk.Construct(this.scope, action.actionProperties.actionName);
        return this._pipeline._attachActionToPipeline(this, action, actionScope);
    }
    renderAction(action) {
        const outputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.outputs) }, { omitEmptyArray: true });
        const inputArtifacts = cdk.Lazy.anyValue({ produce: () => this.renderArtifacts(action.inputs) }, { omitEmptyArray: true });
        return {
            name: action.actionName,
            inputArtifacts,
            outputArtifacts,
            actionTypeId: {
                category: action.category.toString(),
                version: action.version,
                owner: action.owner,
                provider: action.provider,
            },
            configuration: action.configuration,
            runOrder: action.runOrder,
            roleArn: action.role ? action.role.roleArn : undefined,
            region: action.region,
            namespace: action.namespace,
        };
    }
    renderArtifacts(artifacts) {
        return artifacts
            .filter(a => a.artifactName)
            .map(a => ({ name: a.artifactName }));
    }
}
exports.Stage = Stage;
function sanitizeArtifactName(artifactName) {
    // strip out some characters that are legal in Stage and Action names,
    // but not in Artifact names
    return artifactName.replace(/[@.]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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