"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossRegionSupportStack = exports.CrossRegionSupportConstruct = void 0;
const kms = require("../../aws-kms"); // Automatically re-written from '@aws-cdk/aws-kms'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const REQUIRED_ALIAS_PREFIX = 'alias/';
/**
 * A class needed to work around CodePipeline's extremely small (100 characters)
 * limit for the name/ARN of the key in the ArtifactStore.
 * Limits the length of the alias' auto-generated name to 50 characters.
 */
class AliasWithShorterGeneratedName extends kms.Alias {
    generatePhysicalName() {
        let baseName = super.generatePhysicalName();
        if (baseName.startsWith(REQUIRED_ALIAS_PREFIX)) {
            // remove the prefix, because we're taking the last characters of the name below
            baseName = baseName.substring(REQUIRED_ALIAS_PREFIX.length);
        }
        const maxLength = 50 - REQUIRED_ALIAS_PREFIX.length;
        // take the last characters, as they include the hash,
        // and so have a higher chance of not colliding
        return REQUIRED_ALIAS_PREFIX + lastNCharacters(baseName, maxLength);
    }
}
function lastNCharacters(str, n) {
    const startIndex = Math.max(str.length - n, 0);
    return str.substring(startIndex);
}
class CrossRegionSupportConstruct extends cdk.Construct {
    constructor(scope, id) {
        super(scope, id);
        const encryptionKey = new kms.Key(this, 'CrossRegionCodePipelineReplicationBucketEncryptionKey', {
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        const encryptionAlias = new AliasWithShorterGeneratedName(this, 'CrossRegionCodePipelineReplicationBucketEncryptionAlias', {
            targetKey: encryptionKey,
            aliasName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        this.replicationBucket = new s3.Bucket(this, 'CrossRegionCodePipelineReplicationBucket', {
            bucketName: cdk.PhysicalName.GENERATE_IF_NEEDED,
            encryptionKey: encryptionAlias,
        });
    }
}
exports.CrossRegionSupportConstruct = CrossRegionSupportConstruct;
/**
 * A Stack containing resources required for the cross-region CodePipeline functionality to work.
 * This class is private to the aws-codepipeline package.
 */
class CrossRegionSupportStack extends cdk.Stack {
    constructor(scope, id, props) {
        super(scope, id, {
            stackName: generateStackName(props),
            env: {
                region: props.region,
                account: props.account,
            },
            synthesizer: props.synthesizer,
        });
        const crossRegionSupportConstruct = new CrossRegionSupportConstruct(this, 'Default');
        this.replicationBucket = crossRegionSupportConstruct.replicationBucket;
    }
}
exports.CrossRegionSupportStack = CrossRegionSupportStack;
function generateStackName(props) {
    return `${props.pipelineStackName}-support-${props.region}`;
}
//# sourceMappingURL=data:application/json;base64,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