"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EcrSourceAction = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const targets = require("../../../aws-events-targets"); // Automatically re-written from '@aws-cdk/aws-events-targets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * The ECR Repository source CodePipeline Action.
 *
 * Will trigger the pipeline as soon as the target tag in the repository
 * changes, but only if there is a CloudTrail Trail in the account that
 * captures the ECR event.
 */
class EcrSourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'ECR',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            registryId: this.variableExpression('RegistryId'),
            repositoryName: this.variableExpression('RepositoryName'),
            imageDigest: this.variableExpression('ImageDigest'),
            imageTag: this.variableExpression('ImageTag'),
            imageUri: this.variableExpression('ImageURI'),
        };
    }
    bound(_scope, stage, options) {
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['ecr:DescribeImages'],
            resources: [this.props.repository.repositoryArn],
        }));
        this.props.repository.onCloudTrailImagePushed(stage.pipeline.node.uniqueId + 'SourceEventRule', {
            target: new targets.CodePipeline(stage.pipeline),
            imageTag: this.props.imageTag,
        });
        // the Action Role also needs to write to the Pipeline's bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                ImageTag: this.props.imageTag,
            },
        };
    }
}
exports.EcrSourceAction = EcrSourceAction;
//# sourceMappingURL=data:application/json;base64,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