"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFormationDeleteStackAction = exports.CloudFormationCreateUpdateStackAction = exports.CloudFormationCreateReplaceChangeSetAction = exports.CloudFormationExecuteChangeSetAction = void 0;
const cloudformation = require("../../../aws-cloudformation"); // Automatically re-written from '@aws-cdk/aws-cloudformation'
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
/**
 * Base class for Actions that execute CloudFormation
 */
class CloudFormationAction extends action_1.Action {
    constructor(props, inputs) {
        super({
            ...props,
            provider: 'CloudFormation',
            category: codepipeline.ActionCategory.DEPLOY,
            artifactBounds: {
                minInputs: 0,
                maxInputs: 10,
                minOutputs: 0,
                maxOutputs: 1,
            },
            inputs,
            outputs: props.outputFileName
                ? [props.output || new codepipeline.Artifact(`${props.actionName}_${props.stackName}_Artifact`)]
                : undefined,
        });
        this.props = props;
    }
    bound(_scope, _stage, options) {
        const singletonPolicy = SingletonPolicy.forRole(options.role);
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantReadWrite(singletonPolicy);
        }
        else if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(singletonPolicy);
        }
        return {
            configuration: {
                StackName: this.props.stackName,
                OutputFileName: this.props.outputFileName,
            },
        };
    }
}
/**
 * CodePipeline action to execute a prepared change set.
 */
class CloudFormationExecuteChangeSetAction extends CloudFormationAction {
    constructor(props) {
        super(props, undefined);
        this.props2 = props;
    }
    bound(scope, stage, options) {
        SingletonPolicy.forRole(options.role).grantExecuteChangeSet(this.props2);
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_EXECUTE',
                ChangeSetName: this.props2.changeSetName,
            },
        };
    }
}
exports.CloudFormationExecuteChangeSetAction = CloudFormationExecuteChangeSetAction;
/**
 * Base class for all CloudFormation actions that execute or stage deployments.
 */
class CloudFormationDeployAction extends CloudFormationAction {
    constructor(props, inputs) {
        super(props, (props.extraInputs || []).concat(inputs || []));
        this.props2 = props;
    }
    /**
     * Add statement to the service role assumed by CloudFormation while executing this action.
     */
    addToDeploymentRolePolicy(statement) {
        return this.getDeploymentRole('method addToRolePolicy()').addToPolicy(statement);
    }
    get deploymentRole() {
        return this.getDeploymentRole('property role()');
    }
    bound(scope, stage, options) {
        if (this.props2.deploymentRole) {
            this._deploymentRole = this.props2.deploymentRole;
        }
        else {
            const roleStack = cdk.Stack.of(options.role);
            const pipelineStack = cdk.Stack.of(scope);
            if (roleStack.account !== pipelineStack.account) {
                // pass role is not allowed for cross-account access - so,
                // create the deployment Role in the other account!
                this._deploymentRole = new iam.Role(roleStack, `${stage.pipeline.node.uniqueId}-${stage.stageName}-${this.actionProperties.actionName}-DeploymentRole`, {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                    roleName: cdk.PhysicalName.GENERATE_IF_NEEDED,
                });
            }
            else {
                this._deploymentRole = new iam.Role(scope, 'Role', {
                    assumedBy: new iam.ServicePrincipal('cloudformation.amazonaws.com'),
                });
            }
            // the deployment role might need read access to the pipeline's bucket
            // (for example, if it's deploying a Lambda function),
            // and even if it has admin permissions, it won't be enough,
            // as it needs to be added to the key's resource policy
            // (and the bucket's, if the access is cross-account)
            options.bucket.grantRead(this._deploymentRole);
            if (this.props2.adminPermissions) {
                this._deploymentRole.addToPolicy(new iam.PolicyStatement({
                    actions: ['*'],
                    resources: ['*'],
                }));
            }
        }
        SingletonPolicy.forRole(options.role).grantPassRole(this._deploymentRole);
        const capabilities = this.props2.adminPermissions && this.props2.capabilities === undefined
            ? [cloudformation.CloudFormationCapabilities.NAMED_IAM]
            : this.props2.capabilities;
        const actionConfig = super.bound(scope, stage, options);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                // None evaluates to empty string which is falsey and results in undefined
                Capabilities: parseCapabilities(capabilities),
                RoleArn: this.deploymentRole.roleArn,
                ParameterOverrides: cdk.Stack.of(scope).toJsonString(this.props2.parameterOverrides),
                TemplateConfiguration: this.props2.templateConfiguration
                    ? this.props2.templateConfiguration.location
                    : undefined,
                StackName: this.props2.stackName,
            },
        };
    }
    getDeploymentRole(member) {
        if (this._deploymentRole) {
            return this._deploymentRole;
        }
        else {
            throw new Error(`Cannot use the ${member} before the Action has been added to a Pipeline`);
        }
    }
}
/**
 * CodePipeline action to prepare a change set.
 *
 * Creates the change set if it doesn't exist based on the stack name and template that you submit.
 * If the change set exists, AWS CloudFormation deletes it, and then creates a new one.
 */
class CloudFormationCreateReplaceChangeSetAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateReplaceChangeSet(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'CHANGE_SET_REPLACE',
                ChangeSetName: this.props3.changeSetName,
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateReplaceChangeSetAction = CloudFormationCreateReplaceChangeSetAction;
/**
 * CodePipeline action to deploy a stack.
 *
 * Creates the stack if the specified stack doesn't exist. If the stack exists,
 * AWS CloudFormation updates the stack. Use this action to update existing
 * stacks.
 *
 * AWS CodePipeline won't replace the stack, and will fail deployment if the
 * stack is in a failed state. Use `ReplaceOnFailure` for an action that
 * will delete and recreate the stack to try and recover from failed states.
 *
 * Use this action to automatically replace failed stacks without recovering or
 * troubleshooting them. You would typically choose this mode for testing.
 */
class CloudFormationCreateUpdateStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, props.templateConfiguration
            ? [props.templatePath.artifact, props.templateConfiguration.artifact]
            : [props.templatePath.artifact]);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantCreateUpdateStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: this.props3.replaceOnFailure ? 'REPLACE_ON_FAILURE' : 'CREATE_UPDATE',
                TemplatePath: this.props3.templatePath.location,
            },
        };
    }
}
exports.CloudFormationCreateUpdateStackAction = CloudFormationCreateUpdateStackAction;
/**
 * CodePipeline action to delete a stack.
 *
 * Deletes a stack. If you specify a stack that doesn't exist, the action completes successfully
 * without deleting a stack.
 */
class CloudFormationDeleteStackAction extends CloudFormationDeployAction {
    constructor(props) {
        super(props, undefined);
        this.props3 = props;
    }
    bound(scope, stage, options) {
        // the super call order is to preserve the existing order of statements in policies
        const actionConfig = super.bound(scope, stage, options);
        SingletonPolicy.forRole(options.role).grantDeleteStack(this.props3);
        return {
            ...actionConfig,
            configuration: {
                ...actionConfig.configuration,
                ActionMode: 'DELETE_ONLY',
            },
        };
    }
}
exports.CloudFormationDeleteStackAction = CloudFormationDeleteStackAction;
/**
 * Manages a bunch of singleton-y statements on the policy of an IAM Role.
 * Dedicated methods can be used to add specific permissions to the role policy
 * using as few statements as possible (adding resources to existing compatible
 * statements instead of adding new statements whenever possible).
 *
 * Statements created outside of this class are not considered when adding new
 * permissions.
 */
class SingletonPolicy extends cdk.Construct {
    constructor(role) {
        super(role, SingletonPolicy.UUID);
        this.role = role;
        this.statements = {};
        this.grantPrincipal = role;
    }
    /**
     * Obtain a SingletonPolicy for a given role.
     * @param role the Role this policy is bound to.
     * @returns the SingletonPolicy for this role.
     */
    static forRole(role) {
        const found = role.node.tryFindChild(SingletonPolicy.UUID);
        return found || new SingletonPolicy(role);
    }
    grantExecuteChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStacks',
                'cloudformation:DescribeChangeSet',
                'cloudformation:ExecuteChangeSet',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateReplaceChangeSet(props) {
        this.statementFor({
            actions: [
                'cloudformation:CreateChangeSet',
                'cloudformation:DeleteChangeSet',
                'cloudformation:DescribeChangeSet',
                'cloudformation:DescribeStacks',
            ],
            conditions: { StringEqualsIfExists: { 'cloudformation:ChangeSetName': props.changeSetName } },
        }).addResources(this.stackArnFromProps(props));
    }
    grantCreateUpdateStack(props) {
        const actions = [
            'cloudformation:DescribeStack*',
            'cloudformation:CreateStack',
            'cloudformation:UpdateStack',
            'cloudformation:GetTemplate*',
            'cloudformation:ValidateTemplate',
            'cloudformation:GetStackPolicy',
            'cloudformation:SetStackPolicy',
        ];
        if (props.replaceOnFailure) {
            actions.push('cloudformation:DeleteStack');
        }
        this.statementFor({ actions }).addResources(this.stackArnFromProps(props));
    }
    grantDeleteStack(props) {
        this.statementFor({
            actions: [
                'cloudformation:DescribeStack*',
                'cloudformation:DeleteStack',
            ],
        }).addResources(this.stackArnFromProps(props));
    }
    grantPassRole(role) {
        this.statementFor({ actions: ['iam:PassRole'] }).addResources(role.roleArn);
    }
    statementFor(template) {
        const key = keyFor(template);
        if (!(key in this.statements)) {
            this.statements[key] = new iam.PolicyStatement({ actions: template.actions });
            if (template.conditions) {
                this.statements[key].addConditions(template.conditions);
            }
            this.role.addToPolicy(this.statements[key]);
        }
        return this.statements[key];
        function keyFor(props) {
            const actions = `${props.actions.sort().join('\x1F')}`;
            const conditions = formatConditions(props.conditions);
            return `${actions}\x1D${conditions}`;
            function formatConditions(cond) {
                if (cond == null) {
                    return '';
                }
                let result = '';
                for (const op of Object.keys(cond).sort()) {
                    result += `${op}\x1E`;
                    const condition = cond[op];
                    for (const attribute of Object.keys(condition).sort()) {
                        const value = condition[attribute];
                        result += `${value}\x1F`;
                    }
                }
                return result;
            }
        }
    }
    stackArnFromProps(props) {
        return cdk.Stack.of(this).formatArn({
            region: props.region,
            service: 'cloudformation',
            resource: 'stack',
            resourceName: `${props.stackName}/*`,
        });
    }
}
SingletonPolicy.UUID = '8389e75f-0810-4838-bf64-d6f85a95cf83';
function parseCapabilities(capabilities) {
    if (capabilities === undefined) {
        return undefined;
    }
    else if (capabilities.length === 1) {
        const capability = capabilities.toString();
        return (capability === '') ? undefined : capability;
    }
    else if (capabilities.length > 1) {
        return capabilities.join(',');
    }
    return undefined;
}
//# sourceMappingURL=data:application/json;base64,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