"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FilterGroup = exports.EventAction = exports.Source = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const source_types_1 = require("./source-types");
/**
 * Source provider definition for a CodeBuild Project.
 */
class Source {
    constructor(props) {
        this.badgeSupported = false;
        this.identifier = props.identifier;
    }
    static s3(props) {
        return new S3Source(props);
    }
    static codeCommit(props) {
        return new CodeCommitSource(props);
    }
    static gitHub(props) {
        return new GitHubSource(props);
    }
    static gitHubEnterprise(props) {
        return new GitHubEnterpriseSource(props);
    }
    static bitBucket(props) {
        return new BitBucketSource(props);
    }
    /**
     * Called by the project when the source is added so that the source can perform
     * binding operations on the source. For example, it can grant permissions to the
     * code build project to read from the S3 bucket.
     */
    bind(_scope, _project) {
        return {
            sourceProperty: {
                sourceIdentifier: this.identifier,
                type: this.type,
            },
        };
    }
}
exports.Source = Source;
/**
 * A common superclass of all build sources that are backed by Git.
 */
class GitSource extends Source {
    constructor(props) {
        super(props);
        this.cloneDepth = props.cloneDepth;
        this.branchOrRef = props.branchOrRef;
        this.fetchSubmodules = props.fetchSubmodules;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceVersion: this.branchOrRef,
            sourceProperty: {
                ...superConfig.sourceProperty,
                gitCloneDepth: this.cloneDepth,
                gitSubmodulesConfig: this.fetchSubmodules ? {
                    fetchSubmodules: this.fetchSubmodules,
                } : undefined,
            },
        };
    }
}
/**
 * The types of webhook event actions.
 */
var EventAction;
(function (EventAction) {
    /**
     * A push (of a branch, or a tag) to the repository.
     */
    EventAction["PUSH"] = "PUSH";
    /**
     * Creating a Pull Request.
     */
    EventAction["PULL_REQUEST_CREATED"] = "PULL_REQUEST_CREATED";
    /**
     * Updating a Pull Request.
     */
    EventAction["PULL_REQUEST_UPDATED"] = "PULL_REQUEST_UPDATED";
    /**
     * Merging a Pull Request.
     */
    EventAction["PULL_REQUEST_MERGED"] = "PULL_REQUEST_MERGED";
    /**
     * Re-opening a previously closed Pull Request.
     * Note that this event is only supported for GitHub and GitHubEnterprise sources.
     */
    EventAction["PULL_REQUEST_REOPENED"] = "PULL_REQUEST_REOPENED";
})(EventAction = exports.EventAction || (exports.EventAction = {}));
const FILE_PATH_WEBHOOK_COND = 'FILE_PATH';
/**
 * An object that represents a group of filter conditions for a webhook.
 * Every condition in a given FilterGroup must be true in order for the whole group to be true.
 * You construct instances of it by calling the {@link #inEventOf} static factory method,
 * and then calling various `andXyz` instance methods to create modified instances of it
 * (this class is immutable).
 *
 * You pass instances of this class to the `webhookFilters` property when constructing a source.
 */
class FilterGroup {
    constructor(actions, filters) {
        if (actions.size === 0) {
            throw new Error('A filter group must contain at least one event action');
        }
        this.actions = actions;
        this.filters = filters;
    }
    /**
     * Creates a new event FilterGroup that triggers on any of the provided actions.
     *
     * @param actions the actions to trigger the webhook on
     */
    static inEventOf(...actions) {
        return new FilterGroup(new Set(actions), []);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIs(branchName) {
        return this.addHeadBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given branch.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBranchIsNot(branchName) {
        return this.addHeadBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIs(tagName) {
        return this.addHeadTagFilter(tagName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect the given tag.
     *
     * @param tagName the name of the tag (can be a regular expression)
     */
    andTagIsNot(tagName) {
        return this.addHeadTagFilter(tagName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIs(pattern) {
        return this.addHeadRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the event must not affect a Git reference (ie., a branch or a tag)
     * that matches the given pattern.
     *
     * @param pattern a regular expression
     */
    andHeadRefIsNot(pattern) {
        return this.addHeadRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIs(pattern) {
        return this.addActorAccountId(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the account ID of the actor initiating the event must not match the given pattern.
     *
     * @param pattern a regular expression
     */
    andActorAccountIsNot(pattern) {
        return this.addActorAccountId(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIs(branchName) {
        return this.addBaseBranchFilter(branchName, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given base branch.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param branchName the name of the branch (can be a regular expression)
     */
    andBaseBranchIsNot(branchName) {
        return this.addBaseBranchFilter(branchName, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIs(pattern) {
        return this.addBaseRefFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the Pull Request that is the source of the event must not target the given Git reference.
     * Note that you cannot use this method if this Group contains the `PUSH` event action.
     *
     * @param pattern a regular expression
     */
    andBaseRefIsNot(pattern) {
        return this.addBaseRefFilter(pattern, false);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIs(pattern) {
        return this.addFilePathFilter(pattern, true);
    }
    /**
     * Create a new FilterGroup with an added condition:
     * the push that is the source of the event must not affect a file that matches the given pattern.
     * Note that you can only use this method if this Group contains only the `PUSH` event action,
     * and only for GitHub and GitHubEnterprise sources.
     *
     * @param pattern a regular expression
     */
    andFilePathIsNot(pattern) {
        return this.addFilePathFilter(pattern, false);
    }
    /** @internal */
    get _actions() {
        return set2Array(this.actions);
    }
    /** @internal */
    get _filters() {
        return this.filters.slice();
    }
    /** @internal */
    _toJson() {
        const eventFilter = {
            type: 'EVENT',
            pattern: set2Array(this.actions).join(', '),
        };
        return [eventFilter].concat(this.filters);
    }
    addHeadBranchFilter(branchName, include) {
        return this.addHeadRefFilter(`refs/heads/${branchName}`, include);
    }
    addHeadTagFilter(tagName, include) {
        return this.addHeadRefFilter(`refs/tags/${tagName}`, include);
    }
    addHeadRefFilter(refName, include) {
        return this.addFilter('HEAD_REF', refName, include);
    }
    addActorAccountId(accountId, include) {
        return this.addFilter('ACTOR_ACCOUNT_ID', accountId, include);
    }
    addBaseBranchFilter(branchName, include) {
        return this.addBaseRefFilter(`refs/heads/${branchName}`, include);
    }
    addBaseRefFilter(refName, include) {
        if (this.actions.has(EventAction.PUSH)) {
            throw new Error('A base reference condition cannot be added if a Group contains a PUSH event action');
        }
        return this.addFilter('BASE_REF', refName, include);
    }
    addFilePathFilter(pattern, include) {
        return this.addFilter(FILE_PATH_WEBHOOK_COND, pattern, include);
    }
    addFilter(type, pattern, include) {
        return new FilterGroup(this.actions, this.filters.concat([{
                type,
                pattern,
                excludeMatchedPattern: include ? undefined : true,
            }]));
    }
}
exports.FilterGroup = FilterGroup;
/**
 * A common superclass of all third-party build sources that are backed by Git.
 */
class ThirdPartyGitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.webhook = props.webhook;
        this.reportBuildStatus = props.reportBuildStatus === undefined ? true : props.reportBuildStatus;
        this.webhookFilters = props.webhookFilters || [];
    }
    bind(_scope, _project) {
        const anyFilterGroupsProvided = this.webhookFilters.length > 0;
        const webhook = this.webhook === undefined ? (anyFilterGroupsProvided ? true : undefined) : this.webhook;
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                reportBuildStatus: this.reportBuildStatus,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: webhook === undefined ? undefined : {
                webhook,
                filterGroups: anyFilterGroupsProvided ? this.webhookFilters.map(fg => fg._toJson()) : undefined,
            },
        };
    }
}
/**
 * CodeCommit Source definition for a CodeBuild project.
 */
class CodeCommitSource extends GitSource {
    constructor(props) {
        super(props);
        this.badgeSupported = true;
        this.type = source_types_1.CODECOMMIT_SOURCE_TYPE;
        this.repo = props.repository;
    }
    bind(_scope, project) {
        // https://docs.aws.amazon.com/codebuild/latest/userguide/setting-up.html
        project.addToRolePolicy(new iam.PolicyStatement({
            actions: ['codecommit:GitPull'],
            resources: [this.repo.repositoryArn],
        }));
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.repo.repositoryCloneUrlHttp,
            },
            sourceVersion: superConfig.sourceVersion,
        };
    }
}
/**
 * S3 bucket definition for a CodeBuild project.
 */
class S3Source extends Source {
    constructor(props) {
        super(props);
        this.type = source_types_1.S3_SOURCE_TYPE;
        this.bucket = props.bucket;
        this.path = props.path;
        this.version = props.version;
    }
    bind(_scope, project) {
        this.bucket.grantRead(project);
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: `${this.bucket.bucketName}/${this.path}`,
            },
            sourceVersion: this.version,
        };
    }
}
/**
 * GitHub Source definition for a CodeBuild project.
 */
class GitHubSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_SOURCE_TYPE;
        this.httpsCloneUrl = `https://github.com/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, project) {
        const superConfig = super.bind(_scope, project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * GitHub Enterprise Source definition for a CodeBuild project.
 */
class GitHubEnterpriseSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.GITHUB_ENTERPRISE_SOURCE_TYPE;
        this.httpsCloneUrl = props.httpsCloneUrl;
        this.ignoreSslErrors = props.ignoreSslErrors;
    }
    bind(_scope, _project) {
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
                insecureSsl: this.ignoreSslErrors,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
}
/**
 * BitBucket Source definition for a CodeBuild project.
 */
class BitBucketSource extends ThirdPartyGitSource {
    constructor(props) {
        super(props);
        this.type = source_types_1.BITBUCKET_SOURCE_TYPE;
        this.httpsCloneUrl = `https://bitbucket.org/${props.owner}/${props.repo}.git`;
    }
    bind(_scope, _project) {
        // BitBucket sources don't support the PULL_REQUEST_REOPENED event action
        if (this.anyWebhookFilterContainsPrReopenedEventAction()) {
            throw new Error('BitBucket sources do not support the PULL_REQUEST_REOPENED webhook event action');
        }
        // they also don't support file path conditions
        if (this.anyWebhookFilterContainsFilePathConditions()) {
            throw new Error('BitBucket sources do not support file path conditions for webhook filters');
        }
        const superConfig = super.bind(_scope, _project);
        return {
            sourceProperty: {
                ...superConfig.sourceProperty,
                location: this.httpsCloneUrl,
            },
            sourceVersion: superConfig.sourceVersion,
            buildTriggers: superConfig.buildTriggers,
        };
    }
    anyWebhookFilterContainsPrReopenedEventAction() {
        return this.webhookFilters.findIndex(fg => {
            return fg._actions.findIndex(a => a === EventAction.PULL_REQUEST_REOPENED) !== -1;
        }) !== -1;
    }
    anyWebhookFilterContainsFilePathConditions() {
        return this.webhookFilters.findIndex(fg => {
            return fg._filters.findIndex(f => f.type === FILE_PATH_WEBHOOK_COND) !== -1;
        }) !== -1;
    }
}
function set2Array(set) {
    const ret = [];
    set.forEach(el => ret.push(el));
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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